#! /usr/local/bin/perl -U

# Copyright (c) 2004-2005 The Regents of the University of California.
# All Rights Reserved.
# 
# Permission to use, copy, modify, and distribute this software and its
# documentation for educational, research and non-profit purposes,
# without fee, and without a written agreement is hereby granted,
# provided that the above copyright notice, this paragraph and the
# following three paragraphs appear in all copies.
# 
# Permission to incorporate this software into commercial products may
# be obtained from the Office of Technology Licensing, 2150 Shattuck
# Avenue, Suite 510, Berkeley, CA  94704.
# 
# IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
# FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
# INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND
# ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# 
# THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE
# PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
# CALIFORNIA HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,
# UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

# $Id: bsdata.pl,v 1.36 2009/01/13 18:01:57 doug Exp $ 

require 5.002;
use strict;
use FileHandle;
use Getopt::Std;
use Sys::Hostname;
use File::Basename;
use File::Path;
use IO::File;
use IO::Seekable;
use IO::Handle;
use DBI;
use Env qw( ORACLE_HOME NLS_LANG NLS_DATE_FORMAT );

use Qtime;

use vars qw ( $VERSION $cmdname $opt_h $opt_f $opt_t $opt_s $opt_p $opt_D $opt_d $opt_l $opt_k );
use vars qw ( $opt_S $opt_N $opt_C $opt_L $opt_m $opt_v $opt_K $opt_A );
use vars qw ( $opt_i $opt_o $opt_w $opt_b $opt_I $opt_E $opt_W $opt_F $opt_c );
use vars qw ( $debug $DEBUG_POD $DEBUG_CMD $DEBUG_KEEP_QUERY $DEBUG_KEEP_WAVEFORMS $DEBUG_SQL $DEBUG_WAVEFORMS );
use vars qw ( $dbuser $dbpass $dbname $oracle_home $dbh );
use vars qw ( $Waveform_File $Dflt_Blockette_Dir $Day_Window $Month_Window %MaxWaveLen);
use vars qw ( $WF_Window_default $pod_req $miniseed $dataless $verbose );
use vars qw ( $BaseDir $stagedir );
use vars qw ( $qmerge $Make_Blockettes $Pod );
use vars qw ( $qtmax $epsilon_usecs );
use vars qw ( $STAGE_FILE $STAGE_SEGMENT $stage_alg );
use vars qw ( $QUERY_DATALESS $QUERY_SNCL_LIST $QUERY_NOFILE );
$VERSION = "1.36 (2009.013)";

######################################################################
#	print_syntax	- print syntax and exit.
######################################################################
sub print_syntax {
    my ($cmdname) = @_;
    print
"$cmdname version $VERSION
$cmdname - Build SEED (or MiniSEED) file from data archive.
Syntax:
$cmdname [-h] [-f date -t date | -s interval] [-E eventid_list] 
    [-i request_file] [-F netdc|breq_fast|bf]
    [-S station] [-N network] [-C channel] [-L location]
    [-p pod_req] [-D datapath] [-o output_file] [-I] [-W wavetype]
    [-w workdir] [-b podhdrdir] [-l tapelabel ] [-k dcc|dmc]
    [-A file | segment ] [-m] [-v] [-K] [-d n] [-c channelfile] 
    [sncls ...]
where:
    -h		Prints out a summary of the available options.
    -i filename	Specifies the name of an input request file.
		An input file of '-' reads from stdin.
		File format specified by the -F option.
    -F format
		Specifies the format of any input file request.
		Valid formats are:
			netdc
			breq_fast | bf
    -f date	From date - ignore data before this date.
    -t date	To date - ignore data from this date on.
		Date can be in formats:
			yyyy/mm/dd/hh:mm:ss.ffff
			yyyy/mm/dd.hh:mm:ss.ffff
			yyyy/mm/dd,hh:mm:ss.ffff
			yyyy.ddd,hh:mm:ss.ffff
			yyyy,ddd,hh:mm:ss.ffff
		    You may leave off any trailing 0 components of the time.
    -s interval	Span interval.  Alternate way of specifying end time.
		Interval can be an integer followed immediately by
		S, M, H, or d for seconds, minutes, hours, or days.
    -E evidlist	A comma-delimited list of eventids.
    -W wavetype	Waveform type.
		C = continuous archive,
		T = triggered event archive,
		A = all archives (C,T)
    -S station_list
		A comma-delimited list of station names for the specified time.
    -N network_list
		A comma-delimited list of network names for the specified time.
    -C channel_list
		A comma-delimited list of channel names for the specified time.
    -L locationlist
		A list of location names for the specified time.
    -p filename Specifies the name of an additional POD requests file.
    -D datapath	Additional explicit POD DATA_PATH entry.
    -o output_file
		Specifies the name of the output data file.
		Specify a filename of '-' to write to stdout.
    -I		Generates a Dataless SEED volume (seed output),
		or SNCL inventory list (mseed output).
    -w dir	Specifies the path of the working directory.
		If none specified, a temporary working directory
		will be created in /tmp and deleted when done.
    -b podhdrdir
		Specifies the pathname of an existing POD request
		directory structure (HAR000) to be used with POD.
		If not specified, a POD directory structure will be built.
    -l tapelabel
		Specify optional tapelabel for SEED volume.
    -k dmc|dcc	Generate sync file as dmc or dcc instead of SEED volume.
    -m 		Generate MiniSEED instead of SEED volume.
    -v		Verbose option for MiniSEED output.
    -K		Estimate data size in KBytes.
    -d n	Debug option (may be added)
			1-3 POD debug flags
			4   Print commands
			8   Keep query directory
                        16  Keep waveform directory
			1024 Print SQL command
			2048 Print waveform operations and results.
    -c channelfile
		Specify a file containing a list of SNCLs, one per line.
		SNCLs may be in the same format as shown below.
    sncls	One or more optional SNCLS.  Each SNCL can be
			stations.networks.channels.locations
		where
			stations  = comma-delimited list of stations
			networks  = comma-delimited list of networks
			channels  = comma-delimited list of channels
			locations = comma-delimited list of locations
		Any component may include UNIX wildcard characters,
		any SNCL with wildcards specified on the command line
		must be quoted to prevent command line wildcard expansion/
";
       exit(0);
}
    
{
######################################################################
# Initialization.
######################################################################

    $ENV{PATH}	= "/bin:/usr/bin:/usr/local/bin";
    require "/usr/local/lib/dbsetup.pl";
#::     $dbuser	= "netdc";
#::     $dbpass	= "dB2Usr";
#::     $dbname	= "publicdb";

    $oracle_home = "/home/oracle/product/9.2.0";
    $qmerge	= "/usr/local/bin/qmerge";
    $Pod	= "/home/netdc/Local/pod";
    $BaseDir	= "/data/dc6/bsdata";	# TEST VALUE
    $Make_Blockettes = "/home/netdc/Local/Make_Blockettes";
    $WF_Window_default = 3600*24 + 1;
    my($dcname)	= "NCEDC";	# For sync files.

    $Waveform_File = "Waveform_File";
    $Dflt_Blockette_Dir = "HAR000";
    $pod_req = "pod.req";
    my($stage) = hostname;
    $stage .= "." . "$$" . "." . time();
    $stagedir = "$BaseDir/$stage";
    

    $ENV{ORACLE_HOME} = $oracle_home if ($ENV{ORACLE_HOME} eq "");
    $ENV{NLS_LANG} = "american_america.US7ASCII";
    $ENV{NLS_DATE_FORMAT} = "yyyy/mm/dd hh24:mi:ss";

    $ENV{CREATOR} = "Northern California Earthquake Data Center, UC Berkeley";
    $ENV{SEED_LABEL} = "SEED V2.4 volume created by NCEDC";

    $QUERY_DATALESS = 1;
    $QUERY_SNCL_LIST = 2;
    $QUERY_NOFILE = 4;

    STDERR->autoflush(1);
    STDOUT->autoflush(1);
    $DEBUG_POD = 3;
    $DEBUG_CMD = 4;
    $DEBUG_KEEP_QUERY = 8;
    $DEBUG_KEEP_WAVEFORMS = 16;
    $DEBUG_SQL = 1024;
    $DEBUG_WAVEFORMS = 2048;

    $STAGE_FILE = 1;
    $STAGE_SEGMENT = 2;

    $stage_alg = $STAGE_SEGMENT;

    $qtmax = Qtime::set_time ("3000/01/01,00:00:00");
    $epsilon_usecs = 100;
			
    # Use DB_CONNECT environment variable if it is set.
    if ($ENV{DB_CONNECT} ne "") {
	($dbuser,$dbpass,$dbname) = $ENV{DB_CONNECT} =~ m|^(.+)/(.+)\@(.+)$|;
	die ("Error: invalid DB_CONNECT environment variable: $ENV{DB_CONNECT}\n")
		if ($dbuser eq "" || $dbpass eq "" || $dbname eq "");
#:: 	print ("Using:  $dbuser/$dbpass\@$dbname\n");
    }
	
######################################################################
#	Parse command line.
######################################################################
    $cmdname = basename($0);
    &getopts ('hImvKk:d:f:t:s:p:D:i:F:R:o:b:w:S:N:C:L:E:W:l:c:A:');
    &print_syntax($cmdname) if ($opt_h);
    $debug = $opt_d;
    $miniseed = $opt_m;
    $dataless = $opt_I;
    $verbose = $opt_v;
    my($start_str) = $opt_f;
    my($end_str) = $opt_t;
    my($span_str) = $opt_s;
    my($add_podreq) = $opt_p;
    my($add_datapath) = $opt_D;
    my($input_file) = $opt_i if ($opt_i ne "");
    my($input_format) = $opt_F;
    my($seedfile) = $opt_o;
    my($workdir) = $opt_w;
    my($podhdrpath) = $opt_b;
    my($channelfile) = $opt_c;
    my($estimate_size) = $opt_K;
    my(@qinfo);
    my($sync_type);
    if ($opt_A ne "") {
	$stage_alg = ($opt_A =~ /file/i) ? $STAGE_FILE :
		     ($opt_A =~ /seg/i)  ? $STAGE_SEGMENT :
		     $stage_alg;
    }
    $stage_alg = $STAGE_SEGMENT if ($miniseed);

    my(@SNCL);
    push(@SNCL,@ARGV);

    # Convert old-style channel specs into a new style channel spec.
    if ("$opt_S$opt_N$opt_C$opt_L" ne "") {
	my($sncl) = sprintf ("%s.%s.%s.%s",
		($opt_S ne "") ? $opt_S : "*",
		($opt_N ne "") ? $opt_N : "*",
		($opt_C ne "") ? $opt_C : "*",
		($opt_L ne "") ? $opt_L : "*");
	unshift (@SNCL,$sncl);
    }

    # Check for incompatible options, and assign option defaults.
    my($starttime,$endtime) = &gen_timespan ($start_str, $end_str, $span_str);
    if (defined $starttime) {
	$start_str = &date_string($starttime);
	$end_str = &date_string($endtime);
    }
    else {
	$start_str = undef;
	$end_str = undef;
    }
    die ("Error: missing '-o output_file' option\n") 
	unless ($seedfile ne "" or $estimate_size);

    # Read SNCLs from channellist file, and merge with commandline SNCLs.
    push(@SNCL,&sncls_from_channelfile($channelfile)) if ($channelfile ne "");
    
    # SNCL and time checks
    die ("Error: Missing eventlist or request time\n")
	if (@SNCL > 0 && $opt_E eq "" && ! defined $starttime);

    # Require SNCL and times for sync file.
    ($sync_type = $opt_k) =~ tr/[A-Z]/[a-z]/;
    die ("Invalid sync type: $opt_k") if (! $sync_type =~ /^d[mc]c/i);
    die ("Error: Missing channel list or request time\n")
	if ($opt_k ne "" && (@SNCL == 0 || ! defined $starttime));
    $opt_W = "C" if ($opt_k ne "" && $opt_W eq "");

    # Wavetype checks.
    die ("Error: -W C invalid with eventlist\n") 
	if ($opt_W eq "C" && $opt_E ne "");
    warn ("Warning: Wavetype option changed to -W T with eventlist\n"), $opt_W = "T"
	if ($opt_W =~ /[AC]/ && $opt_E ne "");

    # Input checks.
    $input_format = "breq_fast" if ($input_format =~ /^(breq_fast)|(bf)$/i);
    $input_format = "netdc" if ($input_format =~ /^netdc$/i);
    if ($input_file ne "") {
	die ("Error: Invalid input format: $input_format\n") 
	    unless ($input_format =~ /^(breq_fast)|(netdc)$/);
	die ("Error: No input_format specified for input file\n")
	    if ($input_format eq "");
    }

    # Assign defaults.
    $opt_W = ($opt_E ne "") ? "T" : "A" if ($opt_W eq "");
    $opt_W = "C,T" if ($opt_W =~ /[A]/);
    $workdir = $stagedir if ($workdir eq "");

    # Input is either from input file or cmdline, but not both.
    if ($input_file ne "") {
	$opt_W = "C,T";	#:: TO BE CHANGED IN THE FUTURE.
	@qinfo = &parse_input_file($input_format,$input_file,$opt_W);
    }
    else {
	my(@evids);
        if ($opt_E ne "") {
	    (my($evids) = $opt_E) =~ tr/\?\*/_%/;
	    push (@evids,split(/,/,$evids)) if ($evids ne "");
	}
	@qinfo = &build_qinfo ([@SNCL], [@evids], $start_str, $end_str, $opt_W);
    }
    die ("Error: no query info") if (@qinfo <= 0);

    mkdir ("$stagedir", 0755) || die ("Error: unable to create stagedir $stagedir\n");
    my($query_flags) = 0;
    $query_flags = $QUERY_DATALESS if ($dataless);
    $query_flags = $QUERY_SNCL_LIST if ($dataless && $miniseed);
    $query_flags |= $QUERY_NOFILE if ($sync_type ne "");
    my(@dbinfo) = &query_db ($dbname, $dbuser, $dbpass, $query_flags, @qinfo);
    
    if ($query_flags & $QUERY_SNCL_LIST) {
	# Print SNCL list.
	my($status) = &print_sncl_list (@dbinfo);
	&cleanup ($workdir, $stagedir);
	exit($status);
    }

    # If size estimate is requested, generate it and exit.
    if ($estimate_size ne "") {
	my($size) = &estimate_size (@dbinfo);
	printf ("%.0lf\n", $size);
	&cleanup ($workdir, $stagedir);
	exit(0);
    }

    # If sync file requested, generate it and exit.
    if ($sync_type ne "") {
	my($status) = &build_sync ($seedfile, $dcname, $sync_type, @dbinfo);
	&cleanup ($workdir, $stagedir);
	exit($status);
    }

    # If creating SEED volume, limit requests to channels and times with responses.
    @dbinfo = &limit_requests_by_response(@dbinfo) unless ($miniseed);
    if (@dbinfo == 0) {
	&cleanup ($workdir, $stagedir);
	exit(0);
    }

    my($del_workdir) = 0;
    mkpath([$workdir], 0, 0755), $del_workdir=1 if (! -e $workdir);

    # Stage all of the data files.
    @dbinfo = &stage_files ($workdir, "tmp", @dbinfo) unless ($dataless);

    # Generate output.
    if ($miniseed) {
	&build_mseed($seedfile, $workdir, @dbinfo);
    }
    else {
	&build_seed($seedfile, $workdir, $podhdrpath, @dbinfo);
    }

    &cleanup ($workdir, $stagedir);
    exit(0);
}

######################################################################
#   cleanup:
#	Remove workdir and stagedir directories, unless told otherwise.
######################################################################
sub cleanup 
{
    my($workdir,$stagedir) = @_;
    if ($debug & $DEBUG_KEEP_QUERY) {
        print ("Query kept in $workdir\n");
    }
    else {
	rmtree([$workdir],0,0) if (-d $workdir);
    }
    if ($debug & $DEBUG_KEEP_WAVEFORMS) {
	print ("Waveform files left in $stagedir\n");
    }
    else {
	rmtree ([$stagedir],0,0) if (-d $stagedir);
    }
}

######################################################################
#   gen_timespan:
#	Return Qtime starttime and endtime from input strings 
#		specifying starttime and either endtime or timespan.
#	Die on error.
######################################################################
sub gen_timespan
{
    my($start,$end,$span) = @_;
    # No timespan may be valid.
    return (undef,undef) if ($start eq "" and $end eq "" && $span eq "");
    die ("Error: Missing start of timespan\n") if ($start eq "");
    die ("Error: Missing end of timespan\n") if ($end eq "" && $span eq "");
    die ("Error: Can specify endtime or timespan, not both\n") if ($end ne "" && $span ne "");
    my($qs) = Qtime::set_time($start);
    die ("Error: Invalid start time $start\n") if (! defined $qs);
    my($qe) = Qtime::new();
    if ($end ne "") {
	$qe = Qtime::set_time($end);
	die ("Error: Invalid end time $end\n") if (! defined $qe);
    }
    else {
	$qe = Qtime::end_of_span($qs,$span);
	die ("Error: Invalid timespan $span\n") if (! defined $qe);
    }
    return ($qs,$qe);
}

######################################################################
#   sncls_from_channelfile
#	Read SNCLS from channelfile file.
#	Return an array of sncls.
#	Warn on error.
######################################################################
sub sncls_from_channelfile
{
    my($file) = @_;
    my($fh) = new FileHandle ("<$file");
    warn ("Error opening channelfile $file\n") if (! defined $fh);
    my(@sncl);
    while (my $line = <$fh>) {
	chomp $line;
	next if ($line =~ /^\s*$/);
	next if ($line =~ /^\s*#/);
	my(@t) = split(" ",$line);
	warn ("Invalid SNCL in channelfile: $line\n"), next if (@t != 1);
	my(@comp) = split(/\./,$t[0]);
	# There must be at least SNC on the line.
	warn ("Invalid SNCL in channelfile: $t[0]\n"), next 
		if (@comp < 3 || @comp > 4);
    	push(@sncl,$t[0]);
    }
    return (@sncl);
}    

######################################################################
#   parse_input_file
#	Parse an input file.
#	Return an array of request structures.
#	Warn on error.
######################################################################
sub parse_input_file 
{
    my($input_format,$input_file,$wtype) = @_;
    return (parse_netdc_file($input_file,$wtype))
	if ($input_format =~ /netdc/i);
    return (parse_bf_file($input_file,$wtype))
	if ($input_format =~ /breq_fast/i);
    warn ("Error: unknown input file format: $input_format");
}

######################################################################
#   parse_netdc_file
#	Parse a netdc input file.
#	Return an array of request structures.
#	Warn on error.
######################################################################
sub parse_netdc_file 
{
    my($file,$wtype) = @_;
    my(@qlist);
    my($tpat) = '([\w\?\*]+|\"(?:[^\"]*)\")';
    my($fh) = new FileHandle ("<$file");
    warn ("Error opening netdc file $file\n") if (! defined $fh);
    my($line);
    while (defined($fh) && ($line = <$fh>)) {
	chomp $line;
	next if ($line =~ /^\s*$/);
	next if (! ($line =~ /^.DATA/));
	my(@t);
	@t = $line =~ 
	    /^\.DATA\s+$tpat\s+$tpat\s+$tpat\s+$tpat\s+$tpat\s+$tpat\s+$tpat\s*$/;
	warn ("Error parsing netdc line: wrong number of fields: $line\n"), next if (@t != 7);
	my($dc,$nets,$stats,$locs,$chans,$start,$end);

	($dc     = $t[0]) =~ s/"//g;
	($nets   = $t[1]) =~ s/"//g;
	($stats  = $t[2]) =~ s/"//g;
	($locs   = $t[3]) =~ s/"//g;
	($chans  = $t[4]) =~ s/"//g;
	($start  = $t[5]) =~ s/"//g;
	($end    = $t[6]) =~ s/"//g;

	@t = split(' ',$start);
	warn ("Error parsing starttime \"$start\" in netdc line:  $line\n"), next if (@t != 6);
	($start = sprintf ("%04d/%02d/%02d,%02d:%02d:%07.4lf", @t)) =~ tr/ /0/;
	my($qs) = Qtime::set_time($start);
	warn ("Error parsing starttime \"$start\" in netdc line: $line\n"),next if (! defined $qs);
	my($tstart) = Qtime::tepoch_time($qs);

	@t = split(' ',$end);
	warn ("Error parsing endtime \"$end\" in netdc line: $line\n"),next if (@t != 6);
	$end = sprintf ("%04d/%02d/%02d,%02d:%02d:%07.4lf", @t);
	my($qe) = Qtime::set_time($end);
	warn ("Error parsing endtime \"$end\" in netdc line: $line\n"),next if (! defined $qe);
	my($tend) = Qtime::tepoch_time($qe);

	warn ("Error starttime \"$start\" >= endtime \"$end\" in netdc line: $line\n"),next 
	    if ($tend <= $tstart);

	warn ("Error parsing netdc line: wrong number of fields: $line\n"), next if ($end eq "");
#	printf ("dc=%s, n=%s, s=%s, l=%s, c=%s, start=%s, end=%s\n",
#		$dc,$net,$stat,$loc,$chan,$start,$end);

	# Some entries can consist of one or more tokens.
	# Allow for a blank token for location code.
	my(@nets)  = split(" ",$nets);
	my(@stats) = split(" ",$stats);
	my(@chans) = split(" ",$chans);
	my(@locs) = split(" ",$locs);
	@locs = ("  ") if ($locs =~ /^\s*$/);
	
	# Blank trim (on both sides) all tokens.
	for (@nets) { s/ //g; }
	for (@stats) { s/ //g; }
	for (@chans) { s/ //g; }
	for (@locs) { s/ //g; }

	# Some entries may have significant blanks (or blank placeholders).
	for (@locs) { $_ = "  " if ($_ eq "") };
	for (@locs) { tr/-/ /; }

	# Since each token may be wildcarded, we have to perform an
	# explicit cross-product of the net,stat,loc,chan.
	for my $net (@nets) {
	    for my $stat (@stats) {
		for my $loc (@locs) {
		    for my $chan (@chans) {
			my($rec);
			($rec->{net}  = $net)  =~ tr/\?\*/_%/;
			($rec->{stat} = $stat) =~ tr/\?\*/_%/;
			($rec->{loc}  = $loc)  =~ tr/\?\*/_%/;
			($rec->{chan} = $chan) =~ tr/\?\*/_%/;
			$rec->{wtype} = $wtype if ($wtype ne "");

			$rec->{reqstart} = $start;
			$rec->{treqstart} = $tstart;
			$rec->{qreqstart} = $qs;

			$rec->{reqend} = $end;
			$rec->{treqend} = $tend;
			$rec->{qreqend} = $qe;
			# netdc has no concept of eventids.
#::			$rec->{evids} = "";
			push (@qlist,$rec);
		    }
		}
	    }
	}
    }
    return(@qlist);
}

######################################################################
#   parse_bf_file
#	Parse a breq_fast input file.
#	Return an array of request structures.
#	Warn on error.
#
# BREQ_FAST input format:	(NET is optional, Loc is optional)
#
#      STARTING TIME              ENDING TIME
# STA  NET YYYY MM DD HH MM SS.T  YYYY MM DD HH MM SS.T #_CH CH1 CH2 CHn Loc
#
# Per IRIS definition:
# 1.	Station and Channel each have an implicit "*" (match 0-N chars)
#	at the end of the string.
# 2.	Only "?" (match single character) user-supplied wildcard is supported.
#
######################################################################
sub parse_bf_file 
{
    my($file,$wtype) = @_;
    my(@qlist);
    my($tpat) = '([\w,\.\?\*]+|"(?:[^"]*)")';
    my($fh) = new FileHandle ("<$file");
    warn ("Error opening breq_fast file $file\n") if (! defined $fh);
    my($line);
    while (defined($fh) && ($line = <$fh>)) {
	chomp $line;
	my(@t) = split(" ",$line);
	warn ("Error parsing breq_fast line: $line\n"), next
		if (@t < 15);
	my($net,$stat,$loc,$chan,$start,$end,$nchans);
	# Insert wildcard for network if network is not specified.
	# Network not specified if second field is digits (year).
	splice (@t,1,0,"*") if ($t[1] =~ /^(\d+)$/);
	$stat   = uc($t[0]);
	$net    = uc($t[1]);
	($start = sprintf ("%04d/%02d/%02d,%02d:%02d:%07.4lf", @t[2..7])) =~ tr/ /0/;
	($end   = sprintf ("%04d/%02d/%02d,%02d:%02d:%07.4lf", @t[8..13])) =~ tr/ /0/;
	$nchans = $t[14];
	# Pick up optional location identifier.
	# If no location identifier, use wildcard.
	# If "-" or "--", only match blank location code.
	$loc = (scalar(@t) > $nchans+15) ? $t[$nchans+15] : "*";
	$loc =  ($loc eq "") ? "*" :
		($loc =~ /^\-{1,2}$/) ? "  " : $loc;
        # Breq_fast stations have implicit wildcard matching on the end.
	# Add wildcard to the end of the station name.
	$stat .= '*';
        # Breq_fast channels have implicit wildcard matching on the end.
	# Add wildcard if the channel name is < 3 characters.
	warn ("Error: missing channel specifiers: $line\n"), next
		if (scalar(@t) < 14 + $nchans);
	foreach $chan (@t[15 .. 15+$nchans-1]) {
	    $chan .= "*" if (length($chan) < 3);
	    my($rec);
	    ($rec->{net}  = $net)  =~ tr/\?\*/_%/;
	    ($rec->{stat} = $stat) =~ tr/\?\*/_%/;
	    ($rec->{loc}  = $loc)  =~ tr/\?\*/_%/;
	    ($rec->{chan} = $chan) =~ tr/\?\*/_%/;
	    $rec->{wtype} = $wtype if ($wtype ne "");

	    my($qs) = Qtime::set_time($start);
	    warn ("Error parsing starttime in breq_fast line: $line\n"),next if (! defined $qs);
	    my($tstart) = Qtime::tepoch_time($qs);
	    $rec->{reqstart} = $start;
	    $rec->{treqstart} = $tstart;
	    $rec->{qreqstart} = $qs;

	    my($qe) = Qtime::set_time($end);
	    warn ("Error parsing endtime in breq_fast line: $line\n"),next if (! defined $qe);
	    my($tend) = Qtime::tepoch_time($qe);
	    $rec->{reqend} = $end;
	    $rec->{treqend} = $tend;
	    $rec->{qreqend} = $qe;

	    warn ("Error starttime \"$start\" >= endtime \"$end\" in breq_fast line: $line\n"),next 
		if ($tend <= $tstart);

	    # breq_fast has no concept of eventids.
#::	    $rec->{evids} = "";
	    push (@qlist,$rec);
	}
    }
    return(@qlist);
}

######################################################################
#   build_qinfo
#	Build qinfo array from:
#		sncls array, evids array, start time, end time.
#	Return an array of request structures.
#	Die on error.
######################################################################
sub build_qinfo 
{
    my($ref_sncls,$ref_evids,$start,$end,$wtype) = @_;
    my ($qs, $qe, $tstart, $tend);
    my(@qlist);
    if ($start ne "") {
	$qs = Qtime::set_time($start);
	die ("Error parsing start time %start\n") if (! defined $qs);
	$qe = Qtime::set_time($end);
	die ("Error parsing end time %end\n") if (! defined $qe);
	$tstart = Qtime::tepoch_time($qs);
	$tend = Qtime::tepoch_time($qe);
	die ("Error starttime \"$start\" >= endtime \"$end\"\n") if ($tend <= $tstart);
    }
    my(@sncl) = @$ref_sncls;
    @sncl = &expand_sncl(@sncl);
    my(@evid) = @$ref_evids;
    return (@qlist) if (@sncl == 0 && @evid == 0);

    # We have either a sncl list, event list, or both.
    
    if (@sncl > 0) {
	for my $sncl (@sncl) {
	    # Loop over evids, adding entry for each.
	    # Iterate once even if we have no evid.
	    my(@evidlist) = @evid;
	    push (@evidlist,"") if (@evidlist == 0);
	    for my $evid (@evidlist) {
		my($stat,$net,$chan,$loc) = split(/\./,$sncl);
		my($ref);
		$loc = "" if (! defined $loc);
		($ref->{net}  = $net)  =~ tr/\?\*/_%/ if (defined $net);
		($ref->{stat} = $stat) =~ tr/\?\*/_%/;
		($ref->{chan} = $chan) =~ tr/\?\*/_%/;
		($ref->{loc}  = $loc)  =~ tr/\?\*/_%/;
		$ref->{wtype} = $wtype if ($wtype ne "");
		if ($start ne "") {
		    $ref->{reqstart} = $start;
		    $ref->{qreqstart} = $qs;
		    $ref->{treqstart} = $tstart;
		    $ref->{reqend} = $end;
		    $ref->{qreqend} = $qe;
		    $ref->{treqend} = $tend;
		}
		# Add evid to the request only if it is a real evid.
		($ref->{evids} = $evid) =~ tr/\?\*/_%/ if ($evid ne "");
		push (@qlist,$ref);
	    }
	}
    }
    elsif (@evid > 0) {
	# Loop over evids, adding entry for each evid.
	for my $evid (@evid) {
	    my($ref);
	    ($ref->{evids} = $evid) =~ tr/\?\*/_%/;
	    $ref->{wtype} = $wtype if ($wtype ne "");
	    if ($start ne "") {
		$ref->{reqstart} = $start;
		$ref->{qreqstart} = $qs;
		$ref->{treqstart} = $tstart;
		$ref->{reqend} = $end;
		$ref->{qreqend} = $qe;
		$ref->{treqend} = $tend;
	    }
	    # Add entry.
	    push(@qlist,$ref);
	}
    }
    return (@qlist);
}

######################################################################
#   date_string
#	Return date string from Qtime object in YMD format.
######################################################################
sub date_string 
{
    my($qt) = @_;
    my($str) = sprintf ("%04d/%02d/%02d,%02d:%02d:%02d.%04d",
	Qtime::year($qt), Qtime::month($qt), Qtime::day($qt),
	Qtime::hour($qt), Qtime::minute($qt), 
	Qtime::second($qt), Qtime::usec($qt)/$USECS_PER_TICK);
    return ($str);
}

######################################################################
#   jdate_string
#	Return date string from Qtime object in YDoy format suitable
#	for use with POD.  y,d,h:m:s.ffff
######################################################################
sub jdate_string 
{
    my($qt) = @_;
    my($str) = sprintf ("%04d,%03d,%02d:%02d:%02d.%04d",
	Qtime::year($qt), Qtime::doy($qt),
	Qtime::hour($qt), Qtime::minute($qt), 
	Qtime::second($qt), Qtime::usec($qt)/$USECS_PER_TICK);
    return ($str);
}

######################################################################
#   trim:
#	Trim trailing blanks from the string.
#	Return string without the trailing blanks.
######################################################################
sub trim 
{
    my($arg) = @_;
    my($str) = $arg =~ /^(.*[^ ]) *$/;
    return ($str);
}

######################################################################
#   query_db
#	Query the database for the selected info.
#	Return array of dbinfo structures for the query.
######################################################################
sub query_db
{
    my($dbname,$dbuser,$dbpass,$query_flags,@qinfo) = @_;
    my($sth,@pod_req);
    my(@dbinfo);
    my($dbinfo);

    $dbh = DBI->connect("dbi:Oracle:" . $dbname, $dbuser, $dbpass, 
			{RaiseError => 1, AutoCommit => 1});

    # Initialize list of network codes and their max waveform durations.

    if ($query_flags & $QUERY_DATALESS) 
    {
	# Dataless request.  Query Channel_Data table.
	for my $q (@qinfo) {
	    my($sql) = "SELECT DISTINCT net, sta, seedchan, location, ondate, offdate FROM Channel_Data WHERE ";
	    my(@cond);
	    push (@cond, "(sta LIKE '$q->{stat}\')")      if ($q->{stat} ne "");
	    push (@cond, "(net LIKE \'$q->{net}\')")       if ($q->{net}  ne "");
	    push (@cond, "(seedchan LIKE \'$q->{chan}\')") if ($q->{chan} ne "");
	    push (@cond, "(location LIKE \'$q->{loc}\')")  if ($q->{loc}  ne "");
	    push (@cond, "((a.evid IN $q->{evids}) AND (a.wfid IN w.wfid))") if ($q->{evids} ne "");
	    $sql .= join (" AND ", @cond);
	    print ("$sql\n") if ($opt_d & $DEBUG_SQL);

	    eval {
		$sth = $dbh->prepare( $sql);
		$sth->execute;
		my $arefs =  $sth->fetchall_arrayref({});
		for my $aref (@$arefs) {
		    my($ondate,$offdate,$qon,$qoff,$ton,$toff,$jon,$joff,$pod_req);
		    ($ondate  = $aref->{ONDATE})  =~ tr/ /,/;
		    ($offdate = $aref->{OFFDATE}) =~ tr/ /,/;
		    $qon = Qtime::set_time($ondate);
		    $qoff = Qtime::set_time($offdate);
		    $ton = Qtime::tepoch_time ($qon);
		    $toff = Qtime::tepoch_time ($qoff);
		    # Compare using half-open interval.
		    next if ($q->{treqstart} > $toff || $ton >= $q->{treqend});
		    my($rec);
		    $rec->{stat} = $aref->{STA};
		    $rec->{net} = $aref->{NET};
		    $rec->{chan} = $aref->{SEEDCHAN};
		    $rec->{loc} = $aref->{LOCATION};
		    $rec->{sncl} = sprintf ("%s.%s.%s.%s",$rec->{stat},$rec->{net},$rec->{chan},&trim($rec->{loc}));
		    # Request time.
		    # Since we want to go THROUGH the end of the waveform,
		    # set reqend to epsilon_usecs more than offdate.
		    my($qreqend) = Qtime::add_dtime ($qoff,$epsilon_usecs);
		    my($treqend) = Qtime::tepoch_time ($qreqend);
		    my($reqend) = &date_string($qreqend);
		    $rec->{reqstart} = $ondate;
		    $rec->{qreqstart} = $qon;
		    $rec->{treqstart} = $ton;
		    $rec->{reqend} = $reqend;
		    $rec->{qreqend} = $qreqend;
		    $rec->{treqend} = $treqend;
		    # Waveform start time.
		    $rec->{wavstart} = $ondate;
		    $rec->{qwavstart} = $qon;
		    $rec->{twavstart} = $ton;
		    $rec->{wavend} = $offdate;
		    $rec->{qwavend} = $qoff;
		    $rec->{twavend} = $toff;
		    $rec->{wavefile} = $Waveform_File;
		    $rec->{dir} = "./";
		    push (@dbinfo,$rec);
		}
	    };
	    if ($@) {
		warn ("Error: querying waveform table: $@");
		return();
	    }
	}
	return (@dbinfo);
    }
    elsif ($query_flags & $QUERY_SNCL_LIST) 
    {

	# Initialize list of network codes and their max waveform durations.
	init_MaxWaveLen() == 0 || warn "Error: unable to query MaxWaveLenght table.";

	# Data request.  Search over each waveform.
	for my $q (@qinfo) {
	    # Determine the timewindow based on the network.
#:: 	    my($windowmax) = &window_max ($q->{net});	
	    # Decide what tables to search.
	    my(@tables) = ("Waveform w");
	    push (@tables, "AssocWaE a") if ($q->{evids} ne "");
	    my($tables) = join (", ", @tables);

	    # If we have a reqstart or reqend time, we need to expand any wildcard in the net
	    # into a list of waveform arranged by maxwavelength in order to determine the
	    # max bounds for waveform length.

	    my(@len_nets) = ($q->{treqstart} ne "" || $q->{net} ne "") ? &get_maxwavelens_for($q->{net}) : ("");

	    my(@sql);
	    for my $len_nets (@len_nets) {
		my($windowmax,$netlist) = split(/:/,$len_nets);
		my($sql) =  "SELECT DISTINCT w.net, w.sta, w.seedchan, w.location " ;
		$sql .= "FROM $tables WHERE ";
		my(@cond);
		push (@cond, "(w.sta LIKE \'$q->{stat}\')")      if ($q->{stat} ne "");
		push (@cond, "(w.net IN ($netlist))")		 if ($netlist   ne "");
		push (@cond, "(w.seedchan LIKE \'$q->{chan}\')") if ($q->{chan} ne "");
		push (@cond, "(w.location  LIKE \'$q->{loc}\')") if ($q->{loc}  ne "");
		push (@cond, "(w.wavetype = \'$q->{wtype}\')")       if ($q->{wtype} =~ /^.$/);
		push (@cond, "((a.evid IN $q->{evids}) AND (a.wfid = w.wfid))") if ($q->{evids} ne "");
		if ($q->{treqstart} ne "") {
		    my($ts);
		    if ($q->{treqend} - $q->{treqstart} < $windowmax) {
			# The request is less than the max single waveform span.
			# Make sure the request is suitably bounded for better performance.
			# Use explicit >= and < instead of BETWEEN for timespan requests.
			$ts = sprintf ( "((w.datetime_on >= %lf AND w.datetime_on < %lf) OR  (w.datetime_off >= %lf AND w.datetime_off < %lf) OR " . 
				    "((w.datetime_on >= %lf AND w.datetime_on < %lf) AND (w.datetime_off >= %lf AND w.datetime_off < %lf))) ",
				    $q->{treqstart}, $q->{treqend}, $q->{treqstart}, $q->{treqend}, 
				    $q->{treqend}-$windowmax, $q->{treqstart}, $q->{treqend}, $q->{treqstart}+$windowmax);
		    }
		    else {
			# The request exceeds the max single waveform span.
			# Look for any waveform that has a start or end within the request interval.
			# Use explicit >= and < instead of BETWEEN for timespan requests.
			$ts = sprintf ( "((w.datetime_on >= %lf and w.datetime_on < %lf) OR (w.datetime_off >= %lf AND w.datetime_off < %lf))",
				    $q->{treqstart}, $q->{treqend}, $q->{treqstart}, $q->{treqend});
		    }
		    push (@cond, $ts);
			}
		$sql .= join (" AND ", @cond);
		push (@sql,$sql);
	    }
	    # Complete the single sql request.	    
	    my($sql) = join(" UNION ",@sql);
	    $sql .= "ORDER BY net, sta, seedchan, location"
		if ($sql ne "");
	    next if ($sql =~ /^\s*$/);
	    print ("$sql\n") if ($opt_d & $DEBUG_SQL);

	    eval {
		$sth = $dbh->prepare($sql);
		$sth->execute;
		my $arefs =  $sth->fetchall_arrayref({});
		print "# matches = ", scalar(@$arefs), "\n" if ($opt_d & $DEBUG_SQL);
		for my $aref (@$arefs) {
		    my($ondate,$offdate,$qon,$qoff,$ton,$toff,$jon,$joff,$pod_req);
		    # Compare using half-open interval.
#:: 		    next if ($q->{treqstart} ne "" && ( $q->{treqstart} > $aref->{DATETIME_OFF} || $aref->{DATETIME_ON} >= $q->{treqend} ));
#:: 		    $ton = $aref->{DATETIME_ON};
		    my($rec);
		    $rec->{stat} = $aref->{STA};
		    $rec->{net} = $aref->{NET};
		    $rec->{chan} = $aref->{SEEDCHAN};
		    $rec->{loc} = $aref->{LOCATION};
		    $rec->{sncl} = sprintf ("%s.%s.%s.%s",$rec->{stat},$rec->{net},$rec->{chan},&trim($rec->{loc}));
		    push (@dbinfo,$rec);
		}
	    };
	    if ($@) {
		warn ("Error: querying waveform table: $@");
		return();
	    }
	}
	return (@dbinfo);
    }
    else 
    {

	# Initialize list of network codes and their max waveform durations.
	init_MaxWaveLen() == 0 || warn "Error: unable to query MaxWaveLenght table.";

	# Data request.  Search over each waveform.
	for my $q (@qinfo) {
	    # Determine the timewindow based on the network.
#:: 	    my($windowmax) = &window_max ($q->{net});	
	    # Decide what tables to search.
	    my(@tables) = ("Waveform w");
	    push (@tables, "AssocWaE a") if ($q->{evids} ne "");
	    my($tables) = join (", ", @tables);

	    # If we have a reqstart or reqend time, we need to expand any wildcard in the net
	    # into a list of waveform arranged by maxwavelength in order to determine the
	    # max bounds for waveform length.

	    my(@len_nets) = ($q->{treqstart} ne "" || $q->{net} ne "") ? &get_maxwavelens_for($q->{net}) : ("");

	    my(@sql);
	    for my $len_nets (@len_nets) {
		my($windowmax,$netlist) = split(/:/,$len_nets);
		my($sql) =  "SELECT DISTINCT w.net, w.sta, w.seedchan, w.location, " ;
		$sql .= "w.datetime_on, w.datetime_off, w.samprate, w.wavetype, " . 
			"w.fileid, w.nbytes, w.traceoff, w.tracelen, w.status, w.format_id, " .
			"w.wavetype, w.recordsize ";
		$sql .= "FROM $tables WHERE ";
		my(@cond);
		push (@cond, "(w.sta LIKE \'$q->{stat}\')")      if ($q->{stat} ne "");
		push (@cond, "(w.net IN ($netlist))")		 if ($netlist   ne "");
		push (@cond, "(w.seedchan LIKE \'$q->{chan}\')") if ($q->{chan} ne "");
		push (@cond, "(w.location  LIKE \'$q->{loc}\')") if ($q->{loc}  ne "");
		push (@cond, "(w.wavetype = \'$q->{wtype}\')")       if ($q->{wtype} =~ /^.$/);
		push (@cond, "((a.evid IN $q->{evids}) AND (a.wfid = w.wfid))") if ($q->{evids} ne "");
		if ($q->{treqstart} ne "") {
		    my($ts);
		    if ($q->{treqend} - $q->{treqstart} < $windowmax) {
			# The request is less than the max single waveform span.
			# Make sure the request is suitably bounded for better performance.
			# Use explicit >= and < instead of BETWEEN for timespan requests.
			$ts = sprintf ( "((w.datetime_on >= %lf AND w.datetime_on < %lf) OR  (w.datetime_off >= %lf AND w.datetime_off < %lf) OR " . 
				    "((w.datetime_on >= %lf AND w.datetime_on < %lf) AND (w.datetime_off >= %lf AND w.datetime_off < %lf))) ",
				    $q->{treqstart}, $q->{treqend}, $q->{treqstart}, $q->{treqend}, 
				    $q->{treqend}-$windowmax, $q->{treqstart}, $q->{treqend}, $q->{treqstart}+$windowmax);
		    }
		    else {
			# The request exceeds the max single waveform span.
			# Look for any waveform that has a start or end within the request interval.
			# Use explicit >= and < instead of BETWEEN for timespan requests.
			$ts = sprintf ( "((w.datetime_on >= %lf and w.datetime_on < %lf) OR (w.datetime_off >= %lf AND w.datetime_off < %lf))",
				    $q->{treqstart}, $q->{treqend}, $q->{treqstart}, $q->{treqend});
		    }
		    push (@cond, $ts);
			}
		$sql .= join (" AND ", @cond);
		push (@sql,$sql);
	    }
	    # Complete the single sql request.	    
	    my($sql) = join(" UNION ",@sql);
	    $sql .= "ORDER BY net, sta, seedchan, location, datetime_on, datetime_off"
		if ($sql ne "");
	    next if ($sql =~ /^\s*$/);
	    print ("$sql\n") if ($opt_d & $DEBUG_SQL);

	    eval {
		$sth = $dbh->prepare($sql);
		$sth->execute;
		my $arefs =  $sth->fetchall_arrayref({});
		print "# matches = ", scalar(@$arefs), "\n" if ($opt_d & $DEBUG_SQL);
		for my $aref (@$arefs) {
		    my($ondate,$offdate,$qon,$qoff,$ton,$toff,$jon,$joff,$pod_req);
		    # Compare using half-open interval.
		    next if ($q->{treqstart} ne "" && ( $q->{treqstart} > $aref->{DATETIME_OFF} || $aref->{DATETIME_ON} >= $q->{treqend} ));
		    $ton = $aref->{DATETIME_ON};
		    $toff = $aref->{DATETIME_OFF};
		    $qon = Qtime::set_time ($ton . 'T');
		    $qoff = Qtime::set_time ($toff . 'T');
		    $ondate = &jdate_string($qon);
		    $offdate = &jdate_string($qoff);
		    my($path,$wavefile,$tfilstart,$tfilend) = &file_info ($query_flags, $aref) 
			unless ($query_flags & $QUERY_NOFILE);
		    my($rec);
		    $rec->{stat} = $aref->{STA};
		    $rec->{net} = $aref->{NET};
		    $rec->{chan} = $aref->{SEEDCHAN};
		    $rec->{loc} = $aref->{LOCATION};
		    $rec->{sncl} = sprintf ("%s.%s.%s.%s",$rec->{stat},$rec->{net},$rec->{chan},&trim($rec->{loc}));
		    # Waveform time.
		    $rec->{wavstart} = $ondate;
		    $rec->{qwavstart} = $qon;
		    $rec->{twavstart} = $ton;
		    $rec->{wavend} = $offdate;
		    $rec->{qwavend} = $qoff;
		    $rec->{twavend} = $toff;
		    # Ensure request time is within this waveform segment.
		    if ($q->{treqstart} ne "" && $q->{treqstart} < $ton) {
			$rec->{reqstart} = $ondate;
			$rec->{treqstart} = $ton;
			$rec->{qreqstart} = $qon;
		    }
		    else {
			$rec->{reqstart} = $q->{reqstart};
			$rec->{qreqstart} = $q->{qreqstart};
			$rec->{treqstart} = $q->{treqstart};
		    }
		    if ($q->{treqend} ne "" && $q->{treqend} > $toff) {
			# Since we want to go THROUGH the end of the waveform,
			# set reqend to epsilon_usecs more than offdate.
			my($qreqend) = Qtime::add_dtime ($qoff,$epsilon_usecs);
			my($treqend) = Qtime::tepoch_time ($qreqend);
			my($reqend) = &jdate_string($qreqend);
			$rec->{reqend} = $reqend;
			$rec->{qreqend} = $qreqend;
			$rec->{treqend} = $treqend;
#:: 			$rec->{reqend} = $offdate;
#:: 			$rec->{treqend} = $toff;
#:: 			$rec->{qreqend} = $qoff;
		    }
		    else {
			$rec->{reqend} = $q->{reqend};
			$rec->{qreqend} = $q->{qreqend};
			$rec->{treqend} = $q->{treqend};
		    }
		    # Waveform file info.
		    $rec->{wtype} = $aref->{WAVETYPE};
		    $rec->{wavefile} = $wavefile;
		    $rec->{dir} = $path;
		    $rec->{fileid} = $aref->{FILEID};
		    $rec->{nbytes} = $aref->{NBYTES};
		    $rec->{tfilstart} = $tfilstart;
		    $rec->{tfilend} = $tfilend;
		    $rec->{traceoff} = $aref->{TRACEOFF};
		    $rec->{tracelen} = $aref->{TRACELEN};
		    $rec->{samprate} = $aref->{SAMPRATE};
		    push (@dbinfo,$rec);
		}
	    };
	    if ($@) {
		warn ("Error: querying waveform table: $@");
		return();
	    }
	}
	return (@dbinfo);
    }
}

######################################################################
#   file_info
#	Return file info for the specified request.
#   Return:	
#	(dir,filename,tfilstart,tfilend) list for waveform file.
######################################################################
sub file_info
{
    my($query_flags,$ref) = @_;
    my($dir,$pathdir,$subdir,$wavefile,$tfilstart,$tfilend);
    return($dir,$wavefile,$tfilstart,$tfilend) if ($query_flags & $QUERY_NOFILE);

    # First find the waveform filename and subdir.
    my($sql) =  "SELECT DISTINCT f.dfile, s.subdirname, f.datetime_on, f.datetime_off " .
		"FROM Filename f, Subdir s WHERE " . 
		"f.fileid = $ref->{FILEID} and f.subdirid = s.subdirid";
    print ("$sql\n") if ($opt_d & $DEBUG_SQL);
    eval {
	my($sth) = $dbh->prepare($sql);
	$sth->execute;
	my $rref = $sth->fetchrow_hashref;
	$wavefile = $rref->{DFILE};
	$subdir = $rref->{SUBDIRNAME};
	$tfilstart = $rref->{DATETIME_ON};
	$tfilend = $rref->{DATETIME_OFF};
    };
    if ($@) {
	warn ("Error: querying subdir table: $@");
	return();
    }
    # Now look for the file in one of several places.
    # NCEDC archives continuous waveforms and triggered waveforms by different rules.
    $dir = $subdir;
    if ($ref->{WAVETYPE} eq 'C') {
	# Continuous data, archived by network code.
	# Search for all possible waveform files.
	$sql = "SELECT DISTINCT pathname FROM Pathname where net = \'$ref->{NET}\'";
	eval {
	    my($sth) = $dbh->prepare($sql);
	    $sth->execute;
	    my $arefs =  $sth->fetchall_arrayref({});
	    for my $aref (@$arefs) {
		$pathdir = "$aref->{PATHNAME}/$subdir";
		$dir = $pathdir, return if (-f "$pathdir/$wavefile");
	    }
	    $dir = $subdir;	# File not found - perhaps we can find the full dir later.
	};
	if ($@) {
	    warn ("Error: querying pathname table: $@");
	    return();
	}
    }
    elsif ($ref->{WAVETYPE} eq 'T') {
	# Triggered data, archived by ???
	# Search for all possible waveform files.
	# For triggered data, use waveform times as the filetime.
	$tfilstart = $ref->{DATETIME_ON};
	$tfilend = $ref->{DATETIME_OFF};
	$sql = "SELECT DISTINCT pathname FROM Pathname where net = 'EVENT'";
	eval {
	    my($sth) = $dbh->prepare($sql);
	    $sth->execute;
	    my $arefs =  $sth->fetchall_arrayref({});
	    for my $aref (@$arefs) {
		$pathdir = "$aref->{PATHNAME}/$subdir";
		$dir = $pathdir, return if (-f "$pathdir/$wavefile");
	    }
	    $dir = $subdir;	# File not found - perhaps we can find the full dir later.
	};
	if ($@) {
	    warn ("Error: querying pathname table: $@");
	    return();
	}
#:: 	$dir = "$BaseDir/$subdir";	#:: Debugging only.
    }
    return($dir,$wavefile,$tfilstart,$tfilend);
}

######################################################################
#   stage_files
#	Stage the data for the specified requests.
#	Create separate files for event segments.
#	Merge event segment when they overlap with other files.
#   Return:
#	Updated dbinfo array with updated request info.
######################################################################
sub stage_files
{
    my($workdir,$tmp,@dbinfo) = @_;
    mkdir("$workdir/$tmp", 0755) || die ("Error: mkdir $workdir/$tmp") unless (-d "$workdir/$tmp");
    # Sort the requests first by SNCL, then by time.
    # This allows us to process all waveforms at once for each SNCL.
    @dbinfo = sort dbinfo_by_sncl_time @dbinfo;
    my(@done,@todo,$prev,$wend);
    $prev = shift(@dbinfo);
    my($twavend) = $prev->{twavend};
    push (@todo,$prev);
    while (@dbinfo > 0) {
	my($cur) = shift(@dbinfo);
	if ($prev->{sncl} ne $cur->{sncl} ||
	    $cur->{twavstart} > $twavend) {
	    # We have all of the requests for a given SNCL.
	    # Merge the waveforms together.
	    &merge_waveforms($workdir,$tmp,@todo);
	    push(@done,@todo);
	    @todo = ($cur);
	    $prev = $cur;
	    $twavend = $prev->{twavend};
	}
	else {
	    # Collect all of the requests for a single SNCL.
	    push (@todo,$cur);
	    $prev = $cur;
	    $twavend = $prev->{twavend} if ($prev->{twavend} > $twavend);
	}
    }
    if (@todo > 0) {
	# We have all of the requests for the last SNCL.
	# Merge the waveforms together.
	&merge_waveforms($workdir,$tmp,@todo);
	push(@done,@todo);
    }
    return (@done);
}

######################################################################
#   merge_waveforms:
#	Merge waveforms for 1 or more requests for a single SNCL
#	into a single waveform file.
#	Update the wavefile and dir for these requests.
#	a.  Use $workdir/$tmp dir as temporary staging area.
#	b.  Merged files go into $workdir/$sncl directory.
#   Returns:
#	1 on success.
######################################################################
sub merge_waveforms
{
    my($workdir,$tmp,@todo) = @_;
    return (@todo) if (@todo == 0);
    mkdir ("$workdir/$tmp", 0755) || die ("Error: mkdir $workdir/$tmp") unless (-d "$workdir/$tmp");
    my($sncl) = $todo[0]->{sncl};
    mkdir ("$workdir/$sncl") || die ("Error: mkdir $workdir/$sncl") unless (-d "$workdir/$sncl");
    my($n) = 0;

    for my $rec (@todo) {
	# For continuous data (SEED output):
	#   Assume that we have only one copy of the continuous data, 
	#   and name the temp file with the original filename and 
	#   append the starttime of the waveform segment.
	#   This way, if we have multiple requests within this segment,
	#   we only have to stage that segment once.
	# For triggered data or MiniSEED output:
	#   Name the temp files sequentually.  It is possible to have
	#   multiple triggered segments from different files with the
	#   same base filename but with different number of samples,
	#   so we may have to stage them all.
	#   MiniSEED is trimmed to the requested time, so each request
	#   must be processed.
	my($in) = "$rec->{dir}/$rec->{wavefile}";
	my($out) = ($stage_alg == $STAGE_FILE)
		? sprintf ("$tmp/$rec->{wavefile}.%.4lf", $rec->{reqstart})
		: sprintf ("$tmp/$sncl.%d", $n++);
	# If we have already staged this exact segment, don't stage it again.
	if (-f "$workdir/$out") {
	    print ("merge: skip multiple stage of $workdir/$out\n") if ($debug && $DEBUG_WAVEFORMS);
	    next;
	}

	# Limit the data to the requested time if we are generating MiniSEED.
	my($starttime,$endtime);
	if (($stage_alg eq $STAGE_SEGMENT) && defined($rec->{reqstart})) {
	    $starttime = $rec->{reqstart};
	    $endtime = $rec->{reqend};
	}

	# Stage this segment into the tmp directory.
	if ($rec->{wtype} eq 'C') {
	    &stage_data ("$workdir/$out","$in",$rec->{traceoff},$rec->{tracelen},$starttime,$endtime ) 
		|| warn ("Warning: error staging file from $in\n");
	}
	else {
	    &stage_data ("$workdir/$out","$in",$rec->{traceoff},$rec->{tracelen},$starttime,$endtime) 
		|| warn ("Warning: error staging event $sncl data from $in\n");
	}
    }

    # Now merge all of the tmp files into a single output file.
    my($outfile) = "$sncl.$todo[0]->{wavstart}";
    my($cmd) = "$qmerge -o $workdir/$sncl/$outfile $workdir/$tmp/* >/dev/null 2>&1";
    print ("$cmd\n") if ($debug && $DEBUG_WAVEFORMS);
    system ($cmd) == 0 || warn ("Error: retcode $? running: $cmd\n");

    # Update the dir and filename of all of the requests.
    for my $rec (@todo) {
	$rec->{wavefile} = $outfile;
	$rec->{dir} = $sncl;
    }

    # Delete the tmp directory and its contents.
    rmtree(["$workdir/$tmp"],0,0);
    return (1);
}

######################################################################
#   stage_data
#	Stage the data for the specified requests.
#	Create separate files for event segments.
#   Return:
#	1 on success, 0 on failure.
######################################################################
sub stage_data
{
    my($out, $in, $pos, $len, $starttime, $endtime) = @_;
    mkdir ($out, 0755) || die ("Error: mkdir $out") unless (-d dirname($out));
    my ($ifh) = new IO::File ("<$in");
    if (! defined $ifh) {
	warn "Error: unable to open for reading: $in";
	return (0);
    }
    my($ofh);
    if ($starttime ne "" && $endtime ne "") {
	$out = "| $qmerge -T -f $starttime -t $endtime > $out";
	$out .= "  2>/dev/null" unless ($verbose);
    }
    else {
	$out = "> $out";
    }
    $ofh = new IO::File ("$out");
    if (! defined $ofh) {
	warn "Error: unable to open for writing: $out";
	return (0);
    }
    print ("stage_data: in=$in pos=$pos len=$len out=$out\n") if ($debug && $DEBUG_WAVEFORMS);
    if (defined $pos) {
	my($buf);
	$ifh->seek($pos,0);
	$ifh->read($buf,$len);
	$ofh->print($buf);
    }
    else {
	$ofh->print(<$ifh>);
    }	
    $ofh->close;
    $ifh->close;
    return (1);
}

######################################################################
#   build_seed:
#	Build SEED output.
######################################################################
sub build_seed
{
    my($seedfile, $workdir, $podhdrpath, @dbinfo) = @_;
    # Determine whether we have to build a blockette directory.
    my($build_blockette_dir, $blockette_dir);
    if ($podhdrpath eq "") {
	$blockette_dir = $Dflt_Blockette_Dir;
	$build_blockette_dir = 1;
    }
    else {
	$blockette_dir = basename($podhdrpath);
	$build_blockette_dir = 0;
    }

    my(@pod_req);
    @pod_req = &build_pod_req($blockette_dir, @dbinfo);
    @pod_req = &order_pod_req(@pod_req);
    $pod_req = "$workdir/$pod_req";
    open (POD_REQ, ">$pod_req") || die ("Error: unable to open $pod_req\n");
    print (POD_REQ @pod_req);
    close (POD_REQ);
    
    my($cmd);
    # Set up ENV for POD and ancillary programs.
    # Build blockette directory if necessary.
    if ($build_blockette_dir) {
	$ENV{USE_LOOKUP} = "YES";
	$ENV{USE_ABPOLY} = "YES";
	$ENV{DATA_ROOT} = "$workdir/$blockette_dir";
	$ENV{HEADER_PATH} = $ENV{DATA_ROOT};
	$cmd = "$Make_Blockettes -p $pod_req 1>>$workdir/pod.log 2>&1";
	system ($cmd) == 0 || warn ("Error: recode $? running: $cmd");
    }
    else {
	$ENV{HEADER_PATH} = "$podhdrpath";
    }
    $ENV{DATA_PATH} = $workdir;
    $ENV{DEBUG} = $opt_d & $DEBUG_POD if ($opt_d ne 0);
    $ENV{SEED_LABEL} = $opt_l if ($opt_l ne "");
    $ENV{SEED_VERSION} = "2.4";
    $ENV{SEEDOUTPUTNODATA} = "YES" if ($dataless);

    $cmd = "$Pod $seedfile $pod_req 4096 32768 99,999 99,999 >>$workdir/pod.log 2>&1";
    system ($cmd) == 0 || warn ("Error: retcode $? running: $cmd");
    return ($?);
}

######################################################################
#   build_mseed:
#	Build MiniSEED output.
######################################################################
sub build_mseed
{
    my($seedfile, $workdir, @dbinfo) = @_;
    open (OUTFILE, ">$seedfile") || die ("Error: unable to open output file $seedfile\n");
    # Generate unique list of input files.
    my(%infile);
    for my $rec (@dbinfo) {
	# Dates in POD request must be in julian format.
	my($file) = "$workdir/$rec->{dir}/$rec->{wavefile}";
	$infile{$file} = 1;
    }
    for my $infile (sort (keys (%infile))) {
	open (INFILE, "$infile") || warn ("Warning: Unable to open infile $infile\n");
	print ("processing $infile\n") if ($debug && $DEBUG_WAVEFORMS);
	print (OUTFILE <INFILE>);
	close (INFILE);
    }
    close(OUTFILE);
    return (0);
}

######################################################################
#   build_pod_req:
#	Build and return a list of POD requests.
######################################################################
sub build_pod_req
{
    my($blockette_dir,@dbinfo) = @_;
    my(@pod_req);
    @dbinfo = sort dbinfo_by_sncl_time @dbinfo;
    my(@done,@todo,$prev,$wend);
    $prev = shift(@dbinfo);
    my($max_twavend) = $prev->{twavend};
    push (@todo,$prev);
    while (@dbinfo > 0) {
	my($cur) = shift(@dbinfo);
	# Merge requests together if they are for the same SNCL
	# and have overlapping or contiguous times.
	unless ($prev->{sncl} eq $cur->{sncl} &&
		$cur->{twavstart} <= $max_twavend &&
		$cur->{wavefile} eq $prev->{wavefile})
	{
	    # We have all of the requests for a given SNCL, contiguous time, and wavefile.
	    # Merge the todo requests together.
	    my($pod_req) = &gen_pod_req($blockette_dir,@todo);
	    push (@pod_req, $pod_req);
	    @todo = ($cur);
	    $prev = $cur;
	    $max_twavend = $prev->{twavend};
	}
	else {
	    # Collect all of the requests for a single SNCL and timespan.
	    push (@todo,$cur);
	    $prev = $cur;
	    $max_twavend = $prev->{twavend} if ($prev->{twavend} > $max_twavend);
	}
    }	    
    if (@todo > 0) {
	# We have all of the requests for the last SNCL.
	my($pod_req) = &gen_pod_req($blockette_dir,@todo);
	push (@pod_req, $pod_req);
    }
    return (@pod_req);
}

######################################################################
#   gen_pod_req:
#	Return a single pod request from a list of contiguous requests.
######################################################################
sub gen_pod_req 
{
    my($blockette_dir,@dbinfo) = @_;
    my($rec) = shift(@dbinfo);
    if (@dbinfo > 0) {
	# Replace request and waveform end of first rec with last rec.
	my($lrec) = pop(@dbinfo);
	if ($rec->{treqend} < $lrec->{treqend}) {
	    $rec->{reqend} = $lrec->{reqend};
	    $rec->{qreqend} = $lrec->{qreqend};
	    $rec->{treqend} = $lrec->{treqend};
	}
	if ($rec->{twavend} < $lrec->{twavend}) {
	    $rec->{wavend} = $lrec->{wavend};
	    $rec->{qwavend} = $lrec->{qwavend};
	    $rec->{twavend} = $lrec->{twavend};
	}
    }
    # Dates in POD request must be in julian format.
    my($wstart) = &jdate_string ($rec->{qwavstart});
    my($wend) = &jdate_string ($rec->{qwavend});
    # Use waveform time as request time if no request time.
    my($rstart) = (defined $rec->{qreqstart}) ? &jdate_string ($rec->{qreqstart}) : $wstart;
    my($rend) = (defined $rec->{qreqend}) ? &jdate_string ($rec->{qreqend}): $wend;
    my($pod_req) = sprintf ("%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
			    $rec->{stat}, $rec->{net}, $rec->{chan}, &trim($rec->{loc}),
			    $wstart, $wend,
			    "$rec->{dir}/$rec->{wavefile}", $blockette_dir, 
			    $rstart, $rend);
    return ($pod_req);
}

######################################################################
#   order_pod_req:
#	Sort and order list of POD requests.
######################################################################
sub order_pod_req
{
    my(@pod_req) = @_;
    @pod_req = sort pod_req_by_sncl_time @pod_req;
    @pod_req = &earliest_first (@pod_req);
    return (@pod_req);
}

######################################################################
#   earliest_first:
#	Move sncl requests for earliest POD request first in the list.
######################################################################
sub earliest_first
{
    my(@pod_req) = @_;
    return (@pod_req) if (@pod_req <= 1);
    # Find sncl with earliest time.
    my(@t) = split(/\t/,$pod_req[0]);
    my($s,$n,$c,$l,$min_time) = $t[0-4];
    my($i,$k,$n);
    $i = 1;
    while ($i++ < @pod_req) {
	@t = split(/\t/,$pod_req[0]);
	($s,$n,$c,$l,$min_time) = $t[0-4] if (($min_time cmp $t[4]) < 0);
    }
    # Find sncl with earliest time.
    $i = $k = $n = 0;
    sncl_search:
    while ($i++ < @pod_req) {
	@t = split (/\t/,@pod_req[$i]);
	if ($s eq $t[0] && $n eq $t[1] && $c eq $t[2] && $l eq $t[3]) {
	    $n = 1;
	    $k = $i;
	    while ($i++ < @pod_req) {
		@t = split (/\t/,@pod_req[$i]);
		$n++, next if ($s eq $t[0] && $n eq $t[1] && $c eq $t[2] && $l eq $t[3]);
		last scnl_search;
	    }
	}
    }
    # Move all sncl request with earliest time to front.
    if ($n > 0) {
	@t = splice(@pod_req,$k,$n);
	splice (@pod_req,0,0,@t);
    }
    return (@pod_req);
}

######################################################################
#   Sort function for pod_req array.
######################################################################
sub pod_req_by_sncl_time {
    $a cmp $b 
    ;
}
	
######################################################################
#   Sort function for dbinfo array.
######################################################################
sub dbinfo_by_sncl_time {
    $a->{sncl} cmp $b->{sncl} ||
    $a->{twavstart} <=> $b->{twavstart} ||
    ((defined($a->{treqstart}) ? $a->{treqstart} : $a->{twavstart}) <=> 
     (defined($b->{treqstart}) ? $b->{treqstart} : $b->{twavstart})) ||
    ((defined($a->{treqend}) ? $a->{treqend} : $a->{twavend}) <=> 
     (defined($b->{treqend}) ? $b->{treqend} : $b->{twavend}))
    ;
}
	
######################################################################
#   expand_sncl
#	Expand list of sncl_crossproducts into sncl list.
######################################################################
sub expand_sncl
{
    my(@sncl_cp) = @_;
    my(@sncl);
    for my $sncl_cp (@sncl_cp) {
	my($S,$N,$C,$L) = split(/\./,$sncl_cp);
	$L = "  " if ((! defined $L) || ($L eq ""));
	$L = "  " if ($L =~ /^--?$/);
	my(@s) = split(/,/,$S);
	my(@n) = split(/,/,$N);
	my(@c) = split(/,/,$C);
	my(@l) = split(/,/,$L);
	for my $s (@s) {
	    for my $n (@n) {
		for my $c (@c) {
		    for my $l (@l) {
			push (@sncl, "$s.$n.$c.$l");
		    }
		}
	    }
	}
    }
    return (@sncl);
}

######################################################################
#   estimate_size:
#	Estimate size of request in KBytes.
#   Return:	size estimate in KBytes.
######################################################################
sub estimate_size
{
    my(@dbinfo) = @_;
    @dbinfo = sort dbinfo_by_sncl_time @dbinfo;
    my($blksize) = 4096;
    my($size) = 0;
    for my $rec (@dbinfo) {
	my($wavdur) = $rec->{twavend} - $rec->{twavstart};
	$wavdur = 1 if ($wavdur == 0);
	my($reqdur) = $wavdur;
	# Compute duration of request that overlaps with this waveform.
	if (defined($rec->{treqstart})) {
		my($tstart) = $rec->{treqstart} > $rec->{twavstart} ? $rec->{treqstart} : $rec->{twavstart};
		my($tend)   = $rec->{treqend}   < $rec->{twavend}   ? $rec->{treqend}   : $rec->{twavend};
		$reqdur = $tend - $tstart;
	}
	$reqdur = $wavdur if ($reqdur > $wavdur);
	my($wavsiz) = $rec->{nbytes};
	# Scale by request duration.
	$wavsiz *= ($reqdur/$wavdur);		
	# Scale
	$wavsiz = int(($wavsiz+$blksize-1)/$blksize) * $blksize;
	$wavsiz = ($wavsiz/(1024)); 
	$size += $wavsiz;
    }
    return ($size);	
}

######################################################################
#   print_sncl_list:
#	Build and print SNCL list.
#   Return:	0 on success, -1 on failure.
######################################################################
sub print_sncl_list
{
    my(@dbinfo) = @_;
    for my $rec (@dbinfo) {
	print ($rec->{sncl}, "\n");
    }
    return (0);	
}

######################################################################
#   build_sync:
#	Build and print sync records.
#   Return:	0 on success, -1 on failure.
######################################################################
sub build_sync 
{
    my($seedfile,$dcname,$sync_type,@dbinfo) = @_;
    open (OUTFILE, ">$seedfile") || die ("Error: unable to open output file $seedfile\n");
    @dbinfo = sort dbinfo_by_sncl_time @dbinfo;
    my($clock_drift,$sample_rate,$num_samples,$chan_flag,$volname,$tapenum,$volnum,$comment,$dmc_dtm,$dcc_dtm);
    my($qnow) = Qtime::set_time(time() . "N");
    my($now) = sprintf ("%04d,%03d",Qtime::year($qnow),Qtime::doy($qnow));
    my($sync_hdr) = "$dcname|$now";
    my($hdr_printed) = 0;
    $dmc_dtm = $now if ($sync_type =~ /dmc/i);
    $dcc_dtm = $now if ($sync_type =~ /dcc/i);
    for my $rec (@dbinfo) {
	my($qwavend) = Qtime::new($rec->{qwavend});
	my($interval) = 1/$rec->{samprate};
	$qwavend = Qtime::add_dtime($qwavend,$interval*$USECS_PER_SEC);
	my($wavend) = &jdate_string($qwavend);
	my($sync) = join("|",$rec->{net}, $rec->{stat},
			    &trim($rec->{loc}),$rec->{chan},
			    $rec->{wavstart},$wavend,
			    $clock_drift, $sample_rate,
			    $num_samples, $chan_flag,
			    $volname, $tapenum,
			    $volnum, $comment,
			    $dmc_dtm, $dcc_dtm);
	print (OUTFILE "$sync_hdr", "\n") unless ($hdr_printed);
	$hdr_printed = 1;
	print (OUTFILE "$sync", "\n");
    }
    close (OUTFILE);
    return (0);	
}

sub limit_requests_by_response
{
    my(@dbin) = @_;
    # Generate a list of unique channels and check for 
    # min and max response time.
    my(%resp);
    my(@dbout);
    for my $rec (@dbin) {
	my($sncl) = "$rec->{stat}.$rec->{net}.$rec->{chan}.$rec->{loc}";
	my($loc) = &db_location($rec->{loc});
	my($sth);
	if (! defined $resp{$sncl}) {
	    $resp{$sncl}{sncl} = $sncl;
	    $resp{$sncl}{qstart} = Qtime::new($qtmax);
	    $resp{$sncl}{qend} = Qtime::new($qtmax);

	    if (1) {
	    my($sql) =  "SELECT MIN(ondate), MAX(offdate) FROM Channel_Data WHERE (net = '$rec->{net}') AND " . 
			"(sta = '$rec->{stat}') AND (seedchan = '$rec->{chan}') AND (location = '$loc')";
	    print ("$sql\n") if ($opt_d & $DEBUG_SQL);
	    eval {
		$sth = $dbh->prepare( $sql);
		$sth->execute;
		my($ondate,$offdate);
	        while (($ondate,$offdate) = $sth->fetchrow_array) {
		    if (defined $ondate) {
			$resp{$sncl}{qstart} = Qtime::set_time($ondate);
		    }
		    else {
			print ("Channel_data table not found for $sncl\n") if ($opt_d & $DEBUG_SQL);
			last;
		    }
		    if (defined $offdate) {
			$resp{$sncl}{qend} = Qtime::set_time($offdate);
		    }
		    else {
			print ("Channel_data table not found for $sncl\n") if ($opt_d & $DEBUG_SQL);
			last;
		    }
		}
	    };
	    if ($@) {
		warn ("Error: querying channel_data table for %sncl");
		return (@dbout);
	    }
	    }

	    if (0) {
	    my($sql) =  "SELECT MIN(ondate) FROM Channel_Data WHERE (net = '$rec->{net}') AND " . 
			"(sta = '$rec->{stat}') AND (seedchan = '$rec->{chan}') AND (location = '$loc')";
	    print ("$sql\n") if ($opt_d & $DEBUG_SQL);
	    eval {
		$sth = $dbh->prepare( $sql);
		$sth->execute;
		my($ondate);
	        while (($ondate) = $sth->fetchrow_array) {
		    if (defined $ondate) {
			$resp{$sncl}{qstart} = Qtime::set_time($ondate);
		    }
		    else {
			print ("Channel_data table not found for $sncl\n") if ($opt_d & $DEBUG_SQL);
			last;
		    }
		}
	    };
	    if ($@) {
		warn ("Error: querying channel_data table for %sncl");
		return (@dbout);
	    }

 	    $sql =  "SELECT MAX(offdate) FROM Channel_Data WHERE (net = '$rec->{net}') AND " . 
			"(sta = '$rec->{stat}') AND (seedchan = '$rec->{chan}') AND (location = '$loc')";
	    print ("$sql\n") if ($opt_d & $DEBUG_SQL);
	    eval {
		$sth = $dbh->prepare( $sql);
		$sth->execute;
		my($offdate);
	        while (($offdate) = $sth->fetchrow_array) {
		    if (defined $offdate) {
			$resp{$sncl}{qend} = Qtime::set_time($offdate);
		    }
		    else {
			print ("Channel_data table not found for $sncl\n") if ($opt_d & $DEBUG_SQL);
			last;
		    }
		}
	    };
	    if ($@) {
		warn ("Error: querying channel_data table for %sncl");
		return (@dbout);
	    }
	    }	#::
	}
	next if (Qtime::tdiff($resp{$sncl}{qend},$resp{$sncl}{qstart}) <= 0);
	next if (Qtime::tdiff($resp{$sncl}{qstart},$rec->{qwavend}) > 0);
	next if (Qtime::tdiff($resp{$sncl}{qend},$rec->{qwavstart}) < 0);
	$rec->{qreqstart} = $resp{$sncl}{qstart} 
	    if (defined($rec->{qreqstart}) && Qtime::tdiff($resp{$sncl}{qstart},$rec->{qreqstart}) > 0);
	$rec->{qreqend} = $resp{$sncl}{qend} 
	    if (defined($rec->{qreqend}) && Qtime::tdiff($rec->{qreqend},$resp{$sncl}{qend}) > 0);
	next if (defined($rec->{qreqstart}) && Qtime::tdiff($rec->{qreqstart},$rec->{qwavend}) > 0);
	next if (defined($rec->{qreqend}) && Qtime::tdiff($rec->{qwavstart},$rec->{qreqend}) >= 0);
	push (@dbout,$rec);
    }
    printf ("Available responses reduce %d requests to %d requests\n", scalar(@dbin), scalar(@dbout)) if ($opt_d & $DEBUG_SQL);
    return (@dbout);
} 
	    
######################################################################
#   init_MaxWaveLen:
#	Initialize MaxWaveLen hash for each network from database.
#   Return:	0 on success, -1 on failure.
######################################################################
sub init_MaxWaveLen
{
    my($table) = "MaxWaveLength";
    my($sql) = "SELECT DISTINCT net, length from $table";
    eval {
	my($sth) = $dbh->prepare($sql);
	$sth->execute;
	while (my ($net,$maxlen) = $sth->fetchrow_array) {
	    # Initialize global hash.
	    $MaxWaveLen{$net} = $maxlen;
	}
    };
    if ($@) {
	warn ("Error: querying $table table: $@");
	return(-1);
    }
    return(0);
}

######################################################################
#   get_maxwavelens_for:
#	Retrieve maxwavelens for the (optionally wildcarded) network.
#   Return:	array where each element is:
#		maxwavelen:'net1',...,'netN'
#		Note that the net codes in the string are single-quoted.		
######################################################################
sub get_maxwavelens_for
{
    my($netstring) = @_;
    my($netperl) = (defined $netstring) ? $netstring : "*";
    $netperl =~ tr/%_/\*\./;
    my($netstr) = $netperl;
    $netperl =~ s/\*/\.\*/g;
    my(%netsoflen);
    my($found_a_match) = 0;
    for my $net (sort keys %MaxWaveLen) {
	next if (! ($net =~ m/^$netperl$/));
	# Found a match.  Collect its maxwavelen.
	my($len) = $MaxWaveLen{$net};
	$netsoflen{$len} = ($netsoflen{$len} eq "") 
		? "\'$net\'" 
		: join(",",$netsoflen{$len},"\'$net\'");
	$found_a_match = 1;
    }
    # Provide default if no match to known networks.
    $netsoflen{$WF_Window_default} ="\'$netstr\'"
	unless (scalar(keys(%netsoflen)) > 0);
    my(@list);
    for my $len (sort {$a <=> $b} keys %netsoflen) {
	push (@list, join(":",$len,$netsoflen{$len}));
    }
    return (@list);
}


######################################################################
#   db_location:
#	Retrieve location string suitable for database query.
######################################################################
sub db_location
{
    my($location) = @_;
    my($loc) = $location;
    $loc = "  " if ($loc eq "");
    return ($loc);
}

######################################################################
#   print_location:
#	Retrieve location string suitable for printing.
######################################################################
sub print_location 
{
    my($location) = @_;
    my($loc) = $location;    
    $loc = "" if ($loc eq "  ");
    return ($loc);
}

######################################################################
#
# Reference info.
#
######################################################################
# qinfo record: contains one or more of the following fields.
#	If start or end time is present, all time field must be present
# 	rec->{net}		SEED network	(comma-delimited list)
# 	rec->{stat}		SEED station	(comma-delimited list)
#	rec->{chan}		SEED channel	(comma-delimited list)
#	rec->{loc}		SEED location	(comma-delimited list)
#	rec->{wtype}		Waveform type	(waveform type)
#	rec->{evids}		eventids	(comma-delimited list)
#	rec->{reqstart}		start time	(printable time)
#	rec->{qreqstart}	start time	(Qtime object)
#	rec->{treqstart}	start time	(true_epoch)
#	rec->{reqend}		end time	(printable time)
#	rec->{qreqend}		end time	(Qtime object)
#	rec->{treqend}		end time	(true_epoch)
#
# dbinfo record: contains one or more of the following fields.
#	If start or end time is present, all time field must be present
# 	rec->{net}		SEED network	(comma-delimited list)
# 	rec->{stat}		SEED station	(comma-delimited list)
#	rec->{chan}		SEED channel	(comma-delimited list)
#	rec->{loc}		SEED location	(comma-delimited list)
#	rec->{evids}		eventids	(comma-delimited list)
#	rec->{reqstart}		start time	(printable time)
#	rec->{qreqstart}	start time	(Qtime object)
#	rec->{treqstart}	start time	(true_epoch)
#	rec->{reqend}		end time	(printable time)
#	rec->{qreqend}		end time	(Qtime object)
#	rec->{treqend}		end time	(true_epoch)
#	rec->{wavstart}		waveform start
#	rec->{qwavstart}	waveform start
#	rec->{twavstart}	waveform start
#	rec->{wavend}		waveform end
#	rec->{qwavend}		waveform end
#	rec->{twavend}		waveform end
#	rec->{wtype}		Waveform type	(waveform type)
#	rec->{wavefile}		waveform filename
#	rec->{dir}		waveform dir
#	rec->{fileid}		waveform fileid
#	rec->{nbytes}		waveform nbytes
#	rec->{tfilstart}	file starttime
#	rec->{tfilend}		file endtime
#	rec->{traceoff}		waveform offset in file
#	rec->{tracelen}		waveform length (in bytes)
#	rec->{samprate}		sample rate in samples/second;

######################################################################
# POD request file format: tab-separated fields.
# sta net chan loc req_start req_end mseed_file blockette_dir file_start file_end
# For dataless requests:
#   a.	req_times and file_times can be the same.
#   b.	specify any string for the filename.
######################################################################
