/***********************************************************

File Name :
	summarize.C

Programmer:
	Phil Maechling

Description:
	This takes results from a magnitude calculation and
	summarizes the results. It will generate email and pages
	with the summary information.

Creation Date:
	14 September 1995

Modification History:


Usage Notes:


**********************************************************/

#include <iostream.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "basics.h"
#include "solution.h"
#include "network_mag.h"
#include "summarize.h"
#include "qpager.h"
#include "syserr.h"
#include "page.h"
#include "qpager.h"
#include "tvmag.h"
#include "get_cda_ts.h" // for VBB and LG
#include "locks.h"
#include "moment.h"

const float MAXIMUM_ML=8.0;

int summarize_results(struct solution_type&     sol, 
		      struct station_list_type& stalist)
{

  struct network_mag_summary_type netmags;

  summarize_network_results(stalist,
                            netmags);

  if (netmags.number_of_stations_used > 0)
  {
 
     make_magbid(sol,netmags);

    send_pages(sol,netmags);

    update_event_check_magnitudes(sol,netmags);

    make_summary_report(stalist,sol,netmags);
   
    add_event_to_catalog(sol);

//
//  make_full_report(sol,
//		   stalist);

    cout << "Event ID             : " << sol.event_id << endl;
    cout << "Number of Stations   : " << netmags.number_of_stations_used 
                                                              <<endl;

    cout << "Network Ml Magnitude : " << netmags.network_ml << endl;
    cout << "Network Ml RMS       : " << netmags.ml_rms << endl;
    cout << "Network Me Magnitude : " << netmags.network_me << endl;
    cout << "Network Me RMS       : " << netmags.me_rms << endl;

  }

  return(SUCCESS);
}

void add_event_to_catalog(struct solution_type& sol)
{
  char sysline[220];
  char tempfil[20];

  sprintf(tempfil,"%d.email",sol.event_id);

  sprintf(sysline,"cat %s >> %s",tempfil,TVMAG_CATALOG);
 
  system(sysline);

  return;

}


void update_event_check_magnitudes(struct solution_type& sol,
		                   struct network_mag_summary_type& netmags)
{

  FILE* fp;
  char eventid[20];
  char longtempbuf[120];
  int res;

  sprintf(eventid,"%d",sol.event_id);
  strcpy(longtempbuf,TVMAG_RESULTS_DIRECTORY);
  strcat(longtempbuf,eventid);

  res = lock(eventid);

  if (res < 0)
  {
   cout << "Unable to lock event file : " << eventid << endl;
  }


  fp = fopen(longtempbuf,"w");

  if (fp == NULL)
  {
     cout << "Error opening the tvmag event directory" << endl;
     if (res < 0)
     {
       return;
     }
     else
     {
       unlock(eventid);
       return;
     }
  } 

  fprintf(fp,
    " %d %d %d %d %d %d %7.4f %9.4f %5.1f %d %4.1f %d %4.2f %4.1f %d %4.2f",
   sol.year,sol.month,sol.day,sol.hour,sol.minute,(int)sol.seconds,
   sol.latitude,sol.longitude,sol.depth,sol.event_id,
   netmags.network_ml,netmags.number_of_stations_used,netmags.ml_rms,
   netmags.network_me,netmags.number_of_stations_used,netmags.me_rms);

  fclose(fp);

  if (res >= 0)
  {
    unlock(eventid);
  }
  return;
}



void make_magbid(struct solution_type& sol,
                 struct network_mag_summary_type& netmags)
{
  char sysline[220];
  char mlfile[20];
  char mefile[20];
  FILE* fp; 

  sprintf(mlfile,"%d",sol.event_id);
  strcat(mlfile,".ml");

  fp = fopen(mlfile,"w");
  if (fp == NULL)
  {

    syserr("Error opening magbid output file");
    return;
  }
  
  fprintf(fp,"MAGBID %d %4.1f %4.2f %d Ml TERRASCOPE ENGYMAG F\n",
    sol.event_id,netmags.network_ml,netmags.ml_rms,
    netmags.number_of_stations_used);

  fclose(fp);

  sprintf(sysline,
   "rcp %s phil@cajon:cajon\\$dua0:\\[remote.cajon\\]remote.com",mlfile);

  system(sysline);

  sprintf(mefile,"%d",sol.event_id);
  strcat(mefile,".me");

  fp = fopen(mefile,"w");
  if (fp == NULL)
  {
    syserr("Error opening magbid output file");
    return;
  }
  
  fprintf(fp,"MAGBID %d %4.1f %4.2f %d Me TERRASCOPE ENGYMAG F\n",
    sol.event_id,netmags.network_me,netmags.me_rms,
    netmags.number_of_stations_used);

  fclose(fp);

  sprintf(sysline,
   "rcp %s phil@cajon:cajon\\$dua0:\\[remote.cajon\\]remote.com",mefile);

  system(sysline);

  return;

}


void make_summary_report(struct station_list_type& stalist,
 			 struct solution_type& sol,
			 struct network_mag_summary_type& netmags)
{
  FILE *fp;
  char flname[80];
  char sysline[220];
  extern float minpagemag;

  sprintf(flname,"%d",sol.event_id);
  strcat(flname,".email");
 
  fp = fopen(flname,"w");
  if (fp == NULL) 
  {

    syserr("Error opening email summary file");
    return;
  }

  fprintf(fp,"----------------------------------------------------------------------\n");
  fprintf(fp," Caltech/USGS Digital Seismic Network Earthquake Information\n");
  fprintf(fp,"----------------------------------------------------------------------\n");
  fprintf(fp,"\n");
  fprintf(fp,"ISAIAH Event ID :   %d\n", sol.event_id);
  fprintf(fp,"Origin Time     :  %4.4d/%2.2d/%2.2d  %2.2d:%2.2d:%4.1f (UTC) \n",
    sol.year,sol.month,sol.day,sol.hour,sol.minute,sol.seconds);
  fprintf(fp,"Latitude        : %7.2f  Longitude : %8.2f  Depth : %6.2f\n",
    sol.latitude,
    sol.longitude,
    sol.depth);
  fprintf(fp,"Magnitude       : %5.2f\n",
    sol.magnitude);


  fprintf(fp,"----------------------------------------------------------------------\n");

  fprintf(fp,"Summary of Digital Network Magnitude Calculations : \n");
  fprintf(fp,"\n");

  fprintf(fp,"Network Ml Magnitude    : %4.1f  RMS : %4.2f\n", 
    netmags.network_ml,
    netmags.ml_rms);

  fprintf(fp,"Network Me Magnitude    : %4.1f  RMS : %4.2f\n",
    netmags.network_me,
    netmags.me_rms);

  fprintf(fp,
    "Number of Stations Used :  %d\n",netmags.number_of_stations_used);

  fprintf(fp,"----------------------------------------------------------------------\n");
  fprintf(fp,"Detailed Station Results :\n");
  fprintf(fp,"\n");
  fprintf(fp,
    " Stn   D(km)    Ml   Me     Z(Ml  Sec)   N(Ml  Sec)   E(Ml  Sec)  Strm\n");
  fprintf(fp,"----------------------------------------------------------------------\n");
  for (int i = 0; i< netmags.number_of_stations_used ; i++)
  {
    int not_found = TRUE;
    float thisdis = 0.0;
    float zml = 0.0;
    float nml = 0.0;
    float eml = 0.0;
    int thisztime = 0;
    int thisntime = 0;
    int thisetime = 0;
    int res;

    while(not_found)
    {
      for (int x=0;x < stalist.number_of_stations;x++)
      {

        res =  strcmp(stalist.station_data[x].input_data.station_name,
                 netmags.station_name[i]);

        if (res == 0)
        {
          thisdis = stalist.station_data[x].output_data.distance_to_event;

          switch(netmags.sensor_type[i])
          {
            case VBB:
            { 
              if (stalist.station_data[x].output_data.vbb_rate != 0)
              {
                zml = stalist.station_data[x].
                  output_data.vbb_mlz;

                thisztime = stalist.station_data[x].
                  output_data.z_vbb_samples_used/
		  stalist.station_data[x].output_data.vbb_rate;

                nml = stalist.station_data[x].
                  output_data.vbb_mln;

                thisntime = stalist.station_data[x].
                  output_data.n_vbb_samples_used/
		  stalist.station_data[x].output_data.vbb_rate;

                eml = stalist.station_data[x].
                  output_data.vbb_mle;

                thisetime = stalist.station_data[x].
                  output_data.e_vbb_samples_used/
		  stalist.station_data[x].output_data.vbb_rate;
              }
              else
              {
	        cout << "Found rate of zero vbb samples per seconds" <<endl;
              }
	      break;
            }
            case LG:
            {
              if (stalist.station_data[x].output_data.lg_rate != 0)
              {

                zml = stalist.station_data[x].
                  output_data.lg_mlz;

                thisztime = stalist.station_data[x].
                  output_data.z_lg_samples_used/
	          stalist.station_data[x].output_data.lg_rate;

                nml = stalist.station_data[x].
                  output_data.lg_mln;

                thisntime = stalist.station_data[x].
                  output_data.n_lg_samples_used/
	          stalist.station_data[x].output_data.lg_rate;

                eml = stalist.station_data[x].
                  output_data.lg_mle;

                thisetime = stalist.station_data[x].
                  output_data.e_lg_samples_used/
	          stalist.station_data[x].output_data.lg_rate;
              }
              else
              {
	        cout << "Found rate of zero lg samples per seconds" <<endl;
              }
              break;
            }
            default:
            {
              cout << "Unknown sensor type in summary report" << endl;
              thisdis = 0.0;
              thisztime = 0;
              thisntime = 0;
              thisetime = 0;
	      zml = 0.0;
	      nml = 0.0;
	      eml = 0.0;
            }
          } // end if switch
          not_found = FALSE;
        }// end if comparision of station name matches
      } // end for all stations in list of all station

      if (not_found)
      {
        cout << "Could not find station distance in staion list" << endl;
        thisdis = 0.0;
        not_found = FALSE;
      }
    }

    fprintf(fp,
      " %-4s  %5.1f  %4.1f %4.1f     %4.1f  %3d    %4.1f  %3d    %4.1f  %3d",
              netmags.station_name[i],
	      thisdis,
	      netmags.ml[i],
     	      netmags.me[i],
	      zml,
              thisztime,
	      nml,
              thisntime,
	      eml,
 	      thisetime);

    if (netmags.sensor_type[i] == VBB)
    {
      fprintf(fp,"   VBB\n");
    }
    else
    {
      fprintf(fp,"    LG\n");
    }
  }

  fprintf(fp,"----------------------------------------------------------------------\n");
  if (netmags.network_ml > MAXIMUM_ML)
  {
    fprintf(fp,"ML near saturation. Paging Me Magnitude.\n");
    fprintf(fp,"----------------------------------------------------------------------\n");
  }

  fclose(fp);

  sprintf(sysline,"/usr/bin/mailx tmag_all < %s\n",
	flname);

  system(sysline);

  if ((netmags.network_ml >= minpagemag) || (sol.magnitude >= minpagemag))
  {
    sprintf(sysline,"/usr/bin/mailx tmag3_5 < %s\n",
	flname);
    system(sysline);
  }

  return;
}



void send_pages(struct solution_type& sol,
		struct network_mag_summary_type& netmags)
{

  extern float minpagemag;
  char eventid[20];
  char pagefilename[30];
  char sumfilename[30];
  char compfilename[30];

  sprintf(eventid,"%d",sol.event_id);

  if ((netmags.network_ml >= minpagemag) || (sol.magnitude >= minpagemag))
  {
    char beltpage[LENGTH_OF_PAGE+1];
    char comppage[LENGTH_OF_PAGE+1];
    char beltsum[LENGTH_OF_PAGE+1];  
    qpager_data_type qpage;

    //
    // Create and send pages
    //

    strcpy(qpage.message_type,"Event");
    qpage.event_id    = sol.event_id;
    strcpy(qpage.data_source,"Pasadena");
    qpage.event_version = sol.solution_version;
    qpage.year = sol.year;
    qpage.month = sol.month;
    qpage.day   = sol.day;
    qpage.hour  = sol.hour;
    qpage.minute = sol.minute;
    qpage.second = sol.seconds;
    qpage.latitude = sol.latitude;
    qpage.longitude = sol.longitude;
    qpage.depth     = (-1.0) * sol.depth;
    qpage.stations_in_location = sol.solution_phases;
    qpage.phases_in_location   = sol.solution_phases;
    qpage.closest_station = 0.0;
    qpage.time_rms = sol.solution_rms;
    qpage.horizontal_rms = sol.solution_rms;
    qpage.vertical_rms   = sol.solution_rms;
    qpage.az_gap         = 0.0;
    strcpy(qpage.method_of_location,"ISAIAH"); 
    qpage.stations_in_mag = netmags.number_of_stations_used;


    if (netmags.network_ml <= MAXIMUM_ML)
    {
      qpage.magnitude = netmags.network_ml;
      strcpy(qpage.mag_type,"Ml");
      qpage.magnitude_rms   = netmags.ml_rms;
    }
    else
    {
      qpage.magnitude = netmags.network_me;
      strcpy(qpage.mag_type,"Me");
      qpage.magnitude_rms   = netmags.me_rms;
    }


    qevent_to_belt_page(&qpage,beltpage,"T"); 

    qevent_to_computer_page(&qpage,comppage);

    netmag_to_belt_summary_page(netmags,sol.event_id,beltsum);

    //
    // First the belt page
    //

    strcpy(pagefilename,eventid);
    strcat(pagefilename,".page");

    FILE *fp;
    fp = fopen(pagefilename,"w");
    if (fp == NULL)
    {
      syserr("Error opening beltpage");
      return;
    }

    fprintf(fp,"%s",beltpage);

    fclose(fp);

    cout << beltpage << endl;

    //
    // Next the belt summary page
    //

    strcpy(sumfilename,eventid);
    strcat(sumfilename,".sum");

    fp = fopen(sumfilename,"w");
    if (fp == NULL)
    {
      syserr("Error opening beltpage summary");
      return;
    }

    fprintf(fp,"%s",beltsum);

    fclose(fp);

    cout << beltsum << endl;

    //
    // Now the computer page
    //

    strcpy(compfilename,eventid);
    strcat(compfilename,".comp");

    fp = fopen(compfilename,"w");
    if (fp == NULL)
    {
      syserr("Error opening computer page");
      return;
    }

    fprintf(fp,"%s",comppage);

    fclose(fp);

    send_pages("tv_belt_group",pagefilename);
    send_pages("tv_belt_group",sumfilename);
   
    //
    //  send_pages("tv_comp_group","tvcomppage.dat");
    //

    //
    // Use VAX paging on large events
    //

    if(netmags.network_ml > 5.0)
    {
      char sysline[220];
      
      fp = fopen("rpagefile.dat","w");
      if (fp == NULL)
      {
        syserr("Error opening rpagefile in summarize");
        return;
      }

      fprintf(fp,"rpage group \"%s\"\n",beltpage);
      fclose(fp);

      sprintf(sysline,
      "rcp rpagefile.dat phil@cajon:cajon\\$dua0:\\[remote.cajon\\]remote.com");

      system(sysline);

      //
      // Send summary file
      //

      fp = fopen("rsumfile.dat","w");
      if (fp == NULL)
      {
        syserr("Error opening rsumfile in summarize");
        return;
      }

      fprintf(fp,"rpage group \"%s\"\n",beltsum);
      fclose(fp);

      sprintf(sysline,
      "rcp rsumfile.dat phil@cajon:cajon\\$dua0:\\[remote.cajon\\]remote.com");

      system(sysline);
    } // close if mag > 5.0
  } // close if more than 0 stations used
  return;
}


void netmag_to_belt_summary_page(struct network_mag_summary_type& netmags,
			         int  event_id,
				 char* page)
{

  sprintf(page,
   "EVENT: %d  DML=%3.1f   DME=%3.1f   NST=%-3d MLRMS=%4.2f MERMS=%.2f",
   event_id,
   netmags.network_ml,netmags.network_me,
   netmags.number_of_stations_used,netmags.ml_rms,
   netmags.me_rms);

  return;
}



void summarize_network_results(struct station_list_type& stalist,
		               struct network_mag_summary_type& netmags)
{

  int sensor_to_use;

  //
  // No stations used yet
  //

  netmags.number_of_stations_used = 0;

  for(int i = 0;i<stalist.number_of_stations;i++)
  {
    if ( use_station(stalist.station_data[i]))
    {
      sensor_to_use = pick_vbb_or_lg(stalist.station_data[i]);

      switch(sensor_to_use)
      {
        case VBB:
        { 
          strcpy(netmags.station_name[netmags.number_of_stations_used],
                 stalist.station_data[i].input_data.station_name);

          netmags.sensor_type[netmags.number_of_stations_used] = VBB;

          netmags.ml[netmags.number_of_stations_used] = 
                  stalist.station_data[i].output_data.vbb_ml;
	   
          netmags.me[netmags.number_of_stations_used] = 
                  stalist.station_data[i].output_data.vbb_me;

          netmags.number_of_stations_used++;

          break;
        }
        case LG:
        {
          strcpy(netmags.station_name[netmags.number_of_stations_used],
                 stalist.station_data[i].input_data.station_name);

          netmags.sensor_type[netmags.number_of_stations_used] = LG;

          netmags.ml[netmags.number_of_stations_used] =
                  stalist.station_data[i].output_data.lg_ml;
	   
          netmags.me[netmags.number_of_stations_used] =
                  stalist.station_data[i].output_data.lg_me;

          netmags.number_of_stations_used++;

          break;
        } 
        default:
        {
          cout << "Unknown sensor type in summarize routine" << endl;
        }
      } // end switch on sensor type
    } // end use station test
  } // end For every station in station list

 
  //
  // Summarize the collected results
  //

  if (netmags.number_of_stations_used == 0)
  {
    return;
  }
  else if (netmags.number_of_stations_used == 1)
  {
    netmags.network_ml = netmags.ml[0];
    netmags.ml_rms = 0.0;
    netmags.network_me = netmags.me[0];
    netmags.me_rms = 0.0;
  }
  else // calculate the average and the rms
  {
     //
     // Put in guards so calculation of very small 
     // average values is not attempted.
     //

     float msum = 0.0;
     for(int pi= 0; pi< netmags.number_of_stations_used;pi++)
     {
       msum += netmags.ml[pi]; 
     }

     if (msum < (float) netmags.number_of_stations_used)
     {
       netmags.network_ml = 1.0;
       netmags.ml_rms = 0.0;
     }  
     else
     {
       moment(netmags.ml,
              netmags.number_of_stations_used,
              &netmags.network_ml,
              &netmags.ml_rms);
     }

     //
     // Test me results also for low numbers before averaging them
     //

     msum = 0.0;
     for(pi=0; pi<netmags.number_of_stations_used;pi++)
     {
       msum += netmags.me[pi]; 
     }

     if (msum < (float) netmags.number_of_stations_used)
     {
       netmags.network_ml = 1.0;
       netmags.ml_rms = 0.0;
     }
     else
     {
       moment(netmags.me,
              netmags.number_of_stations_used,
              &netmags.network_me,
              &netmags.me_rms);
     }
  }
  return;
}	


void update_event_check_locations(struct solution_type& sol)
{
  char tempbuf[10];
  char longtempbuf[120];
  int res;
 
  FILE* fp;
  
  strcpy(longtempbuf,ISAIAH_EVENT_DIRECTORY);
  sprintf(tempbuf,"%d",sol.event_id);
  strcat(longtempbuf,tempbuf);

  res = lock(tempbuf);

  if (res < 0)
  {
   cout << "Unable to lock event file : " << tempbuf << endl;
  }


  fp = fopen(longtempbuf,"w");

  if (fp == NULL)
  {
     cout << "Error opening the isaiah event directory" << endl;
     if (res < 0)
     {
       return;
     }
     else
     {
       unlock(tempbuf);
       return;
     }
  } 

  fprintf(fp," %d %d %d %d %d %d %7.4f %9.4f %5.1f %3.1f %d",
   sol.year,sol.month,sol.day,sol.hour,sol.minute,(int)sol.seconds,
   sol.latitude,sol.longitude,sol.depth,sol.magnitude,sol.event_id);

  fclose(fp);

  if (res >= 0)
  {
    unlock(tempbuf);
  }
  return;
}

int pick_vbb_or_lg(struct station_data_type& sta)
{
   //
   // This routine should only be called if either
   // lg is useable or vbb is useable
   //

  // 
  // Normal processing. VBB good, not clipped
  //

  if ( (sta.status.vbb_ran) && 
       (!sta.status.vbb_fatal_error) &&
       (!sta.status.vbb_clipped) )
  {
    return(VBB); 
  } 

  //
  // vbb good, but clipped, and lg ok
  //
  if ( (sta.status.vbb_ran) &&
       (!sta.status.vbb_fatal_error) &&
       (sta.status.vbb_clipped) &&
       (sta.status.lg_ran) &&
       (!sta.status.lg_fatal_error) )
  {
    return(LG);
  }

  //
  // vbb good, but clipped, but lg no good.
  // Problems with LG true by virtue of getting here
  // Either it didn't run, or it had a fatal error 
  //
  if ( (sta.status.vbb_ran) &&
       (!sta.status.vbb_fatal_error) )
  {
    return(VBB);
  }

  //
  // some problem with VBB. Try to use lg
  //
  if ( (sta.status.lg_ran) &&
       (!sta.status.lg_fatal_error) )
  {
    return(LG);
  }

  //
  // All cases of "good stations" should have been checked
  // Flag error and return vbb
  //
  cout << "CHECK : should not return via this path " << endl;
  cout << "Used the last chance selection in pick_vbb_of_lg routine" << endl;
  return(VBB);

}


int use_station(struct station_data_type& sta)
{
  if ( ((sta.status.vbb_ran) && (!sta.status.vbb_fatal_error)) ||
       ((sta.status.lg_ran)  && (!sta.status.lg_fatal_error)) )
  {
    return(TRUE);
  }
  else
  {
    return(FALSE);
  }
}


