/***********************************************************

File Name :
	network_mag.C

Programmer:
	Phil Maechling

Description:
	This is the routine that accepts a solution and a station
	list. It calculates the resulting network magnitude,
	then distributes ther results.
	It returns only a SUCCESS or FAILURE result to the caller.

Creation Date:
	11 September 1995

Modification History:


Usage Notes:


**********************************************************/
#include <iostream.h>
#include <sys/types.h>
#include <math.h>
#include <sunmath.h>
#include <unistd.h>
#include <string.h>

#include "basics.h"
#include "qtime.h"
#include "get_cda_ts.h"
#include "network_mag.h"
#include "great.h"
#include "solution.h"
#include "tt.h"
#include "stream_mag.h"

const int MAXIMUM_STATIONS_TO_USE_IN_MAGNITUDE = 20;
const int MAX_SAMPLES_IN_REQUEST = 20000; // 3 minutes of 100 sps data
const int TELEMETRY_DELAY        = 10;
const int S_WINDOW               = 25;


int network_magnitude(
		struct solution_type&     sol,
		struct station_list_type& stalist)
{


  /* Data Request Results */

  int z_array[MAX_SAMPLES_IN_REQUEST];
  INT_TIME z_time;
  int z_samples;

  int n_array[MAX_SAMPLES_IN_REQUEST];
  INT_TIME n_time;
  int n_samples;

  int e_array[MAX_SAMPLES_IN_REQUEST];
  INT_TIME e_time;
  int e_samples;
  
  int syr[3];
  int sday[3];
  int shr[3];
  int smin[3];
  int ssec[3];
  int spts[3];
  
  int fifteen_seconds; 

/* Initialize the results table to  zero */

  initialize_results_list(stalist);
  
/* Create event information structure */

  EXT_TIME eq_time;
  int day_in_year;
  double   dsec;
  struct timeval cstime,cftime;
  double timediff;



//
// Find the starting time for the calculation 
//

  gettimeofday(&cstime,NULL);


/* Convert the Event month day into year and day */
 
   day_in_year = mdy_to_doy(sol.month,
                            sol.day,
                            sol.year);

   eq_time.year   = sol.year;
   eq_time.doy    = day_in_year;
   eq_time.month  = sol.month;
   eq_time.day    = sol.day;
   eq_time.hour   = sol.hour;
   eq_time.minute = sol.minute;
   dsec           = (double) sol.seconds;
   eq_time.second = nint(dsec);
   eq_time.ticks  = 0;
 
   INT_TIME ieq_time;
   ieq_time = ext_to_int(eq_time);

  //
  // Find the distance from each station to event 
  //
  // Define some placeholder values that are returned from the
  // great routine
  //

  float distd;
  float gc,az,baz;

  for (int i=0;i<stalist.number_of_stations;i++)
  {
    great_(&stalist.station_data[i].input_data.latitude,
	   &stalist.station_data[i].input_data.longitude,
           &sol.latitude,
	   &sol.longitude,
           &stalist.station_data[i].output_data.distance_to_event,
           &distd,
	   &gc,&az,&baz);

    if (stalist.station_data[i].output_data.distance_to_event < 0)
    {
      cout << 
        "Great distance calculation returned distance less than zero" << endl;
      return(FAILURE);
    }
  }

//
// Sort stations by distance to event with closest station first on list
// Insertion Sort into ascending distance from event
//

  struct station_data_type stat_dat;
  int j;

  for (j=1;j<stalist.number_of_stations;j++)
  {
    stat_dat = stalist.station_data[j];
    i=j-1; 
    while( (i >= 0) && (stalist.station_data[i].output_data.distance_to_event 
      > stat_dat.output_data.distance_to_event) )
    {
      stalist.station_data[i+1]=stalist.station_data[i];
      i--;
    }
    stalist.station_data[i+1]=stat_dat;
  }  


//
// Calculate the S-wave arrival time at each station
//
//

  float p1,p2,s1,s2;

  for (i=0;i<stalist.number_of_stations;i++)
  {
    travel_t_(&stalist.station_data[i].output_data.distance_to_event,
              &p1,&p2,&s1,&s2);

    if (s1 >= s2)
    {
      stalist.station_data[i].output_data.seconds_to_s_arrival =
	(int) s1;
    }
    else
    {
      stalist.station_data[i].output_data.seconds_to_s_arrival =
	(int) s2;
    }   
  }
  

//
// Main loop that calculates mag for each station 
//

  int continue_processing_network;
  struct timeval curtime;
  int res;
  INT_TIME icurtime;
  INT_TIME ttime;
  EXT_TIME extime;
  int waiting_for_s_wave;
  int samples_per_second;
  int status;
  int rferror,rcerror,rclip,rupdate,rzerr,rnerr,reerr;
  float rmlz,rmln,rmle,ramlav,renmag;
  int sensor_type;


  //
  // Initialize the counters and status flags
  //

  continue_processing_network = TRUE;
  int stations_reporting = 0;
  int stacount = 0;                      // station being processed
  while(continue_processing_network)
  {
    //
    // Check to see if we should quit processing network
    // Stop processing when
    // (1) Stations are more than 300 km away
    // (2) All availalbe stations are processed
    // (3) A maxmimun number of stations to use is reached
    //

    if(stalist.station_data[stacount].output_data.distance_to_event > 300.0)
    {
      stations_reporting = stacount;
      continue_processing_network = FALSE;
      continue;
    }
    else if (stacount >= stalist.number_of_stations)
    {
      stations_reporting = stacount;
      continue_processing_network = FALSE;
      continue;
    }
    else if( stacount >= MAXIMUM_STATIONS_TO_USE_IN_MAGNITUDE)
    {
      stations_reporting = stacount;
      continue_processing_network = FALSE;
      continue;
    }

    //
    // The input stations are sorted by distance to event.
    // Check if the S wave has arrived, then calculate the magnitude
    // for the station if it has.
    //

    waiting_for_s_wave = TRUE;

    while(waiting_for_s_wave)
    {
      res = gettimeofday(&curtime,NULL);
      icurtime = int_time_from_timeval(&curtime);     

      //
      // Add the delays onto the event time and wait until the
      // s wave has arrived
      // 

      ttime    = add_time(ieq_time,
                       (stalist.station_data[i].
                          output_data.seconds_to_s_arrival 
                           + TELEMETRY_DELAY + S_WINDOW),
                        0);

      if ((icurtime.year <= ttime.year ) &&
          (icurtime.second <= ttime.second))
      {
        sleep(1);       
      }
      else
      {
        waiting_for_s_wave = FALSE;
      }
    }

    //
    // It is time to get the waveform from memory. Start with vbb
    //

    sensor_type = VBB;

    int additional_sensors_to_read = TRUE;

    while(additional_sensors_to_read)
    {
      status = get_cda_time_series(ieq_time,
   				 stalist.
                                   station_data[stacount].
				   input_data.station_name,
				 sensor_type,
                                 &samples_per_second,
				 z_array,
				 &z_time,
				 &z_samples,
				 n_array,
				 &n_time,
				 &n_samples,
				 e_array,
				 &e_time,
				 &e_samples);

      if (status == FAILURE)
      {
        printf("Error returned from extract data routine \n");
        if (sensor_type == VBB)
        {
          sensor_type = LG;
        }
        else
        {
          additional_sensors_to_read = FALSE;
        } 
        continue;
      }

      // 
      // Fill in information returned from data extract routine
      //
       
      
      switch (sensor_type)
      {
        case VBB:
        {
          stalist.station_data[stacount].output_data.vbb_rate = 
            samples_per_second;
          stalist.station_data[stacount].output_data.z_vbb_samples_used =
	    z_samples;
          stalist.station_data[stacount].output_data.n_vbb_samples_used =
	    n_samples;
          stalist.station_data[stacount].output_data.e_vbb_samples_used =
	    e_samples;

           break;
        }
	case LG:
        {
          stalist.station_data[stacount].output_data.lg_rate = 
            samples_per_second;
	  stalist.station_data[stacount].output_data.z_lg_samples_used =
	    z_samples;
	  stalist.station_data[stacount].output_data.n_lg_samples_used =
	    n_samples;
	  stalist.station_data[stacount].output_data.e_lg_samples_used =
	    e_samples;

          break;
        }
        default:
        {
         cout << "Error in network_mag. Unknown sensor type : " 
           << sensor_type << endl;
         sensor_type = VBB;
         continue;
        }
      }
    
      //
      // Report on data extraction results
      //

      fifteen_seconds = samples_per_second * 15;
      if ( (z_samples < fifteen_seconds) || (n_samples < fifteen_seconds) ||
	   (e_samples < fifteen_seconds) )
      {
        cout << "Less than 15 seconds of data for : " <<
          stalist.station_data[stacount].input_data.station_name;

        switch (sensor_type)
        {
          case VBB:
          {
            cout << " VBB data" << endl;
            break;
          }
          case LG:
          {
            cout << " LG data" << endl;
            break;
          }
          default:
          {
            cout << "Unknown sensor_type : " << sensor_type << endl;
          }
        }

      }

      spts[0] = z_samples;
      spts[1] = n_samples;
      spts[2] = e_samples;

      extime = int_to_ext(z_time);
      syr[0] = extime.year;
      sday[0]= extime.doy;
      shr[0] = extime.hour;
      smin[0] = extime.minute;
      ssec[0] = extime.second;

      extime = int_to_ext(n_time);
      syr[1] = extime.year;
      sday[1]= extime.doy;
      shr[1] = extime.hour;
      smin[1] = extime.minute;
      ssec[1] = extime.second;

      extime = int_to_ext(e_time);
      syr[2] = extime.year;
      sday[2]= extime.doy;
      shr[2] = extime.hour;
      smin[2] = extime.minute;
      ssec[2] = extime.second;

/*
* 
*   Call a Fortran subroutine that returns the magnitude 
*   and the distance from station to event.
*   Pass all variables by reference. 
*
*/

      stream_magnitude_(
	&stalist.station_data[stacount].input_data.v2c,
	&stalist.station_data[stacount].input_data.amlcor,
	&stalist.station_data[stacount].input_data.vbbgain,
	&stalist.station_data[stacount].input_data.lggain,
        &eq_time.year,
        &day_in_year,
        &eq_time.hour,
        &eq_time.minute,
        &eq_time.second,
        &stalist.station_data[stacount].output_data.distance_to_event,
        &sol.depth,
        &sensor_type,	
        &samples_per_second,
        z_array,
        n_array,
        e_array,
        syr,sday,shr,smin,ssec,
        spts,
        &rferror,
        &rcerror,
        &rclip,
        &rupdate,
        &rzerr,&rnerr,&reerr, 
        &rmlz,&rmln,&rmle, 
        &ramlav,&renmag);

      // 
      // Fill in the status structure
      //
    

      switch (sensor_type)
      {
        case VBB:
        {
          stalist.station_data[stacount].status.vbb_ran = TRUE;
          stalist.station_data[stacount].status.vbb_fatal_error = rferror;
          stalist.station_data[stacount].status.vbb_component_error = rcerror;
          stalist.station_data[stacount].status.vbb_clipped = rclip;
          stalist.station_data[stacount].status.vbb_preliminary   = rupdate;
          stalist.station_data[stacount].status.vbb_z_err   = rzerr;
          stalist.station_data[stacount].status.vbb_n_err   = rnerr;
          stalist.station_data[stacount].status.vbb_e_err   = reerr;

          //
          // Fill in the results structure
          //

          stalist.station_data[stacount].output_data.vbb_mlz = rmlz;
          stalist.station_data[stacount].output_data.vbb_mln = rmln;
          stalist.station_data[stacount].output_data.vbb_mle = rmle;
          stalist.station_data[stacount].output_data.vbb_ml  = ramlav,
          stalist.station_data[stacount].output_data.vbb_me  = renmag;

          break;
        }
        case LG:
        {
          stalist.station_data[stacount].status.lg_ran = TRUE;
          stalist.station_data[stacount].status.lg_fatal_error = rferror;
          stalist.station_data[stacount].status.lg_component_error = rcerror;
          stalist.station_data[stacount].status.lg_clipped = rclip;
          stalist.station_data[stacount].status.lg_preliminary = rupdate;
          stalist.station_data[stacount].status.lg_z_err   = rzerr;
          stalist.station_data[stacount].status.lg_n_err   = rnerr;
          stalist.station_data[stacount].status.lg_e_err   = reerr;

          //
          // Fill in the results structure
          //

          stalist.station_data[stacount].output_data.lg_mlz = rmlz;
          stalist.station_data[stacount].output_data.lg_mln = rmln;
          stalist.station_data[stacount].output_data.lg_mle = rmle;
          stalist.station_data[stacount].output_data.lg_ml  = ramlav,
          stalist.station_data[stacount].output_data.lg_me  = renmag;

          break;
        }
        default:
        {
          cout << "Error in network_mag. Unknown sensor type : " 
            << sensor_type << endl;
          sensor_type = VBB;
          continue;
        }
      } // end of switch statmenttiw

      //
      // Check to see if vbb is clipped. If so process lg data. Else
      //   continue on and process next station.
      //

      switch (sensor_type)
      {
        case VBB:
        {
          if (stalist.station_data[stacount].status.vbb_clipped == TRUE)
          {
	    sensor_type = LG;
          }
          else
          {
            additional_sensors_to_read = FALSE;
          }
          break;
        }
        case LG:
        {
          additional_sensors_to_read = FALSE;
          break;
        }
        default:
        {
          cout << "Error in network_mag. Unknown sensor type : " 
           << sensor_type << endl;
          sensor_type = VBB;
        }
      } // end switch

      continue;
    
    } // end of while additional sensors to read

    stacount++;
   
  } // while more stations to read
  
  cout << "Number of stations processed : " << stations_reporting << endl;

  //
  // Summarize how long it took to calculate the results
  //

  gettimeofday(&cftime,NULL);

  timediff = tdiff(int_time_from_timeval(&cftime),
                   int_time_from_timeval(&cstime));

  cout  << "Event ID : " << sol.event_id << 
    "   Seconds to calculate network magnitude  : " << 
           (int) timediff/10000 << endl;

  timediff = tdiff(int_time_from_timeval(&cftime),
                   ieq_time);

  cout  << "Event ID : " << sol.event_id <<
    "   Seconds since origin time               : " <<
           (int) timediff/10000 << endl;

  return(SUCCESS);
}


void initialize_results_list(struct station_list_type& station_list)
{
  for(int i= 0;i<station_list.number_of_stations;i++)
  {
    station_list.station_data[i].output_data.distance_to_event = 0.0;
    station_list.station_data[i].output_data.seconds_to_s_arrival = 0;
    station_list.station_data[i].output_data.vbb_rate = 0;

    station_list.station_data[i].output_data.z_vbb_samples_used = 0;
    station_list.station_data[i].output_data.vbb_mlz = 0.0;

    station_list.station_data[i].output_data.n_vbb_samples_used = 0;
    station_list.station_data[i].output_data.vbb_mln = 0.0;

    station_list.station_data[i].output_data.e_vbb_samples_used = 0;
    station_list.station_data[i].output_data.vbb_mle = 0.0;

    station_list.station_data[i].output_data.vbb_ml;
    station_list.station_data[i].output_data.vbb_me;

    station_list.station_data[i].output_data.lg_rate = 0;

    station_list.station_data[i].output_data.z_lg_samples_used = 0;
    station_list.station_data[i].output_data.lg_mlz = 0.0;

    station_list.station_data[i].output_data.n_lg_samples_used = 0;
    station_list.station_data[i].output_data.lg_mln = 0.0;

    station_list.station_data[i].output_data.e_lg_samples_used = 0;
    station_list.station_data[i].output_data.lg_mle = 0.0;

    station_list.station_data[i].output_data.lg_ml;
    station_list.station_data[i].output_data.lg_me;


    //
    // Initialize the status section
    //

    station_list.station_data[i].status.vbb_ran             = FALSE;
    station_list.station_data[i].status.vbb_fatal_error     = FALSE;
    station_list.station_data[i].status.vbb_component_error = FALSE;
    station_list.station_data[i].status.vbb_clipped         = FALSE;
    station_list.station_data[i].status.vbb_preliminary     = FALSE;
    station_list.station_data[i].status.vbb_z_err           = FALSE;
    station_list.station_data[i].status.vbb_n_err           = FALSE;
    station_list.station_data[i].status.vbb_e_err           = FALSE;

    station_list.station_data[i].status.lg_ran              = FALSE;
    station_list.station_data[i].status.lg_fatal_error      = FALSE;
    station_list.station_data[i].status.lg_component_error  = FALSE;
    station_list.station_data[i].status.lg_clipped          = FALSE;
    station_list.station_data[i].status.lg_preliminary      = FALSE;
    station_list.station_data[i].status.lg_z_err            = FALSE;
    station_list.station_data[i].status.lg_n_err            = FALSE;
    station_list.station_data[i].status.lg_e_err            = FALSE;

  }
}

