/*
 * file: rddata.c
 * 
 * These routines read (with subroutine getpacket) and decompress data packets
 * using steim decompression.  The samples are time-aligned and held in a
 * circular buffer of size specifed by MAX_POINTS.  Samples are obtained by
 * the routine read_next_sample, which is the controlling routine, i.e.
 * packets are only read when read_next_sample needs them to fulfill the
 * request.  read_next_sample can be used to step backward through the data
 * by setting the number requested in read_next_sample to be less than zero.
 * read_next_sample also gives the time of the first sample it obtains, in
 * seconds since 00:00:00 GMT, Jan. 1, 1970.  The millisecond offset is also
 * obtained (which may be greater than 1000, but is always positive).
 * read_next sample returns the number of samples read, -1 for failure on
 * positive requests, 1 for failure of negative requests.  By linking with
 * the appropriate subroutines, these routines can read packets from disk,
 * shared memory area, UNIX socket, etc.
 * 
 * Note that the following routines must be externally defined (see epackets.h):
 * 
 * void rdsmplerr(int errno, char * errstr) - which is the error routine.  errno
 * is scaled to the severity of the error (>4 is not important, i.e. timing
 * errors).  Errors greater than 32 generally should be fatal.
 * 
 * void get_packet(unsigned char *packet) - reads a packet from source.  See
 * utilshm.c and utilsocket.c for examples of this routine.
 * 
 * 
 * Note that these routines do not care about what packets they are receiving,
 * as long as they are in steim decompression.  There is nothing in these
 * routines specifying station or type of data.
 * 
 * 
 * Compiles under GNU ANSI C
 * 
 * Written by Erik Larson, July-August, 1994
 * Modified by Doug Neuhauser, UC Berkeley, April 1995 for miniSEED
 * 
 */

#define EMPTY 0

#include "epackets.h"
#include "rddata.h"
#include "jsteim.h"
#include <time.h>
#include <sys/time.h>
#include <math.h>
#include <fcntl.h>
#include <memory.h>
#include <string.h>
#include <sunmath.h>

char            errstr[511];
extern int debug;

/* Calls decompression routine to decompress a packet */
int
decompress_data(DATA_HDR *hdr, unsigned char *indata, int *array)
{
    int x0, xn;
    int *diffbuff;
    int numsamp;
    switch (hdr->data_type) {
      case EMPTY:
	fprintf (stdout, "Empty packet all nulls : %d\n:", hdr->data_type);
	numsamp = 0;
	break;
      case STEIM1:
	if ((diffbuff = (int *)malloc(hdr->num_samples * sizeof(int))) == NULL) {
	    fprintf (stderr, "Unable to malloc diffbuff\n");
	    exit(1);
	}
	numsamp = unpack_steim1 ((FRAME *)indata, hdr->num_data_frames * sizeof(FRAME), 
				 hdr->num_samples, hdr->num_samples, 
				 array, diffbuff, &x0, &xn, NULL);
	free (diffbuff);
	break;
      case STEIM2:
	if ((diffbuff = (int *)malloc(hdr->num_samples * sizeof(int))) == NULL) {
	    fprintf (stderr, "Unable to malloc diffbuff\n");
	    exit(1);
	}
	numsamp = unpack_steim2 ((FRAME *)indata, hdr->num_data_frames * sizeof(FRAME), 
				 hdr->num_samples, hdr->num_samples, 
				 array, diffbuff, &x0, &xn, NULL);
	free (diffbuff);
	break;
      default:
	fprintf (stderr, "Unknown datatype: %d\n:", hdr->data_type);
	numsamp = -1;
	break;
    }
    return numsamp;
}

/* Calls get_packet and decompress data and analyzes packet header */
int
read_packet(int *comp, short *comp_num, time_t * stime, int *smsec, int *delt)
{
    static unsigned char *indata = NULL;
    int             samp;
    DATA_HDR	*hdr;

    if (indata == NULL) indata = (unsigned char *)malloc(MAX_BLKSIZE * sizeof(char));
    if (indata == NULL) {
	fprintf (stderr, "Error mallocing indata\n");
	exit(1);
    }
    if ((hdr = get_packet(indata)) == NULL)
      return -1;
    if (strchr("Z1A",hdr->channel_id[2])) {
	*comp_num = CHAN0;
    }
    else if (strchr("N2B",hdr->channel_id[2])) {
	*comp_num = CHAN1;
    }
    else if (strchr("E3C",hdr->channel_id[2])) {
	*comp_num = CHAN2;
    }
    if (statchan[*comp_num][0] == '\0') {
	sprintf(statchan[*comp_num],"%s %s", hdr->station_id, hdr->channel_id);
    }
    *stime = unix_time_from_ext_time (int_to_ext(hdr->begtime));
    *smsec = hdr->begtime.ticks * 1000 / TICKS_PER_SEC;
    if (debug) printf ("packet %s stime=%d smsec=%d\n", hdr->channel_id, *stime, *smsec);

    samp = decompress_data(hdr, ((unsigned char *)&indata[0])+hdr->first_data, comp);
    if (samp != hdr->num_samples) {
	rdsmplerr(errno = 31, "Samp not equal to n_samples\n");
	exit(1);
    }
    if (hdr->sample_rate > 0)
      *delt = nint(1000.0 / hdr->sample_rate);
    else
      *delt = nint(-hdr->sample_rate * 1000.0);
    samp = hdr->num_samples;
    free_data_hdr (hdr);
    return samp;    
}

#define MAX_POINTS 100000

/*  Variables used to implement circular buffer.    */

static int      comparray[NUMCOMPS][MAX_POINTS];	
/* Circular buffer storage for data points. */

static int	curr_point;
/* Index in circular buffer of next data pt */

static int	num_all_have;
/* number of point for which we have all comps. */

static time_t   refsecs;
/* Time (in seconds) of next available point.*/

static int	refmsec,
/* msec offset of reference time.	    */

	        first_start,
/* index of first point ever in circular buffer. ?*/

		valid_pt_max[NUMCOMPS], 
/* index of most recent data in circular buffer.*/

		num_have[NUMCOMPS],
/* number of new data points starting at reftime.*/

		delt;
/* sample interfal (in milliseconds).*/



static BOOLEAN  read_before[NUMCOMPS],  
		first_loop,
/* flag to indicate no wrapping of output from circular buffer.*/

		run_before = FALSE;
/* initialization flag for circular buffer vars. */

static ldiv_t	time_per_buf;
/* timespan of data from most recent data read. */




/* Resets the variables in read_next_sample--called if there is an error */
void reset_vars()
{
    short           i;
    for (i = 0; i < NUMCOMPS; i++) {
	valid_pt_max[i] = -1;
	read_before[i] = FALSE;
	num_have[i] = 0;
    }
    first_loop = FALSE;
    curr_point = num_all_have = first_start = 0;
    refmsec = -1;
}

/* Controlling routine--called by programs, contains circular buffer */
int read_next_sample(int    number, 
                     long   *chan0, 
                     long   *chan1, 
                     long   *chan2, 
                     time_t *rtime, 
                     int    *rmsec)
{
    ldiv_t          ldivtemp;
    time_t          stime, tempsecs;
    int             i, j, temparray[MAX_POINTS], smsec, nsamp, start_point,
    offsetmsec, expect_pt;
    short           compnum;
    char            ttmstr[255];
    struct timeval t;
    int curtime;

    if (run_before == FALSE) {
	run_before = TRUE;
	reset_vars();
    }
    if (number < 0) {
	/* Step back in the circular buffer by abs(number) points.	*/
	number = -number;
	curr_point -= number;
	if (first_loop == TRUE && curr_point < first_start) {
	    sprintf(errstr, "Invalid backwards step: curr_point=%i first_start=%i number=%i\n", 
		    curr_point, first_start, number);
	    rdsmplerr(errno = 4, errstr);
	    return 1;
	}
	*rtime = refsecs;
	*rmsec = refmsec + curr_point * delt;
	if (curr_point < 0) {
	    curr_point += MAX_POINTS;
	    num_all_have += MAX_POINTS;
	    if ((number >= MAX_POINTS) || (curr_point < MAX(num_have[CHAN0], MAX(num_have[CHAN1], num_have[CHAN2])))) {
		sprintf(errstr, "Invalid backwards step: curr_point=%i num_all_have=%i number=%i\n", 
			curr_point, num_all_have, number);
		sprintf(errstr, "%s #  num_have  valid_pt_max\n", errstr);
		for (j = 0; j < 3; j++)
		  sprintf(errstr, "%s%i %i  %i\n", errstr, j, num_have[j], 
			  valid_pt_max[j]);
		curr_point -= MAX_POINTS;
		num_all_have -= MAX_POINTS;
		rdsmplerr(errno = 3, errstr);
		return 1;
	    }
	    for (i = 0; i < 3; i++)
	      num_have[i] += MAX_POINTS;
	    refmsec -= time_per_buf.rem;
	    ldivtemp = ldiv(refmsec, 1000);
	    refsecs += -time_per_buf.quot + ldivtemp.quot;
	    refmsec = ldivtemp.rem;
	}
	return -number;
    }

/* Keep reading more input data until you have enough data in all channels*/
/* to satisfy the requested number of points.*/

    while (num_all_have < curr_point + number) {
	if ((nsamp = 
              read_packet(temparray, &compnum, &stime, &smsec, &delt)) <= 0)
	  continue;

	time_per_buf = ldiv(MAX_POINTS * delt, 1000);
	if (nsamp >= MAX_POINTS) {
	    rdsmplerr(errno = 31, 
              "Too many points read from socket, increase MAX_POINTS.\n");
	    return -1;

	}

/* If this is the first data ever read, establish the reference time.	*/
	if (refmsec == -1) {
	    refmsec = smsec;
	    refsecs = stime;
	}

	gettimeofday(&t,NULL);
	curtime = t.tv_sec;
	if (debug) {
printf("timecheck:  curtime=%d comp=%d stime=%d  smsec=%d  nsamp=%d timfr=%d\n",
		   curtime, compnum, stime, smsec, nsamp, nsamp * delt);
	}

/* Determine offset in circular buffer to start inserting the new data.	*/

	start_point = ((stime - refsecs) * 1000 + smsec - refmsec) / delt;
	ldivtemp = ldiv(start_point, MAX_POINTS);
	start_point = ldivtemp.rem;
	/*
	 * printf("ldivtemp q=%d  rem(start_point)=%d\n",
	 * ldivtemp.quot, ldivtemp.rem); printf("check_skip: oldpt=%d
	 * newstart=%d\n",valid_pt_max[compnum],start_point);
	 */

/* Verify that there are no gaps or overlaps between the last data	*/
/* in the circular buffer and the new data.				*/

	expect_pt = valid_pt_max[compnum] + 1;
	if (expect_pt == MAX_POINTS)
	  expect_pt = 0;
	if ((start_point != expect_pt) && (read_before[compnum] == TRUE)) {
	    sprintf(errstr, "Error in reading times: \n");
	    offsetmsec = nint((double) (expect_pt * 1000.0) / delt);
	    ldivtemp = ldiv(refmsec + offsetmsec, 1000);
	    tempsecs = refsecs + ldivtemp.quot;
	    strftime(ttmstr, 255, "%c", gmtime(&tempsecs));
	    sprintf(errstr, "%sExpecting: %s.%ld  (%ldsecs %d+%dmsecs)\n", 
		    errstr, ttmstr, ldivtemp.rem, refsecs, refmsec, offsetmsec);
	    strftime(ttmstr, 255, "%c", gmtime(&stime));
	    sprintf(errstr, "%sTime read from packet: %s.%d  (%ld.%d)\n", 
		    errstr, ttmstr, smsec, stime, smsec);
	    reset_vars();
	    valid_pt_max[compnum] = start_point + nsamp - 1;
	    rdsmplerr(3, errstr);
	    return -1;
	}
	
/* Set the new end offset in the circular buffer once the new data	*/
/* has been inserted into the circular buffer.				*/    

	valid_pt_max[compnum] = start_point + nsamp - 1;
	/*
	 * printf("temparraytest: start=%d first=%d\n",MAX(0,
	 * start_point),temparray[0]);
	 */

/* Insert the new data into the circular buffer.*/

	for (i = MAX(0, start_point); i < MIN(start_point + nsamp, MAX_POINTS); i++)
	  comparray[compnum][i] = temparray[i - start_point];
	if (start_point + nsamp > MAX_POINTS) {
	    valid_pt_max[compnum] -= MAX_POINTS;
	    for (i = 0; i < start_point + nsamp - MAX_POINTS; i++)
	      comparray[compnum][i] = temparray[i + MAX_POINTS - start_point];
	}

/* Set up initial circular buffer points if this is the first data for	*/
/* this channel.							*/

	if (read_before[compnum] == FALSE) {
	    read_before[compnum] = TRUE;
	    curr_point = MAX(curr_point, start_point);
	    first_start = MAX(first_start, curr_point);
	}

/* Set the number of available data points for this channel, and the	*/
/* number of available data points for which we have all channels.	*/

	num_have[compnum] = (valid_pt_max[compnum] > curr_point) ? 
	    valid_pt_max[compnum] : (valid_pt_max[compnum] + MAX_POINTS);
	num_all_have = MIN(num_have[CHAN0], MIN(num_have[CHAN1], num_have[CHAN2]));

	/*
	 * printf("num check:  num_have %d %d %d  min: %d  need:
	 * %d\n",num_have[CHAN0], num_have[CHAN1],
	 * num_have[CHAN2],num_all_have,curr_point+number);
	 */
    }

/* Fulfill the request for the specified number of data points.	*/
/* Return the time of the first data point.			*/
/* Update reftime to the time of the expected time of the next available*/

    /* data point.								
     * printf("Output from curr_point: %i first vals=%i %i %i\n",
     * curr_point, comparray[CHAN0][curr_point],
     * comparray[CHAN1][curr_point], comparray[CHAN2][curr_point]);
     */
    /* printf("."); */

    for (i = curr_point, j = 0; 
         i < MIN(number + curr_point, MAX_POINTS); 
         i++, j++) 
    {
	chan0[j] = comparray[CHAN0][i];
	chan1[j] = comparray[CHAN1][i];
	chan2[j] = comparray[CHAN2][i];
    }
    curr_point = curr_point + number;
    *rtime = refsecs;
    *rmsec = refmsec + curr_point * delt;
    ldivtemp = ldiv (*rmsec,1000);
    *rtime += ldivtemp.quot;
    *rmsec = ldivtemp.rem;
    if (curr_point > MAX_POINTS) {
	first_loop = FALSE;
	refmsec += time_per_buf.rem;
	ldivtemp = ldiv(refmsec, 1000);
	refsecs += time_per_buf.quot + ldivtemp.quot;
	refmsec = ldivtemp.rem;
	curr_point -= MAX_POINTS;
	num_all_have -= MAX_POINTS;
	for (i = 0; i < 3; i++)
	  num_have[i] -= MAX_POINTS;
	for (i = 0; i < curr_point; i++, j++) {
	    chan0[j] = comparray[CHAN0][i];
	    chan1[j] = comparray[CHAN1][i];
	    chan2[j] = comparray[CHAN2][i];
	}
    }
    return number;
}
