/************************************************************************/
/*  Routine for creating a DRM packet header				*/
/*									*/
/*	Phil Maechling							*/
/*	Seismographic Laboratory					*/
/*	California Institute of Technology				*/
/*	phil@seismo.gps.caltech.edu					*/
/*									*/
/************************************************************************/

/*  
	Transmit Manager

    	This module formats then outputs to a serial line a block of data.
	The data is in a Quanterra packet format.

	At this time, no protocol is preformed. It is assumed
	the the other side receives the data correctly.

-----------------------------------------------------------------------


    Methods Include:


	int initialize_station_out_port(char *,char *)
		Give this a Station name, and the
	serial port name out which the data should be sent.
		0 means problems, 1 means init went fine.


	int send_block(char *,HEADER_DATA *);
	
		Give this program a station name, 
	a block of data, and it will transmit it out the
	appropriate serial port.
		0 means problems, 1 means tx was ok.

----------------------------------------------------------------


  Assumptions:

	Transmitting to /dev/null means nothing is output.
	and is useful for testing.

*/


#include <termio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include "network.h"
#include "txman.h"


int static firsttime = 1;

int initialize_station_out_port(char *station, char *line)
{

/* serial port data */

  struct termios trm;
  int out_port;
  char serial_port[30];
  int res;

/* Open serial port */

  printf("Initializing station %s output port %s\n",station,line);

  res = find_a_station(station);
  if (res == -1)
  {
    printf("init serial port can't find stations\n");
    return(0);
  }

  network[res].port_data.portname = line;

  network[res].port_data.port_descriptor = open(line, O_RDWR);

  if(network[res].port_data.port_descriptor == -1) 
  {
    syserr("Cannot open serial line %s\n",line);
    return(0);
  }

/* Set the speed to 19.2k bps */

  trm.c_cflag = (B19200 | CS8 | CLOCAL );

  res = ioctl(network[res].port_data.port_descriptor,TCSETS,&trm);
  if (res == -1)
  {
    syserr("Error setting output serial line\n");
    exit(0);
  }
  else
  {
    printf("%s attached %s at 19200 baud\n",station,line);
  }
  return(1);
}




int send_packet (char *station, char *comp, HEADER_DATA *hd)
{
  unsigned short chcksum;
  unsigned long crc;

/* Input files */

  char charbuff[520];
  char outbuffer[1046]; /* This is the buffer with escape characters in it */
  int res;

/* Transmit variables */

  int seq_no;
  char thischar;
  int nextchar;
  int charnum;    


  res = find_a_station(station);
  if (res == -1)
  {
    printf("send packet (txman) can't find stations\n");
    return(0);
  }

  if(firsttime)
  { 
    /* Initialize the crc table */

    Gcrcinit();

  }

  seq_no = ++network[res].seq_no%8;
  network[res].seq_no = seq_no;

  /* zero the buffer. Important for the checksum */

  memset(&charbuff,0,sizeof(charbuff));

  /* Now construct outblock */


  charbuff[0]   = ( (unsigned char) seq_no);
  charbuff[1] = ( (unsigned char) SEQUENCE_CONT); 
  memcpy(&charbuff[2],hd,sizeof(HEADER_DATA));
  
  chcksum = Checksum(&charbuff[0],514);
  memcpy(&charbuff[514],&chcksum,2);
   

  crc = Gcrccalc(&charbuff[0],514);
  memcpy(&charbuff[516],&crc,4);


 /* Create Transmit buffer */

  outbuffer[0] = ((unsigned char) SOH);

  outbuffer[1] = ((unsigned char) SYN);

  nextchar = 2;


/* Frame each DLE and SOH */ 


  for(charnum=0;charnum<520;charnum++)
  {
    thischar = ((char) charbuff[charnum]);

    if( (thischar == 0x10) || (thischar == 0x15) )
    {
      outbuffer[nextchar] = 0x10;
      ++nextchar;
    }

    outbuffer[nextchar] = thischar;
    ++nextchar;
  }

  /* Add the ETC at the end */

  outbuffer[nextchar] = ((unsigned char) ETX);

  ++nextchar;

  /* Send via serial cable */

  res = write(network[res].port_data.port_descriptor,outbuffer,nextchar);  
  if(res == -1)
  {
    printf("write to serial port error\n");
    return(0);
  }
  else
  {
    return(1);
  }
}


int send_detection_packet (char *station, COMMO_EVENT *hd)
{
  unsigned short chcksum;
  unsigned long crc;

/* Input files */

  char charbuff[520];
  char outbuffer[1046]; /* This is the buffer with escape characters in it */
  int res;

/* Transmit variables */

  int seq_no;
  char thischar;
  int nextchar;
  int charnum;    


  res = find_a_station(station);
  if (res == -1)
  {
    printf("send packet (txman) can't find stations\n");
    return(0);
  }

  if(firsttime)
  { 
    /* Initialize the crc table */

    Gcrcinit();

  }

  seq_no = ++network[res].seq_no%8;
  network[res].seq_no = seq_no;

  /* zero the buffer. Important for the checksum */

  memset(&charbuff,0,sizeof(charbuff));

  /* Now construct outblock */


  charbuff[0]   = ( (unsigned char) seq_no);
  charbuff[1] = ( (unsigned char) SEQUENCE_CONT); 
  memcpy(&charbuff[2],hd,sizeof(COMMO_EVENT));
  
  chcksum = Checksum(&charbuff[0],514);
  memcpy(&charbuff[514],&chcksum,2);
   

  crc = Gcrccalc(&charbuff[0],514);
  memcpy(&charbuff[516],&crc,4);


 /* Create Transmit buffer */

  outbuffer[0] = ((unsigned char) SOH);

  outbuffer[1] = ((unsigned char) SYN);

  nextchar = 2;


/* Frame each DLE and SOH */ 


  for(charnum=0;charnum<520;charnum++)
  {
    thischar = ((char) charbuff[charnum]);

    if( (thischar == 0x10) || (thischar == 0x15) )
    {
      outbuffer[nextchar] = 0x10;
      ++nextchar;
    }

    outbuffer[nextchar] = thischar;
    ++nextchar;
  }

  /* Add the ETC at the end */

  outbuffer[nextchar] = ((unsigned char) ETX);

  ++nextchar;

  /* Send via serial cable */

  res = write(network[res].port_data.port_descriptor,outbuffer,nextchar);  
  if(res == -1)
  {
    printf("write to serial port error\n");
    return(0);
  }
  else
  {
    return(1);
  }
}


/* Find Station converts the SEED name to an integer */
/* A negative number means not found */

int find_a_station(char *station)
{
  
  int i;
  
  for(i=0;i<NUMBER_OF_STATIONS;i++)
  {
    if ( (strncmp(&stations[i][0],station,3) == 0) )
    { 
      return(i);
    }
  }
  return(-1);
}


/* Find Component converts the SEED name to an integer */
/* A negative number means not found */

int find_a_component(char *comp)
{
  
  int i;
  
  for(i=0;i<NUMBER_OF_STREAMS;i++)
  {
    if ( (strncmp(&streams[i][0],comp,3) == 0) )
    { 
      return(i);
    }
  }
  return(-1);
}


