/*


    Program:
            seed2drm.c

    Purpose:
            Convert 4k seed records to 512 byte drm packets.

    Author:
	    Phil Maechling

			
*/

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include "steim.h"
#include "steimlib.h"
#include "miniseed.h"
#include "qlib.h"
#include "pjm_utils.h"

#define  BINREAD          "rb"
#define  BINWRITE         "wb"


long decode_SEED_micro_header (SEED_data_record *sdr, 
	SHORT *firstframe, SHORT *level, 
	SHORT *flip, SHORT *dframes)
{
      SHORT             hfirstdata, hfirstblockette, h1000 ;
      long              headertotal ;

  *flip = 0 ;
   headertotal = sdr->Head.samples_in_record ;
   hfirstdata = sdr->Head.first_data_byte ;
   hfirstblockette = sdr->Head.first_blockette_byte ;
   h1000 = sdr->Head.DOB.blockette_type ;

/*
 * find the first data frame. note that compressed 
 * data must begin on a frame boundary.
 * abort if an illegal start-of-data is found.
 */

  if ((hfirstdata % sizeof(compressed_frame)) != 0)
  {
            printf ("first data does not begin on frame boundary!\n") ;
            exit(0);
  }
 *firstframe = hfirstdata/sizeof(compressed_frame) - 1 ;
 
/*
* if a data-only blockette is present, read the encoding format and
* record length from it. otherwise, the caller's default 
* level will not be changed.
* similarly, do not overwrite the caller's default number of 
* data frames if the
* blockette 1000 is not present.
*/
  
  if ((hfirstblockette == 48) && (h1000 == 1000))
            {
                switch (sdr->Head.DOB.encoding_format)
                {
                    case 10 :
                        *level = 1 ;
                        break ;
                    case 11 :
                        *level = 2 ;
                        break ;
 /*
  * this format code must be defined by the FDSN SEED working group
  * for level 3 compression.
  *
  */
                    case 20 :
                        *level = 3 ;
                        break ;
                    default :
          		printf ("unknown encoding format %d\n",
	   	          (int)sdr->Head.DOB.encoding_format) ;
                        exit (0) ;
                        break ;
                }
                switch (sdr->Head.DOB.rec_length)
                {
                    case 12 :
                        *dframes = 63 ;
                        break ;
                    case 11 :
                        *dframes = 31 ;
                        break ;
                    case 10 :
                        *dframes = 15 ;
                        break ;
                    case  9 :
                        *dframes = 7 ;
                        break ;
                    case 8 :
                        *dframes = 3 ;
                        break ;
                    case 7 :
                        *dframes = 1 ;
                        break ;
                    default :
                        printf ("unknown record length code %d\n",
			(int)sdr->Head.DOB.rec_length) ;
                        exit (0) ;
                        break ;
                }
            }
      return (headertotal) ;
}



/*
              sdr->Head.samples_in_record = (SHORT)rectotal ;
              sdr->Head.first_data_byte = (firstframe+1)*64 ;
              sdr->Head.DOB.blockette_type = 1000 ;
              sdr->Head.first_blockette_byte = 48 ;
*/





void makedrm(int argc,  char *argv[])
{
      long                      *udata ;
      SEED_data_record          *sdr ;
      SEED_data_record          *osdr ;
      long                      rectotal, headertotal,smallrectotal ;
      long                      total,outtotal,samples_read;
      short                     dstat ;
      short                     firstframe, flip, level,dataframes,adapt;
      dcptype                   dcp;
      short                     idframes,frames;
      FILE                      *outfile ;
      FILE                      *infile ;
      int                       numread ;
      int       		pblksize;
      DATA_HDR			*headerdata; 
      DATA_HDR			newheader;
      SDR_HDR   		seedhead,newseedhead;
      gdptype		gdp;
      BOOLEAN		isdata,isfirst;
      long		sample;
      short		reserved_dataframes; 
      int	seq_no;
      int 	startsecs,startticks;
      int       endsecs,endticks;  	
      char      sequence[7];
      HEADER_DATA drmdata;

  /* check for input parameters */



  /*
  * allocate a data buffer to hold the maximum possible 
  * number of decompressed samples.
  * this typically requires at least a "compact" model on a PC.
  */


      if ((udata = (long *)malloc(sizeof(long)*100000)) == NULL )
        exit (ENOMEM);

 /*
  * allocate a 4K SEED record into which to read from disk
  */
      if ((sdr = (SEED_data_record *)malloc(sizeof(SEED_data_record))) == NULL )
        exit (ENOMEM);


   if ((dcp = init_generic_decompression ()) == NULL )
            exit (ENOMEM) ;

 /* 
  * Set some constant values for this application.
  */


      level = 1 ;
      dataframes = 7;
      adapt = (dataframes+8)/8;
      flip = 0;

 /*
  * allocate a SEED record in the desired length into which to compress. 
  */


      if ((osdr=(SEED_data_record *)malloc(sizeof(compressed_frame)
		*dataframes+sizeof(SEED_fixed_data_record_header)))==NULL)
        fatalsyserr("Error mallocing seed record.\n");


  /*
   * allocate a compression continuity and generic data record 
   * control structure. this
   * would normally be done for each channel. 
   * cast the SEED record pointer into the
   * generic type expected by the compression library.
   */


      if ((gdp = (gdptype)init_generic_compression 
             (1, adapt, dataframes, level, flip, (generic_data_record *)osdr)) 
		== NULL )
        fatalsyserr("Error mallocing generic data record control. \n");


    reserved_dataframes = 0;
    clear_generic_compression(gdp,reserved_dataframes); 


   /*
    * open the SEED compressed file to read.
    */

    if( (infile = fopen( argv[1], BINREAD )) == NULL )
    {
       fatalsyserr("Opening seed file\n");
    }


    if( (outfile = fopen( argv[2], BINWRITE )) == NULL )
    {
       fatalsyserr("Opening output seed file\n");
    }



/*
 * set the defaults that will be used if the SEED header contains 
 * no blockette 1000:
 * level 1, 4K records.
 */



    samples_read = 0;
    isdata = TRUE;
    total = 0 ;
    idframes = 63;
    outtotal = 0;
    seq_no = 0;
    isfirst = TRUE;


/*
 * read the header first, to see how big the record is.
 */

/* MAIN LOOP */


   while ((numread = fread(sdr, sizeof(sdr->Head), 1, infile)) == 1)
   {

/*
 * decode just enough crucial bits from the header.
 * actual time/status/channelID processing will take 
 * place in a real function.
 */



      headertotal = decode_SEED_micro_header 
	(sdr, &firstframe, &level, &flip, &idframes) ;

      pblksize = 4096; /* input block size 4096 or 512 */
      if (isfirst)
      {
        memcpy(&seedhead,&sdr->Head,sizeof(SDR_HDR));
        headerdata = decode_hdr_sdr(&seedhead,&pblksize); 
        isfirst = FALSE;
      }
/*
 * read the data portion of the appropriate size, 
 * based on the decoded information,
 * into the "Frames" area of the record, i.e. following the header.
 */


     if ((numread = fread(sdr->Frames, 
	sizeof(compressed_frame)*idframes, 1, infile)) != 1) 
      {
        printf ("partial or corrupt SEED record!\n") ;
        exit (0) ;
      }

 /*
  * decompress the record into the array "udata".
  */

      rectotal = decompress_generic_record 
	((generic_data_record *)sdr, &udata[total], &dstat, dcp,
           firstframe, headertotal, level, flip, idframes) ;

 /*
  * this just prints the channel ID and the number of 
  * samples decompressed from the record
  */

      printf ("\n%c%c%c = %ld\n",sdr->Head.channel_ID[0],
	sdr->Head.channel_ID[1],sdr->Head.channel_ID[2],rectotal);

 /*
  * this will print any error status returned from processing the record, and
  * abort on a "fatal" error, i.e. a record that cannot be decompressed.
  * there is no need to abort, 
  * (there is nothing going on like memory request denied),
  * so the action is up to the caller. a copy of the status 
  * messages will also be written
  * into the output file if one is being written.
  */


    if (dferrorfatal(dstat, stdout))
    {
      printf ("decompress: Exiting. decompress_SEED_record reports errors!\n") ;
      exit(3) ;
    }

 /*
  * totalize the number of samples. this is valid only if you've 
  * checked errors as above,
  * otherwise the decompression function may return negative values.
  */
    
    total = total + rectotal ;

  }


    isdata = TRUE;


    printf("Total samples decompressed: %d \n",total);


    /* Compress data in udata into 512 byte blocks */


    /* compressed frame is 64 bytes */

  while (isdata)
  {
    do {

      while((isdata == TRUE ) && 
          (peek_threshold_avail(gdp->adp->ccp) < 0))
        {
          sample = udata[samples_read];
          if (peek_write(gdp->adp->ccp,&sample,1) != 1)
          {
            printf("Error filling compression buffer\n");
          }
          samples_read = samples_read + 1;
          if (samples_read >= total)
          {
	    isdata = FALSE;
            break;
          }          
        }


        /* compress a frame into a record */


        frames = compress_generic_record(gdp,reserved_dataframes);

        if (frames == dataframes)
        {
          smallrectotal = generic_record_samples(gdp);
          outtotal += smallrectotal;

          printf("samples: record =%ld  total= %ld \n",smallrectotal,outtotal);

          /* Calculate new header */
          /* change sequence number, endtime,hdrtime,numsamples */
          /* change old header begintime,hdrtime */

	  ++seq_no;

/* Determine end of current record and start time of next record */


          time_interval((smallrectotal-1),headerdata->sample_rate,
	    &endsecs,&endticks);

          time_interval(smallrectotal,headerdata->sample_rate,
            &startsecs,&startticks);


/* Adjust DATA_HDR then convert to DRM header */

          memcpy(&newheader,headerdata,sizeof(DATA_HDR));
          newheader.seq_no = seq_no;
          newheader.num_samples = smallrectotal;
          newheader.endtime = 
            add_time(headerdata->begtime,endsecs,endticks);
         
/* Adjust reference header for next pass */

          headerdata->begtime =
            add_time(headerdata->begtime,startsecs,startticks);
          headerdata->hdrtime = headerdata->begtime;

          
/* Copy to data portion to drmdata */

          memcpy(&drmdata.da_d,
            &osdr->Frames,(sizeof(compressed_frame) *dataframes));


/* Install header data into packet */


          encode_drm_packet_header(&drmdata,newheader);

 
          printf("Writing seq no. : %d \n",seq_no);

          fwrite(&drmdata,
            (sizeof(compressed_frame) * (dataframes +1)),1,outfile);

          clear_generic_compression(gdp,reserved_dataframes);

          if (!isdata && (peek_contents(gdp->adp->ccp) <=0))
          {
            break;
          }

        } /* end of if frame === dataframes */
      } while (TRUE); /* This terminates on a break */
  } /* end of while reads work */


/*
 * close the output file gracefully
 */

   fclose(outfile);

/*
 * Close the input file
 */

   fclose (infile) ;
   printf ("\ndone. total samples: %ld\n",total) ;
}


int main (int   argc,  char  *argv[])
{

  if (argc != 3)
  {
    printf("Usage: seed2drm seedfile  drmfile    \n");
    printf("Usage: seed2drm inputfile outputfile \n");
    exit(1);
  }

  makedrm (argc, argv) ;
  exit (0);
}

