/************************************************************************/
/*									*/
/*	Phil Maechling							*/
/*	Seismological Laboratory				        */
/*	California Institute of Technology				*/
/*	phil@seismo.gps.caltech.edu					*/
/*									*/
/************************************************************************/
/*  

    Network Trace Data

    This program sends data from a network into the system for the rerun
	simulation program.


    Method Include:


      initialize_network(char *)

	Input a configuration file to this routine, and it
	will initialize the network.

	The network file must contain, a list like this

	PAS pas.data pas.detect /dev/sts/ttyD0e
        
	which is Station name, data configuration file, 
          event configuration file, serial port


      int send_network_trace_data();

	Call this program and all it will go through the stations in the
	network and send data from each one that has data ready to send.

	Return 0 if no data sent. Return 1 if data sent.


      int send_network_detection_data();

	Call this program and all it will go through the stations in the
	network and send data from each one that has data ready to send.

	Return 0 if no data sent. Return 1 if data sent.

*/


#include <stdio.h>
#include "qlib.h"
#include "sman.h"
#include "txman.h"
#include "meter.h"
#include "config.h"


static char stalist[NUMBER_OF_STATIONS][4];
static int netsize;


int initialize_network(char *config)
{
  int res;
  FILE *netlist;
  char sta[4];
  char sport[100];
  char data_config[100];
  char detect_config[100];

  netsize = 0;

  netlist = fopen(config,"r");
  while (fscanf(netlist,"%s %s %s %s",sta,data_config,detect_config,sport) == 4)
  {
    res = initialize_station(sta,data_config);
    if (res == 0)
    {
      fatalsyserr("Error initializing station %s\n",sta);
    }

    res = initialize_station_out_port(sta,sport);
    if (res == 0)
    {
      fatalsyserr("Error initializing serial port%s\n",sta);
    }
 
    res = initialize_detection_manager(sta,detect_config);
    if (res == 0)
    {
      fatalsyserr("Error initializing detection manager %s \n",sta);
    }
     
 
    memcpy(&stalist[netsize],sta,sizeof(sta));
    netsize++;
  }
  
  fclose(netlist);

  return(1);
}



int send_network_trace_data()
{

  int res;
  HEADER_DATA datablk; 
  INT_TIME blktime;
  int i;
  char comp[4];
  int datasent = 0;


  for (i=0;i<netsize;i++)
  {
    res = block_to_send((char *)&stalist[i],comp,&datablk,&blktime);
    if (res == 1)
    {
      res = send_packet((char *)&stalist[i],comp,&datablk);
      if(res == 0)
      { 
        printf("Error on transmit : %s. Quitting \n",&stalist[i]);
        exit(0);
      }
      else
      {
        printf("Sent packet for station : %s  component : %s \n",
                  stalist[i],comp);
        datasent = 1;
      }
    }
  }
 
  /* return 1 if data was sent else return 0 */

  if (datasent == 1)
  {
    return(1);
  }
  else
  {
    return(0);
  }

}



int send_network_detection_data()
{
  int res;
  COMMO_EVENT datablk; 
  INT_TIME blktime;
  int i;
  char comp[4];
  int datasent = 0;
  int maxamp;

  for (i=0;i<netsize;i++)
  {
    res = detection_to_send(&stalist[i],&blktime,comp,&maxamp);
    if (res == 1)
    {
      encode_drm_event_packet(&datablk,
			      &blktime,
			      comp,
			      maxamp);

      res = send_detection_packet((char *)&stalist[i],&datablk);
      if(res == 0)
      { 
        printf("Error on transmit : %s. Quitting \n",&stalist[i]);
        exit(0);
      }
      else
      {
        printf("Sent detection for station : %s  component : %s \n",
                  stalist[i],comp);
        datasent = 1;
      }
    }
  }
 
  /* return 1 if data was sent else return 0 */

  if (datasent == 1)
  {
    return(1);
  }
  else
  {
    return(0);
  }

}
