/************************************************************************/
/*									*/
/*	Phil Maechling							*/
/*	Seismological Laboratory				        */
/*	California Institute of Technology				*/
/*	phil@seismo.gps.caltech.edu					*/
/*									*/
/************************************************************************/
/*  Virtual Present Meter

    This object is used to let us do real-time processing against
    a given time reference. 

    So, given a start time, the code determines the difference between
    the start time and the actual present, then it provides routines to
    use the current real-time clock, determine the difference between
    to actual present and virtual present.


-----------------------------------------------------------------------


    Methods Include:

    Initialize_virtual_time_meter()
	This must be called first. It attaches to the shared memroy area.


    Establish_Virtual_Present(date GMT);
 	give it a start time. This is the reference time. When the
        meter is started, calculte the diference between the actual
        preseent and the virtual present.

    Start_meter();
       When called, it calls the system clock and determines the
	difference between the actual time and the virtual present.

    time_until_virtual_present(date gmt) returns seconds;
 	This routine returns the time remaining until the
	vitual present.

    virtual_present() return date GMT;
	returns the current "virtual time"

    vtime() return time_t which is seconds since 1970
        imitates the time() function, but return current
	virtual time in seconds since Jan 1970.

   getvtimeofday() returns virtual time in tm struct format
	imitates the gettimeofday function, but returns
	current time in tm format.

----------------------------------------------------------------


  Assumptions:

    virtual time reference is always in the past ?

    Meter results unreliable if change to or from daylight savings
	time occurs while meter is running.


   
*/

#include <stdio.h>
#include <sys/time.h>
#include <time.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include "qlib.h"


struct vpmeter
{
  INT_TIME virtual_present_reference;
  time_t meter_start_time;
  double time_diff;
};


static struct vpmeter *v_p_meter;

extern int shmget();

void initialize_virtual_time_meter()
{

  key_t key;
  int shmid;
  int size; 
  

  key = 1928;
  size = sizeof(*v_p_meter);

  shmid = shmget(key,size,IPC_CREAT | 0666);
  if (shmid == -1)
  {
    fatalsyserr("Error on shmget in establish v p\n");
  }

  v_p_meter = shmat(shmid,0,SHM_RND);
  if (v_p_meter == NULL)
  {
    fatalsyserr("Error on shmat in start_meter\n");
  }

  return;

}



void establish_virtual_present(INT_TIME time)
{

  v_p_meter->virtual_present_reference = time;
  return;

}



void start_meter()
{

  v_p_meter->meter_start_time = time(NULL);
  v_p_meter->time_diff = 
    difftime(v_p_meter->meter_start_time,
          unix_time_from_int_time
            (v_p_meter->virtual_present_reference));
  return;

}




INT_TIME *virtual_present()
{

  time_t curtime;
  INT_TIME rettime;
  double seconds;

  curtime =  time(NULL);
  seconds = difftime(curtime,v_p_meter->meter_start_time);
  rettime = add_time(v_p_meter->virtual_present_reference,(int)seconds,0);
  return(&rettime); 
 
}


time_t vtime()
{

  INT_TIME *inttime;
  INT_TIME gmttime;

  time_t rettime;
 
  inttime = virtual_present();

  /* Subtract 8 hours because unix_time converts to local time */

  gmttime = add_time(*inttime,-28880,0);

  rettime = unix_time_from_int_time(gmttime);
  return(rettime); 
 
}


int getvtimeofday(struct timeval *tp)
{

  INT_TIME *inttime;
  time_t unitime;
  struct timeval rettime;
  INT_TIME gmttime;
 
  inttime = virtual_present();

  /* Subtract 8 hours because unix_time converts to local time */

  gmttime = add_time(*inttime,-28880,0);

  unitime = unix_time_from_int_time(gmttime);

  tp->tv_sec = unitime + (long) 0;
  tp->tv_usec = 0;

  return(1); 
 
}


int seconds_until_this_virtual_time(INT_TIME vtime)
{

  double diff;
  INT_TIME *virp_pres;

  virp_pres = virtual_present();
  diff = tdiff(vtime,*virp_pres);
  return( (int) (diff/TICKS_PER_SEC) );

}
