/************************************************************************/
/*  Routine for creating a DRM Event packet header			*/
/*									*/
/*	Phil Maechling							*/
/*	Seismographic Station						*/
/*	California Institute of Technology				*/
/*	phil@seismo.gps.caltech.edu					*/
/*									*/
/************************************************************************/

#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>
#include    "qlib.h"
#include    "seismo.h"

/************************************************************************/
/*  encode_drm_event_packet						*/
/*     translate header information into drm event packet format        */ 
/************************************************************************/


/* Number of seconds in a year, ignoring leap seconds.                  */
#define fixed_sec_per_year(yr)  \
    ( (365 + IS_LEAP(yr)) * SEC_PER_DAY )

void encode_drm_event_packet(COMMO_EVENT *event, 
			     INT_TIME *mytime,
  			     char *comp,
	                     int maxamp)
{

  EXT_TIME  et,outtime;
  char station[4];
  long evtsec;
  int n = 0;
  int evtyear;
  int thisyear;
  EXT_TIME tetime;
  INT_TIME titime;


  /* Clear header to all zeros */

  memset(event,0,sizeof(COMMO_EVENT));


  /* set flag bytes */

  event->header_flag = ((long) 0);


  /* Set frame type to dectection result */

  event->eventlog_struc.event_det_result.frame_type = ((unsigned char) 5); 


  /* Set detection type to Murdock Hutt dectection */

  event->eventlog_struc.event_det_result.detection_type = ((unsigned char) 1); 


  /* Set detection report time . Equal to detection time */

  outtime = int_to_ext(*mytime);

  event->eventlog_struc.event_det_result.time_of_rep[0] = (char)
    outtime.year % 100;
  event->eventlog_struc.event_det_result.time_of_rep[1] = (char)
    outtime.month;
  event->eventlog_struc.event_det_result.time_of_rep[2] = (char)
    outtime.day;
  event->eventlog_struc.event_det_result.time_of_rep[3] = (char)
    outtime.hour;
  event->eventlog_struc.event_det_result.time_of_rep[4] = (char)
    outtime.minute;
  event->eventlog_struc.event_det_result.time_of_rep[5] = (char)
    outtime.second;


  /* Start filling out the SQ_REP part of the packet */
  /* Time since 1984 in seconds */

  evtyear = mytime->year; 
  thisyear = 1984;
  evtsec = 0;
  while (evtyear != thisyear)
  {
    n = fixed_sec_per_year(thisyear);
    evtsec += n;
    ++thisyear;
  }

  evtsec = evtsec + mytime->second;

  event->eventlog_struc.event_det_result.sq_rep.jdate = evtsec;


  /* Set milliseconds to zero. Accuracy not needed */

  event->eventlog_struc.event_det_result.sq_rep.millisec = (short) 0;


  /* Component */

  if (strncmp(&comp[2],"Z",1) == 0)
  {
    event->eventlog_struc.event_det_result.sq_rep.component = 
      ((unsigned char) 0);
  }
  else if (strncmp(&comp[2],"N",1) == 0)
  {
    event->eventlog_struc.event_det_result.sq_rep.component = 
      ((unsigned char) 1);
  }
  else if (strncmp(&comp[2],"E",1) == 0)
  {
    event->eventlog_struc.event_det_result.sq_rep.component = 
      ((unsigned char) 2);
  }
  else
  {
    printf("Unknown component type\n");
    exit(0);
  }

/************************************************************************/
/*  Table used to map stream numbers to common stream names.            */
/*  This mapping is currently hard-coded in the Quanterra software.     */
/************************************************************************/
/* 

   "VBB", "VSP", "LG", "MP", "VP", "VLP", "ULP", NULL 

   Stream . Per pasadena vbb =0, lg = 2, lp = 4 

*/

  if (strncmp(&comp[0],"BH",2) == 0)
  {
    event->eventlog_struc.event_det_result.sq_rep.stream = ((unsigned char) 0);
  }
  else if ( (strncmp(&comp[0],"EL",2) == 0) ||
            (strncmp(&comp[0],"HL",2) == 0) )
  {
    event->eventlog_struc.event_det_result.sq_rep.stream = ((unsigned char) 2);
  }
  else
  {
    printf("Unknown stream type\n");
    exit(0);
  }


  /* Set motion quality */

  event->eventlog_struc.event_det_result.sq_rep.motion_qual = (long) 0;


  /* Set peak amplitude */

  event->eventlog_struc.event_det_result.sq_rep.peak_amp = (long) maxamp;


  /* Set period * 100 * .5 seconds = 50 */

  event->eventlog_struc.event_det_result.sq_rep.period100 = (long) 50;


  /* Set back amplitude */

  event->eventlog_struc.event_det_result.sq_rep.back_amp = (long) 10;

  /* Test outputs */

  printf("input time : %s\n",time_to_str(*mytime,MONTH_FMT));

  tetime = det_time_to_ext_time(evtsec,0);

  titime = ext_to_int(tetime);

  printf("output event time : %s \n",time_to_str(titime,MONTH_FMT));

  printf("output size : %d\n",sizeof(COMMO_EVENT));
  return;
}
