/************************************************************************/
/*  Routine for creating a DRM packet header				*/
/*									*/
/*	Phil Maechling							*/
/*	Seismographic Station						*/
/*	California Institute of Technology				*/
/*	phil@seismo.gps.caltech.edu					*/
/*									*/
/************************************************************************/

#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>
#include    "qlib.h"
#include    "seismo.h"

/************************************************************************/
/*  encode_drm_packet_header						*/
/*     translate header information inf drm packet header format        */ 
/************************************************************************/

void encode_drm_packet_header(HEADER_DATA *header, DATA_HDR data)
{

  EXT_TIME  et;
  char station[4];


  /* Clear header to all zeros */

  memset(header,0,(sizeof(HEADER_TYPE) + sizeof(long) + 28));


  /* set flag bytes */

  header->header_type.header_flag = ((long) 0);

  /* Set frame type */
  /* This sets it to data packet , compression level 1*/

  header->header_type.frame_type = ((unsigned char) 1); 


  /* Component */

  if (strncmp(&data.channel_id[2],"Z",1) == 0)
  {
    header->header_type.component = ((unsigned char) 0);
  }
  else if (strncmp(&data.channel_id[2],"N",1) == 0)
  {
    header->header_type.component = ((unsigned char) 1);
  }
  else if (strncmp(&data.channel_id[2],"E",1) == 0)
  {
    header->header_type.component = ((unsigned char) 2);
  }
  else
  {
    printf("Unknown component type\n");
    exit(0);
  }

/************************************************************************/
/*  Table used to map stream numbers to common stream names.            */
/*  This mapping is currently hard-coded in the Quanterra software.     */
/************************************************************************/
/* char    *stream_name[] = {
    "VBB", "VSP", "LG", "MP", "VP", "VLP", "ULP", NULL };
*/ 
/* Stream . Per pasadena vbb =0, lg = 2, lp = 4 */

  if (strncmp(&data.channel_id[0],"BH",2) == 0)
  {
    header->header_type.stream = ((unsigned char) 0);
  }
  else if ( (strncmp(&data.channel_id[0],"EL",2) == 0) ||
            (strncmp(&data.channel_id[0],"HL",2) == 0) )
  {
    header->header_type.stream = ((unsigned char) 2);
  }
  else
  {
    printf("Unknown stream type\n");
    exit(0);
  }


  /* State of Health Always 32 per pasadena recorded data */


  header->header_type.soh = ((unsigned char) 32);

  /* Set Station . Joe pads at start of station name */

  memcpy(&station," ",1);
  memcpy(&station[1],&data.station_id,3);
  memcpy(&header->header_type.station,&station,4);


  /* milliseconds */


  header->header_type.millisec = 
    (short) (data.begtime.ticks / TICKS_PER_MSEC);


  /* sample of which time represents */


  header->header_type.time_mark = ((short) 1);


  /* first word of data . Transfer data from body to header */


  memcpy(&header->header_type.firstword,(char *)&header->da_d+4,4);


  /* clock correction */


  header->header_type.clock_corr = ((short) 0);


  /* number_of_samples */


  header->header_type.num_samples = ((short) data.num_samples);


  /* Set sample rate */


  header->header_type.rate = ((char) data.sample_rate);

  /* Convert internal time to seed time */

  et = int_to_ext (data.begtime);
  header->header_type.time_sample[0] = et.year % 100;
  header->header_type.time_sample[1] = et.month;
  header->header_type.time_sample[2] = et.day;
  header->header_type.time_sample[3] = et.hour;
  header->header_type.time_sample[4] = et.minute;
  header->header_type.time_sample[5] = et.second;


  /* Set the non-wrapping sequence number */


  header->packet_seq = ((long) data.seq_no);


    return;
}



/************************************************************************/
/*  encode_drm_event_packet						*/
/*     translate header information into drm event packet format        */ 
/************************************************************************/


/* Number of seconds in a year, ignoring leap seconds.                  */
#define fixed_sec_per_year(yr)  \
    ( (365 + IS_LEAP(yr)) * SEC_PER_DAY )

int encode_drm_event_packet(COMMO_EVENT *event, 
			     INT_TIME *mytime,
  			     char *comp,
	                     int maxamp)
{

  EXT_TIME  et,outtime;
  char station[4];
  long evtsec;
  int n = 0;
  int evtyear;
  int thisyear;
  EXT_TIME tetime;
  INT_TIME titime;


  /* Clear header to all zeros */

  memset(event,0,sizeof(COMMO_EVENT));


  /* set flag bytes */

  event->header_flag = ((long) 0);


  /* Set frame type to dectection result */

  event->eventlog_struc.event_det_result.frame_type = ((unsigned char) 5); 


  /* Set detection type to Murdock Hutt dectection */

  event->eventlog_struc.event_det_result.detection_type = ((unsigned char) 1); 


  /* Set detection report time . Equal to detection time */

  outtime = int_to_ext(*mytime);

  event->eventlog_struc.event_det_result.time_of_rep[0] = (char)
    outtime.year % 100;
  event->eventlog_struc.event_det_result.time_of_rep[1] = (char)
    outtime.month;
  event->eventlog_struc.event_det_result.time_of_rep[2] = (char)
    outtime.day;
  event->eventlog_struc.event_det_result.time_of_rep[3] = (char)
    outtime.hour;
  event->eventlog_struc.event_det_result.time_of_rep[4] = (char)
    outtime.minute;
  event->eventlog_struc.event_det_result.time_of_rep[5] = (char)
    outtime.second;


  /* Start filling out the SQ_REP part of the packet */
  /* Time since 1984 in seconds */

  evtyear = mytime->year; 
  thisyear = 1984;
  evtsec = 0;
  while (evtyear != thisyear)
  {
    n = fixed_sec_per_year(thisyear);
    evtsec += n;
    ++thisyear;
  }

  evtsec = evtsec + mytime->second;

  event->eventlog_struc.event_det_result.sq_rep.jdate = evtsec;


  /* Set milliseconds to zero. Accuracy not needed */

  event->eventlog_struc.event_det_result.sq_rep.millisec = (short) 0;


  /* Component */

  if (strncmp(&comp[2],"Z",1) == 0)
  {
    event->eventlog_struc.event_det_result.sq_rep.component = 
      ((unsigned char) 0);
  }
  else if (strncmp(&comp[2],"N",1) == 0)
  {
    event->eventlog_struc.event_det_result.sq_rep.component = 
      ((unsigned char) 1);
  }
  else if (strncmp(&comp[2],"E",1) == 0)
  {
    event->eventlog_struc.event_det_result.sq_rep.component = 
      ((unsigned char) 2);
  }
  else
  {
    printf("Unknown component type\n");
    return(0);
  }

/************************************************************************/
/*  Table used to map stream numbers to common stream names.            */
/*  This mapping is currently hard-coded in the Quanterra software.     */
/************************************************************************/
/* 

   "VBB", "VSP", "LG", "MP", "VP", "VLP", "ULP", NULL 

   Stream . Per pasadena vbb =0, lg = 2, lp = 4 

*/

  if (strncmp(&comp[0],"BH",2) == 0)
  {
    event->eventlog_struc.event_det_result.sq_rep.stream = ((unsigned char) 0);
  }
  else if ( (strncmp(&comp[0],"EL",2) == 0) ||
            (strncmp(&comp[0],"HL",2) == 0) )
  {
    event->eventlog_struc.event_det_result.sq_rep.stream = ((unsigned char) 2);
  }
  else
  {
    printf("Unknown stream type\n");
    return(0);
  }


  /* Set motion quality */

  event->eventlog_struc.event_det_result.sq_rep.motion_qual = (long) 0;


  /* Set peak amplitude */

  event->eventlog_struc.event_det_result.sq_rep.peak_amp = (long) maxamp;


  /* Set period * 100 * .5 seconds = 50 */

  event->eventlog_struc.event_det_result.sq_rep.period100 = (long) 50;


  /* Set back amplitude */

  event->eventlog_struc.event_det_result.sq_rep.back_amp = (long) 10;

  /* Test outputs */

  tetime = det_time_to_ext_time(evtsec,0);
  titime = ext_to_int(tetime);
  printf("Detection time : %s \n",time_to_str(titime,MONTH_FMT));
  return;
}
