/***********************************************************************/
/*									*/
/*	Phil Maechling							*/
/*	Seismological Laboratory				        */
/*	California Institute of Technology				*/
/*	phil@seismo.gps.caltech.edu					*/
/*									*/
/************************************************************************/
/*  Isaiah Event Manager

    This object interfaces to a file of event data. In this case,
	the data is in Caltech catalog format.

    When an application wants to know information about an event,
	or needs to send one, it calls functions in this package.

-----------------------------------------------------------------------


    Methods Include:


	int initialize_isaiah_event_list(char *filename,int location_delay)

		Give this a Caltech Catalog file name,
	which contains the events you wish to send into the simulation.
		0 means problems, 1 means init went fine.


	int event_ready()

		Call this program and it will return 1 if it has
	an event to send, that is, if the next event is current,
	past the virtual present at least the location_delay number
	of seconds.

	int event_to_send(char *event_line)
	
	It will return 1 if packet to send, 0 if no data.
	The charater string that is returned is a null terminated
	string from a caltech catalog file.

----------------------------------------------------------------


  Assumptions:

   
*/

#include <string.h>
#include <stdio.h>
#include "qlib.h"
#include "meter.h"
#include "network.h"
#include "time.h"


extern double atof(char *);


int initialize_isaiah_event_list(char *filename,int location_delay)
{


  isaiah.infile = fopen(filename,"r"); 
  if (isaiah.infile == NULL)
  {
    fatalsyserr("Error opening event list\n");
  }

  isaiah.location_delay = location_delay;
 
  return(1);

}


/* 
   Input the component. Return 0 if no data is ready to
   be sent. Return 1 if data is ready to be sent.
*/


int event_ready()
{

  INT_TIME last_data_time;
  int res;

  if (!isaiah.data_ready)
  {
    read_next_event();
  }

  /* 
     this test fails if it tried to read data above, but
     it was out of data
  */

  if (!isaiah.data_ready)
  {
    return(0);
  }


  last_data_time = int_time_from_time_tm(&isaiah.event_time);
 
  if (seconds_until_this_virtual_time(last_data_time) +
      isaiah.location_delay > 1)
  {
    return(0);
  }
  else
  {
    return(1);
  } 

}


int event_to_send(char *event_line)
{

  if (!isaiah.data_ready)
  {
    read_next_event();
  }

  if (!isaiah.data_ready)
  {
    return(0);
  }

  strcpy(event_line,isaiah.event_line);
  isaiah.data_ready = 0;
  return(1);
}


/*
   Return 1 if data has been place int next_data.
   Return 0 if no more data is available 
*/

int read_next_event()
{
  int numread;
  char eline[120];
  int res;
  char *thisfile; 

  thisfile = fgets(eline,sizeof(eline),isaiah.infile);

  if (thisfile == NULL)
  {
    isaiah.data_ready = 0;
    return(0);
  } 

  res = convert_catalog_to_isaiah(eline,isaiah.event_line);

  if (res == 1)
  {
     isaiah.data_ready = 1;
     return(1);
  }
  else
  {
    isaiah.data_ready = 0;
    return(0);
  }

}

int convert_catalog_to_isaiah(char *cline,char *eventline)
{
  int year;
  int month;
  int day;
  int hour;
  int minute;
  int second;
  float latit;
  float longit;
  float depth;
  float magnit;
  int idnum;
 
  int min_lat;
  int min_long;
  float min_lat_min;
  float min_long_min;

  char one[2];
  char two[3];
  char three[4];
  char four[5];
  char five[6];
  char six[7];
  char seven[8];
  char eight[9];

  int results;
  double floater;
  char eline[120];

  memcpy(four,cline,4);
  memset(&four[4],0,1);
  year = atoi(four);

  memcpy(two,&cline[5],2);
  memset(&two[2],0,1);
  month = atoi(two);

  memcpy(two,&cline[8],2);
  memset(&two[2],0,1);
  day = atoi(two);

  memcpy(two,&cline[12],2);
  memset(&two[2],0,1);
  hour = atoi(two);

  memcpy(two,&cline[15],2);
  memset(&two[2],0,1);
  minute = atoi(two);

  memcpy(five,&cline[18],5);
  memset(&five[5],0,1);
  floater = atof(five);
  second = (int) floater; 

/* Latitude */

  memcpy(two,&cline[25],2);
  memset(&two[2],0,1);
  min_lat = atoi(two);

  memcpy(five,&cline[28],5);
  memset(&five[5],0,1);
  floater = atof(five);

  latit = min_lat + ((float) (floater/60.0));

/* Longitude */

  memcpy(four,&cline[33],4);
  memset(&four[4],0,1);
  min_long = atoi(four);

  memcpy(five,&cline[38],5);
  memset(&five[5],0,1);
  floater = atof(five);

  longit = min_long - ((float) (floater/60.0));

/* Magnitude */

  memcpy(three,&cline[46],3);
  memset(&three[3],0,1);
  floater = atof(three);
  magnit = (float) floater;

/* Depth */

  memcpy(five,&cline[54],5);
  memset(&five[5],0,1);
  floater = atof(five);
  depth = (float) floater;
  depth = (-1) * depth;

/* ID number */

  memcpy(seven,&cline[71],7);
  memset(&seven[7],0,1);
  idnum = atoi(seven);

  sprintf(eline,"%d %d %d %d %d %d %7.4f %9.4f %4.1f %3.1f %d",
           year,month,day,hour,minute,second,
           latit,longit,depth,magnit,idnum);

  strcpy(eventline,eline);


  isaiah.event_time.tm_yday = ( day_of_year(year,month,day) -1);
  isaiah.event_time.tm_sec = second;
  isaiah.event_time.tm_min = minute;
  isaiah.event_time.tm_hour = hour;
  isaiah.event_time.tm_mday = day;
  isaiah.event_time.tm_mon = month - 1;
  isaiah.event_time.tm_year = year - 1900;

  return(1);

}
