/***********************************************************************/
/*									*/
/*	Phil Maechling							*/
/*	Seismological Laboratory				        */
/*	California Institute of Technology				*/
/*	phil@seismo.gps.caltech.edu					*/
/*									*/
/************************************************************************/
/*  Component Manager

    This object interfaces to a component of data. In this case,
	a component of data is a file.

    When an application wants to know information about a component,
	or needs to send one, it calls functions in this package.

-----------------------------------------------------------------------


    Methods Include:


	int Initialize_component(char *,char *)

		Give this a SEED comopnent name, and the
	path name to the file which contains the data for that
	component.
		0 means problems, 1 means init went fine.


	int close_component(char *)
	
		Give this program a SEED component name, and
	assuming the file was previoulsy initialized, this
	will close the data file;
		0 means problems, 1 means reset ok.


	int packet_ready(char *);

		Give this program a SEED component name, and
	it will return 1 if it has a packet to send, and it
	will return 0 if it has no packets to send. It will
	return 1, only if the next packet to send is current,
	that is past the virtual present.


        int time_of_next_packet(char *,INT_TIME *)
		Give this program a SEED component name. 
	Component input, INT is output.
	It will return 1 if time is found, 0 if problems.


	int packet_to_send(char *,char *,HEADER_DATA *);
	
		Give this program a SEED compent name, and
	it will return a pointer to a SEED block of data.
	It will return 1 if packet to send, 0 if no data.

----------------------------------------------------------------


  Assumptions:

   
*/

#include <string.h>
#include <stdio.h>
#include "qlib.h"
#include "meter.h"
#include "network.h"




int initialize_component(char *station,char *comp,char *file)
{
  int comp_num;
  int stat_num;

  stat_num = find_station(station);
  if(stat_num <0)
  {
   printf("Station not found: %s\n",station);
   return(0);
  }

  comp_num = find_comp(comp);
  if(comp_num <0)
  {
   printf("Component not found: %s\n",comp);
   return(0);
  }

  network[stat_num].comp_array[comp_num].filename=file;

  if ( (network[stat_num].comp_array[comp_num].infile 
          = fopen(file,"rb")) == NULL)
  {
    fatalsyserr("Error opening data file\n");
  }

  network[stat_num].comp_array[comp_num].data_ready = 0;
 
  return(1);

}


int close_component(char *station, char *comp)
{

  int comp_num;
  int stat_num;

  stat_num = find_station(station);
  if(stat_num <0)
  {
   printf("Station not found: %s\n",station);
   return(0);
  }

  comp_num = find_comp(comp);
  if(comp_num <0)
  {
   printf("Component not found: %s\n",comp);
   return(0);
  }

  fclose(network[stat_num].comp_array[comp_num].infile);
  return(1);

}


/* 
   Input the component. Return 0 if no data is ready to
   be sent. Return 1 if data is ready to be sent.
*/


int packet_ready(char *station,char *comp)
{
  int comp_num;
  int stat_num;
  INT_TIME last_data_time;
  DATA_HDR *qda;
  int blksize;

  stat_num = find_station(station);
  if(stat_num <0)
  {
   printf("Station not found: %s\n",station);
   return(0);
  }

  comp_num = find_comp(comp);
  if(comp_num <0)
  {
   printf("Component not found: %s\n",comp);
   return(0);
  }
 
  if (!network[stat_num].comp_array[comp_num].data_ready)
  {
    read_next_data(station,comp);
  }

  /* this test fails if it tried to read data above, but
     it was out of data
  */
  if (!network[stat_num].comp_array[comp_num].data_ready)
  {
    return(0);
  }
 
  blksize = 512;

  qda = decode_hdr_qda((QDA_HDR *) 
          &network[stat_num].comp_array[comp_num].next_packet,&blksize);

  last_data_time = qda->endtime;

  if (seconds_until_this_virtual_time(last_data_time) < 1)
  {
    return(1);
  }
  else
  {
    return(0);
  } 

}


int packet_to_send(char *station,char *comp,HEADER_DATA *hd)
{

  int comp_num;
  int stat_num;

  stat_num = find_station(station);
  if(stat_num <0)
  {
   printf("Station not found: %s\n",station);
   return(0);
  }

  comp_num = find_comp(comp);
  if(comp_num <0)
  {
   printf("Component not found: %s\n",comp);
   return(NULL);
  }
 
  if (!network[stat_num].comp_array[comp_num].data_ready)
  {
    read_next_data(station,comp);
  }

  if (!network[stat_num].comp_array[comp_num].data_ready)
  {
    return(0);
  }

  network[stat_num].comp_array[comp_num].data_ready = 0;
  *hd = network[stat_num].comp_array[comp_num].next_packet;
   return(1);
}



int time_of_next_packet(char *station,char *comp,INT_TIME *pt)
{
  int comp_num;
  int stat_num;
  int blksize;
  DATA_HDR *qda;

  stat_num = find_station(station);
  if(stat_num <0)
  {
   printf("Station not found: %s\n",station);
   return(0);
  }

  comp_num = find_comp(comp);
  if(comp_num <0)
  {
   printf("Component not found: %s\n",comp);
   return(0);
  }
 
  if (!network[stat_num].comp_array[comp_num].data_ready)
  {
    read_next_data(station,comp);
  }

  if (!network[stat_num].comp_array[comp_num].data_ready)
  {
    return(0);
  }

  blksize = 512;
  qda = decode_hdr_qda((QDA_HDR *) 
          &network[stat_num].comp_array[comp_num].next_packet,&blksize);

  *pt = qda->endtime;
  return(1);
}

/* Find Station converts the SEED name to an integer */
/* A negative number means not found */

int find_station(char *station)
{
  
  int i;
  
  for(i=0;i<NUMBER_OF_STATIONS;i++)
  {
    if ( (strncmp(&stations[i][0],station,3) == 0) )
    { 
      return(i);
    }
  }
  return(-1);
}



/* Find component converts the SEED name to an integer */
/* A negative number means not found */

int find_comp(char *strea)
{
  
  int i;

  
  for(i=0;i<NUMBER_OF_STREAMS;i++)
  {
    if ( (strncmp(&streams[i][0],strea,3) == 0) )
    { 
      return(i);
    }
  }
  return(-1);
}



/* 
   Return 1 if data has been place int next_data.
   Return 0 if no more data is available 
*/

int read_next_data(char *station,char *comp)
{

  int numread;
  int comp_num;
  int stat_num;

  stat_num = find_station(station);
  if(stat_num <0)
  {
   printf("Station not found: %s\n",station);
   return(0);
  }

  comp_num = find_comp(comp);
  if (comp_num <0)
  {
    printf("No such component\n");
    return(0);
  } 
 
  numread = fread(&network[stat_num].comp_array[comp_num].next_packet,
              sizeof(HEADER_DATA),1,
                network[stat_num].comp_array[comp_num].infile);

  if (numread == 1)
  {
     network[stat_num].comp_array[comp_num].data_ready = 1;
     return(1);
  }
  else
  {
    network[stat_num].comp_array[comp_num].data_ready = 0;
    return(0);
  }

}
