/***********************************************************

File Name :
	plotmag.c

Programmer:
	Phil Maechling

Description:
	This is a plotting program intended to continuously plot
	magnitudes from a TERRAscope station.

Creation Date:
	4 October 1995

Modification History:


Usage Notes:


**********************************************************/
/*						*/
/* Header files are in /usr/openwin/share/include/X11 */
/*						*/
#include <stdio.h>
#include <string.h>
#include "syserr.h"
#include "basics.h"
#include "larson.h"
#include "rddata.h"
#include <time.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>


#define	debug0(text)	0

/* #define GAIN_FACTOR 9.62e-8 */
#define GAIN_FACTOR 1 
#define	UPDATE_INTERVAL	20000	/*  target display interval in milliseconds */

#define DEFAULT_EXIT_TEXT	"Exit"
#define DEFAULT_BGCOLOR		"white"
#define DEFAULT_FGCOLOR		"black"
#define DEFAULT_BDWIDTH		1
#define DEFAULT_FONT		"fixed"
#define DEFAULT_HEIGHT	300
#define DEFAULT_WIDTH	710
#define MINHEIGHT	50
#define REALMAG_CONFIG_FILE "plotmag.cfg"



int drawseismo(Display *, Window *, GC, int );
int Draw_seismo(Display *,Window ,GC ,float *,int ,char * );
int maxmin(float *, int , float *, float *);
int irint(double); 

#define MAX(a,b) (((a)>(b))?(a):(b))

/*
rdsmplerr(int errcode,char *string)
{
    fprintf(stderr,"%s\n",string);
    if(errcode>12)
      exit(errcode);
}
*/

typedef struct XRTS_PARAMS
{
    char *name;
    char **p_value_string;
}  XRTS_PARAMS;

char *mbgcolor   =	DEFAULT_BGCOLOR,
     *mfgcolor   =	DEFAULT_FGCOLOR,
     *mfont      =	DEFAULT_FONT,
     *ebgcolor   =	DEFAULT_BGCOLOR,
     *efgcolor   =	DEFAULT_FGCOLOR,
     *efont      =	DEFAULT_FONT,
     *mgeom_rsrc =	NULL,
     *extext_rsrc =	NULL,
     *display_name =	NULL,
     *extext_cline =	NULL,
     *mgeom_cline =	NULL,
     *extext       =	DEFAULT_EXIT_TEXT,
     *mgeom       =	NULL,
     *verbose_string =	NULL;

XRTS_PARAMS   resources[] = {
    {"background", &mbgcolor},
    {"forground", &mfgcolor},
    {"font", &mfont},
    {"geometry", &mgeom_rsrc},
    {"exit.background", &ebgcolor},
    {"exit.foreground", &efgcolor},
    {"exit.font", &efont},
    {"exit.text", &extext_rsrc}
};
int             num_resources = sizeof(resources) / sizeof(XRTS_PARAMS);

XRTS_PARAMS   options[] = {
    {"-display", &display_name},
    {"-d", &display_name},
    {"-geometry", &mgeom_cline},
    {"-g", &mgeom_cline},
    {"-extext", &extext_cline},
    {"-e", &extext_cline},
    {"-foreground", &mfgcolor},
    {"-verbose", &verbose_string},
    {"-v", &verbose_string},
};

int  num_options = sizeof(options) / sizeof(XRTS_PARAMS);

char *app_name="plotmag";
XFontStruct	*mfontstruct,*efontstruct;
unsigned long	mbgpix, mfgpix, ebgpix, efgpix;
unsigned int	ewidth, eheight;
int		ey, ex, extxt, eytxt;
XWMHints	xwmh;
XSizeHints	xsh;
Display		*p_disp;
Window		Main, Exit, TimeWin,StaWin;
GC		theGC,exitGC;
XEvent		theEvent;
int		Done = 0;
char		default_geometry[80];
void usage();
int		target_delay = 0;
int		update_interval = UPDATE_INTERVAL;
int		debug = 0;
int             verbose = 0;

/* Make these parameters external so draw routines can see them */

  char sta[6];         /* Station name */
  char  comp[2];       /* component name */
  char  itype[2];      /* instrument type s or b */
  int   instr_type;    /* integer instrument type */
  float sps;           /* seconds per sample */
  float f0,h0,gf;      /* instrument response */
  float stacorml;      /* ml station correction */
  float stacoren;      /* station correction for energy mag */
  float ff,hf;         /* corner frequency and damping constant */
  float fwa,hwa,gfwa;  /* wa constants usually 1.24 0.8 2800.0 */
  float tt0;           /* duration of data for setting baseline */
  float tw;            /* duration of data for max amplitude search */
  float rml100,rme100; /* returned magnitude values */
  long rsecs,rmsecs;  /* returned time values */
  float rxav;
  float  rywa[3],ryv[3],rya[3],ryen[3],rx[3];
  int nrepeat;

  char timestr[30];
  int pcount;
  int ireset;


int main(int argc,char *argv[])
{
    int             i, j, Sx[NUMCOMPS], Sheight,tx,twidth,stax,stawidth;
    char           *tmpstr;
    Colormap        default_cmap;
    XColor          color;
    int             bitmask;
    XGCValues       gcv;
    XSetWindowAttributes xswa;
    Window          SWin[NUMCOMPS];
    char	    statext[80],title_str[255],icon_ttl[255];


  FILE* fp;
  char inline[120];
  float fdummy;       /* dummy time between read time and mark */
  int status;
  int station_not_found;
  int res,res1;
  int plotting;

 
  if (argc < 2)
  {
    usage();
  }

  fp = fopen(REALMAG_CONFIG_FILE,"r");
  if (fp == NULL)
  {
    fatalsyserr("Error opening realmag configuration file\n");
  }

  station_not_found = TRUE;

  while(station_not_found)
  {
    fgets(inline,sizeof(inline),fp);

    /* Test for end of file found */

    if (feof(fp))
    {
      fclose(fp);
      printf("Station %s : Component : %s   not found in realmag \
	configuration file\n",argv[1],argv[2]);
      return(FAILURE);
    }

    /* test for comment */
    if ( (status = strncmp(inline,"#",1)) == 0)
    {
      continue;
    }

    status = sscanf(inline,"%s %s %s %f %f %f %f %f %f %f %f %f %f %f %f %f %d",
                        sta,comp,itype,&sps,
                        &f0,&h0,&gf,
                        &stacorml,&stacoren,
                        &fwa,&hwa,&gfwa,
                        &ff,&hf,
                        &tt0,&tw,&nrepeat);
    if (status != 17)
    {
      printf("Error scanning realmag configuration file\n");
      continue;
    }
    else 
    {
      res = strncasecmp(sta,argv[1],4);
      if  (res == 0) 
      {
        station_not_found = FALSE;
        continue;
      }
    }
  } /* end of while config not found */

  fclose(fp);

  if ( (status = strncasecmp(itype,"b",1)) == 0)
  {
    instr_type = 0;
  }
  else if ( (status = strncasecmp(itype,"s",1)) == 0)
  {
   instr_type = 1;
  }
  else
  {
    printf("Unknown instrument type : %s \n",itype);
    return(FAILURE);
  }

  pcount = 0;
  ireset = TRUE;


  init_input(argv[1]);

    for (i = 2; i < argc; i += 2) 
    {
	for (j = 0; j < num_options; j++) 
        {
	    if (strcmp(options[j].name, argv[i]) == 0) 
            {
                if ( (strcmp(argv[i],"-v") == 0) ||
                     (strcmp(argv[i],"-verbose") == 0) )
                {
		  *options[j].p_value_string = "verbose";
                  --i; /* This kludge is because verbose is a single param */
                }
                else
                {
		  *options[j].p_value_string = argv[i + 1];
                }
		break;
	    }
	}

	if (j >= num_options)
	  usage();
    }

    if (verbose_string) verbose = TRUE;
    if ((p_disp = XOpenDisplay(display_name)) == NULL) {
	fprintf(stderr, "%s: can't open display named %s\n",
		argv[0], XDisplayName(display_name));
	exit(1);
    }

    for (i =0 ; i< num_resources; i++) {
	if((tmpstr = XGetDefault(p_disp,app_name,resources[i].name)) != NULL )
	  *resources[i].p_value_string=tmpstr;
    }

    if((mfontstruct = XLoadQueryFont(p_disp, mfont)) == NULL) {
	fprintf( stderr, "%s: display %s cannot load font %s\n",
		app_name, DisplayString(p_disp), mfont);
	exit(1);
    }
    if((efontstruct = XLoadQueryFont(p_disp, efont)) == NULL) {
	fprintf( stderr, "%s: display %s cannot load font %s\n",
		app_name, DisplayString(p_disp), efont);
	exit(1);
    }

    default_cmap = DefaultColormap(p_disp,DefaultScreen(p_disp));

    if (XParseColor(p_disp,default_cmap,mbgcolor,&color) == 0 ||
	XAllocColor(p_disp,default_cmap,&color) == 0) 
      mbgpix = WhitePixel(p_disp,DefaultScreen(p_disp));
    else
      mbgpix = color.pixel;

    if (XParseColor(p_disp,default_cmap,mfgcolor,&color) == 0 ||
	XAllocColor(p_disp,default_cmap,&color) == 0) 
      mfgpix = BlackPixel(p_disp,DefaultScreen(p_disp));
    else
      mfgpix = color.pixel;

    if (XParseColor(p_disp,default_cmap,ebgcolor,&color) == 0 ||
	XAllocColor(p_disp,default_cmap,&color) == 0) 
      ebgpix = WhitePixel(p_disp,DefaultScreen(p_disp));
    else
      ebgpix = color.pixel;

    if (XParseColor(p_disp,default_cmap,efgcolor,&color) == 0 ||
	XAllocColor(p_disp,default_cmap,&color) == 0) 
      efgpix = BlackPixel(p_disp,DefaultScreen(p_disp));
    else
      efgpix = color.pixel;


    if (extext_cline != NULL) extext = extext_cline;
    else if(extext_rsrc != NULL ) extext = extext_rsrc;


    extxt = efontstruct->max_bounds.width / 2;
    eytxt = efontstruct->max_bounds.ascent + efontstruct->max_bounds.descent;
    ewidth = extxt + XTextWidth(efontstruct,extext,strlen(extext)) + 4;
    eheight = eytxt + 4;

    xsh.flags = (PPosition | PSize | PMinSize );
    xsh.height=DEFAULT_HEIGHT;
    xsh.width=DEFAULT_WIDTH;
    xsh.min_height = MINHEIGHT;
    xsh.width = MAX(xsh.width,ewidth);
    xsh.min_width = xsh.width;
    xsh.x = (DisplayWidth(p_disp, DefaultScreen(p_disp)) - xsh.width) /2;
    xsh.y = (DisplayHeight(p_disp, DefaultScreen(p_disp)) - xsh.height) /2;
    sprintf(default_geometry, "%dx%d+%d+%d",
	    xsh.width, xsh.height, xsh.x, xsh.y);
    mgeom = default_geometry;

    if(mgeom_cline != NULL ) mgeom = mgeom_cline ;
    else if( mgeom_rsrc != NULL ) mgeom = mgeom_rsrc;

    bitmask = XGeometry(p_disp, DefaultScreen(p_disp), mgeom, default_geometry,
			DEFAULT_BDWIDTH, mfontstruct->max_bounds.width, 
			mfontstruct->max_bounds.ascent + 
			mfontstruct->max_bounds.descent, 
			1, 1, &(xsh.x), &(xsh.y), &(xsh.width), &(xsh.height));
    if(bitmask & (XValue | YValue )) xsh.flags |= USPosition;
    if(bitmask & (WidthValue | HeightValue )) xsh.flags |=USSize;


    Main = XCreateSimpleWindow(p_disp, DefaultRootWindow(p_disp),
			       xsh.x, xsh.y, xsh.width, xsh.height, 
			       DEFAULT_BDWIDTH, mfgpix, mbgpix);

    sprintf(title_str,"Real-time Magnitude  (Ml(100) and Me(100)) for %s",argv[1]);
    sprintf(icon_ttl,"plotmag : %s",argv[1]);
    XSetStandardProperties(p_disp,Main,title_str,icon_ttl,None,argv,argc,&xsh);

    xwmh.flags = (InputHint | StateHint );
    xwmh.input = False;
    xwmh.initial_state = NormalState;
    XSetWMHints(p_disp,Main, &xwmh);

    gcv.font = mfontstruct->fid;
    gcv.foreground = mfgpix;
    gcv.background = mbgpix;
    theGC = XCreateGC(p_disp, Main, (GCFont|GCForeground|GCBackground ), &gcv);

    xswa.colormap = DefaultColormap(p_disp,DefaultScreen(p_disp));
    xswa.bit_gravity = CenterGravity;
    XChangeWindowAttributes(p_disp, Main, (CWColormap | CWBitGravity), &xswa);
    
    XSelectInput(p_disp, Main, ExposureMask | ResizeRedirectMask);

    XMapWindow(p_disp,Main);


    ex = 1;
    ey = 1;
    Exit = XCreateSimpleWindow(p_disp, Main, ex, ey, ewidth, eheight, 
			       DEFAULT_BDWIDTH, efgpix, ebgpix);
    XSelectInput(p_disp, Exit, ExposureMask | ButtonPressMask);
    XMapWindow(p_disp, Exit);
	
    stax=ex+ewidth;
    sprintf(statext,"Station: %s",argv[1]);
    stawidth=extxt+XTextWidth(efontstruct,statext,strlen(statext))+4;
    StaWin = XCreateSimpleWindow(p_disp, Main, stax, ey, stawidth, eheight, 
				 DEFAULT_BDWIDTH, efgpix, ebgpix);
    XSelectInput(p_disp, StaWin, ExposureMask);
    XMapWindow(p_disp, StaWin);
    tx=stax+stawidth;
    twidth=xsh.width-tx-3;
    TimeWin = XCreateSimpleWindow(p_disp, Main, tx, ey, twidth, eheight, 
				  DEFAULT_BDWIDTH, efgpix, ebgpix);
    XSelectInput(p_disp, TimeWin, NoEventMask);
    XMapWindow(p_disp, TimeWin);

    Sx[0]=eheight+ey+2;
    Sheight=(xsh.height-Sx[0]-8)/(NUMCOMPS-1);
    Sx[1]=Sx[0]+Sheight+2;
    Sx[2]=Sx[1]+Sheight+2;
    for (i = 0; i < NUMCOMPS-1; i++) {
	SWin[i] = XCreateSimpleWindow(p_disp, Main, 1,Sx[i],xsh.width-2, 
				      Sheight,0, mfgpix, mbgpix);
	XSelectInput(p_disp, SWin[i], NoEventMask );
	XMapWindow(p_disp,SWin[i]);
    }

    gcv.font = efontstruct->fid;
    gcv.foreground = efgpix;
    gcv.background = ebgpix;

    exitGC = XCreateGC(p_disp, Exit, (GCFont|GCForeground|GCBackground), &gcv);

    while (!Done) {
	if (XPending(p_disp) > 0) {
	    XNextEvent(p_disp, &theEvent);
	    if (theEvent.xany.window == Exit) {
		switch (theEvent.type) {
		  case Expose:
		    debug0("Expose for Exit event");
		    if (theEvent.xexpose.count == 0) {
			XClearWindow(p_disp, Exit);
			XDrawString(p_disp, Exit, exitGC, extxt, eytxt, extext,
				    strlen(extext));
			XFlush(p_disp);
		    }
		    break;
		  case ButtonPress:
		    Done = 1;
		}
	    }
	    if (theEvent.xany.window == StaWin) {
		switch (theEvent.type) {
		  case Expose:
		    debug0("Expose for StaWin event");
		    if (theEvent.xexpose.count == 0) {
			XClearWindow(p_disp, StaWin);
			XDrawString(p_disp, StaWin, exitGC, extxt, eytxt, 
				    statext, strlen(statext));
			XFlush(p_disp);
		    }
		    break;
		  case ButtonPress:
		    Done = 1;
		}
	    }
	    /*
	    if (theEvent.xany.window == Main) {
		if (theEvent.type == ResizeRequest) {
		    debug0("ResizeRequest event");
		    if (XGetGeometry(p_disp, Main, &wtemp, &itemp, &itemp, 
				     &width, &height, &utemp, &utemp) == 0)
		      exit(1);
		    Sx[0] = eheight + ey + 2;
		    Sheight = ((height - Sx[0]) / (NUMCOMPS-1)) - 8;
		    Sx[1] = Sx[0] + Sheight + 2;
		    Sx[2] = Sx[1] + Sheight + 2;
		    for(i=0;i<NUMCOMPS-1;i++)
		      XMoveResizeWindow(p_disp, SWin[i], 1, Sx[i],width-2, 
					Sheight);
		}
	    }
	    */
	} else {
	    drawseismo(p_disp, SWin, theGC, xsh.width);
	}

    }

    XFreeGC(p_disp,theGC);
    XFreeGC(p_disp,exitGC);
    XDestroyWindow(p_disp,Main);
    XCloseDisplay(p_disp);
    exit(0);
}

void usage()
{
	fprintf(stderr, "usage: %s station_name [-verbose] [-display host:display] [-geometry geom] [-extext text]\n", app_name);
	exit(1);
}


#define SAMPDISPRATE 25

int drawseismo(Display * tdisp, Window * tWins, GC tGC, int points)
{
    int             i, j, newlen=0, itemp;
    long stime1,smsec1;
    static float compdata[NUMCOMPS][1500];
    static long     stime, smsec, delt;
    static char     run_before = LARSON_FALSE;
    ldiv_t ldivtemp;
    char outstr[155],outstr2[63];
    struct timeval t;
    int diff;
    int csec, cmsec;
    float fdummy;
    int x;


    if (run_before == LARSON_FALSE) 
    {
      run_before = LARSON_TRUE;

      pcount = 0;
      rxav = 0; /* set the initial xav to zero */

      for(x = 0;x<3;x++)
      {
        rywa[x] = 0.0;
        ryv[x] = 0.0;
        rya[x] = 0.0;
        ryen[x] = 0.0;
        rx[x] = 0.0;
      }

      ireset = TRUE;

      realmag_(&ireset,
	     &instr_type,&sps,
  	     &f0,&h0,&gf,
	     &stacorml,&stacoren,
             &fwa,&hwa,&gfwa,
	     &ff,&hf,
             &tt0,&tw,
	     &rxav,
	     rywa,ryv,rya,ryen,rx,
	     &rml100,&rme100,&rsecs,&rmsecs);

      if(verbose)
      {
         fprintf(stdout,"Station: %s  Component: %s\n Sample Period : %f\n",
           sta,comp,sps); 
         fprintf(stdout,"Instrument Type: %s\n",itype);
         fprintf(stdout,"f0: %f  h0: %f  gf : %f\n",f0,h0,gf); 
         fprintf(stdout,"stacorml : %f  stacoren: %f\n",stacorml,stacoren); 
         fprintf(stdout,"fwa: %f  hwa: %f  gfwa: %f\n",fwa,hwa,gfwa); 
         fprintf(stdout,"ff: %f  hf: %f\n",ff,hf); 
         fprintf(stdout,"tt0 : %f  tw: %f  nrepeat : %d\n",tt0,tw,nrepeat); 
  
         fdummy = 0.0;
         tmstr_qqt5_(&rsecs,&rmsecs,&fdummy,timestr);
         fprintf(stdout,
           "Current Ml : %4.2f Current Me : %4.2f Time : %s \n",
           rml100,rme100,timestr);
      }

        stime1 = rsecs;
        smsec1 = rmsecs;
        compdata[0][0] = rml100;
        compdata[1][0] = rme100;
        
        /* Zero compdata arrays */ 
        for(i=1;i<1500;i++)
        {
          compdata[0][i] = 0.0;
          compdata[1][i] = 0.0;
        }

        ireset = FALSE;

        realmag_(&ireset,
	     &instr_type,&sps,
  	     &f0,&h0,&gf,
	     &stacorml,&stacoren,
             &fwa,&hwa,&gfwa,
	     &ff,&hf,
             &tt0,&tw,
	     &rxav,
	     rywa,ryv,rya,ryen,rx,
	     &rml100,&rme100,&rsecs,&rmsecs);

        stime = rsecs;
        smsec = rmsecs;
        compdata[0][1] = rml100;
        compdata[1][1] = rme100;

       if(verbose)
       {
            fdummy = 0.0;
            tmstr_qqt5_(&rsecs,&rmsecs,&fdummy,timestr);
            fprintf(stdout,
            "Current Ml : %4.2f Current Me : %4.2f Time : %s \n",
               rml100,rme100,timestr);
       }


    } /* End of if not run before */

    while (XPending(p_disp) == 0) 
    {
            realmag_(&ireset,
	     &instr_type,&sps,
  	     &f0,&h0,&gf,
	     &stacorml,&stacoren,
             &fwa,&hwa,&gfwa,
	     &ff,&hf,
             &tt0,&tw,
	     &rxav,
	     rywa,ryv,rya,ryen,rx,
	     &rml100,&rme100,&rsecs,&rmsecs);

           if(verbose)
           {
            fdummy = 0.0;
            tmstr_qqt5_(&rsecs,&rmsecs,&fdummy,timestr);
            fprintf(stdout,
            "Current Ml : %4.2f Current Me : %4.2f Time : %s \n",
               rml100,rme100,timestr);
           }
             stime = rsecs;
             smsec = rmsecs;
            
             for(x=1500-1; x>0; x-- )
             {
               compdata[0][x] = compdata[0][x-1];
               compdata[1][x] = compdata[1][x-1];
             }

             compdata[0][0] = rml100;
             compdata[1][0] = rme100;

             ++pcount;
 
             if(pcount >= nrepeat)
             {
               pcount = 0;
               ireset = TRUE;
             }
             else
             {
               ireset = FALSE;
             } 
 
            fflush(stdout);

	for (i = 0; i < NUMCOMPS - 1; i++)
        {
	  Draw_seismo(tdisp, tWins[i], tGC, compdata[i],
		      points, statchan[i]);
        }

        stime = stime - 28800;
	strftime(outstr,80,"Last Magnitude: %a %h %e, %Y %T",gmtime(&stime));
	strftime(outstr2,80,"Local Time: %T",localtime(&stime));
	sprintf(outstr,"%s.%03ld GMT  %s.%03ld",
		outstr,smsec,outstr2,smsec);

	XClearWindow(tdisp,TimeWin);
	XDrawString(tdisp,TimeWin,tGC,extxt,eytxt,outstr,strlen(outstr));
	XFlush(p_disp);	/* debug */
    }
    return 0;
}


int Draw_seismo(Display * tdisp, Window tWin, GC tGC,
                float *data,
	        int points, char *text)
{
	int             i, j,itemp;
        float           maxim,min;
	float           yscale, zero_offset, xscale;
	unsigned int    width, height, utemp;
	Window          wtemp;
	char outstr[80];
        int*  temp;
        static int cou = 0;
        int sampcount;

	if (XGetGeometry(tdisp, tWin, &wtemp, &itemp, &itemp, &width,
			 &height, &utemp, &utemp) == 0)
		exit(1);

        temp = (int *) calloc(width,sizeof(int));

	maxmin(data, (width-2), &maxim, &min);

        if (cou == 0)
        { 
	  sprintf(outstr,
          "Current Ml Magnitude : %4.2f   Peak Ml : %4.2f    Time in Window : %4.1fm",data[0],maxim,((width*tw)/60.0));
           cou = 1;
        }
        else
        {
	  sprintf(outstr,
          "Current Me Magnitude : %4.2f   Peak Me : %4.2f    Time in Window : %4.1fm",data[0],maxim,((width*tw)/60.0));
          cou = 0;
        }

	yscale = ((float) (height - 2) / 8 );
	xscale = (float) 1500/width;
        zero_offset = 1 + (yscale * 8);

	/* This loop reverses the plotting order so it plots */
	/* from right to left */

        sampcount = 0;
	for (i = width-1; i > 0; --i) 
        {
           ++sampcount;
	   temp[sampcount] = irint((-data[i] * yscale) + zero_offset);
	}

	XClearWindow(tdisp,tWin);
	for (i = 1; i < width-1; i++)
        {
		XDrawLine(tdisp, tWin, tGC, i, temp[i], i + 1, temp[i + 1]);
        }
	XDrawString(tdisp, tWin, tGC, 0, 10, outstr, strlen(outstr));
	free(temp);
	return 0;
}


int maxmin(float *array, int arrlen, float *maxim, float *min)
{
	int             i;

	*maxim = array[0];
	*min = array[1];
	for (i = 1; i < arrlen; i++) {
		if (array[i] > *maxim)
			*maxim = array[i];
		if (array[i] < *min)
			*min = array[i];
	}
	if(*min==*maxim)
		return 1;
	else
		return 0;
}

