/***********************************************************

File Name :
	qpager.c

Programmer:
	Phil Maechling

Description:
	This defines the qpage creation routines.

	These routines convert into and out of an "event" structure
	called qevent.

	The available routines are:

	zero_qevent();
	int rtpmsg_to_qevent(char *rtpmsg,struct qpager_data_type *qevent)

	int qevent_to_computer_page
          (struct qpager_data_type *qevent,char *comppage)

	int qevent_to_belt_page(struct qpager_data_type *qevent,
				char *beltpage,
				char *source)
 

Creation Date:
	9 September 1995

Modification History:


Usage Notes:


**********************************************************/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include "cbasics.h"
#include "qpager.h"
#include "wheresub.h"
#include "fcs.h"


int zero_qevent(struct qpager_data_type *qevent)
{
  memset(qevent->message_type,0,SIZE_OF_QFIELD);
  qevent->event_id = 0;
  memset(qevent->data_source,0,SIZE_OF_QFIELD);
  qevent->event_version = 0;
  qevent->year = 0;
  qevent->month = 0;
  qevent->day = 0;
  qevent->hour = 0;
  qevent->minute = 0;
  qevent->second = 0.0;
  qevent->latitude = 0.0;
  qevent->longitude = 0.0;
  qevent->depth = 0.0;
  qevent->magnitude = 0.0;
  qevent->stations_in_location = 0;
  qevent->phases_in_location = 0;
  qevent->closest_station = 0.0;
  qevent->time_rms = 0.0; 
  qevent->horizontal_rms = 0.0;
  qevent->vertical_rms = 0.0;
  qevent->az_gap = 0.0;
  memset(qevent->mag_type,0,SIZE_OF_QFIELD);
  qevent->stations_in_mag = 0;
  qevent->magnitude_rms = 0.0;
  memset(qevent->method_of_location,0,SIZE_OF_QFIELD);
  memset(qevent->checksum,0,SIZE_OF_QFIELD);
  return(SUCCESS);
}


int clear_page(char *page)
{

  memset(page,0,LENGTH_OF_PAGE+1);
  return(SUCCESS);
}



int get_field(char *rtpmsg, int position,int length,char *field)
{

  char *p;

  /* Field is always larger than max field in rtpmsg */
  /* Initializing it to zero insures all fields */
  /* end with a null terminator */

  memset(field,0,(SIZE_OF_QFIELD + 1) );

  memcpy(field,&rtpmsg[position],length);

  p = strrchr(field,'*');

  if (p == NULL)
  {

    /* no *'s found */

    return(SUCCESS);
  }
  else
  {
    /* *'s found */

    return(FAILURE); 
  }  

}



int rtpmsg_to_qevent(char *rtpmsg,struct qpager_data_type *qevent)
{

char field[SIZE_OF_QFIELD];
int results;
int status; 
int getstat;

/* Zero Structure. Load default values */

  zero_qevent(qevent);

/* Message Type */

  getstat = get_field(rtpmsg,0,2,field);
  if(getstat != SUCCESS)
  {
    printf("error in message type\n");
    return(FAILURE);
  }

  status = strncmp("E",field,1);
  if (status != 0)
  {
    printf("Unknown message type\n");
    return(FAILURE);
  }
  else
  {
   strcpy(qevent->message_type,"Event");
  }


/* Event ID */

  getstat = get_field(rtpmsg,2,8,field);
  if(getstat != SUCCESS)
  {
    printf("error in event id\n");
    return(FAILURE);
  }

  results = atoi(field);

  if( results < 0)
  {
    printf("Negative Event ID\n");
    return(FAILURE);
  }
  else
  {
    qevent->event_id = results;
  }

/* Data Source */

  getstat = get_field(rtpmsg,10,2,field);
  if(getstat != SUCCESS)
  {
    printf("error in data source \n");
    return(FAILURE);
  }

  if (strncmp("CI",field,2) == 0 )
  {
    strcpy(qevent->data_source,"Pasadena");
  }
  else if (strncmp("NC",field,2) == 0 )
  {
    strcpy(qevent->data_source,"Menlo");
  }
  else if (strncmp("US",field,2) == 0)
  {
    strcpy(qevent->data_source,"Golden");
  }
  else if (strncmp("UW",field,2) == 0)
  {
    strcpy(qevent->data_source,"Seattle");
  }
  else
  {
    printf("Unknown data source\n");
    return(FAILURE);
  }

/* Event Version Number */

  getstat = get_field(rtpmsg,12,1,field);
  if(getstat != SUCCESS)
  {
    printf("error in version number \n");
    return(FAILURE);
  }
 
  results = atoi(field);

  if (results < 0)
  {
    printf("Error reading Event Version Number \n");
    return(FAILURE);
  }
  else
  {
    qevent->event_version = results;
  }

/* Year */

  getstat = get_field(rtpmsg,13,4,field);
  if(getstat != SUCCESS)
  {
    printf("error in year \n");
    return(FAILURE);
  }

  results = atoi(field);

  if( (results < 1900) || (results > 2020) )
  {
   printf("Event dated before 1900 or after 2020 \n");
   return(FAILURE);
  }
  else
  {
   qevent->year = results;
  }

/* Month */

  getstat = get_field(rtpmsg,17,2,field);
  if(getstat != SUCCESS)
  {
    printf("error in month \n");
    return(FAILURE);
  }

  results = atoi(field);
  if ( (results < 1) || (results > 12) )
  {
    printf("Invalid month value\n");
    return(FAILURE);
  }
  else
  {
    qevent->month = results;
  }

/* Day */ 

  getstat = get_field(rtpmsg,19,2,field);
  if(getstat != SUCCESS)
  {
    printf("error in day \n");
    return(FAILURE);
  }

  results = atoi(field);
  if ( (results < 1) || (results > 31) )
  {
    printf("Invalid day value\n");
    return(FAILURE);
  }
  else
  {
    qevent->day = results;
  }

/* Hour */

  getstat = get_field(rtpmsg,21,2,field);
  if(getstat != SUCCESS)
  {
    printf("error in hour \n");
    return(FAILURE);
  }

  results = atoi(field);
  if ( (results < 0) || (results > 23) )
  {
    printf("Invalid hour value\n");
    return(FAILURE);
  }
  else
  {
    qevent->hour = results;
  }

/* Minute */

  getstat = get_field(rtpmsg,23,2,field);
  if(getstat != SUCCESS)
  {
    printf("error in minute \n");
    return(FAILURE);
  }

  results = atoi(field);
  if ( (results < 0) || (results > 59) )
  {
    printf("Invalid minute value\n");
    return(FAILURE);
  }
  else
  {
    qevent->minute = results;
  }

/* Seconds */

  getstat = get_field(rtpmsg,25,3,field);
  if(getstat != SUCCESS)
  {
    printf("error in seconds \n");
    return(FAILURE);
  }

  results = atoi(field);
  if ( (results < 0) || (results > 599) )
  {
    printf("Invalid second value\n");
    return(FAILURE);
  }
  else
  {
    qevent->second = (float)results / 10.0;
  }

/* Latitude */

  getstat = get_field(rtpmsg,28,7,field);
  if(getstat != SUCCESS)
  {
    printf("error in latitude \n");
    return(FAILURE);
  }

  results = atoi(field);
  if ( (results < -900000 ) || (results > 900000) )
  {
    printf("Invalid latitude value\n");
    return(FAILURE);
  }
  else
  {
    qevent->latitude = (float)results / 10000.0;
  }

/* Longitude */

  getstat = get_field(rtpmsg,35,8,field);
  if(getstat != SUCCESS)
  {
    printf("error in longitude \n");
    return(FAILURE);
  }

  results = atoi(field);
  if ( (results < -1800000) || (results > 1800000) )
  {
    printf("Invalid longitude value\n");
    return(FAILURE);
  }
  else
  {
    qevent->longitude = (float)results / 10000.0;
  }

/* Depth. First non-required field */

  getstat = get_field(rtpmsg,43,4,field);
  if(getstat == SUCCESS)
  {
    results = atoi(field);
    if ( (results < -200 ) || (results > 200) )
    {
      printf("Invalid depth value\n");
    }
    else
    {
      qevent->depth = (float)results / 10.0;
    }
  }
  else
  {
    qevent->depth = -6.0;
  }

/* Magnitude */

  getstat = get_field(rtpmsg,47,2,field);
  if(getstat == SUCCESS)
  {  
    results = atoi(field);
    if ( (results < 0 ) || (results > 120 ) )
    {
      printf("Invalid magnitude value\n");
    }
    else
    {
      qevent->magnitude = (float)results / 10.0;
    }
  }
  else
  {
    qevent->magnitude = -1.0;
  }

/* Stations in Location */

  getstat = get_field(rtpmsg,49,3,field);
  if(getstat == SUCCESS)
  {
    results = atoi(field);
    if ( (results < 0 ) || (results > 600) )
    {
      printf("Invalid stations in location value\n");
    }
    else
    {
      qevent->stations_in_location = results;
    }
  }
  else
  {
    qevent->stations_in_location = -1;
  }

/* Phases in Location */

  getstat = get_field(rtpmsg,52,3,field);
  if(getstat == SUCCESS)
  {
    results = atoi(field);
    if ( (results < 0 ) || (results > 3000) )
    {
      printf("Invalid phases in location value\n");
    }
    else
    {
      qevent->phases_in_location = results;
    }
  }
  else
  {
    qevent->phases_in_location = -1;
  }

/* Distance to Closest Station */

  getstat = get_field(rtpmsg,55,4,field);
  if(getstat == SUCCESS)
  {
    results = atoi(field);
    if ( (results < 0 ) || (results > 3000) )
    {
      printf("Invalid closest station value\n");
    }
    else
    {
      qevent->closest_station = results/10.0;
    }
  }
  else
  {
    qevent->closest_station = -1;
  }


/* Time Error rms */

  getstat = get_field(rtpmsg,59,4,field);
  if(getstat == SUCCESS)
  {
    results = atoi(field);
    if ( (results < 0 ) || (results > 1000) )
    {
      printf("Invalid time error value\n");
    }
    else
    {
      qevent->time_rms = (float) results/100.0;
    }
  }
  else
  {
    qevent->time_rms = -1.0;
  }

/* Horizontal error rms */

  getstat = get_field(rtpmsg,63,4,field);
  if(getstat == SUCCESS)
  {
    results = atoi(field);
    if ( (results < 0 ) || (results > 1000) )
    {
      printf("Invalid horizontal error value\n");
    }
    else
    {
      qevent->horizontal_rms = (float) results/10.0;
    }
  }
  else
  {
    qevent->horizontal_rms = -1.0;
  }

/* Vertical error rms */

  getstat = get_field(rtpmsg,67,4,field);
  if(getstat == SUCCESS)
  {
    results = atoi(field);
    if ( (results < 0 ) || (results > 1000) )
    {
      printf("Invalid vertical error value\n");
    }
    else
    {
      qevent->vertical_rms = (float) results/10.0;
    }
  }
  else
  {
    qevent->vertical_rms = -1.0;
  }


/* Max Azmuthal Gap between stations . Percentage of Circle */

  getstat = get_field(rtpmsg,71,2,field);
  if(getstat == SUCCESS)
  {
    results = atoi(field);
    if ( (results < 0 ) || (results > 100 ) )
    {
      printf("Invalid azmuthal gap value\n");
    }
    else
    {
      qevent->az_gap = results * 3.6;
    }
  }
  else
  {
    qevent->az_gap = -1;
  }


/* Magnitude type */

  getstat = get_field(rtpmsg,73,1,field);
  if(getstat == SUCCESS)
  {
    if (strncmp("G",field,1) == 0 )
    {
      strcpy(qevent->mag_type,"Mg");
    }
    else if (strncmp("B",field,1) == 0 )
    {
      strcpy(qevent->mag_type,"Mb");
    }
    else if (strncmp("C",field,1) == 0 )
    {
      strcpy(qevent->mag_type,"Mca");
    }
    else if (strncmp("D",field,1) == 0)
    {
      strcpy(qevent->mag_type,"Md");
    }
    else if (strncmp("E",field,1) == 0)
    {
      strcpy(qevent->mag_type,"Me");
    }
    else if (strncmp("L",field,1) == 0)
    {
      strcpy(qevent->mag_type,"Ml");
    }
    else if (strncmp("S",field,1) == 0)
    {
      strcpy(qevent->mag_type,"Ms");
    }
    else
    {
      printf("Unknown magnitude type \n");
    }
  }
  else
  {
    strcpy(qevent->mag_type,"*");
  }
/* Number of stations used in Magnitude */

  getstat = get_field(rtpmsg,74,2,field);
  if(getstat == SUCCESS)
  {
    results = atoi(field);
    if ( (results < 00 ) || (results > 100) )
    {
      printf("Invalid number of stations in magnitude value\n");
    }
    else
    {
      qevent->stations_in_mag = results;
    }
  }
  else
  {
    qevent->stations_in_mag = -1;
  }

/* Magnitude rms  */

  getstat = get_field(rtpmsg,76,2,field);
  if(getstat == SUCCESS)
  {
    results = atoi(field);
    if ( (results < 0 ) || (results > 900) )
    {
      printf("Invalid magnitude rms value\n");
    }
    else
    {
      qevent->magnitude_rms = (float) results/10.0;
    }
  }
  else
  {
    qevent->magnitude_rms = -1.0;
  }

/* Location Method */

  getstat = get_field(rtpmsg,78,1,field);
  if(getstat == SUCCESS)
  {
    if (strncmp("R",field,1) == 0 )
    {
      strcpy(qevent->method_of_location,"RTP");
    }
    else if (strncmp("I",field,1) == 0 )
    {
      strcpy(qevent->method_of_location,"ISAIAH");
    }
    else if (strncmp("A",field,1) == 0)
    {
      strcpy(qevent->method_of_location,"Auryn");
    }
    else if (strncmp("H",field,1) == 0)
    {
      strcpy(qevent->method_of_location,"Human");
    }
    else if (strncmp("T",field,1) == 0)
    {
      strcpy(qevent->method_of_location,"TERRAscope");
    }
    else
    {
      printf("Unknown location method \n");
    }
  }
  else
  {
    strcpy(qevent->method_of_location,"*");
  }

 return(SUCCESS);

}


int qevent_to_computer_page(struct qpager_data_type *qevent,char *comppage)
{

char field8[9];
char field7[8];
char field6[7];
char field5[6];
char field4[5];
char field3[4];
char field2[3];
char field1[2];

int results;
int status; 
int getstat;

/* Clear page.  All blanks */

  clear_page(comppage);


/* Message Type */

  if (  strncmp(qevent->message_type,"Event",5) == 0)
  {
    memcpy(&comppage[0],"E ",2);
  }
  else
  {
    printf("Unknown message type\n");
    return(FAILURE);
  } 

/* Event ID */

  sprintf(field8,"%c%7.7d",'T',qevent->event_id);
  
  memcpy(&comppage[2],field8,8);

/* Data Source */

  if (strncmp("Pasadena",qevent->data_source,8) == 0 )
  {
    sprintf(field2,"CI");
    memcpy(&comppage[10],field2,2);
  }
  else if (strncmp("Menlo",qevent->data_source,5) == 0 )
  {
    sprintf(field2,"NC");
    memcpy(&comppage[10],field2,2);
  }
  else if (strncmp("Golden",qevent->data_source,6) == 0)
  {
    sprintf(field2,"US");
    memcpy(&comppage[10],field2,2);
  }
  else if (strncmp("Seattle",qevent->data_source,7) == 0)
  {
    sprintf(field2,"UW");
    memcpy(&comppage[10],field2,2);
  }
  else
  {
    printf("Unknown source of data source\n");
    return(FAILURE);
  }

/* Event Version Number */

  sprintf(field1,"%1.1d",qevent->event_version);
  memcpy(&comppage[12],field1,1);

/* year */
  
  sprintf(field4,"%4.4d",qevent->year);
  memcpy(&comppage[13],field4,4);

/* month */

  sprintf(field2,"%2.2d",qevent->month);
  memcpy(&comppage[17],field2,2);

/* day */

  sprintf(field2,"%2.2d",qevent->day);
  memcpy(&comppage[19],field2,2);

/* hour */

  sprintf(field2,"%2.2d",qevent->hour);
  memcpy(&comppage[21],field2,2);

/* minute */

  sprintf(field2,"%2.2d",qevent->minute);
  memcpy(&comppage[23],field2,2);

/* second */

  sprintf(field3,"%3.3d",  (int) (qevent->second * 10 ) );
  memcpy(&comppage[25],field3,3);

/* latitude */

  sprintf(field7,"%7.7d", (int) (qevent->latitude * 10000) );
  memcpy(&comppage[28],field7,7);

/* longitude */

  sprintf(field8,"%8d", (int) (qevent->longitude * 10000)  );
  memcpy(&comppage[35],field8,8);

/* depth . First optional field */

  if (qevent->depth < 0)
  {
    memcpy(&comppage[43],"****",4);
  }
  else
  {
    sprintf(field4,"%4.4d",  (int) (qevent->depth * 10) );
    memcpy(&comppage[43],field4,4);
  }

/* magnitude */

  if (qevent->magnitude < 0)
  {
    memcpy(&comppage[47],"**",2);
  }
  else
  {
    sprintf(field2,"%2.2d", (int) (qevent->magnitude * 10) );
    memcpy(&comppage[47],field2,2);

    /* now insert the magnitude for the media cube confirmation field */

    memcpy(&comppage[1],&comppage[47],1);
  }

/* stations in locations */

  if (qevent->stations_in_location < 0 ) 
  {
    memcpy(&comppage[49],"***",3);
  }
  else
  {
    sprintf(field3,"%3.3d",qevent->stations_in_location);
    memcpy(&comppage[49],field3,3);
  }

/* Number of phases in location */

  if (qevent->phases_in_location < 0 )
  {
    memcpy(&comppage[52],"***",3);
  }
  else
  {
    sprintf(field3,"%3.3d",qevent->phases_in_location);
    memcpy(&comppage[52],field3,3);
  }

/* Distance to closest stations */

  if (qevent->closest_station < 0 )
  {
    memcpy(&comppage[55],"****",4);
  }
  else
  {
    sprintf(field4,"%4.4d", (int) (qevent->closest_station *10) );
    memcpy(&comppage[55],field4,4);
  }

/* time rms */

  if (qevent->time_rms < 0 )
  {
    memcpy(&comppage[59],"****",4);
  }
  else
  {
    sprintf(field4,"%4.4d", (int) (qevent->time_rms *100) );
    memcpy(&comppage[59],field4,4);
  }

/* horizontal rms */

  if (qevent->horizontal_rms < 0 )
  {
    memcpy(&comppage[63],"****",4);
  }
  else
  {
    sprintf(field4,"%4.4d", (int) (qevent->horizontal_rms * 10) );
    memcpy(&comppage[63],field4,4);
  }

/* vertical rms */

  if (qevent->vertical_rms < 0 )
  {
    memcpy(&comppage[67],"****",4);
  }
  else
  {
    sprintf(field4,"%4.4d", (int) (qevent->vertical_rms * 10) );
    memcpy(&comppage[67],field4,4);
  }

/* Azmuthal gap */

  if (qevent->az_gap < 0 )
  {
    memcpy(&comppage[71],"**",2);
  }
  else
  {
    sprintf(field2,"%2.2d", (int) qevent->az_gap/3.6 );
    memcpy(&comppage[71],field2,2);
  }

/* Magnitude Type */

  if (strncmp(qevent->mag_type,"*",1) == 0)
  {
    memcpy(&comppage[73],"*",1);
  }
  else
  {
    if ( strncmp(qevent->mag_type,"Mg",2) == 0)
    {
      memcpy(&comppage[73],"G",1);
    }
    else if ( strncmp(qevent->mag_type,"Mb",2) == 0)
    {
      memcpy(&comppage[73],"B",1);
    }
    else if (strncmp(qevent->mag_type,"Mca",3) == 0)
    {
      memcpy(&comppage[73],"C",1);
    }
    else if (strncmp(qevent->mag_type,"Md",2) == 0)
    {
      memcpy(&comppage[73],"D",1);
    }
    else if (strncmp(qevent->mag_type,"Me",2) == 0)
    {
      memcpy(&comppage[73],"E",1);
    }
    else if (strncmp(qevent->mag_type,"Ml",2) == 0)
    {
      memcpy(&comppage[73],"L",1);
    }
    else if (strncmp(qevent->mag_type,"Ms",2) == 0)
    {
      memcpy(&comppage[73],"S",1);
    }
    else 
    {
      printf("Unknown magnitude type\n");
      memcpy(&comppage[73],"*",1);
    }
  }

/* Number of stations in mag */

  if (qevent->stations_in_mag < 0 )
  {
    memcpy(&comppage[74],"**",2);
  }
  else
  {
    sprintf(field2,"%2.2d", qevent->stations_in_mag );
    memcpy(&comppage[74],field2,2);
  }

/* standard error of magnitude  */

  if (qevent->magnitude_rms < 0 )
  {
    memcpy(&comppage[76],"**",2);
  }
  else
  {
    sprintf(field2,"%2.2d",(int) (qevent->magnitude_rms *10) );
    memcpy(&comppage[76],field2,2);
  }

/* Location method */

  if (strncmp(qevent->method_of_location,"*",1) == 0)
  {
    memcpy(&comppage[78],"*",1);
  }
  else
  {
    if ( strncmp(qevent->method_of_location,"RTP",3) == 0)
    {
      memcpy(&comppage[78],"R",1);
    }
    else if (strncmp(qevent->method_of_location,"ISAIAH",6) == 0)
    {
      memcpy(&comppage[78],"I",1);
    }
    else if (strncmp(qevent->method_of_location,"Auryn",5) == 0)
    {
      memcpy(&comppage[78],"A",1);
    }
    else if (strncmp(qevent->method_of_location,"Human",5) == 0)
    {
      memcpy(&comppage[78],"H",1);
    }
    else if (strncmp(qevent->method_of_location,"TERRAscope",10) == 0)
    {
      memcpy(&comppage[78],"T",1);
    }
    else 
    {
      printf("Unknown location method type\n");
      memcpy(&comppage[78],"*",1);
    }
  }

/* Checksum */

  comppage[79] = 36 + cksum(&comppage[0])%91;


/* Return Zero if conversion was successful */

 return(SUCCESS);
}


int qevent_to_belt_page(struct qpager_data_type *qevent,
                        char *beltpage,
			char *source)
{


char field8[9];
char field7[8];
char field6[7];
char field5[6];
char field4[5];
char field3[4];
char field2[3];
char field1[2];
char *wp;
char wherestr[80];
double f;
double ih;
float dtemp;
int flat_min, flong_min;
int startof,endof,lenof;

/* TERRAscope flag */

  memcpy(&beltpage[0],source,1);

/* Magnitude */

  sprintf(field3,"%1.1f", qevent->magnitude);
  memcpy(&beltpage[1],field3,3);

/* space */

  memcpy(&beltpage[4]," ",1);

/* hour */

  sprintf(field2,"%2.2d",qevent->hour);
  memcpy(&beltpage[5],field2,2);

/* colon */

  memcpy(&beltpage[7],":",1);

/* minute */

  sprintf(field2,"%2.2d",qevent->minute);
  memcpy(&beltpage[8],field2,2);

/* space */

  memcpy(&beltpage[10]," ",1);

/* month */

  sprintf(field2,"%2.2d",qevent->month);
  memcpy(&beltpage[11],field2,2);

/* slash */

  memcpy(&beltpage[13],"/",1);
  
/* day */

  sprintf(field2,"%2.2d",qevent->day);
  memcpy(&beltpage[14],field2,2);

/* space */

  memcpy(&beltpage[16]," ",1);

/* latitude */
/* Find degree's */

  f = modf((double) qevent->latitude,&ih); 
  sprintf(field2,"%02d",(int) ih);
  memcpy(&beltpage[17],field2,2);

/* space */

  memcpy(&beltpage[19]," ",1);

/* Find Degree minutes */

  dtemp = (float) f;
  flat_min = (int) (dtemp * 60.0); 
  sprintf(field2,"%2.2d",flat_min);
  memcpy(&beltpage[20],field2,2);

/* space */

  memcpy(&beltpage[22]," ",1);

/* Longitude */

  f = modf((double) qevent->longitude,&ih);
  sprintf(field3,"%03d", (int) ((-1) * ih) );
  memcpy(&beltpage[23],field3,3);

/* space */

  memcpy(&beltpage[26]," ",1);

/* Longitude minutes */

  dtemp = (float) f;
  flong_min = (int) (dtemp * (-1) * 60.0); 
  sprintf(field2,"%2.2d",flong_min);
  memcpy(&beltpage[27],field2,2);

/* space */

  memcpy(&beltpage[29]," ",1);

/* Depth */

  sprintf(field2,"%2.2d",(int) qevent->depth);
  memcpy(&beltpage[30],field2,2);

/* space */

  memcpy(&beltpage[32]," ",1);


/* ID number */

  sprintf(field8,"%08d",qevent->event_id);
  memcpy(&beltpage[33],&field8[4],4);

/* space */

  memcpy(&beltpage[37]," ",1);

/* Where string */

  wp = wheresub(qevent->latitude,qevent->longitude);
  strcpy(wherestr,wp);

  startof = 0;
  while( strncmp(&wherestr[startof]," ",1) == 0)
  {
    ++startof;
  } 

  endof = strlen(wherestr);
  while (strncmp(&wherestr[endof-1]," ",1) == 0)
  {
    --endof;
  }

  lenof = endof - startof;

  if ( lenof < (size_t) 43)
  { 
    memcpy(&beltpage[38],&wherestr[startof],lenof);
    memset(&beltpage[38 + lenof],0,1);
  }
  else
  {
    memcpy(&beltpage[38],&wherestr[startof],42);
    memset(&beltpage[80],0,1);
  } 

return(SUCCESS); 

}
