/***********************************************************

File Name :
	qpacket.C

Programmer:
	Phil Maechling

Description:
	This defines the operations on Quanterra (aka Shear)
	packets which are available.

Limitations or Warnings:


Creation Date:
	31 January 1996

Modification History:

**********************************************************/
#include <stdio.h>
#include <string.h>
#include "basics.h"
#include "qpacket.h"
#include "jsteim.h"


//*******************************************************
// Constructor
//*******************************************************

QPacket::QPacket(CByteArray *packetData)
{
   // constructor
   if (packetData == NULL)
   {
      // grow the array to Q512_PACKET_SIZE bytes
      SetSize(Q512_PACKET_SIZE);

      // Zero the header
      memset(m_pData,0,Q512_PACKET_SIZE);
   }
   else
   {
      RemoveAll();
      InsertAt(0,packetData);
   }

   initialize_crc();
}


//*******************************************************
// No Destructor as it will be done in c byte array.
// Or is this a memory leak ?
//*******************************************************
QPacket::~QPacket()
{



}

//*******************************************************
// Calculate checksum routine
//*******************************************************

void QPacket::SetChecksum()
{

  WORD sum = 0;
  BYTE cs[2];
  int size=514;
  int start_point = 2;

  while (size-- > 0)
  {
    sum += GetDataAt(start_point);
    ++start_point;
  } 

  memcpy(cs,&sum,2);

  SetDataAt(516,cs[0]);
  SetDataAt(517,cs[1]);

  return;
}

//*******************************************************
// Initalize CRC routine
//*******************************************************

void QPacket::initialize_crc()
{
  unsigned long   *crcp;          /* pointer to current entry in CRC-table */
  short   byte_count;             /* for all bytes 0 ... 255 */
  char    bit_count;              /* bits 0...7 in a byte */
  unsigned long   crc;            /* computed crc-byte value */
  unsigned long   lbyte;          /* byte-value in 32-bit to be shifted */
 
 
 
  /* for all possible 8-Bit bytes */
  for (crcp = crc_table, byte_count = 0; 
             byte_count < 256; 
             crcp++, byte_count++) 
  {
 
            /* make byte a 32-Bit quantity to get the MSBit */
            lbyte = (byte_count << 24); /* shift 3 bytes left */
 
            crc = 0;                    /* clear byte crc */
 
            /* for all bits in a byte */
            for (bit_count = 0; bit_count < 8; bit_count++) 
            {
 
                if ((lbyte ^ crc) & 0x80000000L)
                    crc = (crc << 1) ^ CRC_POLYNOMIAL;
                else
                    crc <<= 1;
 
                lbyte <<= 1;
            }
 
            /* store precalculated byte crc in table */
            *crcp = crc;
  }
}

//*******************************************************
// Calculate CRC routine
//*******************************************************
void QPacket::SetCRC()
{
 
  BYTE crcw[4];
  unsigned long crc;     /* computed returned crc-32-bit */

  crc = 0L;               /* clear return-value */
  int len = 514;
  BYTE buf;
  int start_point = 2;

  while (len-- > 0)      /* until end of buffer */
  {
 
            /* take the most significant 8 bits from current crc-value
               and XOR this with the next buffer byte.
               Use this 8-Bit value (&0xff is not necessary because
               both operands are unsigned) as an index to the
               crc_table and get the crc-mask. Then XOR this
               long with the current crc shifted 8 bits left. */
  
    buf = GetDataAt(start_point);

    crc = crc_table [ *((unsigned char*)&crc) ^ buf ] ^ 
                  (crc << 8);
    ++start_point;

  }

  memcpy(crcw,&crc,4);

  SetDataAt(518,crcw[0]);
  SetDataAt(519,crcw[1]);
  SetDataAt(520,crcw[2]);
  SetDataAt(521,crcw[3]);
  return;
}
 
//*******************************************************
// Set Data byte at a time
//*******************************************************

void QPacket::SetDataAt(int nDataIndex, BYTE data)
{
   // Sets the data array element at the specified data index.
   // Automatically grows the array, if necessary, and modifies the 
   // packet dataLength

   // convert to the real array index
   int nIndex = nDataIndex;

   SetAt(nIndex, data);
}

//*******************************************************
// Get Data a byte at a time.
//*******************************************************

BYTE QPacket::GetDataAt(int nDataIndex)
{
   // Returns the array element at the specified data index
   // convert to the real array index
   int nIndex = nDataIndex;
   return (GetAt(nIndex));
}


//*******************************************************
// Get a Pointer to data
//*******************************************************

BYTE* QPacket::GetDataPointerAt(int nDataIndex)
{
   // Returns a temporary pointer to the data element at
   // the specified data index. nDataIndex = 0 refers to first data
   // byte.

   // convert to the real array index
   int nIndex = nDataIndex;
   return (&m_pData[nIndex]); 
}

//*******************************************************
// Increment the Modulo 8 packet sequence counter
//*******************************************************


void QPacket::SetLinkSeqNo()
{

  BYTE seqno;

  seqno = (BYTE) (link_level_seq_no % SHEAR_SEQ_MODULUS);
 
  SetDataAt(2,seqno);

  ++link_level_seq_no;   

  return; 
}

//*******************************************************
// Set the component. Translate from kff to Shear convention
//*******************************************************

void QPacket::SetComponent(int comp_num)
{

  BYTE comp;

  comp = (BYTE) (comp_num % 3);

  switch(comp)
  {
    case 0:
    {
      SetAt(9,0x01); // North
      break;
    }
    case 1:
    {
      SetAt(9,0x00); // Vertical
      break;
    }
    case 2:
    {
      SetAt(9,0x02); // East
      break;
    }
    default:
    {
      cout << "Unknown component found " << endl;
    }
  }
  return; 
}

//*******************************************************
// Set the stream . Assume lg on fba and vsp on other channels
//*******************************************************

void QPacket::SetStream(int stream_num)
{

  switch(stream_num)
  {
    case 0:
    case 1:
    case 2:
    {
      SetAt(10,0x02); // Hard code stream 2 = lg
      break;
    }
    case 3:
    case 4:
    case 5:
    {
      SetAt(10,0x01); // Hard code stream 2 = vsp
      break;
    }
    default:
    {
      cout << "Unknown Stream type found " << endl;
    }
  }
  return; 
}

//*******************************************************
// Set the Station Name.
//*******************************************************

void QPacket::SetStationName(char* station)
{
  BYTE sta[4];
  int i;
  int leng;
  int startloc;

  startloc =0;
  leng = strlen(station);
  for (i=0; i<4-leng; i++)
  {
     sta[i] = ' ';
     ++startloc;
  }

  for (i=0;i<leng; i++)
  {
    sta[startloc] = station[i];
    ++startloc;
  }

  SetAt(12,sta[0]);
  SetAt(13,sta[1]);
  SetAt(14,sta[2]);
  SetAt(15,sta[3]);

  return;
}

//*******************************************************
// Move the first 4 bytes of data into header.
//*******************************************************

void QPacket::SetFirstWordOfData(int data)
{
  BYTE da[4];
  
  memcpy(da,&data,4);

  SetAt(20,da[0]);
  SetAt(21,da[1]);
  SetAt(22,da[2]);
  SetAt(23,da[3]);

  return;
}

//*******************************************************
// Set the total samples into header
//*******************************************************

void QPacket::SetTotalSamples(int num_samples)
{
  BYTE samples[2];
  WORD sampls;

  sampls =(WORD) num_samples;
 
  memcpy(samples,&sampls,2);

  SetAt(26,samples[0]);
  SetAt(27,samples[1]);

  return;
}

//*******************************************************
// Set data rate
//*******************************************************

void QPacket::SetDataRate(int sps)
{
  BYTE bsps;
  
  bsps = (BYTE) sps;
 
  SetAt(28,bsps);

  return;
}

//*******************************************************
// Set milliseconds
//*******************************************************

void QPacket::SetMilliseconds(int msecs)
{
  WORD wmsecs;
  BYTE bmsecs[2];
  
  wmsecs = (WORD) msecs;
 
  memcpy(bmsecs,&wmsecs,2);
 
  SetAt(16,bmsecs[0]);
  SetAt(17,bmsecs[1]);

  return;
}

//*******************************************************
// Set Time stamp
//*******************************************************

void QPacket::SetTime(int yr, int mnth, int day, 
		      int hr, int minute, int sec)
{

  BYTE this_time;
  int short_yr;

  short_yr = yr - CURRENT_CENTURY;

  this_time = (BYTE) short_yr;
  SetAt(30,this_time);

  this_time = (BYTE) mnth;
  SetAt(31,this_time);

  this_time = (BYTE) day;
  SetAt(32,this_time);

  this_time = (BYTE) hr;
  SetAt(33,this_time);

  this_time = (BYTE) minute;
  SetAt(34,this_time);

  this_time = (BYTE) sec;
  SetAt(35,this_time);

  return;
}

//*******************************************************
// Set Packet Sequence
//*******************************************************

void QPacket::SetDataSequenceNumber(unsigned int seqno)
{

  BYTE b4[4];
 
  memcpy(b4,&seqno,4);
   
  SetAt(36,b4[0]); 
  SetAt(37,b4[1]); 
  SetAt(38,b4[2]); 
  SetAt(39,b4[3]); 

  return;
}


//*******************************************************
// Set Data fields and first long of data into header
//*******************************************************

void QPacket::SetData(gdptype data)
{

  BYTE bdata[448];
  int  lw;

  memcpy(bdata,data->gdr->frames,448);
  for (int n=0;n<448;n++)
  {
    SetAt(n+68,bdata[n]);
  }
  memcpy(&lw,data->gdr->frames,4);
  SetFirstWordOfData(lw);
  return;
}

