/***********************************************************

File Name :
	packet.C

Programmer:
	Phil Maechling

Description:
	This defines the operations on packets which are
	available.

Limitations or Warnings:


Creation Date:
	28 March 1995

Modification History:

**********************************************************/
#include <stdio.h>
#include <string.h>
#include "basics.h"
#include "packet.h"
#include "verbose.h"

Verbosity pget;

//-------------------------------------------------------------//
//

CPacket *CPacket::New(CByteArray *packetData)
{
   // Creates a new instance of CPacket
   return (new CPacket(packetData));
}

// 
CPacket *CPacket::New(BYTE typeCode, WORD source, WORD dest)
{
   return (new CPacket(typeCode, source, dest));
}

//-------------------------------------------------------------//

BYTE CPacket::CalcHeaderChecksum(void)
{
   // Returns checksum of header
   int i;
   WORD chksum = 0;
   BYTE putback;
   for (i=0; i < PACKET_HDR_SIZE; i++)
      chksum += GetAt(i);
   putback = (BYTE) chksum; // Should return lower half of word
   return (putback);
}

//-------------------------------------------------------------//

WORD CPacket::CalcDataChecksum(void)
{
   int i;
   WORD chksum = 0;
   int n = GetSize() - PACKET_HDR_SIZE - 3;
   int dataLength = GetDataLength();
   if (n != (int)GetDataLength() || n <= 0)
      return 0;
   int upperBound = GetUpperBound()-2;
   for (i=PACKET_HDR_SIZE+1; i <= upperBound; i++)
      chksum += GetAt(i);
   return chksum;
}

//-------------------------------------------------------------//

CPacket::CPacket(BYTE typeCode, WORD source, WORD destination)
{
   // constructor
   SetSize(PACKET_HDR_SIZE+1);
   memset(m_pData,0,PACKET_HDR_SIZE+1); // zero the header
   SetTypeCode(typeCode);
   SetSource(source);
   SetDestination(destination);
}

//-------------------------------------------------------------//

CPacket::CPacket(CByteArray *packetData)
{
   // constructor
   if (packetData == NULL)
   {
      // grow the array to PACKET_HDR_SIZE+1 bytes
      SetSize(PACKET_HDR_SIZE+1);

      // Zero the header
      memset(m_pData,0,PACKET_HDR_SIZE+1);
   }
   else
   {
      RemoveAll();
      InsertAt(0,packetData);
   }
}

//-------------------------------------------------------------//

BOOL CPacket::Verify(void)
{
   // Checks packet integrity of a received packet
   // Returns TRUE if the packet is OK

   // Check header size + checksum byte
   if (GetSize() < PACKET_HDR_SIZE+1) 
   {
     if (pget.verbose_mode())
     {
       cout << "Packet header too short " << endl;
     }
     return FALSE;
   }      

   // Check header checksum
   if (CalcHeaderChecksum() != GetHeaderChecksum())
   {
      if (pget.verbose_mode())
      {
        cout << "Packet header checksum does not match " << endl;
      }
      return FALSE;
   }
         
   // Check data checksum, if any
   if (GetDataLength())
   {
      if (CalcDataChecksum() != GetDataChecksum())
      {
        if (pget.verbose_mode())
        {
          cout << "Packet data checksum does not match " << endl;
        }
        return FALSE;
      }
    }
   return TRUE;
}

//-------------------------------------------------------------//

void CPacket::SetDataAt(int nDataIndex, BYTE data)
{
   // Sets the data array element at the specified data index.
   // Automatically grows the array, if necessary, and modifies the 
   // packet dataLength

   // convert to the real array index
   int nIndex = nDataIndex+PACKET_HDR_SIZE+1;

   if (nIndex > GetUpperBound()-2)
      SetDataLength(nDataIndex+1);

   SetAt(nIndex, data);
}

//-------------------------------------------------------------//

void CPacket::SetDataAt(int nDataIndex, CByteArray& array)
{
   // Sets the data array element at the specified data index.
   // Automatically grows the array, if necessary, and modifies the 
   // packet dataLength

   SetDataAt(nDataIndex, &array[0], array.GetSize());
}
//-------------------------------------------------------------//

void CPacket::SetDataAt(int nDataIndex, BYTE *pData, int nBytes)
{
   // Sets the data array element at the specified data index.
   // Automatically grows the array, if necessary, and modifies the 
   // packet dataLength
   // convert to the real array index
   int nIndex = nDataIndex+PACKET_HDR_SIZE+1;

   if (nIndex > GetUpperBound()-nBytes-1)
      SetDataLength(nDataIndex+nBytes);

   memcpy(&m_pData[nIndex], pData, nBytes);
}

//-------------------------------------------------------------//

BYTE CPacket::GetDataAt(int nDataIndex)
{
   // Returns the array element at the specified data index
   // convert to the real array index
   int nIndex = nDataIndex+PACKET_HDR_SIZE+1;
   return (GetAt(nIndex));
}
//-------------------------------------------------------------//

BYTE *CPacket::GetDataPointerAt(int nDataIndex)
{
   // Returns a temporary pointer to the data element at
   // the specified data index. nDataIndex = 0 refers to first data
   // byte.

   // convert to the real array index
   int nIndex = nDataIndex+PACKET_HDR_SIZE+1;
   return (&m_pData[nIndex]); 
}

//-------------------------------------------------------------//

void CPacket::GetData(CByteArray *array)
{
   // Returns data in array
   array->SetSize(GetDataLength());
   memcpy(&(array->ElementAt(0)),&ElementAt(PACKET_HDR_SIZE+1),
      GetDataLength());
}
//-------------------------------------------------------------//

void CPacket::SetDataLength(WORD dataLength)
{
  BYTE hibyte;
  BYTE lowbyte;

  lowbyte = (BYTE) dataLength;
  hibyte = ((BYTE) (dataLength >> 8));
  SetAt(OFFSET_DATALENGTH, hibyte);
  SetAt(OFFSET_DATALENGTH+1, lowbyte);
  SetSize(dataLength+PACKET_HDR_SIZE+3);
}

//-------------------------------------------------------------//

void CPacket::SetHeaderChecksum(void)
{
   BYTE chksum = CalcHeaderChecksum();
   SetAt(PACKET_HDR_SIZE,chksum);
}

//-------------------------------------------------------------//

void CPacket::SetDataChecksum(void)
{
  BYTE hibyte;
  BYTE lowbyte;

   if (GetDataLength() == 0)
      return;
   WORD chksum = CalcDataChecksum();
   int i = GetUpperBound();
   lowbyte = (BYTE) chksum; // converts to low portion
   hibyte  = ((BYTE) (chksum >> 8)); // shift upper byte right then convert
   SetAt(i-1,hibyte);
   SetAt(i,lowbyte);
}

//-------------------------------------------------------------//

WORD CPacket::GetDataChecksum(void)
{
   // Returns data checksum which is located at the
   // last two bytes of the packet
   int index = GetUpperBound()-1;
   return (WORD)( (GetAt(index) << 8) + GetAt(index+1));
}

//-------------------------------------------------------------//


void CPacket::SetSource(WORD source)
{
  BYTE hibyte;
  BYTE lowbyte;

  lowbyte = (BYTE) source; // conversion give low bytes
  hibyte  = ((BYTE) (source >> 8));
  SetAt(OFFSET_SOURCE, hibyte);
  SetAt(OFFSET_SOURCE+1, lowbyte); 
}

//-------------------------------------------------------------//

void CPacket::SetDestination(WORD dest)
{ 
  BYTE hibyte;
  BYTE lowbyte;

  lowbyte = (BYTE) dest; // conversion give low bytes
  hibyte  = ((BYTE) (dest >> 8));
  SetAt(OFFSET_DESTINATION, hibyte);
  SetAt(OFFSET_DESTINATION+1, lowbyte); 
}

//-------------------------------------------------------------//

void CPacket::SetDataString(char *setstring)
{
  int len;

  len = strlen(setstring);
  if(len <=0)
  {
    return;
  }
  else
  {
    for (int i=0;i<len;i++)
    {
       SetDataAt(i,(BYTE) setstring[i] );
    }
    SetDataAt(i,(BYTE) 0); // Set terminating Null;
  }
}

void CPacket::GetDataString(char *outstring)
{
  int len;
 
  outstring[0]=0;

  len = GetDataLength();

  if (len <=0)
  {
    return;
  }
  else
  {
    for(int i=0;i<len;i++)
    {
      outstring[i] = (BYTE) GetDataAt(i);
    }
    outstring[i] = 0;
    return;
  }

}
