/***********************************************************

File Name :
	k2time.C

Programmer:
	Phil Maechling

Description:
	This program is the k2time class. 

Limitations or Warnings:


Creation Date:
	14 March 1995

Modification History:

**********************************************************/

// This file implements the CK2Time class which
// decodes the K2 time format (a 4 byte integer) to
// calendar time.  The K2 time format is a 4 byte integer
// that represents seconds since Jan. 1, 1980.

#include <iostream.h>
#include <stdio.h>
#include <string.h>
#include "k2time.h"

/*---------------------------------------------------------*/

CK2Time::CK2Time(unsigned long rawTime)
{
   Seconds2Time(rawTime, &m_nYear, &m_nDayOfYear, &m_nHour,
      &m_nMin, &m_nSec);
   Day2Month(m_nYear, m_nDayOfYear, &m_nMonth, &m_nDayOfMonth);
   m_nRawTime = rawTime;
}
/*---------------------------------------------------------*/

CK2Time::CK2Time(int year, int month, int dayOfMonth, int hour, int minute,
         int sec)
{
   Month2Day(year,month,dayOfMonth, &m_nDayOfYear);
   m_nRawTime=Time2Seconds(year,m_nDayOfYear, hour, minute, sec);
   Seconds2Time(m_nRawTime, &m_nYear, &m_nDayOfYear, &m_nHour,
      &m_nMin, &m_nSec);
   Day2Month(m_nYear, m_nDayOfYear, &m_nMonth, &m_nDayOfMonth);
}

/*---------------------------------------------------------*/
// Algorithms below were taken from K2 code, file TIME.C
/*---------------------------------------------------------*/

static const int mon[] =
   {0,31,28,31,30,31,30,31,31,30,31,30,31}; /* January = 1 */


// static
BOOL CK2Time::IsLeapYear(int year)
{
   /* Returns 1 if year is a leap year
      Leap years are year divisible by 4 except century years.
      Leap years are years divisible by 400.
    */
   return (((year%100 == 0) && (year%400 == 0)) ||
            ((year%100 != 0) && (year%4 == 0)));
}

/*---------------------------------------------------------*/

// static
unsigned long CK2Time::Time2Seconds(int iyear, int iday_of_year,
   int ihour, int iminute, int isec)
{
   /* Converts calendar time/date to seconds
      since 00:00:00 (midnight) on Jan. 1, 1980.
      This algorithm is good for years between 1980 through 2100.
      (Year 2100 is not a leap year and requires special handling.)
    */
   unsigned long x;
   register int i;

   /* x is days since 1980 */
   i = iyear - 1980;
   x = (i >> 2) * (1461L); /* 1461 = # of days in 4 years */
   if (i & 3) /* not a leap year ? */
      x += (i & 3) * 365L + 1;

    return (((x + iday_of_year - 1) * 24L * 60L * 60L) +
            (ihour * 60L * 60L) + (iminute * 60L) + isec);
}

/*---------------------------------------------------------*/

// static
void CK2Time::Seconds2Time(unsigned long time, int *iyear, int *iday_of_year, int *ihour,
   int *imin, int *isec)
{
   /* Converts seconds since Jan. 1, 1980 to
      calendar date and time
      This algorithm is good for years between 1980 and 2100.
      (Year 2100 is not a leap year and requires special handling.)
   */
   *isec = (int)(time % 60L);
   time /= 60L; /* time in minutes */
   *imin = (int)(time % 60L);
   time /= 60L; /* time in hours */
   *ihour = (int)(time % 24L);
   time /= 24L; /* time in days */
   *iyear = 1980 + (int)((time/1461L) << 2); /* 1461 days in 4 years */
   time %= 1461L;
   if (time >= 366)
   {
      time -= 366;
      (*iyear)++;
      *iyear += (int)(time / 365L);
      time %= 365L;
   }
   *iday_of_year = (int)(time+1); /* add 1 since day 1 == Jan. 1 */
}

/*---------------------------------------------------------*/

// static 
void CK2Time::Day2Month(int year, int day_of_year, int *month, int *day_of_month)
{
    /*
     * Given year and day of year, calculates month and day of month
     * day_of_year = 1 for January 1
     * *month = 1, *day_of_month = 1 for January 1
     */
    int i;

    for ( ; ; (year)++)
    {
        if (IsLeapYear(year))
        {
            if (day_of_year > 60)
               day_of_year--;
            else if (day_of_year == 60)
            {
               *month = 2;
               *day_of_month = 29;
               return;
            }
        }

        for (i=1; i <= 12; i++)
        {
            if (day_of_year <= mon[i])
            {
                *day_of_month = day_of_year;
                *month = i;
                return;
            }
            day_of_year -= mon[i];
        }
    }
}

/*-----------------------------------------*/

// static
void CK2Time::Month2Day(int year, int month, int day_of_month,
   int *day_of_year)
{
    /* converts month/day format to day_of_year
     * day_of_year = 1 for January 1
     * month = 1, day_of_month = 1 for January 1
     */

   register int i;


   *day_of_year = day_of_month;

   for (i=1; i < month; i++)
      *day_of_year += mon[i];

   if (IsLeapYear(year) && month > 2)
      (*day_of_year)++;
}
