/***********************************************************

File Name :
	k2defs.h

Programmer:
	Phil Maechling

Description:
	This is the CIT Unix version of the kinemetrics k2defs.h file
	These are the structure definitions for all the kinemetrics
	status and configuration structures. The routines which
	can manipulate these structures are in k2.h and k2.cc


Limitations or Warnings:


Creation Date:
	28 March 1995

Modification History:

**********************************************************/
#ifndef k2sizes_H
#define k2sizes_H

#include "k2sizes.h"


/* K2 Parameter structures and Data Definitions */
/* This file should contain portable code */
/* This file is common to both the K2 HC16 code and the PC code */

/**************************************************************

REVISION HISTORY:

Version 1.00 (July 5, 1994)
-------------------------
Initial release with support for 12 channels.

Version 1.10 (August 25, 1994)
----------------------------
NOTE: all structure sizes and previous parameter offsets remain the same.

Used expansion space in CHANNEL_RW_PARMS to add the following:
   unsigned short sensorSerialNumberExt;  (high word of sensor serial number)
   float resp1,resp2,resp3,resp4,resp5,resp6; (sensor response)

Used expansion space in MISC_RW_PARMS to add the following:
   unsigned char minBatteryVoltage; (minimum battery voltage x 10) 

Added new sensor code definitions for FBA 4g, 2g, 1g, 0.5g, 0.25g and 0.1g
for parameter CHANNEL_RW_PARMS.sensorType.  The code definitions are
   #define SENSOR_FBA_4G 11  
   #define SENSOR_FBA_2G 12
       etc. 

Changed MISC_RO_PARMS.batteryVoltage type from unsigned short to signed
short.  MISC_RO_PARMS.batteryVoltage has negative value if charging. 
(For example, MISC_RO_PARMS.batteryVoltage = -130 indicates 13.0 V 
and charging).

Redefined the meaning of STREAM_RW_PARMS.filterFlag.
Redefined the meaning of MODEM_RW_PARMS.flags.
Changed TIMING_RW_PARMS.gpsTurnOnInterval from hours to minutes.

New parameters:
   MISC_RO_PARMS.temperature
   MISC_RO_PARMS.restartSource
   MISC_RW_PARMS.networkFlag
   FRAME_HEADER.msec
   STREAM_RO_PARMS.nscans
   STREAM_RO_PARMS.startTimeMsec
   STREAM_RO_PARMS.triggerTimeMsec
   CHANNEL_RO_PARMS.aqOffset

   Added on 8/26/94:
   TIMING_RO_PARMS.gpsLongitude
   TIMING_RO_PARMS.gpsLatitude
   TIMING_RO_PARMS.gpsAltitude

   Added 9/9/94:
   MODEM_RW_PARMS.flags: added more definitions for bit fields

****************************************************************/

/* Type Definitions:
      char: 1 byte integer
      short: 2 byte integer
      float: 4 byte floating point (IEEE format)
      long: 4 byte integer

   All structures are 16-bit aligned 
*/

/* Recorder specific */
#define MAX_CHANNELS 12
#define MAX_STREAMS 1
#define PARAMETER_VERSION 110 /* =1.10; header version number x 100 */
#define INSTRUMENT_CODE 9 /* for K2 */

/* SENSOR TYPE CODES */
#define SENSOR_UNKNOWN 0
#define SENSOR_ACCELERATION 1
#define SENSOR_VELOCITY 2
#define SENSOR_FBA11 10
#define SENSOR_FBA_4G 11 /* 4g FBA */
#define SENSOR_FBA_2G 12 /* 2g FBA */
#define SENSOR_FBA_1G 13 /* 1g FBA */
#define SENSOR_FBA_0_5G 14 /* 0.5g FBA */
#define SENSOR_FBA_0_25G 15 /* 0.25g FBA */
#define SENSOR_FBA_0_1G 16 /* 0.1g FBA */
#define SENSOR_FBA23 20
#define SENSOR_WR1 30

/* Voter type codes */
#define VOTER_UNDEFINED 0
#define VOTER_CHANNEL 'C'    
#define VOTER_STREAM 'S'
#define VOTER_NETWORK 'N'
#define VOTER_KEYBOARD 'K'
#define VOTER_EXTERNAL 'E'
#define STREAM_MAX_VOTERS (MAX_CHANNELS+3) /* channels+network+keyboard+ext */

/* Restart sources for use with MISC_RO_PARMS.restartSource */
#define RESTART_UNKNOWN 0
#define RESTART_FROM_POWERSWITCH 1 /* someone flipped the power switch */
#define RESTART_FROM_COMMAND 2 /* user command to reset */
#define RESTART_FROM_WATCHDOG 3 /* watchdog caught a system crash */
#define RESTART_FROM_DSPFAILURE 4 /* DSP failure caused reset */
#define RESTART_FROM_BATTERYFAILURE 5 /* battery failure caused reset */
#define RESTART_FROM_MEMORYERROR 6 /* memory error caused reset */

/*---------------------------------------------------*/

/* Voting information */
struct VOTER_INFO {
   unsigned char type; /* voter type code */
   unsigned char number; /* channel number, stream number, etc. */
   short weight; /* voting weight: range is -100 to 100 */
};

/*---------------------------------------------------*/

/* Channel read/write parameters */
struct CHANNEL_RW_PARMS {
#define CHANNEL_ID_LENGTH 5
   char id[CHANNEL_ID_LENGTH]; /* NULL terminated */
   char bytepad; /* for expansion */
   unsigned short sensorSerialNumberExt; /* high word of serial number */
   short north; /* displacement */
   short east; /* displacement */
   short up; /* displacement */
   short altitude;
   short azimuth;
   unsigned short sensorType;
   unsigned short sensorSerialNumber; /* low word of serial number */
   unsigned short gain;
   float resp1; /* sensor response; poles and zeroes */
   float resp2;
   float fullscale; /* volts */
   float sensitivity; /* in volts per unit (e.g., g's) */
   float damping; /* fraction of critical */
   float naturalFrequency; /* hz */
   float triggerThreshold; /* fraction of fullscale */
   float detriggerThreshold; /* fraction of fullscale */
   float alarmTriggerThreshold; /* fraction of fullscale */
   float resp3; 
   float resp4;
   float resp5;
   float resp6;
};


/*---------------------------------------------------*/

/* Read only parameters */
struct CHANNEL_RO_PARMS {
   long maxPeak; /* raw sample counts */
   unsigned long maxPeakOffset; /* offset from start of file */
   long minPeak;
   unsigned long minPeakOffset; /* offset from start of file */
   long mean; /* raw sample counts */
   long aqOffset;
   long dwordpad[3]; /* for expansion */
};

/*---------------------------------------------------*/

/* Stream read/write information */
struct STREAM_RW_PARMS {
   unsigned char filterFlag; /* BIT0 = 1 if filtered data;
                                BIT1 = 1 if auto FT after Event
                                BIT2 = 1 if compressed */
   unsigned char primaryStorage; /* = 0 for drive A: ,etc. */
   unsigned char secondaryStorage; /* = 1 for drive B:, etc. */
   unsigned char bytepad[5]; /* for expansion */
   unsigned short eventNumber; /* *NOT USED* */
   unsigned short sps; /* sampling rate */
   unsigned short apw; /* array propagation window, in seconds */
   unsigned short preEvent; /* in seconds */
   unsigned short postEvent; /* in seconds */
   unsigned short minRunTime; /* in seconds */
   short triggerThreshold; /* signed number */
   short detriggerThreshold;
   short wordpad[4]; /* for expansion */

   long dwordpad[4]; /* for expansion */

   /* Voter info: one entry for each channel, network, user command */
   /* Use functions IsStreamVoter() and GetStreamVoterWeight() in
      STRMPAR.C to access the members.
   */
   struct VOTER_INFO voterInfo[STREAM_MAX_VOTERS];
};

/*---------------------------------------------------*/

/* Stream read only parameters */
struct STREAM_RO_PARMS {
   unsigned long startTime; /* first sample time, includes PEM */
   unsigned long triggerTime;
   unsigned long duration; /* in # of frames; note: frames may have different sizes */
   unsigned short errors;
   unsigned short flags; /* Bit 0 = 1 if functional test */
   unsigned short startTimeMsec;
   unsigned short triggerTimeMsec;
   unsigned long nscans; /* # of scans in the event */
   unsigned long pad[2]; /* for expansion */
};

/*---------------------------------------------------*/

/* Miscellaneous read/write parameters */
struct MISC_RW_PARMS {
   unsigned short serialNumber;
   unsigned short nchannels; /* number of channels used */
#define STN_ID_LENGTH 5
   char stnID[STN_ID_LENGTH]; /* null terminated */
#define COMMENT_LENGTH 33
   char comment[COMMENT_LENGTH]; /* NULL terminated */
   short elevation; /* meters above sea level */
   float latitude; /*  degrees North */
   float longitude; /* degrees East */
   short userCodes[4];

   /* Cutler grabber */
   unsigned char cutlerCode; /* 0: Cutler Off,
                                1: Decimation=4, 4800 baud, 50 sps
                                2: Decimation=2, 9600 buad, 100 sps
                                3: Decimation=1, 19200 buad, 200 sps
                              */

   unsigned char minBatteryVoltage; /* minimum alarm battery voltage x 10 */

   char bpad[14]; /* 14 bytes */
   short wpad[7]; /* 14 bytes */

   /* Network */
   unsigned char externalTrigger; /* either 1 (on) or 0 (off) */
   unsigned char networkFlag; /* BIT0 = 0 (Master) or = 1 (Slave) */

};

/*---------------------------------------------------*/

/* Miscellaneous read only parameters */
struct MISC_RO_PARMS {
   unsigned char a2dBits; /* A/D bits per sample; = A2DBITS */
   unsigned char sampleBytes; /* bytes per sample; = 3 */
   unsigned char restartSource; /* code of restart source:
       0 = unknown
       1 = power switch; 2 = user command,
       3 = software watchdog, 4 = DSP failure, 5 = battery failure
       6 = memory error */
   char bytepad[5]; /* for expansion */

   unsigned short maxChannels; /* physical number of channels */
   unsigned short sysBlkVersion; /* sys block version * 100 */
   unsigned short bootBlkVersion; /* boot block version * 100 */
   unsigned short appBlkVersion;  /* application block version * 100 */
   unsigned short dspBlkVersion; /* DSP version * 100 */
   
   /* System Status */
   short batteryVoltage; /* voltage * 10; negative value indicates charging */

   unsigned short crc; /* 16-bit CRC of entire file, with this word set to 0xffff */
                       /* NOTE: this parameter is not used at the moment.
                          For integrity checking, the header and each frame are preceeded by
                          a structure tag which contains a checksum */
   unsigned short flags; /* bit 0 = 1 if DSP system error */
   short temperature; /* degrees C x 10 */
   short wordpad[3]; /* for expansion */

   long dwordpad[4]; /* for expansion */
};

/*---------------------------------------------------*/
struct TIMING_RW_PARMS
{
   unsigned char gpsTurnOnInterval; /* minutes between GPS update checking */
   unsigned char gpsMaxTurnOnTime; /* max time in minutes GPS tries to lock before giving up */
   unsigned char bytepad[6];
   short localOffset; /* hours ahead of UTC; if this parameter is non-zero
                         then all times are in UTC time. If this parameter
                         is zero, then all times are based on local time.
                      */
   short wordpad[3];
   long dwordpad[4];
};

/*---------------------------------------------------*/

struct TIMING_RO_PARMS
{
   unsigned char clockSource; /* 0 = RTC from cold start
                                       1 = keyboard
                                       2 = Sync w/ ext. ref. pulse
                                       3 = Internal GPS
                                     */

   unsigned char gpsStatus; /* Bit 0=1 if currently checking for presence
                                       of GPS board
                               Bit 1=1 if GPS board present
                               Bit 2=1 if error communicating with GPS
                               Bit 3=1 if failed to lock within an allotted time (gpsMaxTurnOnTime)
                               Bit 4=1 if not locked 
                               Bit 5=1 when GPS power is ON
                               Bits 6,7=undefined
                             */

   unsigned char gpsSOH; /* Current state of health; same as Acutime SOH code */
   unsigned char bytepad[5]; /* for expansion */

   unsigned short gpsLockFailCount; /* # of times GPS failed to locked within gpsMaxTurnOnTime */
   unsigned short gpsUpdateRTCCount; /* # of times GPS actually updated the RTC */
   short acqDelay; /* time in msec between actual A/D conversion and DSP output */
   short gpsLatitude; /* latitude x 100 , degrees North */
   short gpsLongitude; /* longitude x 100, degrees East */
   short gpsAltitude; /* altitude in meters */
   short wordpad[2]; /* for expansion */
   short gpsLastDrift[2]; /* in msec.; e.g. 5 = RTC was 5 msec faster than GPS */

   unsigned long gpsLastTurnOnTime[2]; /* time when GPS was last turned on */
   unsigned long gpsLastUpdateTime[2]; /* time of last RTC update */
   unsigned long gpsLastLockTime[2]; /* time of last GPS lock */
   long dwordpad[4]; /* for expansion */
};

/*---------------------------------------------------*/
#define MODEM_INITCMD_LENGTH 64
#define MODEM_DIALPREFIX_LENGTH 16
#define MODEM_DIALSUFFIX_LENGTH 16
#define MODEM_HANGUPCMD_LENGTH 16
#define MODEM_AUTOANSWERON_LENGTH 16
#define MODEM_AUTOANSWEROFF_LENGTH 16
#define MODEM_PHONES 4
#define MODEM_PHONENUMBER_LENGTH 24

struct MODEM_RW_PARMS {
   /* reserved space for modems */
   char initCmd[MODEM_INITCMD_LENGTH]; /* initialization string */
   char dialingPrefix[MODEM_DIALPREFIX_LENGTH]; /* dialing prefix */
   char dialingSuffix[MODEM_DIALSUFFIX_LENGTH];
   char hangupCmd[MODEM_HANGUPCMD_LENGTH];
   char autoAnswerOnCmd[MODEM_AUTOANSWERON_LENGTH];
   char autoAnswerOffCmd[MODEM_AUTOANSWEROFF_LENGTH];
   char phoneNumber[MODEM_PHONES][MODEM_PHONENUMBER_LENGTH]; /* 4 phone numbers */

   unsigned char waitForConnection; /* secs */
   unsigned char pauseBetweenCalls; /* secs */
   unsigned char maxDialAttempts;
   char bytepad[17];

   unsigned short flags; /* BIT0=1 to enable auto call out
                            BIT1 = 1 to call out on battery < 12 V
                            BIT2 = 1 to call out on battery charge failed
                            BIT3 = 1 to call out on extreme temperature
                            BIT4 = 1 to call out on event
                            BIT5 = 1 to call out on GPS lock failure
                          */
   unsigned short wordpad[7];

   unsigned long dwordpad[8];
};

/*---------------------------------------------------*/

/* Group read/write parms into one structure */
struct RW_PARMS {
   struct MISC_RW_PARMS misc;
   struct TIMING_RW_PARMS timing;
   struct CHANNEL_RW_PARMS channel[MAX_CHANNELS];
   struct STREAM_RW_PARMS stream;
   struct MODEM_RW_PARMS modem;
};

/*---------------------------------------------------*/

/* Group read only parms into one structure */
#define FILE_DESCRIPTOR_SIZE 8 /* size of file descriptor */ 
struct RO_PARMS {
   /* first 8 bytes is a common file descriptor */
   char id[3]; /* = 'KMI' to denote a Kinemetrics file */
   unsigned char instrumentCode; /* = '9' for K2 */
   unsigned short headerVersion; /* header version * 100 */
   unsigned short headerBytes; /* size of header following (includes RW_PARMS) */

   /* actual header follows */
   struct MISC_RO_PARMS misc;
   struct TIMING_RO_PARMS timing;
   struct CHANNEL_RO_PARMS channel[MAX_CHANNELS];
   struct STREAM_RO_PARMS stream;
};

/* This is how the K2 header looks like */
struct K2_HEADER {
   struct RO_PARMS roParms;
   struct RW_PARMS rwParms;
};

/*---------------------------------------------------*/
/* K2 File System Definitions */
/*---------------------------------------------------*/

//
// This is redefined because it doesn't allocate properly
// on a 32 bit system.
//

/* dir entry structure for PKR_DIR packets */
#define K2_DIRENTRY_SIZE 22  // size in bytes
//struct K2_DIRENTRY {  /* 22 bytes exactly */
//   unsigned long time; /* creation time in seconds since Jan 1, 1980 */
//   unsigned long size; /* file size, in bytes */
//   char ext[4]; /* extension,  NULL terminated */
//   char name[9]; /* file name, NULL terminated */
//   unsigned char attribute; /* file attribute */
//};

struct K2_DIRENTRY
{
  char data[BYTES_IN_K2_DIRENTRY];
};

/*---------------------------------------------------*/
/* EVENT SUMMARY */
/*---------------------------------------------------*/
struct EVT_SUMMARY {
   short nchan;
   short a2dBits; /* resolution of A/D in bits */
   unsigned short flags; /* bit 0 = FT flag */
   unsigned short sps; /* sampling rate */
   unsigned long nscans; /* # of scans */
   unsigned long triggerTime;
   unsigned short triggerTimeMsec;
   unsigned long startTime;
   unsigned short startTimeMsec;
   struct {
      long maxPeak;
      unsigned long maxPeakOffset;
      long minPeak;
      unsigned long minPeakOffset;
      float sensitivity;
      float fullScale;
      unsigned short gain;
      char id[CHANNEL_ID_LENGTH];
   } channel[MAX_CHANNELS];
};

/*---------------------------------------------------*/
/* K2 Event Frame Structure */
/*---------------------------------------------------*/

#define FRAME_HEADER_SIZE 32 /* bytes */
#define FRAME_HEADER_TYPE 3
#define TIMECODE_BYTES 13
//struct FRAME_HEADER { /* 32 bytes */
//   unsigned char frameType; /* like a version #, = FRAME_HEADER_TYPE */
//   unsigned char instrumentCode; /* instrument code */
//   unsigned short recorderID;
//   unsigned short frameSize; /* # of frame bytes, includes 32 byte header */
//   unsigned long blockTime; /* block time */
// unsigned short channelBitMap; /* 1 bit for each ch. in use. Ch. 1 is bit 0 */
//   unsigned short streamPar; /* see comments below */
//   unsigned char frameStatus; /* see comments below */
//   unsigned char pad1; /* for expansion */
//   unsigned short msec; /* 0..999 */
//   unsigned char pad4; /* expansion */
//   unsigned char timeCode[TIMECODE_BYTES]; /* time code, see comment below */
//};

/* REdefine the frame header. It doesn't allocate properly on a 32 bit system */

struct FRAME_HEADER
{
  BYTE data[BYTES_IN_FRAME_HEADER];
};

/* FRAME_HEADER Description:
   frameStatus:
      Bits 0-3 = frame sequence number 0..9
      Bit 4    = stream triggered flag. Set if frame exceeds stream trigger level.
      Bit 5    = Compressed flag. Set if compressed frame.
      Bits 6-7 = Sample size. 1=16 bits (2 bytes), 2=24 bits (3 bytes)
                              3=32 bits (4 bytes)
                 Samples are expressed as a signed integer of digital counts.

   streamPar:
      Bits 0-11 = Stream sampling rate, 1..4095
      Bits 12-15 = Stream number 0..15

   timeCode[]:
      Time code sampled every millisecond.
      Bits 4-7 of timeCode[0] = time code type (TBD)
      Bit 3 of timeCode[0] = time code bit sampled at first msec of frame (0=low, 1=high)
      Bit 2 of timeCode[0] = time code bit at 2nd msec.
      Bit 0 of timeCode[12] = last (100th msec) bit of time code
*/

/*---------------------------------------------------*/
/* K2 STATUS INFO Structure */
/*---------------------------------------------------*/
#define SF_FAULT 1
#define SF_OK 0
#define SF_NOT_READY 0x0400
#define SF_KB 0x0800
#define SF_MB 0x1000
#define SF_GB 0x2000

struct STATUS_INFO {
   unsigned long systemTime;
   unsigned short serialNumber;
   unsigned short events;
   unsigned short recordingErrors;
   unsigned short triggerStatus; /* bit 0= ACQ STATUS (ON=1, OFF=0)
                                    bit 1= STREAM TRIG (1), DETRIG (0)
                                    bit 2 = KEYBRD TRIG (1), DETRIG (0)
                                    bit 3 = ALARM TRIG (1), DETRIG (0)
                                    bit 4 = EXTERNAL TRIG (1), DETRIG (0)
                                  */
   unsigned char batteryStatus; /* volts * 10; 0 means charging */
   unsigned char hardwareStatus; /* one of: SF_FAULT ("FAULT")
                                            SF_OK ("OK")
                                 */
   unsigned short driveStatus[2]; /* for both drives.
                                     Bits 0-9 is space remaining
                                     Bits 10-15 are flags :
                                       Bit 10 Set: SF_NOT_READY = Drive not ready
                                       Bit 11 Set: SF_KB = space in kilobytes
                                       Bit 12 Set: SF_MB = space in megabytes
                                       Bit 13 Set: SF_GB = space in gigabytes */
                                 
   char stnID[STN_ID_LENGTH]; /* null terminated */
   unsigned char clockSource; /* same code as TIMING_RO_PARMS.clockSource */
};

/*---------------------------------------------------*/
/* K2 ALARM structure  */
/*---------------------------------------------------*/
struct K2_ALARM {
   unsigned short channelBitMap; /* each bit represents a channel (bit 0=ch1)
                                  a set bit means the channel alarm is triggered */
};

/*---------------------------------------------------*/
/*---------------------------------------------------*/

#endif
