/***********************************************************

File Name :
	ikpp.C

Programmer:
	Phil Maechling

Description:
	Incoming K2 Packet Processor (ikpp)

	This is a class definition for a class that can input
	a received K2 packet, and process it.

Creation Date:
	27 January 1995

Modification History:


Usage Notes:


**********************************************************/
#include <string.h>
#include <iostream.h>
#include "k2rt.h"
#include "ikpp.h"
#include "packet.h"
#include "packet.h"
#include "jsteim.h"
#include "steimlib.h"
#include "compress.h"

//************************************************************
// Constructor
//************************************************************

Incoming_K2_Packet_Processor::Incoming_K2_Packet_Processor()
{

  chans.c_0 = TRUE;
  chans.c_1 = TRUE;
  chans.c_2 = TRUE;
  chans.c_3 = FALSE;
  chans.c_4 = FALSE;
  chans.c_5 = FALSE;
 
  // Scan for channel inversion information
  // get_inversion_config(struct channel_inversion_type invert);

  invert.invert_0 = TRUE;
  invert.invert_1 = TRUE;
  invert.invert_2 = TRUE;
  invert.invert_3 = FALSE;
  invert.invert_4 = FALSE;
  invert.invert_5 = FALSE;

  // Scan for station name
  // get_station_name(char* staname); 

  strcpy(station_name,"K2X");


}


//************************************************************
// Destructor
//************************************************************
Incoming_K2_Packet_Processor::~Incoming_K2_Packet_Processor()
{


}


//***********************************************************
// Process each valid packet
//***********************************************************

void Incoming_K2_Packet_Processor::process_k2_packet(CCommControl& k2_port,
						     CPacket&   valid_packet,
						     SocketControl& dl_socket,
						     QPacket&   outpacket)
{
  int msg_type;

  msg_type = valid_packet.GetTypeCode();

  switch(msg_type)
  {
    case PKR_STRDATA:
    {
//      cout << "Got Data Packet" << endl;
      convert_data_packet(valid_packet,
	                  dl_socket,
		          outpacket);
      break;
    }
    case PKC_MSG:
    {
//      cout << "Got Message Packet" << endl;
      break;
    }
    case PKC_SETALARM:
    {
//      cout << "Got Alarm Packet" << endl;
      break;
    }
    case PKR_ACK:
    {
//      cout << "Got Acknowlegment Packet" << endl;
      break;
    }
    case PKR_FILE:
    {
//      cout << "Got File Packet" << endl;
      break;
    }
    case PKR_STATUS:
    {
//      cout << "Got Status Packet" << endl;
      break;
    }
    default:
    {
      if(debug == TRUE)
      { 
        cout << "Unknown Message Type Received" << endl;
      }
    }
  } // End switch on message type

  return;
}


//***********************************************************
// This converts a data packet into a q512 packet and send it.
//***********************************************************

int Incoming_K2_Packet_Processor::convert_data_packet(CPacket& k2_packet,
						      SocketControl& dl_socket,
					              QPacket& q_packet)
{

  int res;

  // Decode the Packet Header

  res = DecodePacketHeader(k2_packet,packet_header);
  if (res != SUCCESS)
  {
    return(FAILURE);
  }

  // Decode the Data block header
  res = DecodeDataBlockHeader(k2_packet,data_header);
  if (res != SUCCESS)
  {
    return(FAILURE);
  }

  // Decompress the data 

  res = DecodeSamples(datablock,(PACKET_DATA *)k2_packet.GetDataPointerAt(0));
  if (res != SUCCESS)
  {
    return(FAILURE);
  }

  // Get component number

//  cout << "Data Packet Length in Bytes : " 
//    << packet_header.data_length << endl;


  // Insert Non Specific packet field

  q_packet.SetHeaderTrailer();
  q_packet.SetLinkSeqNo();
  q_packet.SetComponent(data_header.stream_number);
  q_packet.SetStream(data_header.stream_number);
  q_packet.SetStationName(station_name);
  q_packet.SetDataRate(100); // Find where this is in packet

  // Set up several required variables
      
  long sample;
  int samples_used = 0;
  int StartLoc = 0;
  int EndLoc = SAMPLES_IN_K2_PACKET;
  int isdata = TRUE;
  dframes = 0;
  dataframes = 7;

  switch(data_header.stream_number)
  {
    case 0:
    {
      if(chans.c_0 == TRUE)
      {
//        cout << "LG N Stream : Seq No : " << 
//          data_header.stream_sequence_number << endl;

        if (data_header.stream_sequence_number != 
              (unsigned int) (last_seq_no_0 + 1) )
        {
          cout << "Data Gap for Stream : " << 
            data_header.stream_number << endl;
        }
        last_seq_no_0 = data_header.stream_sequence_number;

        if(invert.invert_0 == TRUE)
        {
          invert_data(datablock,SAMPLES_IN_K2_PACKET);
        }


         // Preseve last two samples for continuity

        slast_0  = datablock[99];
        snlast_0 = datablock[98];

        while (  (isdata == TRUE) ||
                ( (isdata == FALSE) && (dframes != dataframes) ) )
        {
          //loop while we produce a complete frame 
        
          while(isdata && (peek_threshold_avail(gdp_0->adp->ccp) < 0) )
          {
            sample = datablock[StartLoc];

            if (peek_write(gdp_0->adp->ccp,&sample,1) != 1)
            {
              cout << "Error filling Compression Buffer" << endl;
            }

            StartLoc++;
   
            if (StartLoc >= EndLoc)
            {
              isdata = FALSE;
              break; 
            }
          }
          
          /* Compress a frame into a record */

          dframes = compress_generic_record(gdp_0,reserved_dataframes);
          
          if (dframes == dataframes)
          {
            // Here we have the proper number of frames for a packet

            int this_pass = generic_record_samples(gdp_0);
            samples_used += this_pass; 

            clear_generic_compression(gdp_0,reserved_dataframes);

            if (peek_contents(gdp_0->adp->ccp) > 0 )
            {
              cout << "Full packet but data left over" << endl;
              isdata = FALSE;
            }

            clear_running_compression(gdp_0->adp->ccp,level,
				      slast_0,snlast_0);
          } 
        }   

	q_packet.SetTotalSamples(samples_used);

        q_packet.SetData(gdp_0);

        q_packet.SetDataSequenceNumber(data_header.stream_sequence_number);
        
        CK2Time packet_time(data_header.seconds);

        q_packet.SetTime(packet_time.GetYear(),
                         packet_time.GetMonth(),
			 packet_time.GetDayOfMonth(),
                         packet_time.GetHour(),
	                 packet_time.GetMinute(),
		         packet_time.GetSec() );

        q_packet.SetMilliseconds(data_header.milliseconds);

        q_packet.SetChecksum();

        q_packet.SetCRC();

        dl_socket.Send(q_packet);

      }
      else
      {
        cout << "Data packet for Unconfigured Stream : " << 
          data_header.stream_number << endl;
      }
      break;
    }
    case 1:
    {
      if(chans.c_1 == TRUE)
      {
 //       cout << "LG Z Stream : Seq No : " << 
 //         data_header.stream_sequence_number << endl;

        if (data_header.stream_sequence_number != 
              (unsigned int) (last_seq_no_1 + 1) )
        {
          cout << "Data Gap for Stream : " << 
            data_header.stream_number << endl;
        }

        last_seq_no_1 = data_header.stream_sequence_number;

        if(invert.invert_1 == TRUE)
        {
          invert_data(datablock,SAMPLES_IN_K2_PACKET);
        }


         // Preseve last two samples for continuity

        slast_1  = datablock[99];
        snlast_1 = datablock[98];

        while (  (isdata == TRUE) ||
                ( (isdata == FALSE) && (dframes != dataframes) ) )
        {
          //loop while we produce a complete frame 
        
          while(isdata && (peek_threshold_avail(gdp_1->adp->ccp) < 0) )
          {
            sample = datablock[StartLoc];

            if (peek_write(gdp_1->adp->ccp,&sample,1) != 1)
            {
              cout << "Error filling Compression Buffer" << endl;
            }

            StartLoc++;
   
            if (StartLoc >= EndLoc)
            {
              isdata = FALSE;
              break; 
            }
          }
          
          /* Compress a frame into a record */

          dframes = compress_generic_record(gdp_1,reserved_dataframes);
          
          if (dframes == dataframes)
          {
            // Here we have the proper number of frames for a packet

            int this_pass = generic_record_samples(gdp_1);
            samples_used += this_pass; 

            clear_generic_compression(gdp_1,reserved_dataframes);

            if (peek_contents(gdp_1->adp->ccp) > 0 )
            {
              cout << "Full packet but data left over" << endl;
              isdata = FALSE;
            }

            clear_running_compression(gdp_1->adp->ccp,level,
				      slast_1,snlast_1);
          } 
        }   

	q_packet.SetTotalSamples(samples_used);

        q_packet.SetData(gdp_1);

        q_packet.SetDataSequenceNumber(data_header.stream_sequence_number);
        
        CK2Time packet_time(data_header.seconds);

        q_packet.SetTime(packet_time.GetYear(),
                         packet_time.GetMonth(),
			 packet_time.GetDayOfMonth(),
                         packet_time.GetHour(),
	                 packet_time.GetMinute(),
		         packet_time.GetSec() );

        q_packet.SetMilliseconds(data_header.milliseconds);

        q_packet.SetChecksum();

        q_packet.SetCRC();

        dl_socket.Send(q_packet);

      }
      else
      {
        cout << "Data packet for Unconfigured Stream : " << 
          data_header.stream_number << endl;
      }
      break;
    }
    case 2:
    {
      if(chans.c_2 == TRUE)
      {
//        cout << "LG E Stream : Seq No : " << 
//          data_header.stream_sequence_number << endl;

        if (data_header.stream_sequence_number != 
              (unsigned int) (last_seq_no_2 + 1) )
        {
          cout << "Data Gap for Stream : " << 
            data_header.stream_number << endl;
        }

        last_seq_no_2 = data_header.stream_sequence_number;

        if(invert.invert_2 == TRUE)
        {
          invert_data(datablock,SAMPLES_IN_K2_PACKET);
        }


         // Preseve last two samples for continuity

        slast_2  = datablock[99];
        snlast_2 = datablock[98];

        while (  (isdata == TRUE) ||
                ( (isdata == FALSE) && (dframes != dataframes) ) )
        {
          //loop while we produce a complete frame 
        
          while(isdata && (peek_threshold_avail(gdp_2->adp->ccp) < 0) )
          {
            sample = datablock[StartLoc];

            if (peek_write(gdp_2->adp->ccp,&sample,1) != 1)
            {
              cout << "Error filling Compression Buffer" << endl;
            }

            StartLoc++;
   
            if (StartLoc >= EndLoc)
            {
              isdata = FALSE;
              break; 
            }
          }
          
          /* Compress a frame into a record */

          dframes = compress_generic_record(gdp_2,reserved_dataframes);
          
          if (dframes == dataframes)
          {
            // Here we have the proper number of frames for a packet

            int this_pass = generic_record_samples(gdp_2);
            samples_used += this_pass; 

            clear_generic_compression(gdp_2,reserved_dataframes);

            if (peek_contents(gdp_2->adp->ccp) > 0 )
            {
              cout << "Full packet but data left over" << endl;
              isdata = FALSE;
            }

            clear_running_compression(gdp_2->adp->ccp,level,
				      slast_2,snlast_2);
          } 
        }   

	q_packet.SetTotalSamples(samples_used);

        q_packet.SetData(gdp_2);

        q_packet.SetDataSequenceNumber(data_header.stream_sequence_number);
        
        CK2Time packet_time(data_header.seconds);

        q_packet.SetTime(packet_time.GetYear(),
                         packet_time.GetMonth(),
			 packet_time.GetDayOfMonth(),
                         packet_time.GetHour(),
	                 packet_time.GetMinute(),
		         packet_time.GetSec() );

        q_packet.SetMilliseconds(data_header.milliseconds);

        q_packet.SetChecksum();

        q_packet.SetCRC();

        dl_socket.Send(q_packet);

      }
      else
      {
        cout << "Data packet for Unconfigured Stream : " << 
          data_header.stream_number << endl;
      }
      break;
    }
    case 3:
    {
      if(chans.c_3 == TRUE)
      {
//        cout << "VBB N Stream : Seq No : " << 
//          data_header.stream_sequence_number << endl;

        if (data_header.stream_sequence_number != 
              (unsigned int) (last_seq_no_3 + 1) )
        {
          cout << "Data Gap for Stream : " << 
            data_header.stream_number << endl;
        }

        last_seq_no_3 = data_header.stream_sequence_number;

        if(invert.invert_3 == TRUE)
        {
          invert_data(datablock,SAMPLES_IN_K2_PACKET);
        }


         // Preseve last two samples for continuity

        slast_3  = datablock[99];
        snlast_3 = datablock[98];

        while (  (isdata == TRUE) ||
                ( (isdata == FALSE) && (dframes != dataframes) ) )
        {
          //loop while we produce a complete frame 
        
          while(isdata && (peek_threshold_avail(gdp_3->adp->ccp) < 0) )
          {
            sample = datablock[StartLoc];

            if (peek_write(gdp_3->adp->ccp,&sample,1) != 1)
            {
              cout << "Error filling Compression Buffer" << endl;
            }

            StartLoc++;
   
            if (StartLoc >= EndLoc)
            {
              isdata = FALSE;
              break; 
            }
          }
          
          /* Compress a frame into a record */

          dframes = compress_generic_record(gdp_3,reserved_dataframes);
          
          if (dframes == dataframes)
          {
            // Here we have the proper number of frames for a packet

            int this_pass = generic_record_samples(gdp_3);
            samples_used += this_pass; 

            clear_generic_compression(gdp_3,reserved_dataframes);

            if (peek_contents(gdp_3->adp->ccp) > 0 )
            {
              cout << "Full packet but data left over" << endl;
              isdata = FALSE;
            }

            clear_running_compression(gdp_3->adp->ccp,level,
				      slast_3,snlast_3);
          } 
        }   

	q_packet.SetTotalSamples(samples_used);

        q_packet.SetData(gdp_3);

        q_packet.SetDataSequenceNumber(data_header.stream_sequence_number);
        
        CK2Time packet_time(data_header.seconds);

        q_packet.SetTime(packet_time.GetYear(),
                         packet_time.GetMonth(),
			 packet_time.GetDayOfMonth(),
                         packet_time.GetHour(),
	                 packet_time.GetMinute(),
		         packet_time.GetSec() );

        q_packet.SetMilliseconds(data_header.milliseconds);

        q_packet.SetChecksum();

        q_packet.SetCRC();

        dl_socket.Send(q_packet);

      }
      else
      {
        cout << "Data packet for Unconfigured Stream : " << 
          data_header.stream_number << endl;
      }
      break;
    }
    case 4:
    {
      if(chans.c_4 == TRUE)
      {
 //       cout << "VBB Z Stream : Seq No : " << 
 //         data_header.stream_sequence_number << endl;

        if (data_header.stream_sequence_number != 
              (unsigned int) (last_seq_no_4 + 1) )
        {
          cout << "Data Gap for Stream : " << 
            data_header.stream_number << endl;
        }

        last_seq_no_4 = data_header.stream_sequence_number;

        if(invert.invert_4 == TRUE)
        {
          invert_data(datablock,SAMPLES_IN_K2_PACKET);
        }

         // Preseve last two samples for continuity

        slast_4  = datablock[99];
        snlast_4 = datablock[98];

        while (  (isdata == TRUE) ||
                ( (isdata == FALSE) && (dframes != dataframes) ) )
        {
          //loop while we produce a complete frame 
        
          while(isdata && (peek_threshold_avail(gdp_4->adp->ccp) < 0) )
          {
            sample = datablock[StartLoc];

            if (peek_write(gdp_4->adp->ccp,&sample,1) != 1)
            {
              cout << "Error filling Compression Buffer" << endl;
            }

            StartLoc++;
   
            if (StartLoc >= EndLoc)
            {
              isdata = FALSE;
              break; 
            }
          }
          
          /* Compress a frame into a record */

          dframes = compress_generic_record(gdp_4,reserved_dataframes);
          
          if (dframes == dataframes)
          {
            // Here we have the proper number of frames for a packet

            int this_pass = generic_record_samples(gdp_4);
            samples_used += this_pass; 

            clear_generic_compression(gdp_4,reserved_dataframes);

            if (peek_contents(gdp_4->adp->ccp) > 0 )
            {
              cout << "Full packet but data left over" << endl;
              isdata = FALSE;
            }

            clear_running_compression(gdp_4->adp->ccp,level,
				      slast_4,snlast_4);
          } 
        }   

	q_packet.SetTotalSamples(samples_used);

        q_packet.SetData(gdp_4);

        q_packet.SetDataSequenceNumber(data_header.stream_sequence_number);
        
        CK2Time packet_time(data_header.seconds);

        q_packet.SetTime(packet_time.GetYear(),
                         packet_time.GetMonth(),
			 packet_time.GetDayOfMonth(),
                         packet_time.GetHour(),
	                 packet_time.GetMinute(),
		         packet_time.GetSec() );

        q_packet.SetMilliseconds(data_header.milliseconds);

        q_packet.SetChecksum();

        q_packet.SetCRC();

        dl_socket.Send(q_packet);

      }
      else
      {
        cout << "Data packet for Unconfigured Stream : " << 
          data_header.stream_number << endl;
      }
      break;
    }
    case 5:
    {
      if(chans.c_5 == TRUE)
      {
//        cout << "VBB E Stream : Seq No : " << 
//          data_header.stream_sequence_number << endl;

        if (data_header.stream_sequence_number != 
              (unsigned int) (last_seq_no_5 + 1) )
        {
          cout << "Data Gap for Stream : " << 
            data_header.stream_number << endl;
        }

        last_seq_no_5 = data_header.stream_sequence_number;

        if(invert.invert_5 == TRUE)
        {
          invert_data(datablock,SAMPLES_IN_K2_PACKET);
        }


         // Preseve last two samples for continuity

        slast_5  = datablock[99];
        snlast_5 = datablock[98];

        while (  (isdata == TRUE) ||
                ( (isdata == FALSE) && (dframes != dataframes) ) )
        {
          //loop while we produce a complete frame 
        
          while(isdata && (peek_threshold_avail(gdp_5->adp->ccp) < 0) )
          {
            sample = datablock[StartLoc];

            if (peek_write(gdp_5->adp->ccp,&sample,1) != 1)
            {
              cout << "Error filling Compression Buffer" << endl;
            }

            StartLoc++;
   
            if (StartLoc >= EndLoc)
            {
              isdata = FALSE;
              break; 
            }
          }
          
          /* Compress a frame into a record */

          dframes = compress_generic_record(gdp_5,reserved_dataframes);
          
          if (dframes == dataframes)
          {
            // Here we have the proper number of frames for a packet

            int this_pass = generic_record_samples(gdp_5);
            samples_used += this_pass; 

            clear_generic_compression(gdp_5,reserved_dataframes);

            if (peek_contents(gdp_5->adp->ccp) > 0 )
            {
              cout << "Full packet but data left over" << endl;
              isdata = FALSE;
            }

            clear_running_compression(gdp_5->adp->ccp,level,
				      slast_5,snlast_5);
          } 
        }   

	q_packet.SetTotalSamples(samples_used);

        q_packet.SetData(gdp_5);

        q_packet.SetDataSequenceNumber(data_header.stream_sequence_number);
        
        CK2Time packet_time(data_header.seconds);

        q_packet.SetTime(packet_time.GetYear(),
                         packet_time.GetMonth(),
			 packet_time.GetDayOfMonth(),
                         packet_time.GetHour(),
	                 packet_time.GetMinute(),
		         packet_time.GetSec() );

        q_packet.SetMilliseconds(data_header.milliseconds);

        q_packet.SetChecksum();

        q_packet.SetCRC();

        dl_socket.Send(q_packet);

      }
      else
      {
        cout << "Data packet for Unconfigured Stream : " << 
          data_header.stream_number << endl;
      }
      break;
    }
    default:
    {
      cout << "Unknown Stream Found " << endl;
      break;
    }
  } // End switch on stream number


  return(SUCCESS);
}

//***********************************************************
// This is a utility that decompress a data packet into an array
// of samples
//***********************************************************

int Incoming_K2_Packet_Processor::DecodeSamples(int* Samples, 
						PACKET_DATA* PktData) 
{
   BYTE sampbyte; 
   BYTE currentsample[4];
   int currentint;
   int referenceint = 0;
   int currentbyte = 15;

   int OutSample= 0;

   //
   //

   for (int i= 0; i < SAMPLES_IN_K2_PACKET; i++) 
   {

     memset(currentsample,0,4); // Clear current sample

     sampbyte = PktData->Data[++currentbyte];

     if (sampbyte & 0x80)
     {

       currentsample[1]  = sampbyte;
       currentsample[2]  = PktData->Data[++currentbyte];
       currentsample[3]  = PktData->Data[++currentbyte];

       if (currentsample[1] & 0x40)
       {
         currentsample[0] = 0xFF;   
         currentsample[1] = currentsample[1] | 0x80;
       } 
       else
       {
         currentsample[1] = currentsample[1] & 0x7F; //Mask out the flag
       }
       memcpy(&currentint,currentsample,4);
       currentint = currentint<<1; // Shift to the left one bit
     }
     else  // This means high bit was not set
     {

       currentsample[2]  = sampbyte;
       currentsample[3] = PktData->Data[++currentbyte];

       if (currentsample[2] & 0x40) 
       {
         currentsample[0] = 0xFF;   
         currentsample[1] = 0xFF;   
         currentsample[2] = currentsample[2] | 0x80;
       } 

       memcpy(&currentint,currentsample,4);
       currentint = currentint << 1; // Shift one bit to the left
       currentint = referenceint + currentint;
     }

     referenceint = currentint;
     Samples[OutSample++] = currentint;
   }
   return(SUCCESS);
}


//***********************************************************
// This moves the header contents from the packet into a structure
//***********************************************************

int Incoming_K2_Packet_Processor::DecodePacketHeader(CPacket& rxpacket,
                           struct k2_packet_header_type& pack_header)
{
  BYTE bvalue;
  WORD wvalue;
  int ivalue;


  bvalue = rxpacket.GetTypeCode();
  pack_header.packet_type = (int) bvalue;

  bvalue = rxpacket.GetSeqNo();
  pack_header.link_level_seq_no = (int) bvalue;

  wvalue = rxpacket.GetSource();
  pack_header.source_id = (int) wvalue;
  
  wvalue = rxpacket.GetDestination();
  pack_header.destination_id = (int) wvalue;

  wvalue = rxpacket.GetDataLength();
  pack_header.data_length = (int) wvalue;

  return(SUCCESS);
}

//***********************************************************
// This move the header contents into a structure
//***********************************************************
 
int Incoming_K2_Packet_Processor::DecodeDataBlockHeader(CPacket& rxpacket,
                              struct k2_data_block_header_type& data_b_header)
{

  BYTE bvalue;
  WORD wvalue;
  unsigned int  ivalue;
  unsigned char b2[2];
  unsigned char b4[4];
  unsigned char b8[8];


  // Instrument serial number

  b2[0] = rxpacket.GetDataAt(0);
  b2[1] = rxpacket.GetDataAt(1);

  memcpy(&wvalue,b2,2);
  data_b_header.instrument_serial_number = (int) wvalue; 

  // Stream Number

  bvalue = rxpacket.GetDataAt(2);
  data_b_header.stream_number = (int) bvalue;

  // Status fields

  bvalue = rxpacket.GetDataAt(3);

  if ( (bvalue & 0x80) > 0)
  {
    data_b_header.gps_fault = TRUE;
  }
  else
  {
    data_b_header.gps_fault = FALSE;
  }


  if ( (bvalue & 0x40) > 0)
  {
    data_b_header.hardware_fault = TRUE;
  }
  else
  {
    data_b_header.hardware_fault = FALSE;
  }

  if ( (bvalue & 0x20) > 0)
  {
    data_b_header.memory_fault = TRUE;
  }
  else
  {
    data_b_header.memory_fault = FALSE;
  }

  if ( (bvalue & 0x10) > 0)
  {
    data_b_header.battery_low = TRUE;
  }
  else
  {
    data_b_header.battery_low = FALSE;
  }

  BYTE nbvalue = (bvalue & 0x0F);
  data_b_header.compression_mode = (int) nbvalue;

  // Sequence Number

  b4[0] = rxpacket.GetDataAt(4);
  b4[1] = rxpacket.GetDataAt(5);
  b4[2] = rxpacket.GetDataAt(6);
  b4[3] = rxpacket.GetDataAt(7);

  memcpy(&ivalue,b4,4);

  data_b_header.stream_sequence_number = (unsigned int) ivalue;

  // K2 Time

  b4[0] = rxpacket.GetDataAt(8);
  b4[1] = rxpacket.GetDataAt(9);
  b4[2] = rxpacket.GetDataAt(10);
  b4[3] = rxpacket.GetDataAt(11);

  memcpy(&ivalue,b4,4);
  data_b_header.seconds = (unsigned int) ivalue;


  b4[0] = 0x00;
  b4[1] = 0x00;
  b4[2] = rxpacket.GetDataAt(12);
  b4[3] = rxpacket.GetDataAt(13);

  memcpy(&ivalue,b4,4);
  data_b_header.milliseconds = (unsigned int) ivalue;

 return(SUCCESS);
}


//***********************************************************
// This inverts the data by mulitiplying by negative one.
//***********************************************************

void Incoming_K2_Packet_Processor::invert_data(int dblock[], 
                                               int num_samples)
{
  for(int i=0;i<num_samples;i++)
  {
    dblock[i] = (-1) * dblock[i]; 
  }
}
