/***********************************************************

File Name :
	cba.C

Programmer:
	Phil Maechling

Description:
	This is the CIT version of the byte array class. Microsoft
	delivers this on the PC. It was rewritten here for the unix
	machine.

Limitations or Warnings:


Creation Date:
	28 March 1995

Modification History:

**********************************************************/

#include <iostream.h>
#include <string.h>
#include <assert.h>
#include "verbose.h"
#include "cba.h"


/////////////////////////////////////////////////////////////////////////////
//
// Implementation of parameterized Array
//
/////////////////////////////////////////////////////////////////////////////
// NOTE: we allocate an array of 'm_nMaxSize' elements, but only
//  the current size 'm_nSize' contains properly constructed
//  objects.



/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

Verbosity get;

CByteArray::CByteArray()
{
  m_pData = NULL;
  m_nSize = m_nMaxSize = m_nGrowBy = 0;

  if (get.verbose_mode())
  {
    cout << "Completing cbyte array construction " << endl;
  }
}

CByteArray::~CByteArray()
{
  assert(this);
  delete [] (BYTE*)m_pData;

  if (get.verbose_mode())
  {
    cout << "Completing cbyte array destruction " << endl;
  }
}

void CByteArray::SetSize(int nNewSize, int nGrowBy /* = -1 */)
{
	assert(this);
	assert(nNewSize >= 0);

	if (nGrowBy != -1)
	  m_nGrowBy = nGrowBy;  // set new size

	if (nNewSize == 0)
	{
	  // shrink to nothing
	  delete [] (BYTE*)m_pData;
	  m_pData = NULL;
	  m_nSize = m_nMaxSize = 0;

          if (get.verbose_mode())
          {
            cout << "In CByteArray::SetSize "<< endl;
            cout << "m_nSize "<< m_nSize << endl;
            cout << "m_nMaxSize "<< m_nMaxSize << endl;
            cout << "m_nGrowBy " << m_nGrowBy << endl;
          }
	}
	else if (m_pData == NULL)
	{
            // check for exceeding maximum size
            assert((long)nNewSize * sizeof(BYTE) <= SIZE_T_MAX);

	    // create one with exact size
	    m_pData = (BYTE*) new BYTE[nNewSize * sizeof(BYTE)];
	    memset(m_pData, 0, nNewSize * sizeof(BYTE));  // zero fill
	    m_nSize = m_nMaxSize = nNewSize;

	    if (get.verbose_mode())
            {
              cout << "In CByteArray::SetSize "<< endl;
              cout << "m_nSize "<< m_nSize << endl;
              cout << "m_nMaxSize "<< m_nMaxSize << endl;
              cout << "m_nGrowBy " << m_nGrowBy << endl;
             }
	}
	else if (nNewSize <= m_nMaxSize)
	{
	  // it fits
	  if (nNewSize > m_nSize)
	  {
	    // initialize the new elements

	    memset(&m_pData[m_nSize], 0, (nNewSize-m_nSize) * sizeof(BYTE));
           
            if (get.verbose_mode())
            {
	      cout << "In CByteArray::SetSize "<< endl;
              cout << "m_nSize "<< m_nSize << endl;
              cout << "m_nMaxSize "<< m_nMaxSize << endl;
              cout << "m_nGrowBy " << m_nGrowBy << endl;
            }
	  }

	  m_nSize = nNewSize;

          if(get.verbose_mode())
          {
            cout << "In CByteArray::SetSize "<< endl;
            cout << "m_nSize "<< m_nSize << endl;
            cout << "m_nMaxSize "<< m_nMaxSize << endl;
            cout << "m_nGrowBy " << m_nGrowBy << endl;
          }
	}
	else
	{
	  // Otherwise grow array
	  int nNewMax;

	  if (nNewSize < m_nMaxSize + m_nGrowBy)
          {
	    nNewMax = m_nMaxSize + m_nGrowBy;  // granularity
          }
	  else
          {
	    nNewMax = nNewSize;  // no slush
	  }

            assert((long) nNewMax * sizeof(BYTE) <= SIZE_T_MAX);

	    BYTE* pNewData = (BYTE*) new BYTE[nNewMax * sizeof(BYTE)];

	    // copy new data from old
	    memcpy(pNewData, m_pData, m_nSize * sizeof(BYTE));

	    // construct remaining elements
	    assert(nNewSize > m_nSize);

            memset(&pNewData[m_nSize], 0, (nNewSize-m_nSize) * sizeof(BYTE));

	    // get rid of old stuff (note: no destructors called)
	    delete [] (BYTE*)m_pData;
	    m_pData = pNewData;
	    m_nSize = nNewSize;
	    m_nMaxSize = nNewMax;

	    if (get.verbose_mode())
	    {
              cout << "In CByteArray::SetSize "<< endl;
              cout << "m_nSize "   << m_nSize    << endl;
              cout << "m_nMaxSize "<< m_nMaxSize << endl;
              cout << "m_nGrowBy " << m_nGrowBy  << endl;
            }
	}
}


int CByteArray::GetSize() const
{

  return(m_nSize);

}

int CByteArray::GetUpperBound() const
{

  return(m_nSize - 1);

}

BYTE CByteArray::GetAt(int nIndex) const
{

  if (nIndex < m_nSize)
  {
    return (m_pData[nIndex]);
  }
  else
  {
    return(NULL);
  }
}

void CByteArray::SetAt(int nIndex, BYTE newElement)
{

  if (nIndex < m_nSize)
  {
    m_pData[nIndex] = newElement;
  }
}

void CByteArray::FreeExtra()
{
	assert(this);
       
        if (get.verbose_mode())
        { 
          cout << "Start of FreeExtra " << endl;
          cout << m_nMaxSize << endl;
        }

	if (m_nSize != m_nMaxSize)
	{
	  // shrink to desired size
	  assert((long)m_nSize * sizeof(BYTE) <= SIZE_T_MAX);  // no overflow
	  BYTE* pNewData = NULL;
	  if (m_nSize != 0)
	  {
	    pNewData = (BYTE*) new BYTE[m_nSize * sizeof(BYTE)];
	    // copy new data from old
	    memcpy(pNewData, m_pData, m_nSize * sizeof(BYTE));
	   }

	  // get rid of old stuff (note: no destructors called)
	  delete [] (BYTE*)m_pData;
	  m_pData = pNewData;
	  m_nMaxSize = m_nSize;

	  if (get.verbose_mode())
          {
            cout << "End of FreeExtra " << endl;
            cout << m_nMaxSize << endl;
          }

	}
}


void CByteArray::RemoveAll()
{
  m_nSize = 0;
  m_nGrowBy = 0; 
}


BYTE& CByteArray::ElementAt (int nIndex)
{
  assert(m_nSize);
  return (m_pData[nIndex]);
}


void CByteArray::SetAtGrow(int nIndex, BYTE newElement)
{
	assert(this);
	assert(nIndex >= 0);

	if (nIndex >= m_nSize)
		SetSize(nIndex+1);
	m_pData[nIndex] = newElement;
}


int CByteArray::Add(BYTE newElement)
{

  if ((m_nSize) >= SIZE_T_MAX)
  {
    cout << "Trying to add elements over max array size";
    return(0); 
  }
  assert(m_nSize < SIZE_T_MAX);
 
  SetAtGrow((m_nSize),newElement);
  
  return(GetUpperBound());
   
}

BYTE CByteArray::operator[](int nIndex) const
{

  assert(this);
  if (nIndex < m_nSize)
  {
    return(m_pData[nIndex]);
  }
  else
  {
    cout << "Trying to return data outside array range " << endl;
    return(0);
  }
}


BYTE& CByteArray::operator[](int nIndex) 
{

  assert(this);
  if (nIndex < m_nSize)
  {
    return(ElementAt(nIndex));
  }
  else
  {
    cout << "Trying to return data outside array range " << endl;
    return(ElementAt(0));
  }
}


void CByteArray::InsertAt(int nIndex, BYTE newElement, int nCount /*=1*/)
{
	assert(this);
	assert(nIndex >= 0);    // will expand to meet need
	assert(nCount > 0);     // zero or negative size not allowed

	if (nIndex >= m_nSize)
	{
		// adding after the end of the array
		SetSize(nIndex + nCount);  // grow so nIndex is valid
	}
	else
	{
		// inserting in the middle of the array
		int nOldSize = m_nSize;
		SetSize(m_nSize + nCount);  // grow it to new size
		// shift old data up to fill gap
		memmove(&m_pData[nIndex+nCount], &m_pData[nIndex],
			(nOldSize-nIndex) * sizeof(BYTE));

		// re-init slots we copied from

		memset(&m_pData[nIndex], 0, nCount * sizeof(BYTE));

	}

	// insert new value in the gap
	assert(nIndex + nCount <= m_nSize);
	while (nCount--)
		m_pData[nIndex++] = newElement;
}

void CByteArray::RemoveAt(int nIndex, int nCount /* = 1 */)
{
	assert(this);
	assert(nIndex >= 0);
	assert(nCount >= 0);
	assert(nIndex + nCount <= m_nSize);

	// just remove a range
	int nMoveCount = m_nSize - (nIndex + nCount);

	if (nMoveCount)
		memcpy(&m_pData[nIndex], &m_pData[nIndex + nCount],
			nMoveCount * sizeof(BYTE));
	m_nSize -= nCount;
}

void CByteArray::InsertAt(int nStartIndex, CByteArray* pNewArray)
{
	assert(this);
	assert(pNewArray != NULL);
	assert(pNewArray);
	assert(nStartIndex >= 0);

     if (pNewArray->GetSize() > 0)
     {
	InsertAt(nStartIndex, pNewArray->GetAt(0), pNewArray->GetSize());
	for (int i = 0; i < pNewArray->GetSize(); i++)
		SetAt(nStartIndex + i, pNewArray->GetAt(i));
     }
}


//////////////////////////////////////////////////////////////////////////
// Diagnostics

void CByteArray::Dump() const
{

  int i;
  int inloop;
  assert(this);

  inloop = 0;
  for (i=0; i < m_nSize; i++)
  {
    cout << (BYTE *) m_pData[i] << " ";
    inloop++;
    if (inloop == 10)
    {
      inloop = 0;
      cout << endl;
    }
  }
  cout << endl;
}
