/*   Client Test Program. Gets data from server.
     Copyright 1994 Quanterra, Inc.
     Written by Woodrow H. Owens
     Modified for datasock by Douglas Neuhauser, UC Berkeley Seismographic Station

Edit History:
   Ed Date      By  Changes
   -- --------- --- ---------------------------------------------------
    0 11 Mar 94 WHO First created.
    1 30 Mar 94 WHO Uses new service calls.
*/
#include <stdio.h>

char *syntax[] = {
"    [-v n] [-h] [-p port] station channel_list",
"    where:",
"	-p port		Specify port number for server.",
"			If no port is specified, assume port is already open",
"			on unit 0 (from inetd).",
"	-v n		Set verbosity level to n.",
"	-h		Help - prints syntax message.",
"	station		station name.",
"	channel_list	list of channels (comma-delimited).",
NULL };


#include <stdlib.h>
#include <termio.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <sys/time.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <math.h>
double log2(double);

#include "qlib.h"

#include "dpstruc.h"
#include "seedstrc.h"
#include "stuff.h"
#include "timeutil.h"
#include "service.h"
#include "cfgutil.h"

#define	TIMESTRLEN  40


/************************************************************************/
/*  Externals required in multiple files.				*/
/************************************************************************/
int port = -1;			/* No explicit port means the program	*/
				/* should inherit the socket on stdout.	*/
FILE *info = stdout;		/* Default FILE for messages.		*/
char	*cmdname;		/* Program name.			*/
short data_mask =		/* data mask for cs_setup.		*/
    (CSIM_DATA);
static char name[5] = "DSOC" ;	/* Default client name			*/
char sname[5] = "*" ;		/* Default station list.		*/
tstations_struc stations ;
typedef char char23[24] ;
typedef char char5[6] ;

char23 stats[11] = { "Good", "Enqueue Timeout", "Service Timeout", "Init Error",
                       "Attach Refused", "No Data", "Server Busy", "Invalid Command",
                       "Server Dead", "Server Changed", "Segment Error" } ;

char *gmt_clock_string(time_t clock);
void finish_handler(int sig);
int terminate_program (int error);
static int terminate_proc;
static int verbosity;
static pclient_struc me = NULL;

typedef struct _sel {		/* Structure for storing selectors.	*/
    int nselectors;		/* Number of selectors.			*/
    seltype *selectors;		/* Ptr to selector array.		*/
} SEL;

static SEL sel[NUMQ];

/************************************************************************/
/*  print_syntax:							*/
/*	Print the syntax description of program.			*/
/************************************************************************/
void
print_syntax(cmd)
    char *cmd;
{
    int i;
    fprintf (info, "%s", cmd);
    for (i=0; syntax[i] != NULL; i++) fprintf (info, "%s\n", syntax[i]);
}



/************************************************************************/
/*  main procedure							*/
/************************************************************************/
main (int argc, char *argv[], char **envp)
    begin
      pclient_station this ;
      short j, k, err ;
      boolean alert ;
      pdata_user pdat ;
      seed_record_header *pseed ;
      pselarray psa ;
      int sd;

      config_struc cfg;
      char str1[160], str2[160], station_dir[160], station_desc[60], source[160];
      char filename[160];
      char time_str[TIMESTRLEN];
      int i ;

    /* Variables needed for getopt. */
    extern char	*optarg;
    extern int	optind, opterr;
    int		c;

    cmdname = argv[0];
    while ( (c = getopt(argc,argv,"hv:p:")) != -1)
	switch (c) {
	case '?':
	case 'h':   print_syntax (cmdname); exit(0);
	case 'v':   verbosity=atoi(optarg); break;
	case 'p':   port=atoi(optarg); break;
	}

    /*	Skip over all options and their arguments.			*/
    argv = &(argv[optind]);
    argc -= optind;
    if (port < 0) info = stderr;

/* Allow override of station name on command line */
      if (argc > 0)
        then
          begin
            strncpy(sname, argv[0], 4) ;
            sname[4] = '\0' ;
          end
        else
	  begin
	    fprintf (stderr, "Missing station name\n");
	    exit(1);
	  end
      upshift(sname) ;

/* Skip over station name to channel name(s).	*/
    ++argv;
    --argc;

/* open the stations list and look for that station */
      strcpy (filename, "/etc/stations.ini") ;
      if (open_cfg(&cfg, filename, sname))
        then
	  begin 
            fprintf (stderr,"Could not find station\n") ;
	    exit(1);
	  end

/* Try to find the station directory, source, and description */
      do
        begin
          read_cfg(&cfg, str1, str2) ;
          if (str1[0] == '\0')
            then
              break ;
          if (strcmp(str1, "DIR") == 0)
            then
              strcpy(station_dir, str2) ;
          else if (strcmp(str1, "DESC") == 0)
            then
              begin
                strcpy(station_desc, str2) ;
                station_desc[59] = '\0' ;
		if (port >= 0) {
		    fprintf (info, "%s %s startup - %s\n", 
			     localtime_string(dtime()),
			     sname, station_desc) ;
		}
              end
          else if (strcmp(str1, "SOURCE") == 0)
            then
              strcpy(source, str2) ;
        end
      while (1) ;
      close_cfg(&cfg) ;
      
      terminate_proc = 0;
      signal (SIGINT,finish_handler);
      signal (SIGTERM,finish_handler);
      /* Set up a condition handler for SIGPIPE, since a write to a	*/
      /* close pipe/socket raises a alarm, not an error return code.	*/
      signal (SIGPIPE,finish_handler);

/* Generate an entry for all available stations */      
      cs_setup (&stations, name, sname, TRUE, TRUE, 10, 7, data_mask, 6000) ;

/* Create my segment and attach to all stations */      
      me = cs_gen (&stations) ;

/* Set up special selectors. */
      save_selectors (0, argv[0]);
      set_selectors (me);

/* Show beginning status of all stations */
/*::      strcpy(time_str, gmt_clock_string(time(NULL)));*/
      strcpy(time_str, localtime_string(dtime()));
      for (j = 0 ; j < me->maxstation ; j++)
        begin
          this = (pclient_station) ((long) me + me->offsets[j]) ;
	  this->seqdbuf = CSQ_LAST;
	  if (port >= 0) {
	    fprintf (info, "%s - [%s] Status=%s\n", time_str, long_str(this->name.l), 
		     &(stats[this->status])) ;
	  }
        end
      fflush (info);

    /* Either setup the socket from the specified port number,		*/
    /* or assume that the socket was inherited from parent (inetd).	*/
    if (port >= 0)
	sd = setup_socket(port);
    else
	sd = 0;


/* Send a message every second to all stations. Sending an attach command is harmless */
      do
        begin
          j = cs_scan (me, &alert) ;
          if (j != NOCLIENT)
            then
              begin
                this = (pclient_station) ((long) me + me->offsets[j]) ;
                if (alert)
                  then
		    begin
		      strcpy(time_str, localtime_string(dtime()));
		      if (port >= 0) {
			fprintf (info, "%s - New status on station %s is %s\n", time_str,
				 long_str(this->name.l), &(stats[this->status])) ;
			fflush (info);
		      }
		    end
                if (this->valdbuf)
                  then
                    begin
                      pdat = (pdata_user) ((long) me + this->dbufoffset) ;
                      for (k = 0 ; k < this->valdbuf ; k++)
                        begin
                          pseed = (pvoid) &pdat->data_bytes ;
			  if (verbosity & 1)
			  then
			    begin
			      if (port >= 0) {
				fprintf (info, "[%-4.4s] <%2d> %s recvtime=%s ",
					 &this->name, k, seednamestring(&pseed->channel_id, 
									&pseed->location_id), localtime_string(pdat->reception_time)) ;
				printf("hdrtime=%s\n", time_string(pdat->header_time)) ;
				fflush (info);
			      }
			    end
			  write_seed(sd, pseed);
                          pdat = (pdata_user) ((long) pdat + this->dbufsize) ;
                        end
                    end
              end
            else
	      begin
		if ((verbosity & 2) && port >= 0) {
		    fprintf (info, "sleeping...");
		    fflush (info);
		}
		sleep (1) ; /* Bother the server once every second */
		if ((verbosity & 2) && port >= 0) {
		    fprintf (info, "awake\n");
		    fflush (info);
		}
	      end
        end
      while (! terminate_proc) ;
    terminate_program (0);
      return(0);
    end

void finish_handler(int sig)
{
    terminate_proc = 1;
    signal (sig,finish_handler);    /* Re-install handler (for SVR4)	*/
}

char *gmt_clock_string(time_t clock)
{
    static char time_str[TIMESTRLEN];
    strftime(time_str, TIMESTRLEN, "%Y/%m/%d,%T GMT", gmtime(&clock));
    return(time_str);
}

/************************************************************************/
/*  terminate_program							*/
/*	Terminate prog and return error code.  Clean up on the way out.	*/
/************************************************************************/
int terminate_program (int error) 
{
    pclient_station this ;
    char time_str[TIMESTRLEN];
    int j;
    boolean alert ;

    strcpy(time_str, localtime_string(dtime()));
    if ((verbosity & 2) && port >= 0) {
	fprintf (info, "%s - Terminating program.\n", time_str);
	fflush (info);
    }

    /* Perform final cs_scan for 0 records to ack previous records.	*/
    /* Detach from all stations and delete my segment.			*/
    if (me != NULL) {
	for (j=0; j< me->maxstation; j++) {
	    this = (pclient_station) ((long) me + me->offsets[0]) ;
	    this->reqdbuf = 0;
	}
	strcpy(time_str, localtime_string(dtime()));
	if (port >= 0) {
	    fprintf (info, "%s - Final scan to ack all received packets\n", time_str);
	    fflush (info);
	}
	cs_scan (me, &alert);
	cs_off (me) ;
    }

    strcpy(time_str, localtime_string(dtime()));
    if (port >= 0) {
	fprintf (info, "%s - Terminated\n", time_str);
    }
    exit(error);
}

char *selector_type_str[] = {"DAT", "DET", "CAL"};
/************************************************************************/
/*  save_selectors:							*/
/*	Parse and save selectors for specific types of info.		*/
/************************************************************************/
save_selectors(int type, char *str)
{
    char *token;
    int i;
    seltype *selectors = NULL;
    char *p = str;
    int n = 0;

    if (str == NULL || (int)strlen(str) <= 0) return;
    sel[type].nselectors = 0;
    if (sel[type].selectors) free (sel[type].selectors);
    while (token = strtok(p,",")) {
	if ((int)strlen(token) > 5) {
	    fprintf (info, "Error in selector list for %s\n",
		     selector_type_str[type]);
	    if (selectors) free (selectors);
	    return;
	}
	selectors = (selectors == NULL) ? (seltype *)malloc(sizeof(seltype)) : 
	    (seltype *)realloc (selectors, (n+1)*sizeof(seltype));
	if (selectors == NULL) {
	    fprintf (info, "Error allocating selector space for %s\n",
		     selector_type_str[type]);
	    return;
	}
	strcpy(selectors[n++],lead(5,'?',token));
	p = NULL;
    }
    sel[type].selectors = selectors;
    sel[type].nselectors = n;
    return;
}

/************************************************************************/
/*  set_selectors:							*/
/*	Set selector values for the single station.			*/
/*	Assume sufficient selectors have been reserved.			*/
/************************************************************************/
set_selectors (pclient_struc me)
{
    pclient_station this ;
    pselarray psa ;
    int nsel = 1;
    int type, n, i;

    this = (pclient_station) ((long) me + me->offsets[0]) ;
    psa = (pselarray) ((long) me + this->seloffset) ;
	
    for (type=0; type<CHAN; type++) {
	n = sel[type].nselectors;
	if (nsel+n > this->maxsel) {
	    fprintf (info, "Error: Require %d selectors, allocated %d\n",
		     nsel+n, this->maxsel);
	    return;
	}
	if (n > 0) {
	    this->sels[type].first = nsel;
	    this->sels[type].last = nsel + n - 1;
	    memcpy ((void *)&((*psa)[nsel]), (void *)sel[type].selectors, n*sizeof(seltype));
	    if (port >= 0) {
		fprintf (info, "selector type = %d, nselectors = %d\n", type, n);
		for (i=0; i<n; i++) fprintf (info, "%s\n",sel[type].selectors[i]);
	    }
	}
	nsel += n;
    }
}

int setup_socket (int port)
{
    struct sockaddr_in sin;
    struct sockaddr_in from;
    int from_len = sizeof(from);
    int status;
    int ruflag;
    int sd;
    int ld;

    if ((ld = socket (AF_INET, SOCK_STREAM, 0)) == -1) {
	fprintf (stderr, "Error creating the socket\n");
	terminate_program (1);
    }

    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = INADDR_ANY;
/*   sin.sin_addr.s_addr = htonl(INADDR_ANY);*/
    sin.sin_port = port;

    ruflag = 1;
    if (setsockopt (ld, SOL_SOCKET, SO_REUSEADDR, (char *)&ruflag, sizeof(ruflag))) {
	fprintf (stderr, "Error setting REUSEADDR socket option\n");
	terminate_program(1);
    }
    ruflag = 1;
    if (setsockopt (ld, SOL_SOCKET, SO_KEEPALIVE, (char *)&ruflag, sizeof(ruflag))) {
	fprintf (stderr, "Error setting KEEPALIVE socket option\n");
	terminate_program(1);
    }

    if (bind (ld, (struct sockaddr *)&sin, sizeof(sin)) < 0) {
        syserr("Error binding to Socket\n");
	fprintf (stderr, "Error binding to socket\n");
	terminate_program (1);
    }

    listen (ld,1);

    if ((sd = accept (ld, (struct sockaddr *)&from, &from_len)) < 0) {
	fprintf (stderr, "Error accepting socket connection\n");
	terminate_program (1);
    }
    return (sd);
}


#define	SEED_BLKSIZE	512
int write_seed (int sd, seed_record_header *pseed)
{
    int n;

    if ((n = xcwrite (sd, (char *)pseed, SEED_BLKSIZE)) != SEED_BLKSIZE) {
	fprintf (stderr, "Error writing seed record - returned %d\n", n);
	terminate_program (1);
    }
    if (verbosity && port >= 0) fprintf (info, "seed packet sent\n");
    fflush (info);
    return (SEED_BLKSIZE);
}

#define MAX_RETRIES 20
/************************************************************************/
/*  xcwrite:								*/
/*	Write output buffer.  Continue writing until all N bytes are	*/
/*	written or until error.						*
/************************************************************************/
int xcwrite (fd, buf, n)
    int fd;
    char buf[];
    int n;
{
    int nw;
    int left = n;
    int retries = 0;
    while (left > 0) {
	if ( (nw = write (fd, buf+(n-left), left)) <= 0 && errno != EINTR) {
	    fprintf (stderr, "error writing output, errno = %d\n", errno);
	    return (nw);
	}
	if (nw == -1) {
	    fprintf (stderr, "Interrupted write, retry %d.\n", retries);
	    ++retries;
	    if (retries > MAX_RETRIES) {
		fprintf (stderr, "Giving up...\n");
		return(n-left);
	    }
	    continue;
	}
	left -= nw;
    }
    return (n);
}
