/***********************************************************

File Name :
	tdata_client.c

Programmer:
	Phil Maechling

Description:
	This a client that reads TERRAscope data from a socket and places
	it into a shared memory area on the client machine.

	This is program uses TCP/IP sockets to connect to a TERRAscope
        data server somewhere in the Lab. The tdata_client.c reads 
	compressed SEED packets, decompresses them and puts them
	into shared memory.

	Currently, this always reads six componets of data for a
	stations, 20 sps VBB data, and 80-100 sps LG data.
	

Creation Date:
	17 September 1995

Modification History:


Usage Notes:

	SYNTAX: tdata_client.c <remote-node> <station-abbreviation> 

**********************************************************/

#include <iostream.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdlib.h>
#include <unistd.h> // for close()
#include "basics.h"
#include "tdata.h"
#include "mem_map.h"
#include "ts_station_mgr.h"
#include "cssubs.h"
#include "syserr.h"
#include "qlib.h"
//#include "qutils.h"
//#include "sdr_utils.h"



#define MIN_BLKSIZE 128
#define MAX_BLKSIZE 4096

#define TCLIENT_SERVICES "tdata_services"

// Define these as external "file scope" variables so that the 
// write to memory routines can call them, and also so that they
// do not go through thr constructor cycle on each call.


TS_Station_Manager tscope;
int rawdata[512];         // This receives the decompressed samples
			  // before they are written out to memmory
		          // this size assumes worst case (more than
			  // worse case) that each byte in 512 byte block
			  // decompresses to a 32 bit (int) sample
int verbosity = FALSE;    // Setting this TRUE will give a more verbose
			  // output from the stream manager.


int main(int argc, char *argv[])
{
  int 	socket_channel;
  FILE  *fp;
  int   not_found; 
  int   res;
  char  sta[10];
  char  servicenum[20];
  char  inpline[120];
  int   status;
  char  compname[120];
  char  in_data[MAX_BLKSIZE * sizeof(char)];
  DATA_HDR* data_hdr;

  // Following are memmory map variables

  Memory_Map mmap;
  char* cmap;
  int*  tmap;

  char  s1[5];
  char  s2[5];
  char  s3[5];
  char  s4[5];
  char  s5[5];
  char  s6[5];


  if (argc < 2) 
  {
    printf (" SYNTAX: tdata_client <station-name> \n");	
    exit (0);
  } 
  else 
  { 
    printf (" The requested station is: %s\n", argv[1]);
  }


  fp = fopen(TCLIENT_SERVICES,"r");
  if (fp == NULL)
  {
    printf("Unable to open the TCLIENT_SERVICES configuration file: %s.\n",
	TCLIENT_SERVICES);
    exit(FAILURE);
  }

  not_found = TRUE;

  while ( (not_found) && !feof(fp) )
  {
    fgets(inpline,120,fp);
    res = sscanf(inpline,"%s %s %s",sta,compname,servicenum);
    res = strcmp(argv[1],sta);
    if(res == 0)
    {
      printf("Using Station : (%s) from (%s) at service : (%s). \n",
	sta,compname,servicenum);
      not_found = FALSE;
    }
    if (feof(fp))
    {
      printf("Unable to find station %s in tdata_services file.\n",sta);
      exit(FAILURE);
    }

  }


  /* Attach to the memmory area */

  res = initialize_cda(sta,s1,s2,s3,s4,s5,s6);

  if (res != SUCCESS)
  {
    cout << "Error initializing contiguous data area." << endl;
    exit(FAILURE);
  }

  cmap = mmap.Start_of_Memory_Area();
  tmap = (int*) cmap;

  tscope.channel_1.Initialize_Stream(cmap,sta,s1);
  tscope.channel_2.Initialize_Stream(cmap,sta,s2);
  tscope.channel_3.Initialize_Stream(cmap,sta,s3);
  tscope.channel_4.Initialize_Stream(cmap,sta,s4);
  tscope.channel_5.Initialize_Stream(cmap,sta,s5);
  tscope.channel_6.Initialize_Stream(cmap,sta,s6);

  /* cda configuration done */

  /* Open the socket connection */

  int connecting_to_server = TRUE;

  while (connecting_to_server)
  {

    status = open_socket(compname,		
               	         servicenum,
               	         &socket_channel);


    if (status != SUCCESS) 
    {
      printf("Error on open_socket call.\n");
      sleep(60);
      continue;
    }

   
/*  
 * 
 *   Reading socket to get hypo_centers
 *   status == 0 indicates end-of-file (socket closed by remote), else its 
 *   the number of bytes received
 *
 */

    int reading_packets = TRUE;

    while(reading_packets)
    {
      data_hdr = get_packet(socket_channel,(char *) &in_data);
      if(data_hdr == NULL)
      {
        continue;
      }
      else if((int)data_hdr == FAILURE)
      {
        reading_packets = FALSE;
        continue;
      }
      res = write_to_cda(data_hdr,(char *)&in_data);
      if (res == FAILURE)
      {
        printf("Error writing to cda\n");
      }
    }

    close(socket_channel);
    sleep(60);
  }
  return(FAILURE);
}

/*

   Inputs :

    server : This is the name of the machine on which the
      desired service resides.

    service : This is the name of the service to connect to
      
  Outputs :
    sunit:
      If the function return value is negative, the open failed. 
      If return value is positive, then the returned parameter
      sunit is a socket descriptor to the services desired.

*/


#include <ctype.h>			/* or 'isalpha' function*/


#define SERVER_NAME_LENGTH  50
#define SERVICE_NAME_LENGTH 50

int open_socket(char *server, 	/* remote node name */
                char *service, 	/* remote service name */
                int *sunit)
{


  struct sockaddr_in server_sock;
  struct hostent *hp;
  struct servent *sp;
  unsigned long  ipadd;
  int s;

/*
 * Create an socket on which to make the connection.
 *
 * This function call says we want our communications to take
 * place in the internet domain (AF_INET) (the other possible domain
 * is the Unix domain).
 *
 */

  s = socket(AF_INET, SOCK_STREAM, 0);

  if (s < 0) 
  {
    printf(" %% Error opening socket in client.\n");
    return(FAILURE);
  }

/*
 * Get the IP address of the remote client. It must
 * be in the local machines host table or this call fails. (?)
 *
 * This call returns th 'hp' structure representing the
 * IP address of the remote client.
 *
 */

  if (isalpha(server[0])) 
  {
    hp = gethostbyname(server);	/* probe name server by name */
  } 
  else 
  {
    ipadd = inet_addr(server);	/* convert IP to binary format */
    hp = gethostbyaddr((char*)&ipadd, sizeof(ipadd), AF_INET);	
                               /*probe name server by address */
  }

  if (hp == NULL) 
  {
    printf(" %% Remote host is unknown to name server.\n");
    return(FAILURE); 
  } 
  else 
  {
    printf(" %% Remote host: %s\n", hp->h_name);
  }

/* 
 * If specified as a name the service/port must be defined in the local host 
 * table
 */

  if (isalpha(service[0])) 
  {
    sp = getservbyname(service, "tcp");	/* probe service by name */
    if (sp == NULL) 
    {
      printf(" %% Service not in local host table.\n");
      return(-1);
    } 
    else 
    {
      server_sock.sin_port = sp->s_port;
    }

  } 
  else 
  {
    server_sock.sin_port = htons(atoi(service)); /* convert ASCII to int. */
  }

/* 
 * create a 'sockaddr_in' internet address structure to use with connect 
 */

  server_sock.sin_family = hp->h_addrtype;
  strncpy((char*)&server_sock.sin_addr, hp->h_addr, hp->h_length);

/* Print IP address in xx.xx.xx.xx format */

  printf(" %% IP address: %s\n", inet_ntoa(server_sock.sin_addr));
  printf(" %% Service port # is %d. \n", ntohs(server_sock.sin_port));

/*
 *  Connect to the socket address.
 */

  if (connect( s, (struct sockaddr *) &server_sock, sizeof(server_sock)) < 0) 
  {
    printf(" %% Attempt to connect to remote host failed.\n ");
    return(FAILURE);
  }

  *sunit = s;		/* return socket channel # as a calling paramter */
  return(SUCCESS);      /* return SUCCESS value */

}



DATA_HDR* get_packet(int sd,char* indata)
{
  int            nr, blksize;
  DATA_HDR      *hdr;
  BS            *bs = NULL;
  BLOCKETTE_HDR *bh = NULL;
  int goodpacket;

  goodpacket = 0;

  while(!goodpacket)
  {
          nr = xread(sd, (char *)indata, MIN_BLKSIZE);
          if (nr == 0)
          {
            fprintf (stderr, "Found end of file on socket read: %d\n",nr);
            return((DATA_HDR *)FAILURE);
          }
          if (nr < MIN_BLKSIZE) 
          {
            fprintf (stderr, "Error reading from socket: expected %d got %d\n",
	      MIN_BLKSIZE, nr);
            continue;
           }
           if ((hdr = decode_hdr_sdr ((SDR_HDR *)indata, &blksize)) == NULL) 
           {
             fprintf (stderr, "Error decoding SEED data hdr\n");
             continue;
           }
           if (blksize > MIN_BLKSIZE) 
           {
              nr = xread (sd, (char *)indata + MIN_BLKSIZE, blksize-MIN_BLKSIZE);
              if (nr < blksize-MIN_BLKSIZE) 
              {
                fprintf (stderr, "Error reading SEED data\n");
                continue;
               }
            }
        
            goodpacket = 1;

            /* Fill in the number of data frames.   */
            if ((bs = find_blockette(hdr,1001)) &&
                (bh = (BLOCKETTE_HDR *)(bs->pb))) 
            {
                hdr->num_data_frames = ((BLOCKETTE_1001 *)bh)->frame_count;
                /* Explicitly set num_data_frames for SHEAR stations.   */
                if (hdr->num_data_frames == 0 && hdr->sample_rate != 0)
                    hdr->num_data_frames = (blksize - hdr->first_data) / sizeof(FRAME);
             }
             else 
             {
                hdr->num_data_frames =
                (hdr->sample_rate == 0) ? 0 :
                (blksize - hdr->first_data) / sizeof(FRAME);
             }
  } /* End while !goodpacket */
  return (hdr);
}


int write_to_cda(DATA_HDR *pseed,char* indata)
{
    int             samp;
    int tickspersample;
    INT_TIME itime;
    int msecs;
    int dummysec;
    int channel_number;
    BS *bs = NULL;
    BLOCKETTE_HDR *bh = NULL;


    if (pseed == NULL)
    {
      return (FAILURE);
    }

    itime = pseed->begtime; // internal time
    msecs = pseed->begtime.ticks * 1000 / TICKS_PER_SEC;

    if (verbosity) printf ("packet %s stime=%s msecs=%d\n", 
                       pseed->channel_id, time_to_str(itime,1), msecs);

    samp = decompress_data(pseed, 
                           ((char *)&indata[0])+pseed->first_data, 
                           (int *) &rawdata);

    if (samp != pseed->num_samples) 
    {
	printf("Samp not equal to n_samples\n");
	return(FAILURE);
    }

    /* Calculate the time interval in one sample */
    /* This doesn't handle the low data rates properly */

    time_interval(1, pseed->sample_rate,
                  &dummysec,&tickspersample);

    if (strncmp("Z",&pseed->channel_id[2],1) == 0)
    {
      if (strncmp("H",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 1;
      }
      else if (strncmp("L",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 4;
      }
      else
      {
        printf("Invalid channel name found : %s \n",pseed->channel_id);
        return(FAILURE);
      }
    }
    else if (strncmp("N",&pseed->channel_id[2],1) == 0)
    {
      if (strncmp("H",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 2;
      }
      else if (strncmp("L",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 5;
      }
      else
      {
        printf("Invalid channel name found : %s \n",pseed->channel_id);
        return(FAILURE);
      }
    }
    else if (strncmp("E",&pseed->channel_id[2],1) == 0)
    {
      if (strncmp("H",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 3;
      }
      else if (strncmp("L",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 6;
      }
      else
      {
        printf("Invalid channel name found : %s \n",pseed->channel_id);
        return(FAILURE);
      }
    } 
    else
    {
      printf("Invalid channel name found : %s \n",pseed->channel_id);
      return(FAILURE);
    }

    for(int i=0;i<samp;i++)
    {
      switch(channel_number)
      {
        case 1:
        {
          tscope.channel_1.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
	case 2:
        {
          tscope.channel_2.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
	case 3:
        {
          tscope.channel_3.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
        case 4:
        {
          tscope.channel_4.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
	case 5:
        {
          tscope.channel_5.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
	case 6:
        {
          tscope.channel_6.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
	default:
        {
          cout << "Channel that is not handled" << endl;
          return(FAILURE);
        }
      } // end switch
    }

    if (verbosity)
    {
      fprintf (stdout, "wrote data to cda sent\n");
    }
    fflush (stdout);
    return (samp);
}


int decompress_data(DATA_HDR *hdr, char *indata, int *array)
{
    int x0, xn;
    int *diffbuff;
    int numsamp;
    switch (hdr->data_type) {
      case EMPTY:
	fprintf (stdout, "Empty packet all nulls : %d\n:", hdr->data_type);
	numsamp = 0;
	break;
      case STEIM1:
	if ((diffbuff = (int *)malloc(hdr->num_samples * sizeof(int))) == NULL) {
	    syserr("Unable to malloc diffbuff");
	    return(FAILURE);
	}
	numsamp = unpack_steim1 ((FRAME *)indata, 
                                 hdr->num_data_frames * sizeof(FRAME), 
				 hdr->num_samples, hdr->num_samples, 
				 array, diffbuff, &x0, &xn, NULL);
	free (diffbuff);
	break;
      case STEIM2:
	if ((diffbuff = (int *)malloc(hdr->num_samples * sizeof(int))) == NULL) {
	    syserr("Unable to malloc diffbuff");
	    return(FAILURE);
	}
	numsamp = unpack_steim2 ((FRAME *)indata, 
                                 hdr->num_data_frames * sizeof(FRAME), 
				 hdr->num_samples, hdr->num_samples, 
				 array, diffbuff, &x0, &xn, NULL);
	free (diffbuff);
	break;
      default:
	cout << "Unknown datatype: " << hdr->data_type << endl;
        fflush(stdout);
	numsamp = -1;
	break;
    }
    return numsamp;
}
