/***********************************************************

File Name :
	stream_mgr.C

Programmer:
	Phil Maechling

Description:
	This defines the stream manager class. Several of these
	would be expected for a station process.
	Given a memory map location, this routines tries to
	locate the information about its particulary stream.
	Once it locates its shared memory header, then it
	can add data to the memory area, and update the header.

Limitations or Warnings:
	The mem map functions should be called before this 
	function is called. Only minimal checking is done
	to confirm this fact. 

Creation Date:
	22 March 1995

Modification History:

**********************************************************/
#include <sys/shm.h>
#include <string.h>


#include "stream_hdr.h"
#include "stream_mgr.h"
#include "qtime.h"
#include "basics.h"

extern int verbosity;

Stream_Manager::Stream_Manager()
{

}


Stream_Manager::~Stream_Manager()
{

}



//
// This method takes a station and stream name and tries to determine
// the stream number for that stream.
//
// Once the stream number if found, the address of the header
// and the address of the sample array are determined.
//

int Stream_Manager::Initialize_Stream(char* map,char* station, char* stream)
{
  int compres;
  unsigned int this_stream_header_offset;
  int* number_of_streams_ptr;


  start_of_memmap = map;          // establish address at which this routine
		                  // is attached to shared memory segement.

  number_of_streams_ptr = (int*) map; //establish integer pointer
 
  //
  // Establish starting address for the stream headers
  //
 
  this_stream_header_offset = (sizeof(int));
  this_stream_header = (Stream_Header *)
                          (start_of_memmap + this_stream_header_offset);

  //
  // Loop through and try to match given sta,stream with on in
  // shared memory
  //

  for(int i=0 ; i < *number_of_streams_ptr; i++ )
  {
    compres = strcmp(this_stream_header->station,station);
    if (compres == 0)
    {
      compres = strcmp(this_stream_header->stream,stream);
      if(compres == 0)
      {

        // 
        // This is the correct exit route
        // If we exit here, then we know that the
        // pointer "this_stream_ptr" is correctly pointing
        // to the header area we want.
        //
        return(SUCCESS);
      }
    } 

    //
    // Cast to a char pointer so ptr arithemtic is in bytes
    // then cast back to desired stream header type
    //

    this_stream_header_offset = this_stream_header_offset + 
                                  sizeof(Stream_Header);

    this_stream_header = (Stream_Header *)
                            (start_of_memmap + this_stream_header_offset);

  }
  return(FAILURE);  
}



int Stream_Manager::Put_Sample(const INT_TIME time_of_sample,const int counts)
{
  double tickdiff;
  double number_of_samples;
  Sample* samp_ptr;   
  char* temp_ptr;

  tickdiff = tdiff(time_of_sample,this_stream_header->time_of_newest_sample);

  if (tickdiff <0.0)
  { 
    cout << "Currently no support for negative time overwriting data" << endl;
    return(1); 
  }
 
  number_of_samples = 
    dsamples_in_time(this_stream_header->samples_per_second,
         tickdiff);

  if(number_of_samples < 0.0)
  {
    if (verbosity > 0)
    {
      cout << "Number of samples calcuated to be less than 0" << endl;
      cout << "number_of_samples : " << number_of_samples << endl;
    }
    return(FAILURE);
  }
  else if(number_of_samples == 0.0)
  {
    if (verbosity > 0)
    {
     cout << "Number of samples calculated to be 0" << endl;
     cout << "number_of_samples : " << number_of_samples << endl;
    }
    return(FAILURE);
  }
  else if(number_of_samples < 1.0)
  {
    if (verbosity > 0)
    {
      cout << "Less than 1 full sample offset" << endl;
      cout << "number_of_samples : " << number_of_samples << endl;
    }
    return(FAILURE);
  }
  else if(number_of_samples == 1.0)
  {

    // Find times outside the locked section to reduce the time things
    // are locked, and to reduce chance of problems while we own the lock.
    //

    new_time = Find_Next_Newest_Time(number_of_samples,
                 this_stream_header->samples_per_second);

    old_time = Find_Next_Oldest_Time(number_of_samples,
                 this_stream_header->samples_per_second);

    // lock memory
    
    semaphore.Acquire_Semaphore();

    //
    // Increment to next sample locations
    //
   
    Move_Sample_Ptrs_One_Sample();

    //
    // update sample count and valid flag. Use intermediate
    // char ptr so ptr arithmetic works.
    //

    temp_ptr = (start_of_memmap + 
                  this_stream_header->offset_to_newest_sample);
    samp_ptr = (Sample*) temp_ptr;
    samp_ptr->counts = counts;
    samp_ptr->valid  = TRUE;

    //
    // Update time
    //
    
    this_stream_header->time_of_newest_sample = new_time;
    this_stream_header->time_of_oldest_sample = old_time;

    // 
    // Unlock memory
    //

    semaphore.Release_Semaphore();
   
    //
    return(SUCCESS);  

  } 
  else // Else number of sample != 1. There is a time gap or time offset.
  {

    if (verbosity > 0)
    {
      cout << endl;
      cout << "Time Gap Found. Difference in ticks : "<< tickdiff << endl;
      cout << "Oldest : " <<
        time_to_str(this_stream_header->time_of_oldest_sample,0) << endl;
      cout << "Newest : " <<
        time_to_str(this_stream_header->time_of_newest_sample,0) << endl;
      cout << "Sample Time : " <<
        time_to_str(time_of_sample,0) << endl;
      cout << "Number_of_samples between : " << number_of_samples << endl;
    }
  
    //
    // Conditions to handle are 
    // (1) Even number of samples difference
    //      Fill in with invalid. Then add valid sample
    // (2) Limited Sample difference plus tick offset
    //       Fill in with invalid. Add valid sample. Then adjust oldest time
    //        so there is no offset, so next sample will be even samples
    //        difference
    // (1) Large Sample difference greater then total samples in memory.
    //       Make newest sample also oldest sample
    // (1) Large Sample difference with sample offset.
    //       Making newest sample, oldest sample elimiates any offest problem. 
    //

    if (number_of_samples > this_stream_header->stream_samples_in_memory)
    {
      //
      // Find times outside the locked section to reduce the time things
      // are locked, and to reduce chance of problems while we own the lock.
      //

      new_time = time_of_sample;
      old_time = time_of_sample;

      // lock memory
    
      semaphore.Acquire_Semaphore();


      this_stream_header->time_of_newest_sample = new_time;
      this_stream_header->time_of_oldest_sample = old_time;
      this_stream_header->offset_to_newest_sample = this_stream_header->
						    offset_to_first_sample;
      this_stream_header->offset_to_oldest_sample = this_stream_header->
						    offset_to_first_sample;


      temp_ptr = (start_of_memmap + 
                  this_stream_header->offset_to_newest_sample);
      samp_ptr = (Sample*) temp_ptr;
      samp_ptr->counts = counts;
      samp_ptr->valid  = TRUE;

      // 
      // Unlock memory
      //

      semaphore.Release_Semaphore();

      if (verbosity > 0)
      {
        cout << "All data but newest data dropped from cda" << endl;
      }
      return(SUCCESS);
    }

    //
    // This is the processing that fills in the time gap with invalid data.
    // First check for an sample offset. That is, a time difference 
    // between sample of less than an integer number of samples.
    //

     int rem;
     int tickspersample;

     if (verbosity > 0)
     { 
       cout << endl;
       cout << "Start of time gap processing " << endl;
       cout << "Starting diff between samples in ticks : " << 
        (int) tickdiff << endl;
     }

     tickspersample = 
      (int) (10000.0 * (1.0 / this_stream_header->samples_per_second));

     rem = (int) tickdiff % tickspersample;

     if (rem != 0)
     {
       if (verbosity > 0)
       { 
         cout << "Found time gap between samples. " << endl;
         cout << "Gap is not a even multiple of sample period " << endl;
         cout << "number_of_samples : " << number_of_samples << endl;
         cout << "tickdiff  : "  << tickdiff << endl;
         cout << "remainder : "  << rem << endl;
         cout << "Try to correct time alignment " << endl;
         cout << "Sample rate from data : " << this_stream_header->
	   samples_per_second << endl;
       }

       new_time = 
         add_time(this_stream_header->time_of_newest_sample,0,rem);
       old_time = 
         add_time(this_stream_header->time_of_oldest_sample,0,rem);

       if (verbosity > 0)
       {
         cout << "Adjusting time of newest sample from : to : " << endl;
         cout << time_to_str(this_stream_header->time_of_newest_sample,0) 
           << endl;
         cout << time_to_str(new_time,0) << endl;
         cout << "Adjusting time of oldest sample from : to : " << endl;
         cout << time_to_str(this_stream_header->time_of_oldest_sample,0) 
           << endl;
         cout << time_to_str(old_time,0) << endl;
       }

       tickdiff = 
         tdiff(time_of_sample,new_time);
    
       rem = (int) tickdiff % tickspersample;

       if (rem != 0)
       {
         if (verbosity > 0)
         {
           cout << "Attempt to adjust sample offset failed " << endl;
           cout << "rem = : " << rem << endl;
         }
       }
       else
       {

         // Adjusting time of oldest sample
         //
         if (verbosity > 0)
         {
           cout << "Assigning corrected value to time of oldest sample" << endl;
         }
         semaphore.Acquire_Semaphore();
         this_stream_header->time_of_newest_sample = new_time;
         this_stream_header->time_of_oldest_sample = old_time;
         semaphore.Release_Semaphore();
       }

       number_of_samples = 
         dsamples_in_time(this_stream_header->samples_per_second,
           tickdiff);

       new_time = Find_Next_Newest_Time(number_of_samples,
                    this_stream_header->samples_per_second);

       old_time = Find_Next_Oldest_Time(number_of_samples,
                    this_stream_header->samples_per_second);

       // 
       // lock mememory
       //

       semaphore.Acquire_Semaphore();

       //
       // Fill in the invalid data first.
       // Invalidate all the samples between the old one
       // and the newest one.
       // Update sample count and valid flag
       //

       if (number_of_samples > 1)
       {
         for(int i = 0;i<(number_of_samples-1);i++)
         {
           Move_Sample_Ptrs_One_Sample();
      
           temp_ptr = start_of_memmap + 
                   this_stream_header->offset_to_newest_sample;

           samp_ptr = (Sample*) temp_ptr;
           samp_ptr->counts = 0;
           samp_ptr->valid  = FALSE;
         }

       }

      //
      // Put in the last, and only valid sample
      //

      Move_Sample_Ptrs_One_Sample();

      temp_ptr = start_of_memmap + 
                 this_stream_header->offset_to_newest_sample;

      samp_ptr = (Sample*) temp_ptr;
      samp_ptr->counts = counts;
      samp_ptr->valid  = TRUE;
  
      //
      // Update time
      //
    
      this_stream_header->time_of_newest_sample = new_time;
      this_stream_header->time_of_oldest_sample = old_time;

      // 
      // Unlock memory
      //

      semaphore.Release_Semaphore();

      // 
      //
      //
      return(SUCCESS);
     }
     else
     {
       // Determine the new data times since the difference is an even
       // number of samples.
       // Find times outside the locked section to reduce the time things
       // are locked, and to reduce chance of problems while we own the lock.
       //
  
       if (verbosity > 0)
       { 
         cout << "Found time gap between samples. " << endl;
         cout << "Gap is an even multiple of sample period " << endl;
         cout << "number_of_samples : " << number_of_samples << endl;
       }

       new_time = Find_Next_Newest_Time(number_of_samples,
                    this_stream_header->samples_per_second);

       old_time = Find_Next_Oldest_Time(number_of_samples,
                    this_stream_header->samples_per_second);
     }

    // 
    // lock mememory
    //

    semaphore.Acquire_Semaphore();

    //
    // Fill in the invalid data first.
    // Invalidate all the samples between the old one
    // and the newest one.
    // Update sample count and valid flag
    //

    for(int i = 0;i<(number_of_samples-1);i++)
    {
      Move_Sample_Ptrs_One_Sample();
      
      temp_ptr = start_of_memmap + 
                   this_stream_header->offset_to_newest_sample;

      samp_ptr = (Sample*) temp_ptr;
      samp_ptr->counts = 0;
      samp_ptr->valid  = FALSE;
    }

    //
    // Put in the last, and only valid sample
    //

    Move_Sample_Ptrs_One_Sample();

    temp_ptr = start_of_memmap + 
                 this_stream_header->offset_to_newest_sample;

    samp_ptr = (Sample*) temp_ptr;
    samp_ptr->counts = counts;
    samp_ptr->valid  = TRUE;
  
    //
    // Update time
    //
    
    this_stream_header->time_of_newest_sample = new_time;
    this_stream_header->time_of_oldest_sample = old_time;

    // 
    // Unlock memory
    //

    semaphore.Release_Semaphore();

    // 
    //
    //
    return(SUCCESS);
  }
}

void Stream_Manager::Move_Sample_Ptrs_One_Sample()
{

  //
  // Increment the newest data ptr;
  //

  this_stream_header->offset_to_newest_sample =
    this_stream_header->offset_to_newest_sample +
      sizeof(Sample);
  
  if(this_stream_header->offset_to_newest_sample >
      this_stream_header->offset_to_last_sample)
  {
    this_stream_header->offset_to_newest_sample = 
      this_stream_header->offset_to_first_sample;
  }

  //
  // Now increment the oldest data ptr if needed
  //

  if (this_stream_header->offset_to_oldest_sample ==
        this_stream_header->offset_to_newest_sample)
  {
    //
    // Move oldest data sample ptr
    //
    this_stream_header->offset_to_oldest_sample =
      this_stream_header->offset_to_oldest_sample + sizeof(Sample);
  
    if(this_stream_header->offset_to_oldest_sample > 
         this_stream_header->offset_to_last_sample)
    {
      this_stream_header->offset_to_oldest_sample = 
        this_stream_header->offset_to_first_sample;
    }
  }
}


int Stream_Manager::Samples_to_Move_Oldest_Time(int number_of_new_samples)
{
  int nomovecount=0;
  unsigned int temp_offset;

 
  temp_offset = this_stream_header->offset_to_newest_sample;

  for(int i = 0;i<number_of_new_samples;i++)
  {

    temp_offset = temp_offset + sizeof(Sample);

    if(temp_offset > this_stream_header->offset_to_last_sample)
    {
      temp_offset = this_stream_header->offset_to_first_sample;
    }

    if(temp_offset == this_stream_header->offset_to_oldest_sample)
    {
      break;
    }
    ++nomovecount; 
  }
  return(number_of_new_samples - nomovecount);
}




INT_TIME Stream_Manager::Find_Next_Newest_Time(int number_of_new_samples, 
                                               int samples_per_second)
{
  int newsecs;
  int newticks;

  //
  //Warning. Add test for overflow of time interval routine
  //

  time_interval(number_of_new_samples,samples_per_second,
                &newsecs,&newticks);

  return(add_time(this_stream_header->time_of_newest_sample,
                      newsecs,newticks));
}


INT_TIME Stream_Manager::Find_Next_Oldest_Time(int number_of_new_samples, 
                               int samples_per_second)
{
  int newsecs;
  int newticks;
  int samples_to_move;


  samples_to_move = Samples_to_Move_Oldest_Time(number_of_new_samples);

  //Warning. Add test for overflow of time interval routine

  time_interval(samples_to_move,samples_per_second,
                &newsecs,&newticks);

  return(add_time(this_stream_header->time_of_oldest_sample,
                      newsecs,newticks));
}


int Stream_Manager::Get_Samples_Per_Second()
{
  return(this_stream_header->samples_per_second);
}
