/***********************************************************

File Name :
	mem_map.cc

Programmer:
	Phil Maechling

Description:
	This is the definition of the memory map routines.
	These routines do the creation, and destruction of
	the shared memory area's.
	They also do the initialization of the area.

Limitations or Warnings:


Creation Date:
	22 March 1995

Modification History:

**********************************************************/

#include <iostream.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <stdio.h>
#include <errno.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>

#include "sample.h"
#include "stream_hdr.h"
#include "syserr.h"
#include "mem_map.h"
#include "qtime.h"
#include "basics.h"




Memory_Map::Memory_Map()
{

  int size;
  int first = FALSE;
  void* memmap; 

  //
  // All objects have to read the configuration file and get
  // the shared memory key.
  //

  Get_Memory_Key();

  //
  // Test to see if the memory region exists. If not, then 
  // create it an initialize it. If the return value is -1,
  // then the region needs to be created.
  //

  shmid = shmget(mem_key,0,0);
  if (shmid == -1)
  {

    Determine_Size_of_Memory_Area(); 
                      // This reads the configuration file and fills
  		      // in the values for total_stream and total samples.
                      // And it determines the mem_key value from the
	              // configuration file.

    size = ( sizeof(unsigned int ) + 
           (sizeof(Stream_Header) * total_streams_in_list) +
           (sizeof(Sample) * total_samples_in_array) );
  
    cout << "Size of : " << size << endl;


    //
    //
    // Read configuration file and determine the following items
    // key, # of streasm, # of samples in each stream
    //


    shmid = shmget(mem_key,size,0666 | IPC_CREAT);
    if (shmid == -1)
    {
      fatalsyserr("Error getting shmid in mem_map");
    }
    cout << "Memory Allocated : " << endl;

    first = TRUE;

  }

  //
  // All calling processes must attach to the shared memory segment
  //

  memmap = shmat(shmid,(void *)0,0);
  if (memmap == NULL)
  {
    fatalsyserr("Error getting memmap address in mem_map");
  }

  start_of_memmap = (char*)memmap;

  //
  // If we are the first calling program, initialize the 
  // shared memory header
  // 

  if (first)
  {
    Populate_Header();
    cout << "Completed Initialization of Header " << endl;
  }

}

Memory_Map::~Memory_Map()
{
  // on exit,
  // Unattach from the shared memory area
  //
  int res = shmdt(start_of_memmap);
}


Memory_Map::Delete_Memory_Area()
{

  int retnum;

  cout << "Ready to release memory " << endl;
  retnum = shmctl(shmid,IPC_RMID,0);
  if (retnum == FAILURE)
  {
    fatalsyserr("Error releasing shared memory");
  }
  cout << "Segment destroyed" << endl;
  return(SUCCESS);
}


int Memory_Map::Determine_Size_of_Memory_Area()
{
  char* fname;
  FILE* filep;  
  char  keyword[20];
  char  colan[20];
  int   scanres;
  int   compres;
  char* defaultname = "/home/rtem/cda/memmap.cfg";
  int   noerr;
  int   thiskey;

  fname = getenv("MEM_MAP"); 
   
  if (fname == NULL)
  {
    fname = defaultname;
  }

  filep = fopen(fname,"r");
  
  if (filep == NULL)
  {
    cout << "Unable to open Memory Map configuration file" << endl;
    cout << "Tried file : " << fname << endl;
    exit(-1);
  }

  noerr = TRUE;

  while(!feof(filep) && (noerr))
  {

    scanres = fscanf(filep,"%s %s %d",keyword,colan,&thiskey);
    if (scanres != 3)
    {
      cout << "Error scanning memmap file : " << fname << endl;
      cout << "mem_key field not found. " << endl;
      noerr = FALSE;
      continue;
    }

    compres = strcmp(keyword,"mem_key");
    if (compres != 0)
    {
      cout << "Error scanning memmap file : " << fname << endl;
      cout << "mem_key field not found. " << endl;
      noerr = FALSE;
      continue;
    }

    if (thiskey < 1)
    { 
      cout << "Memory key too low :  " << thiskey << endl;
      noerr = FALSE;
    }
    else
    {
      cout << "Shared memory key found : " << thiskey << endl;
    }

  // Now go through station list and just count up two things,
  //	(1) number of stations,
  //	(2) total amount of data bytes to be stored in memory array


    total_streams_in_list = 0;
    total_samples_in_array = 0;

    while(!feof(filep) && (noerr))
    {
      int line1,line2,line3,line4;
      char sta[8];
      char str[8];
      int sps;
      int sim;

      line1 = fscanf(filep,"%s %s %s",keyword,colan,sta);
      line2 = fscanf(filep,"%s %s %s",keyword,colan,str);
      line3 = fscanf(filep,"%s %s %d",keyword,colan,&sps);
      line4 = fscanf(filep,"%s %s %d",keyword,colan,&sim);
     
      if ( (line1 !=3) || (line2 != 3) || (line3 != 3) ||
            (line4 != 3) )
      {
        if(feof(filep))
        {
          continue;
        }
        else
        {  
          noerr = FALSE;
          continue;
        }
      }
      else
      {
         total_samples_in_array = total_samples_in_array +
                                   (sps * sim);
         ++total_streams_in_list;
         cout << "Total samples in array : " << total_samples_in_array << endl;
         cout << "Total streams in list  : " << total_streams_in_list << endl;
      }

    } // end while counting up station totals
  }   // end while reading not end of entire file

  fclose(filep);
   
  if (noerr)
  {
    return(SUCCESS);
  }
  else
  {
    cout << "Exiting on configuration scan error" << endl;
    exit(FAILURE);
  }

}

/*
This function returns with a pointer to the
number of streams integer at the start of the
shared memory data header
*/

char* Memory_Map::Start_of_Memory_Area()
{

  return(start_of_memmap);

}

int Memory_Map::Get_Memory_Key()
{


  char* fname;
  FILE* filep;  
  char  keyword[20];
  char  colan[20];
  int   scanres;
  int   compres;
  char* defaultname = "/home/rtem/cda/memmap.cfg";
  int   noerr;
  int   thiskey;

  fname = getenv("MEM_MAP"); 
   
  if (fname == NULL)
  {
    fname = defaultname;
  }

  filep = fopen(fname,"r");
  
  if (filep == NULL)
  {
    cout << "Unable to open Memory Map configuration file" << endl;
    cout << "Tried file : " << fname << endl;
    exit(-1);
  }

  noerr = TRUE;

  while(!feof(filep) && (noerr))
  {

    scanres = fscanf(filep,"%s %s %d",keyword,colan,&thiskey);
    if (scanres != 3)
    {
      cout << "Error scanning memmap file : " << fname << endl;
      cout << "mem_key field not found. " << endl;
      noerr = FALSE;
      continue;
    }

    compres = strcmp(keyword,"mem_key");
    if (compres != 0)
    {
      cout << "Error scanning memmap file : " << fname << endl;
      cout << "mem_key field not found. " << endl;
      noerr = FALSE;
      continue;
    }

    if (thiskey < 1)
    { 
      cout << "Memory key too low :  " << mem_key << endl;
      noerr = FALSE;
    }
    else
    {
      cout << "Shared memory key found : " << thiskey << endl;
      mem_key = (key_t) thiskey;
      break;
    }
  }   // end while reading not end of entire file

  fclose(filep);
   
  if (noerr)
  {
    return(SUCCESS);
  }
  else
  {
    cout << "Exiting on configuration scan error" << endl;
    exit(FAILURE);
  }
}



/*
  This function is called by the object that creates the
  shared memory area, which is the first caller.
*/

int Memory_Map::Populate_Header()
{

  char* fname;
  FILE* filep;  
  char  keyword[20];
  char  colan[20];
  int   scanres;
  int   compres;
  char* defaultname = "/home/rtem/cda/memmap.cfg";
  int   noerr;
  int   thiskey;

  fname = getenv("MEM_MAP"); 
   
  if (fname == NULL)
  {
    fname = defaultname;
  }

  filep = fopen(fname,"r");
  
  if (filep == NULL)
  {
    cout << "Unable to open Memory Map configuration file" << endl;
    cout << "Tried file : " << fname << endl;
    exit(-1);
  }

  noerr = TRUE;

  //
  // Define these counters at this scope so they can be used to
  // check consitency after file read is done.
  //

  int streams_in_list = 0;
  int samples_in_memory = 0;



  while(!feof(filep) && (noerr))
  {

    scanres = fscanf(filep,"%s %s %d",keyword,colan,&thiskey);
    if (scanres != 3)
    {
      cout << "Error scanning memmap file : " << fname << endl;
      cout << "mem_key field not found. " << endl;
      noerr = FALSE;
      continue;
    }

    compres = strcmp(keyword,"mem_key");
    if (compres != 0)
    {
      cout << "Error scanning memmap file : " << fname << endl;
      cout << "mem_key field not found. " << endl;
      noerr = FALSE;
      continue;
    }

    if (thiskey < 1)
    { 
      cout << "Memory key too low :  " << (int) mem_key << endl;
      noerr = FALSE;
    }
    else
    {
      cout << "Shared memory key found : " << thiskey << endl;
    }

  //
  // Now go through stream list and set the header with several pieces of
  // information.
  //	(1) station name
  //    (2) stream name
  //	(3) samples per second
  //    (4) seconds in memory
  //    (5) total samples in memory
  //    (6) ticks (10th of milliseconds) per sample
  //    (6) first sample of data
  //    (7) last sample of data
  //    (7) location of newest data
  //    (8) location of oldest data
  //    
  //     Then we want to make all samples invalid, and set the start
  //       time and end time to 1980.
  //


    Stream_Header* current_header;
    unsigned int start_of_data_offset;
    int cummulative_bytes = 0;

    start_of_data_offset =   ( sizeof(int) +
                         (total_streams_in_list * sizeof (Stream_Header)));


    while(!feof(filep) && (noerr))
    {
      int line1,line2,line3,line4;
      char sta[8];
      char str[8];
      int sps;
      int sim;

      line1 = fscanf(filep,"%s %s %s",keyword,colan,sta);
      line2 = fscanf(filep,"%s %s %s",keyword,colan,str);
      line3 = fscanf(filep,"%s %s %d",keyword,colan,&sps);
      line4 = fscanf(filep,"%s %s %d",keyword,colan,&sim);
     
      if ( (line1 !=3) || (line2 != 3) || (line3 != 3) ||
            (line4 != 3) )
      {
        if(feof(filep))
        {
          continue;
        }
        else
        {
          noerr = FALSE;
          continue;
        } 
      }
      else
      {

         //
         // calculate the location for each header information
         //

         //
         // Use a character pointer to make the pointer operations
         // work on bytes, not structures
         //

         unsigned int temp_offset;
         char* temp_ptr;

         temp_offset = ( sizeof(int) +
                      (streams_in_list * sizeof(Stream_Header)));

         temp_ptr = (char*) start_of_memmap + temp_offset;

         current_header = (Stream_Header *) temp_ptr;
          
         strcpy(current_header->station,sta);
         strcpy(current_header->stream,str);
         current_header->samples_per_second  = sps;
         current_header->seconds_in_memory   = sim;
         current_header->stream_samples_in_memory    = (sps * sim);

         //
         // Calculate the location of the data portion of the
         // shared memory area
         //


         temp_offset = start_of_data_offset + cummulative_bytes;

         current_header->offset_to_newest_sample = temp_offset;
         current_header->offset_to_oldest_sample = temp_offset;
         current_header->offset_to_first_sample  = temp_offset;

         temp_offset = (start_of_data_offset + cummulative_bytes +
                    ((current_header->stream_samples_in_memory *
                       sizeof(Sample)) - sizeof(Sample)));

         current_header->offset_to_last_sample   = temp_offset;

         //
         // Fill the header with start time
         // Start all data at Jan 1, 1995
         // 

         INT_TIME start_time;
         time_t outtime;
         tm* mytime;

	 //
	 // Use current time to put reasaonable values in start time
         // Then move start time back to start of TERRAscope era.  
         //

         outtime = time(NULL);
         mytime  = gmtime(&outtime);
         start_time = int_time_from_time_tm(mytime);
         start_time.year = 1990;
         start_time.second = 0;
         
         current_header->time_of_newest_sample = start_time;
         current_header->time_of_oldest_sample = start_time;
 
         //
         // Mark all samples invalid
         // 

         temp_ptr = (char *) (start_of_memmap +
                               current_header->offset_to_first_sample);

         Sample* samp_ptr = (Sample *) temp_ptr;

	//
	// This incrementing works because it does not roll over
	//   the end of the data area.
        //

         for(int i=0; i < current_header->stream_samples_in_memory;i++)
         {
           samp_ptr->counts = 0;
           samp_ptr->valid  = FALSE;
             
           ++samp_ptr;  // ptr arithmetic to next sample;
         }


         //
         // Find cummulative total of bytes in data area
         //

	 cummulative_bytes = cummulative_bytes + 
           (current_header->stream_samples_in_memory * sizeof(Sample));

         //
         // Continue to next stream
         // 

         ++streams_in_list;

         samples_in_memory = samples_in_memory +
           current_header->stream_samples_in_memory;

         cout << "Total samples in array : " << samples_in_memory << endl;
         cout << "Total streams in list  : " << streams_in_list << endl;

      }
    } // end while counting up station totals
  }   // end while reading not end of entire file

  fclose(filep);
  
  if (streams_in_list != total_streams_in_list)
  {
     cout << "Error while populating shared memory header " << endl;
     cout << "Population scan doesn't match creation scan " << endl;
     cout << "Streams in list is : " << streams_in_list << endl;
     cout << "Streams in list was :" << total_streams_in_list << endl;
     exit(-1);
  }
  else
  {
    // This is where the value at the start of the memory
    // area is set. All function use this to determine the
    // size of the current shared memory area.
    // Cast the address to a int then assign the value
    //

    int* temp_ptr = (int *) start_of_memmap;
    *temp_ptr = total_streams_in_list;
  }


  if (samples_in_memory != total_samples_in_array)
  {
     cout << "Error while populating shared memory header " << endl;
     cout << "Population scan doesn't match creation scan " << endl;
     cout << "Samples in array is : " << samples_in_memory << endl;
     cout << "Samples in array was : " << total_samples_in_array << endl;
     exit(-1);
  }
 
  if (noerr)
  {
    return(SUCCESS);
  }
  else
  {
    cout << "Exiting on configuration scan error" << endl;
    exit(FAILURE);
  }

}
