/***********************************************************


File Name :
	getdata.C


Programmer:
	Phil Maechling


Description:
	This defines the data retreival class.
	This routine reads the memmap.cfg file to determine
	the networks shared memory area. Then it attaches
	to the shared memory area. Once attached, its member
	routines can be called to retreive data from the
	shared memory area. This routine takes care of
	all the attaching and locking needed to safely
	get data from shared memory.


Limitations or Warnings:
	The mem map functions should be called before this 
	function is called. Only minimal checking is done
	to confirm this fact. 


Creation Date:
	11 April 1995


Modification History:

**********************************************************/
#include <iostream.h>
#include <sys/shm.h>
#include <string.h>
#include "syserr.h"
#include "qtime.h"
#include "qclass.h"
#include "getdata.h"
#include "basics.h"
#include "stream_hdr.h"




GetData::GetData()
{

  int* int_ptr;

  start_of_memmap = mmap.Start_of_Memory_Area();
  int_ptr = (int *) start_of_memmap;

  if (*int_ptr < 1)
  {
    cout << "Error in memmap header" << endl;
    fatalsyserr("Error in getdata constructor.");
  }
  else
  {
    number_of_streams = *int_ptr;
  }
}



GetData::~GetData()
{



}

//
// This routine will get data from the cda. It will start at the 
// indicated time, or anytime up to the end time.
// It will stop getting data if it finds an invalid sample.
// It will not wait for any data to come in. It will get what is
// available and not wait
//

int GetData::get_segment(
         char*     sta, 
         char*     stream,
         INT_TIME  reqtime,
         int       duration, 
         int*      sps,
         INT_TIME* outtime,
         int*      number_of_out_samples,
	 int       retarray[])
{

  int streamoffset;
  INT_TIME endtime;
  INT_TIME starttime;
  INT_TIME lasttime;

//
// Set the output to to the request time.
// Change it if data is found.
// Otherwise return it and show no data with no data pts.
//

  *outtime = reqtime;
  *number_of_out_samples = 0;

  streamoffset = FindStationStreamHeader(sta,stream);

  if (streamoffset == FAILURE)
  {
    cout << "Station : " << sta << "  Stream : " << stream << 
      "  not found in memory map." << endl;
    return(FAILURE);
  }

  char* temp_ptr;
  temp_ptr = start_of_memmap + sizeof(int) + streamoffset;

  this_stream_header = (Stream_Header *) temp_ptr;

  endtime = add_time(reqtime,duration,0); // Find end time

//  cout << "start of data request window : " << time_to_str(reqtime,0) << endl;
//  cout << "end of data request window : " << time_to_str(endtime,0) << endl;

  semaphore.Acquire_Semaphore();

//  cout << "oldest sample in cda : " <<
//    time_to_str(this_stream_header->time_of_oldest_sample,0) << endl;

//   cout << "newest sample in cda : " <<
//    time_to_str(this_stream_header->time_of_newest_sample,0) << endl;

  //
  // First return the samples per second value
  //

  *sps = this_stream_header->samples_per_second;

  //
  // First two tests check for (1) requested data segement older than
  // oldest data  and (2) requested data newer than newest data
  //

  if (qt.A_lt_or_eq_B(endtime,this_stream_header->time_of_oldest_sample))
  {
    semaphore.Release_Semaphore();
    return(FAILURE);
  }
  else if
    (qt.A_gt_or_eq_B(reqtime,this_stream_header->time_of_newest_sample))
  {
    semaphore.Release_Semaphore();
    return(FAILURE);
  }


  //
  // Now test for the various combinations of time in our out of 
  // segment in memory.
  //


  if ( ( qt.A_lt_or_eq_B(reqtime,this_stream_header->time_of_oldest_sample))
    &&
    (qt.A_gt_or_eq_B(endtime,this_stream_header->time_of_newest_sample) ) ) 
  {
    
     starttime = this_stream_header->time_of_oldest_sample;
     lasttime = this_stream_header->time_of_newest_sample;

  }
  else
  if ( ( qt.A_lt_or_eq_B(reqtime,this_stream_header->time_of_oldest_sample))
    &&
    (qt.A_lt_or_eq_B(endtime,this_stream_header->time_of_newest_sample) ) ) 
  {
    
     starttime = this_stream_header->time_of_oldest_sample;
     lasttime = endtime;
  }
  else
  if ( ( qt.A_gt_or_eq_B(reqtime,this_stream_header->time_of_oldest_sample))
    &&
    (qt.A_lt_or_eq_B(endtime,this_stream_header->time_of_newest_sample) ) ) 
  {
    
     starttime = reqtime;
     lasttime = endtime;
  }
  else
  if ( ( qt.A_gt_or_eq_B(reqtime,this_stream_header->time_of_oldest_sample))
    &&
    (qt.A_gt_or_eq_B(endtime,this_stream_header->time_of_newest_sample) ) ) 
  {
    
     starttime = reqtime;
     lasttime = this_stream_header->time_of_newest_sample;
  }
  else
  {
    semaphore.Release_Semaphore();
    cout << "Unknown time request permutation " << endl;
    cout << "reqtime" << endl;
    qt.print_inttime(reqtime);
    cout << "duration " << duration << endl;
    cout << "endtime " << endl;
    qt.print_inttime(endtime);
    cout << "oldest sample" << endl;
    qt.print_inttime(this_stream_header->time_of_oldest_sample);
    cout << "newest sample" << endl;
    qt.print_inttime(this_stream_header->time_of_newest_sample);
    return(FAILURE);
  }


  int samples_from_oldest_to_start;
  int number_of_samples_to_extract;

  if (qt.A_eq_B(starttime,this_stream_header->time_of_oldest_sample))
  {
    samples_from_oldest_to_start = 0;
  }
  else
  {
    double tdif;
    int itdif;

    
    tdif = tdiff(starttime,this_stream_header->time_of_oldest_sample);
    itdif = (int) tdif;
 //   cout << "time dif in seconds : " << (int) itdif /10000 << endl;

    samples_from_oldest_to_start =
      samples_in_time(this_stream_header->samples_per_second,itdif);

 //  cout << "Number of samples from oldest data to start of requested data : " 
 //   << samples_from_oldest_to_start << endl;
  }

  // 
  // Find number of samples to extract
  //

  double tdif;
  int itdif;

  tdif = tdiff(lasttime,starttime);
  itdif = (int) tdif;

// cout << "Seconds of requested data as integers : " << itdif/10000 << endl;
// cout << "Sample rate : " << this_stream_header->samples_per_second << endl;

  number_of_samples_to_extract = 
    samples_in_time(this_stream_header->samples_per_second,itdif);

  cout << "Number of requested samples currently in memory : " 
    << number_of_samples_to_extract << endl;

  // 
  // Set current sample ptr to oldest sample
  //

  current_sample_offset = this_stream_header->offset_to_oldest_sample;

  //
  // Move pointer ahead to first sample
  //

  if (samples_from_oldest_to_start == 0)
  {
    char* t_ptr;

    t_ptr = start_of_memmap + current_sample_offset;
    current_sample = (Sample*) t_ptr;
  }
  else
  {
    for(int i=0;i<samples_from_oldest_to_start;i++)
    {
      MoveCurrentSamplePtrOneSample();
    }
  }

  *outtime = starttime;

  int number_of_samples_extracted;
  number_of_samples_extracted = 0;

  BOOL nogaps;
  nogaps = TRUE;

  while( (number_of_samples_extracted < number_of_samples_to_extract) &&
        nogaps )
  {
    if(current_sample->valid)
    {
      retarray[number_of_samples_extracted] = current_sample->counts;
      ++number_of_samples_extracted;
    }
    else
    {
      cout << "time gap found, trucating extaction " << endl;
      nogaps = FALSE;
    }
    MoveCurrentSamplePtrOneSample();
  }


  semaphore.Release_Semaphore();

  *number_of_out_samples = number_of_samples_extracted; 

  return(SUCCESS);

}



int GetData::FindStationStreamHeader(char* sta,char* stream)
{
  char* temp_ptr;

  // 
  // Point to first stream header in memory
  //

  temp_ptr = start_of_memmap + sizeof(int);

  this_stream_header = (Stream_Header *) temp_ptr;

  for (int i=0;i<number_of_streams;i++)
  {
    if( (strcmp(sta,this_stream_header->station) == 0) &&
      (strcmp(stream,this_stream_header->stream)==0) )
    {
      return( i * sizeof(Stream_Header) );
    }
    else
    {
      temp_ptr = temp_ptr + sizeof(Stream_Header);
      this_stream_header = (Stream_Header *) temp_ptr;
    }
  }
  return(FAILURE);
}



void GetData::MoveCurrentSamplePtrOneSample()
{
  char* temp_ptr;

  //
  // Increment the newest data ptr;
  //

  current_sample_offset = current_sample_offset +
    sizeof(Sample);

  if(current_sample_offset >
      this_stream_header->offset_to_last_sample)
  {
    current_sample_offset =
      this_stream_header->offset_to_first_sample;
  }

  temp_ptr = start_of_memmap + current_sample_offset;
  current_sample = (Sample*) temp_ptr;
}
