/***********************************************************

File Name :
	csfill.C

Programmer:
	Phil Maechling

Description:
	comserv client that fills a shared memory segment with
	time series.

Limitations or Warnings:
	

Creation Date:
	29 Aug 1995

Modification History:

     Client Test Program. Gets data from server.
     Copyright 1994 Quanterra, Inc.
     Written by Woodrow H. Owens
	
     Converted to C++ for use with the cda routines.


Usage Notes:
	read the memcgf.dat file to get stations streams


**********************************************************/
#include <iostream.h>

#define NULL 0

const int MAX_SAMPLES_IN_PACKET = 512;

char *syntax[] = {
"    [-v n] [-h] station",
"    where:",
"	-v n		Set verbosity level to n.",
"	-h		Help - prints syntax message.",
"	station		station name.",
NULL };

#include <sunmath.h>
#include <stdlib.h>
#include <termio.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <unistd.h> // for sleep()
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/shm.h>
#include <sys/time.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <math.h>

#include "basics.h"

double log2(double);

#include "qlib.h"
#include "dpstruc.h"
#include "seedstrc.h"
#include "stuff.h"
#include "timeutil.h"
#include "service.h"
#include "cfgutil.h"
#include "cssubs.h"
#include "mem_map.h"
#include "ts_station_mgr.h"
#include "qtime.h"
#include "unpack.h"
#include "syserr.h"

#define	TIMESTRLEN  40

/************************************************************************/
/*  Externals required in multiple files.				*/
/************************************************************************/
FILE *info = stdout;		/* Default FILE for messages.		*/
char	*cmdname;		/* Program name.			*/
short data_mask =		/* data mask for cs_setup.		*/
    (CSIM_DATA);
static char name[5] = "CFIL" ;	/* Default client name			*/
char sname[5] = "*" ;		/* Default station list.		*/
tstations_struc stations ;
typedef char char23[24] ;
typedef char char5[6] ;


char23 stats[11] = { "Good", "Enqueue Timeout", "Service Timeout", 
		       "Init Error",
                       "Attach Refused", "No Data", "Server Busy", 
		       "Invalid Command",
                       "Server Dead", "Server Changed", "Segment Error" } ;


void finish_handler(int sig);
void terminate_program (int error);
static int terminate_proc;
int verbosity;
static pclient_struc me = NULL;

extern int save_selectors(int , char *);
extern int set_selectors (pclient_struc );
extern int write_to_cda(seed_record_header *pseed);
extern int decompress_data(DATA_HDR*, unsigned char *, int *);

extern "C" {
char* seednamestring (seed_name_type *, location_type *);
}


typedef struct _sel {		/* Structure for storing selectors.	*/
    int nselectors;		/* Number of selectors.			*/
    seltype *selectors;		/* Ptr to selector array.		*/
} SEL;

static SEL sel[NUMQ];

/************************************************************************/
/*  print_syntax:							*/
/*	Print the syntax description of program.			*/
/************************************************************************/
void print_syntax(char *cmd)
{
    int i;
    cout << cmd << endl;
    for (i=0; syntax[i] != NULL; i++) 
    {
      cout << syntax[i] << endl;
    }
}



/* Define these at file scope so that the write routine can see them */

TS_Station_Manager tscope;
static int rawdata[512]; // assume each byte in 512 byte block
			 // decompresses to a 32 bit sample
/************************************************************************/
/*  main procedure							*/
/************************************************************************/
main (int argc, char *argv[], char **envp)
{
      pclient_station thist;
      short j, k;
      boolean alert ;
      pdata_user pdat ;
      seed_record_header *pseed ;
      int res;

      config_struc cfg;
      char str1[160], str2[160], station_dir[160];
      char station_desc[60], source[160];
      char filename[160];
      char time_str[TIMESTRLEN];
      char default_selectors[] = "BHZ,BHN,BHE,HLZ,HLN,HLE";

    /* Variables needed for getopt. */
    extern char	*optarg;
    extern int	optind, opterr;
    int		c;

    /* CDA objects */

    Memory_Map mmap;
    char* cmap;
    int* tmap;

    char s1[5];
    char s2[5];
    char s3[5];
    char s4[5];
    char s5[5];
    char s6[5];

    cmdname = argv[0];
    while ( (c = getopt(argc,argv,"hv:p:")) != -1)
	switch (c) {
	case '?':
	case 'h':   print_syntax (cmdname); exit(0);
	case 'v':   verbosity=atoi(optarg); break;
	}

    /*	Skip over all options and their arguments. */

    argv = &(argv[optind]);
    argc -= optind;
    info = stdout;

/* Allow override of station name on command line */
      if (argc > 0)
        then
          begin
            strncpy(sname, argv[0], 4) ;
            sname[4] = '\0' ;
          end
        else
	  begin
	    fprintf (stderr, "Missing station name\n");
	    exit(1);
	  end
      upshift(sname) ;

/* open the stations list and look for that station */
      strcpy (filename, "/etc/stations.ini") ;
      if (open_cfg(&cfg, filename, sname))
        then
	  begin 
            fprintf (stderr,"Could not find station\n") ;
	    exit(1);
	  end

/* Try to find the station directory, source, and description */
      do
        begin
          read_cfg(&cfg, str1, str2) ;
          if (str1[0] == '\0')
            then
              break ;
          if (strcmp(str1, "DIR") == 0)
            then
              strcpy(station_dir, str2) ;
          else if (strcmp(str1, "DESC") == 0)
            then
              begin
                strcpy(station_desc, str2) ;
                station_desc[59] = '\0' ;
	        fprintf (info, "%s %s startup - %s\n", 
			     localtime_string(dtime()),
			     sname, station_desc) ;
              end
          else if (strcmp(str1, "SOURCE") == 0)
            then
              strcpy(source, str2) ;
        end
      while (1) ;
      close_cfg(&cfg) ;
     
/* Read the cda config file and initialize all the specified streams */
/* Currently, this init assume 6 channels.vbb and lg */

      res = initialize_cda(sname,s1,s2,s3,s4,s5,s6);

      if (res != SUCCESS)
      {
        cout << "Error initializing contiguous data area." << endl;
        exit(FAILURE);
      }

      cmap = mmap.Start_of_Memory_Area();
      tmap = (int*) cmap;

      tscope.channel_1.Initialize_Stream(cmap,sname,s1);
      tscope.channel_2.Initialize_Stream(cmap,sname,s2);
      tscope.channel_3.Initialize_Stream(cmap,sname,s3);
      tscope.channel_4.Initialize_Stream(cmap,sname,s4);
      tscope.channel_5.Initialize_Stream(cmap,sname,s5);
      tscope.channel_6.Initialize_Stream(cmap,sname,s6);

/* cda config done */
 
      terminate_proc = 0;
      signal (SIGINT,finish_handler);
      signal (SIGTERM,finish_handler);

/* Set up a condition handler for SIGPIPE, since a write to a	*/
/* close pipe/socket raises a alarm, not an error return code.	*/

      signal (SIGPIPE,finish_handler);

/* Generate an entry for all available stations */      

      cs_setup (&stations, name, sname, TRUE, TRUE, 10, 7, data_mask, 6000) ;

/* Create my segment and attach to all stations */      
      me = cs_gen (&stations) ;

/* Set up special selectors. */

      save_selectors (0, default_selectors);
      set_selectors (me);

/* Show beginning status of all stations */

      strcpy(time_str, localtime_string(dtime()));
      for (j = 0 ; j < me->maxstation ; j++)
        begin
          thist = (pclient_station) ((long) me + me->offsets[j]) ;
	  thist->seqdbuf = CSQ_LAST;
	  fprintf (info, "%s - [%s] Status=%s\n", 
	           time_str, long_str(thist->name.l), 
		   &(stats[thist->status])) ;
        end
      fflush (info);

/* Send a message every second to all stations. */
/* Sending an attach command is harmless */

      do
        begin
          j = cs_scan (me, &alert) ;
          if (j != NOCLIENT)
            then
              begin
                thist = (pclient_station) ((long) me + me->offsets[j]) ;
                if (alert)
                  then
		    begin
		      strcpy(time_str, localtime_string(dtime()));
			fprintf (info, "%s - New status on station %s is %s\n",
			 time_str,
			 long_str(thist->name.l), &(stats[thist->status])) ;
			fflush (info);
		    end
                if (thist->valdbuf)
                  then
                    begin
                      pdat = (pdata_user) ((long) me + thist->dbufoffset) ;
                      for (k = 0 ; k < thist->valdbuf ; k++)
                        begin
                          pseed = (seed_record_header*) &pdat->data_bytes ;
			  if (verbosity & 1)
			  then
			    begin
			      fprintf (info, "[%-4.4s] <%2d> %s recvtime=%s ",
				 &thist->name, k, 
			         seednamestring(&pseed->channel_id,
				 &pseed->location_id), 
                                 localtime_string(pdat->reception_time)) ;

				 printf("hdrtime=%s\n", 
				   time_string(pdat->header_time)) ;

				fflush (info);
			    end
			  write_to_cda(pseed);
                          pdat = (pdata_user) ((long) pdat + thist->dbufsize) ;
                        end
                    end
              end
            else
	      begin
		if (verbosity & 2) {
		    fprintf (info, "sleeping...");
		    fflush (info);
		}
		sleep (1) ; /* Bother the server once every second */
		if (verbosity & 2)  {
		    fprintf (info, "awake\n");
		    fflush (info);
		}
	      end
        end
      while (! terminate_proc) ;
    terminate_program (0);
      return(0);
}



void finish_handler(int sig)
{
    terminate_proc = 1;
    signal (sig,finish_handler);    /* Re-install handler (for SVR4)	*/
}

/************************************************************************/
/*  terminate_program							*/
/*	Terminate prog and return error code.  Clean up on the way out.	*/
/************************************************************************/
void terminate_program (int error) 
{
    pclient_station thist;
    char time_str[TIMESTRLEN];
    int j;
    boolean alert ;

    strcpy(time_str, localtime_string(dtime()));
    if (verbosity & 2)  {
	fprintf (info, "%s - Terminating program.\n", time_str);
	fflush (info);
    }

    /* Perform final cs_scan for 0 records to ack previous records.	*/
    /* Detach from all stations and delete my segment.			*/
    if (me != NULL) {
	for (j=0; j< me->maxstation; j++) {
	    thist = (pclient_station) ((long) me + me->offsets[0]) ;
	    thist->reqdbuf = 0;
	}
	strcpy(time_str, localtime_string(dtime()));
	    fprintf (info, "%s - Final scan to ack all received packets\n", 
		time_str);
	    fflush (info);
	cs_scan (me, &alert);
	cs_off (me) ;
    }

    strcpy(time_str, localtime_string(dtime()));
	fprintf (info, "%s - Terminated\n", time_str);

    cout << "Exiting through signal handler" << endl;
    exit(error);
}


char *selector_type_str[] = {"DAT", "DET", "CAL"};
/************************************************************************/
/*  save_selectors:							*/
/*	Parse and save selectors for specific types of info.		*/
/************************************************************************/
int save_selectors(int type, char *str)
{
    char *token;
    seltype *selectors = NULL;
    char *p = str;
    int n = 0;

    if (str == NULL || (int)strlen(str) <= 0) return(FAILURE);
    sel[type].nselectors = 0;
    if (sel[type].selectors) free (sel[type].selectors);
    while (token = strtok(p,",")) {
	if ((int)strlen(token) > 5) {
	    fprintf (info, "Error in selector list for %s\n",
		     selector_type_str[type]);
	    if (selectors) free (selectors);
	    return(FAILURE);
	}
	selectors = (selectors == NULL) ? (seltype *)malloc(sizeof(seltype)) : 
	    (seltype *)realloc (selectors, (n+1)*sizeof(seltype));
	if (selectors == NULL) {
	    fprintf (info, "Error allocating selector space for %s\n",
		     selector_type_str[type]);
	    return(FAILURE);
	}
	strcpy(selectors[n++],lead(5,'?',token));
	p = NULL;
    }
    sel[type].selectors = selectors;
    sel[type].nselectors = n;
    return(SUCCESS);
}

/************************************************************************/
/*  set_selectors:							*/
/*	Set selector values for the single station.			*/
/*	Assume sufficient selectors have been reserved.			*/
/************************************************************************/
int set_selectors (pclient_struc me)
{
    pclient_station thist;
    pselarray psa ;
    int nsel = 1;
    int type, n, i;

    thist = (pclient_station) ((long) me + me->offsets[0]) ;
    psa = (pselarray) ((long) me + thist->seloffset) ;
	
    for (type=0; type<CHAN; type++) {
	n = sel[type].nselectors;
	if (nsel+n > thist->maxsel) {
	    fprintf (info, "Error: Require %d selectors, allocated %d\n",
		     nsel+n, thist->maxsel);
	    return(FAILURE);
	}
	if (n > 0) {
	    thist->sels[type].first = nsel;
	    thist->sels[type].last = nsel + n - 1;
	    memcpy ((void *)&((*psa)[nsel]), (void *)sel[type].selectors, n*sizeof(seltype));
	  fprintf (info, "selector type = %d, nselectors = %d\n", type, n);
	  for (i=0; i<n; i++) fprintf (info, "%s\n",sel[type].selectors[i]);
	}
	nsel += n;
    }
    return(SUCCESS);
}

#define	SEED_BLKSIZE	512
int write_to_cda(seed_record_header *tseed)
{
    int             samp;
    int tickspersample;
    INT_TIME itime;
    int msecs;
    int dummysec;
    DATA_HDR *pseed;
    int channel_number;
    int blksize;
    BS *bs = NULL;
    BLOCKETTE_HDR *bh = NULL;

    if ((pseed = decode_hdr_sdr((SDR_HDR *)tseed, &blksize)) == NULL)
    {
      printf("Error decoding SEED data hdr\n");
       return(FAILURE);
    }

    if(blksize != SEED_BLKSIZE)
    {
      printf("SEED block size not 512\n");
      free_data_hdr(pseed);
      return(FAILURE);
    }

    /* Fill in the number of data frames.   */
    if ((bs = find_blockette(pseed,1001)) &&
		(bh = (BLOCKETTE_HDR *)(bs->pb))) {
		pseed->num_data_frames = ((BLOCKETTE_1001 *)bh)->frame_count;
		/* Explicitly set num_data_frames for SHEAR stations.	*/
		if (pseed->num_data_frames == 0 && pseed->sample_rate != 0)
		    pseed->num_data_frames = (blksize - pseed->first_data) / 
			sizeof(FRAME);
	}
	else pseed->num_data_frames =
		(pseed->sample_rate == 0) ? 0 :
		(blksize - pseed->first_data) / sizeof(FRAME);

    if (pseed == NULL)
    {
      return (FAILURE);
    }

    itime = pseed->begtime; // internal time
    msecs = pseed->begtime.ticks * 1000 / TICKS_PER_SEC;

    if (verbosity) printf ("packet %s stime=%s msecs=%d\n", 
                       pseed->channel_id, time_to_str(itime,1), msecs);

    samp = decompress_data(pseed, 
                           ((unsigned char *)&tseed[0])+pseed->first_data, 
                           (int *) &rawdata);

    if (samp != pseed->num_samples) 
    {
	printf("Samp not equal to n_samples\n");
        free_data_hdr(pseed);
	return(FAILURE);
    }

    /* Calculate the time interval in one sample */
    /* This doesn't handle the low data rates properly */

    time_interval(1, pseed->sample_rate,
                  &dummysec,&tickspersample);

    if (strncmp("Z",&pseed->channel_id[2],1) == 0)
    {
      if (strncmp("H",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 1;
      }
      else if (strncmp("L",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 4;
      }
      else
      {
        printf("Invalid channel name found : %s \n",pseed->channel_id);
        free_data_hdr(pseed);
        return(FAILURE);
      }
    }
    else if (strncmp("N",&pseed->channel_id[2],1) == 0)
    {
      if (strncmp("H",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 2;
      }
      else if (strncmp("L",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 5;
      }
      else
      {
        printf("Invalid channel name found : %s \n",pseed->channel_id);
        free_data_hdr(pseed);
        return(FAILURE);
      }
    }
    else if (strncmp("E",&pseed->channel_id[2],1) == 0)
    {
      if (strncmp("H",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 3;
      }
      else if (strncmp("L",&pseed->channel_id[1],1) == 0)
      {
        channel_number = 6;
      }
      else
      {
        printf("Invalid channel name found : %s \n",pseed->channel_id);
        free_data_hdr(pseed);
        return(FAILURE);
      }
    } 
    else
    {
      printf("Invalid channel name found : %s \n",pseed->channel_id);
      free_data_hdr(pseed);
      return(FAILURE);
    }


    for(int i=0;i<samp;i++)
    {
      switch(channel_number)
      {
        case 1:
        {
          tscope.channel_1.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
	case 2:
        {
          tscope.channel_2.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
	case 3:
        {
          tscope.channel_3.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
        case 4:
        {
          tscope.channel_4.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
	case 5:
        {
          tscope.channel_5.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
	case 6:
        {
          tscope.channel_6.Put_Sample(itime,rawdata[i]);
          itime = add_time(itime,dummysec,tickspersample);
          break;
        }
	default:
        {
          cout << "Channel that is not handled" << endl;
          free_data_hdr(pseed);
          return(FAILURE);
        }
      } // end switch
    }

    free_data_hdr(pseed);
    if (verbosity) fprintf (info, "wrote data to cda sent\n");
    fflush (info);
    return (samp);
}

int decompress_data(DATA_HDR *hdr, unsigned char *indata, int *array)
{
    int x0, xn;
    int numsamp;
    int diffbuff[MAX_SAMPLES_IN_PACKET];

    switch (hdr->data_type) {
      case EMPTY:
	fprintf (stdout, "Empty packet all nulls : %d\n:", hdr->data_type);
	numsamp = 0;
	break;
      case STEIM1:
	numsamp = unpack_steim1 ((FRAME *)indata, 
                                 hdr->num_data_frames * sizeof(FRAME), 
				 hdr->num_samples, hdr->num_samples, 
				 array, diffbuff, &x0, &xn, NULL);
	break;
      case STEIM2:
	numsamp = unpack_steim2 ((FRAME *)indata, 
                                 hdr->num_data_frames * sizeof(FRAME), 
				 hdr->num_samples, hdr->num_samples, 
				 array, diffbuff, &x0, &xn, NULL);
	break;
      default:
	cout << "Unknown datatype: " << hdr->data_type << endl;
        fflush(stdout);
	numsamp = -1;
	break;
    }
    return numsamp;
}
