/************************************************************************/
/*  msunpack -								*/
/*	Unpack packed miniSEED files (such as LOG, TIM, DET, CAL files)	*/
/*	that were retrieved from a Quanterra datalogger using		*/
/*	Quanterra's RETRIEVE program into full miniSEED blocks.		*/
/*									*/
/*	Douglas Neuhauser						*/
/*	Seismological Laboratory					*/
/*	University of California, Berkeley				*/
/*	doug@seismo.berkeley.edu					*/
/*									*/
/*  Assumptions:							*/
/*  1.  Packet look like standard MiniSEED blocks, but packets are	*/
/*	multiple of PACKSIZE (currently 128) bytes.			*/
/*									*/
/*  2.  An integer number of packets are packed into a block of fixed	*/
/*	size.  Blksize is power of 2 (as with miniSEED), and may range	*/
/*	from PACKSIZE to maximum miniSEED blksize.			*/
/*	Currently the blksize is 4096.					*/
/*									*/
/*  3.  Packets contain a blockette 1000, but the blocksize contained	*/
/*	in the blockette is not to be believed.  It appears to bear	*/
/*	no resemblance to either the packet size or the overall blksize.*/
/*									*/
/*  4.  Packets are never split across block boundaries.  Therefore,	*/
/*	there may be one or more PACKSIZE packets of NULLS between	*/
/*	blocks, in order to ensure a block boundary every 4096 bytes.	*/
/*									*/
/*  Reference:	"Quanterra Ultra SHEAR SEED Retrieval" document.	*/
/*									*/
/*  Input:	MSG packet miniSEED from Quanterra Retrieve program.	*/
/*  Output:	miniSEED file.						*/
/*									*/
/************************************************************************/

/*
 * Copyright (c) 1996-2000 The Regents of the University of California.
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for educational, research and non-profit purposes,
 * without fee, and without a written agreement is hereby granted,
 * provided that the above copyright notice, this paragraph and the
 * following three paragraphs appear in all copies.
 * 
 * Permission to incorporate this software into commercial products may
 * be obtained from the Office of Technology Licensing, 2150 Shattuck
 * Avenue, Suite 510, Berkeley, CA  94704.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
 * INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND
 * ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 * CALIFORNIA HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#ifndef lint
static char sccsid[] = "@(#)msunpack.c	1.5 10/26/00 13:39:31";
#endif

#include <stdio.h>

#include "version.h"
#define	PACKSIZE    128
#define	OUTPUT_BLKSIZE	    4096

char *syntax[] = {
"%s version " VERSION,
"%s [-p n] [-b n] [-p n] [-h] [infile [outfile]]",
"    where:",
"	-b blksize  Specify output blksize.  Default blksize is 4096.",
"	-p packsize Specify input packsize.   Default packsize is 128.",
"	-d n	    Debug output (OR of any of the following values:)",
"			1 = packet info",
"	-h	    Help - prints syntax message.",
"	infile	    Name of input file.  If none specified, read from stdin.",
"	outfile	    Name of output file.  If none specified, write to stdout.",
"		    You must specify an infile if you specify an outfile.",
NULL };

#include <memory.h>
#include <string.h>
#include <math.h>
double pow(double x, double y);	/* Not included in some math.h files.	*/

#include "qlib2.h"

char *cmdname;			/* program name.			*/
FILE *info;			/* information file ptr.		*/
int debug_option;		/* debug options.			*/
int output_blksize = 4096;	/* default output blksize.		*/
int packsize = PACKSIZE;	/* base input size (bytes) for packets.	*/

#define	MAX_BLKSIZE		8192
#define	DEBUG
#define	DEBUG_PACKET		1
#define	DEBUG_ANY		(DEBUG_PACKET)
#define debug(val)		(debug_option & (val))
#define ERROR			-2
#define	FIXED_DATA_HDR_SIZE	48

/************************************************************************/
/*  get_blksize:							*/
/*	Parse blksize string and return blocksize.  Ensure it is valid, */
/*	ie, it is a power of 2, and within the valid blksize boundaries.*/
/*	Return 0 on error.						*/
/************************************************************************/
int get_blksize
   (char	*str)
{
    int blksize, n, m;
    char *p;
    double l2_blksize;
    blksize = strtol (str, &p, 10);
    switch (*p) {
      case 0:	break;
      case 'k':
      case 'K':	blksize *= 1024; break;
      default:	blksize = 0;
    }
    if (blksize <= 0 || blksize > MAX_BLKSIZE) return (0);
    l2_blksize = log2((double)blksize);
    n = (int)l2_blksize;
    m = (int)pow(2.0,(double)n);
    if (m != blksize) return (0);
    return (blksize);
}

/************************************************************************/
/*  main - main program.						*/
/************************************************************************/
main (int argc, char **argv)
{
    FILE *in = stdin;		/* input file ptr.			*/
    FILE *out = stdout;		/* output file ptr.			*/
    char *infile = "<stdin>";	/* filename of input file.		*/
    char *outfile = "<stdout>";	/* filename of output file.		*/
    char *blksize_str = NULL;	/* string for new blksize.		*/
    char *packsize_str = NULL;	/* string for new packsize.		*/
    DATA_HDR *hdr;		/* ptr to unpacked data header.		*/
    char buf[8192];		/* buffer for full packet.		*/
    int hdrbytes;		/* number of bytes read processing hdr.	*/
    int nframes;		/* number of frames in packet.		*/
    int nbytes;			/* number of bytes in packet.		*/
    int blksize;		/* output blksize of packet.		*/
    int log2_output_blksize;	/* log2(output_blksize).		*/
    int npad;			/* number of bytes to pad packet.	*/
    int n;			/* return code from get_ms_hdr.		*/
    int blkno = 0;		/* packet number.			*/
    char *p;			/* generic character ptr.		*/
    BS *bs;			/* ptr to blockette structure.		*/
    double l2_blksize;

    /* Variables needed for getopt. */
    extern char	*optarg;
    extern int	optind, opterr;
    int		c;

    cmdname = tail(*argv);
    info = stdout;
    /*	Parse command line options.					*/
    while ( (c = getopt(argc,argv,"hd:b:p:")) != -1)
	switch (c) {
	case '?':
	case 'h':   print_syntax (cmdname, syntax, info); exit(0); break;
	case 'd':   debug_option = atoi(optarg); break;
	case 'b':   blksize_str = optarg; break;
	case 'p':   packsize_str = optarg; break;
	default:
	    fprintf (info, "Invalid option: -%c\n", c);
	    exit(1);
	    break;
	}

    if (blksize_str) {
	output_blksize = get_blksize(blksize_str);
	if (output_blksize <= 0 || output_blksize > MAX_BLKSIZE) {
	    fprintf (stderr, "Invalid blksize: %s\n", blksize_str);
	    exit(1);
	}
    }
    if (packsize_str) {
	packsize = get_blksize(packsize_str);
	if (packsize <= 0 || packsize > MAX_BLKSIZE) {
	    fprintf (stderr, "Invalid packsize: %s\n", packsize_str);
	    exit(1);
	}
    }

    if (packsize > output_blksize) {
	fprintf (stderr, "Error - packsize %d > blksize %d\n",
		 packsize, output_blksize);
	exit(1);
    }

    /*	Skip over all options and their arguments.			*/
    argv = &(argv[optind]);
    argc -= optind;

    if (argc-- > 0) {
	infile = *argv++;
	if ((in=fopen(infile,"r")) == NULL) {
	    fprintf (stderr, "Unable to open infile %s\n", infile);
	    exit(1);
	}
    }
    if (argc-- > 0) {
	outfile = *argv++;
	if ((out=fopen(outfile,"w")) == NULL) {
	    fprintf (stderr, "Unable to open outfile %s\n", outfile);
	    exit(1);
	}
    }
    else (info = stderr);
    l2_blksize = log2(output_blksize);
    log2_output_blksize = (int)l2_blksize;
    blksize = pow(2.0,(double)log2_output_blksize);
    if (blksize != output_blksize ||
	output_blksize < packsize || output_blksize > 8192) {
	fprintf (info, "Error - invalid blksize %d.\n", output_blksize);
	exit(1);
    }

    while ((n=get_ms_hdr(&hdr, buf, &blksize, &hdrbytes, in)) != EOF) {
	if (n == ERROR) {
	    nbytes = packsize;
	    fread (buf+hdrbytes, nbytes-hdrbytes, 1, in);
	    continue;
	}
	++blkno;
	if (hdr == NULL) {
	    fprintf (info, "Error decoding header for block %d\n", blkno);
	    exit(1);
	}
	/* Compute the number of additional bytes to read.		*/
	/* If there is a b1001 with number of frames, use that.		*/
	/* If rate == 0, use num_samples.				*/
	/* if num_frames == 0 && rate != 0 && num_samples != 0,		*/
	/* flag it as an error (for now).				*/
	nframes = 0;
	if ((bs = find_blockette (hdr, 1001)) != NULL) {
	    nframes = pow(2.0,((BLOCKETTE_1001 *)(bs->pb))->frame_count);
	}
	if (nframes > 0 ) {
	    if (hdr->num_samples > 0) {
		/* Compressed data.					*/
		nbytes = hdr->first_data + (nframes * sizeof(FRAME));
	    }
	    else {
		fprintf (stderr, "Error - nframes > 0, but no samples\n");
	    }
	}
	else {
	    if (sps_rate(hdr->sample_rate, hdr->sample_rate_mult) == 0.0) {
		/* Non-timeseries data.					*/
		if (hdr->num_samples > 0) {
		    /* Uncompressed data (such as LOG files.		*/
		    nbytes = hdr->first_data + hdr->num_samples;
		}
		else {
		    /* No data values - only header and blockettes.	*/
		    nbytes = hdrbytes;
		}
	    }
	    else {
		/* Uncompressed timeseries.				*/
		fprintf (stderr, "Error - unable to process uncompressed timeseries at this time\n");
		exit(1);
	    }
	}
	/* Round up the total number of bytes to multiple of packsize.	*/
	nbytes = ((nbytes+packsize-1)/packsize)*packsize;
	if (nbytes-hdrbytes > 0) {
	    n = fread (buf+hdrbytes, nbytes-hdrbytes, 1, in);
	    if (n != 1) {
		fprintf (stderr, "Error reading contents of block\n");
		exit(1);
	    }
	}
	if (debug(DEBUG_PACKET)) {
	    fprintf (info, "seq=%d psize=%d ns=%d b1000.blksize=%d, nframes=%d\n",
		     hdr->seq_no, nbytes, hdr->num_samples, blksize, nframes);
	}
	/* Explicitly set blksize.	*/
	if (blksize != output_blksize) {
	    buf[54] = log2_output_blksize;
	    blksize = output_blksize;
	}
	npad = output_blksize - nbytes;
	if (npad < 0) {
	    fprintf (info, "Error - block %d too large (%d) for output blksize %d\n",
		     blkno, nbytes, output_blksize);
	    exit(1);
	}
	memset (buf+nbytes, 0, npad);
	fwrite (buf, output_blksize, 1, out);
    }
    if (n != EOF) {
	++blkno;
	fprintf (info, "Error reading file, blkno = %d retcode = %d\n", blkno, n);
	exit(1);
    }
    return (0);
}

/************************************************************************/
/*  get_ms_hdr -							*/
/*	Get and parse the miniSEED header and blockettes.		*/
/************************************************************************/
int get_ms_hdr
   (DATA_HDR	**phdr,		/* ptr to ptr to DATA_HDR (returned)	*/
    char	buf[],		/* buffer or MiniSEED record.		*/
    int		*pblksize,	/* ptr to blksize (returned).		*/
    int		*phdrbytes,	/* ptr to # bytes in header (returned).	*/
    FILE	*fp)		/* input FILE.				*/
{
    DATA_HDR *hdr;		/* pointer to DATA_HDR.			*/
    BS *bs;			/* ptr to blockette structure.		*/
    int nskip = 0;
    int offset = 0;
    int nread;
    int	blksize;		/* blocksize of miniSEED record.	*/
    int bl_limit;		/* offset of data (blksize if no data).	*/

    /* Read and decode SEED Fixed Data Header.				*/
    *phdr = (DATA_HDR *)NULL;
    *phdrbytes = 0;
    *pblksize = 0;
    if ((nread = fread(buf, FIXED_DATA_HDR_SIZE, 1, fp)) != 1)
	return ((nread == 0) ? EOF : ERROR);
    offset = FIXED_DATA_HDR_SIZE;
    *phdrbytes = offset;
    if ((hdr = decode_fixed_data_hdr((SDR_HDR *)buf)) == NULL) return (ERROR);
    *phdrbytes = FIXED_DATA_HDR_SIZE;

    /* Read blockettes.  MiniSEED should have at least blockette 1000.	*/
    if (hdr->num_blockettes > 0) {
	if (hdr->first_blockette < offset) {
	    free_data_hdr(hdr);
	    return(ERROR);
	}
	if (hdr->first_blockette > offset) {
	    nskip = hdr->first_blockette - offset;
	    if (fread (buf+offset, nskip, 1, fp) != 1) {
		free_data_hdr(hdr);
		return (ERROR);
	    }
	    offset += nskip;
	}
	*phdrbytes = offset;
	if ((offset = read_ms_bkt (hdr, buf, fp)) < 0) {
	    return (ERROR);
	}
    }

    /* Determine blocksize and data format from the blockette 1000.	*/
    /* If we don't have one, it is an error.				*/
    if ((bs = find_blockette (hdr, 1000)) == NULL) {
	return (ERROR);
    }
    blksize = pow(2.0,((BLOCKETTE_1000 *)(bs->pb))->data_rec_len);

    /* Skip over space between blockettes (if any) and data.		*/
    /* For packed files, limit skip to multiple of packsize,		*/
    /* not blksize, since there may be only blockettes and no data.	*/
    bl_limit = (hdr->first_data) ? hdr->first_data : 
	((offset+packsize-1)/packsize)*packsize;
    if (bl_limit < offset) {
	    free_data_hdr(hdr);
	    return(ERROR);
	}
    if (bl_limit > offset) {
	nskip = bl_limit - offset;
	if (fread (buf+offset, nskip, 1, fp) != 1) {
	    free_data_hdr(hdr);
	    return (ERROR);
	}
	offset += nskip;
    }

    *phdr = hdr;
    *phdrbytes = offset;
    *pblksize = blksize;
    return (1);		/* Header successfully read.			*/
}
