#! __PERL_PATH__

# Copyright (c) 2007-2011 The Regents of the University of California.
# All Rights Reserved.
# 
# Permission to use, copy, modify, and distribute this software and its
# documentation for educational, research and non-profit purposes,
# without fee, and without a written agreement is hereby granted,
# provided that the above copyright notice, this paragraph and the
# following three paragraphs appear in all copies.
# 
# Permission to incorporate this software into commercial products may
# be obtained from the Office of Technology Licensing, 2150 Shattuck
# Avenue, Suite 510, Berkeley, CA  94704.
# 
# IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
# FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES,
# INCLUDING LOST PROFITS, ARISING OUT OF THE USE OF THIS SOFTWARE AND
# ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN
# ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# 
# THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE
# PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
# CALIFORNIA HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,
# UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

# prototype waveform request server
#
# Syntax:

# GET_WAVEFORM net station chan loc start end eventid eventdomain format trimflag
#
# GET_INVENTORY net station channel loc start end eventid eventdomain 
#	N, S, C, and L can be wildcarded.	
#	Return the number of lines containing S N C L

# @(#)swc.pl	1.9 01/16/13 15:12:38

require 5.002;
use strict;
use FileHandle;
use Getopt::Std;
use File::Spec;
use File::Basename;
use IO::Socket::INET;
use Digest::MD5;

use lib qw ( USELIBDIR );

use Qtime;
use Waveform::swc;

use vars qw ( $VERSION $opt_h $opt_f $opt_t $opt_s $opt_o $opt_v $opt_q $opt_E $opt_D );
use vars qw ( $opt_T $opt_K $opt_S $opt_H $opt_P $opt_d $opt_I $opt_e );
use vars qw ( $request $station $network $channel $location );
use vars qw ( $starttime $endtime $id $iddomain $format $trimflag $use_event_dir );
use vars qw ( $RESP_OK $RESP_ERROR $DEFAULT_TRIMFLAG $MENU $WAVEFORM $op );
use vars qw ( $DEFAULT_SERVER $DEFAULT_PORT %servermap $serverlist $quiet );
use vars qw ( $use_qmerge $leapseconds );

$VERSION = "1.09 (2013.016)";

########################################################################
#	print_syntax	- print syntax and exit.
########################################################################
sub print_syntax {
    my ($cmdname) = @_;
    printf	
"$cmdname version $VERSION
$cmdname - Simple Waveform Client.
Syntax:
$cmdname  [-h] [-M] [-T|-K] [-f starttime [ -t endtime | -s timespan ]]
	[-E eventid -D eventid_domain]
	[-o - | outputfile] [-e] [-d N]
	[-S server] [-H hostname ][-P port] sncl1 ... snclN
where:
	-h	Help - prints this help message.
	-I	Request channel Inventory (menu list) instead of waveform data.
	-T|-K	Trim waveforms to specified time (-T) or Keep data from 
		records within specified time interval (-K).
		Default is -T.
	-f starttime
		Start time of waveform.
		The date and time are specified in a single string, and can
		 can be provided in one of several formats:
			yyyy/mm/dd,hh:mm:ss.ffff
			yyyy.doy,hh:mm:ss.ffff
		The date must be fully specified.  The time portion is optional,
		and may be specified to hour, minute, second, or subsecond
		resolution.
	-t endtime
		End time of waveform.
	-s timespan
		Timespan of request.  Timespan is an integer string terminated
		with one of the following characters (case sensitive):
			S = seconds	d = days
			M = minutes	m = months
			H = hours	y = years
		If no terminater character is used, seconds are assumed.
	-E eventid
		Specify an eventid for retrieving event waveforms.
		Not all Simple Waveform Servers support retrieval by event.
		The NCEDC DART does NOT know about events.
	-D event_domain
		Specify the domain (catalog source) for the eventid.
		The NCEDC currently supports domains NC and EGS.
		The default domain at the NCEDC is NC if not specified.
	-o - | outputfile
		Specifies an explicit output file, or '-' for stdout.
		If this option is not specified, an output file based
		on the SNCL and date_time will be created.
	-e	Output event data into a directory <eventid.event_domain>
		instead of the current directory.  Does NOT override '-o' option.
	-S servername
		Generic name of waveform server. Valid servernames are:
		$serverlist.
		Default servername is $DEFAULT_SERVER.
		See .swc-init.pl file in home directory for configuration.
	-H hostname
		Explicit hostname or IP address of waveform server.
	-P portnumber
		Port number or port name of server.
		Default port number is 4999.
	-d N	Debug value.  Current values are:
		1 = display request and response.
	sncl1 ... snclN
		A list of one or more SNCL-strings, each containing
		dot-separated sections:
			Station.Network.Channel.Location
		Each section may be a single component (eg BKS) or a 
		comma-delimited list of components such as BKS,CMB or HHE,HHN,HHZ.  
	        If any section contains a comma-delimited list, the SNCL-string
		is expanded to multiple SNCLs by performing a cross-product of
		the each components the the section with every other other section
		of the SNCL-string.  For blank location codes, you may optionally
		omit the .Location portion of the SNCL-string.

Notes:
1.  Wildcard characters used within a SNCL must be quoted to avoid having
    the Unix/Linux shell perform the wildcard expansion.  
    For Unix/Linux, use a backslash character before the  wildcard character, 
    or enclose the SNCL string in quotes.
Examples:
1.  $cmdname -S ncedc_archive -T -f 2010/02/34,05:15 -S 5M HOPS.BK.HHZ CMB.BK.HHZ
2.  $cmdname -S ncedc_archive -T -f 2010.123,12:05:15.67 -s 5M 'HOPS,CMB.BK.BHE,BHN,BHZ.*'
	expands to the list of SNCLs:
		HOPS.BK.BHE.* HOPS.BK.BHN.* HOPS.BK.BHE.*
		CMB.BK.BHE.*  CMB.BK.BHN.*  CMB.BK.BHE.*
	The waveform server will perform the wildcard expansion.
3.  $cmdname -S dart -T -f 2011.054,12:05:15.67 -s 5M 'CMB.BK.BHZ.*'
4.  $cmdname -S ncedc_archive -E 71365875 -e
5.  $cmdname -S ncedc_archive -E 71365875 -D NC -e
6.  $cmdname -S ncedc_archive -E 2002000 -D EGS -e
";
    exit(0);
}

{

####################################################################
# Initialization.
####################################################################

    STDERR->autoflush(1);
    STDOUT->autoflush(1);
    $RESP_OK = 200;
    $RESP_ERROR = 550;
    $DEFAULT_TRIMFLAG = "exact";
    $DEFAULT_SERVER = "ncedc_archive";
    $DEFAULT_PORT = 4999;
    $MENU = "menu";
    $WAVEFORM = "waveform";
    $op = $WAVEFORM;
    $leapseconds = qw ( LEAPSECONDSFILE ) ;
    $ENV{LEAPSECONDS} = $leapseconds if ($leapseconds ne "");
    my($ALL_CHANNELS) = '*.*.*.*';

    # Items that can be overridden in user's .swc-init.pl file.
    $use_qmerge = 0;
    %servermap = (
	dart		=>	"dart.ncedc.org:$DEFAULT_PORT",
	ncedc_archive	=>	"archive.ncedc.org:$DEFAULT_PORT",
	);

    # Load user's init file if found.
    my($HOME) = $ENV{HOME};
    if ($HOME ne "" &&  -f "$HOME/.swc-init.pl") {
	require "$HOME/.swc-init.pl";
    }

    $serverlist = join(", ",sort(keys(%servermap)));

########################################################################
#	Parse command line.
########################################################################
    &getopts ('hTKIef:t:s:o:S:H:P:d:E:D:q');
    &print_syntax($0) if ($opt_h);
    die ("Error: specified both -T and -K\n") if ($opt_T) && ($opt_K);
    die ("Error: missing endtime\n") if ($opt_t eq "" and $opt_s eq "" && $opt_E == 0);
    die ("Error: specified both -t endtime and -s timespan\n") if 
	($opt_t ne "" and $opt_s ne "");
    die ("Error: specified both -S service and -H hostname\n") if 
	($opt_S ne "" and $opt_H ne "");
    $opt_t = &compute_endtime ($opt_f, $opt_s) if ($opt_s ne "" && $opt_f ne "");
    $opt_f = "" if (! defined $opt_f);
    $opt_t = "" if (! defined $opt_t);
    $op = $MENU if ($opt_I);
    $quiet = $opt_q;
    $quiet = 1 if (! $use_qmerge);
    $id = ($opt_E ne "") ? $opt_E : 0;
    $iddomain = ($opt_D ne "") ? $opt_D : "";
    $use_event_dir = $opt_e;
    
    my($serverinfo, $servername, $serverhost, $serverport,$domain);
    # First look for services, then for explicit hostname.
    $servername = ($opt_S ne "") ? $opt_S : $DEFAULT_SERVER;
    ($servername = $servername) =~ tr/[A-Z]/[a-z]/;
    $serverinfo = $servermap{$servername};
    die ("Error: Unknown server name: $opt_S\n") if ($serverinfo eq "");
    ($serverhost,$serverport,$domain) = split(/:/,$serverinfo);
    # Optionally override the server host and port.
    $serverhost = $opt_H if ($opt_H ne "");
    $serverport = $opt_P if ($opt_P ne "");
    $iddomain = $domain if ($domain ne "" && $iddomain eq "");

    # Open global file or inherit STDOUT if explicit output is specified.

    # Menu goes to STDOUT unless explicitly sent elsewhere.
    # Data gets piped to sdrsplit unless explicitly sent elsewhere.

    my($fh) = new FileHandle;
    $opt_o = "-" if ($opt_o eq "" and $op eq $MENU);
    if ($opt_o eq "-") {
	$fh->fdopen(fileno(STDOUT),"w") || die ("Error: Unable to inherit STDOUT\n");
    }
    elsif ($opt_o ne "") {
	$fh->open(">$opt_o") || die ("Error: Unable to open file $opt_o\n");
    }
    else {
#:: 	$fh->open("|sdrsplit") || die "Error: Unable to pipe output to sdrsplit\n";
#:: 	$file = &gen_filename ($n, $s, $c, $l, $opt_f);
#::	$fh->open(">$file") || die "Error: Unable to open output file $file\n";
    }

    my($sncl);
    my($file);
    my($error);
    # Default for request by eventid is for all channels if no channel list specified.
    push (@ARGV, $ALL_CHANNELS) if ($id ne "" && @ARGV == 0);
    my(@sncl_list) = sort(&parse_channel_spec (@ARGV));
    for my $req (@sncl_list) {
	my(@result);
	if ($op eq $MENU) {
	    # Menu request.
	    ($error,@result) = &ch_request ($req,$serverhost,$serverport);
	    print ($fh @result);
	}
	elsif ($op eq $WAVEFORM) {
	    # Waveform request.
	    if ($req =~ /[\*\?]/) {
		# Ask server to expand channel wildcard.
		($error,@result) = &ch_request ($req,$serverhost,$serverport);
		chomp(@result);
	    }
	    else {
		@result = ($req);
	    }
	    for my $sncl (@result) {
		&wf_request ($sncl,$serverhost,$serverport,$fh);
	    }
	}
	else {
	    die ("Error: Unknown command\n");
	}
    }

    close ($fh) if ($opt_o eq "");
    unlink ($file) if ($error);

    exit(0);
}

sub ch_request 
{
    my($sncl,$serverhost,$serverport) = @_;
    my($s,$n,$c,$l) = split(/\./,$sncl);
    my($request);
    $request = Waveform::swc::ch_request ($n,$s,$c,$l,$opt_f,$opt_t,$id,$iddomain);
    die "Invalid client request\n" if (! defined $request);
    print (STDERR "$serverhost $serverport $request") if ($opt_d);

    my($client, $socket);
    $socket = IO::Socket::INET->new(PeerAddr	=> $serverhost,
				    PeerPort	=> $serverport,
				    Proto	=> "tcp",
				    Type      => SOCK_STREAM)
	or die "Could not open connection to $serverhost:$serverport : $!\n";

    # Send request.
    $socket->print($request) || die ("Error: connection rejected by serverhost.\n");

    my($response_str);
    my($error) = 1;
    my(@data);
    # Read response string with server's linecound and md5.
    # Compute md5 as data is read.
    # Compare received bytecount with server's bytecount.
    # Compare computed md5 with server's md5.
    $response_str = $socket->getline;
    die ("Error: connection rejected by server.\n") if ($response_str eq "");
    print (STDERR $response_str) if ($opt_d);
    my($resp_num,$tlines,$md5);
    ($resp_num, $tlines, $md5) = &parse_response ($response_str);
    if ($tlines >= 0 && $resp_num == $RESP_OK) {
	my($context) = Digest::MD5->new;
	my($nlines, $computed_md5);
	@data = $socket->getlines;
#::		printf ("# data elements = %d\n", scalar (@data)) if ($opt_d);
	for (my($i)=0; $i<@data; $i++) {
	    $context->add($data[$i]);
	    print (STDERR $data[$i]) if ($opt_d);
	    $nlines++;
	}
	if ($nlines != $tlines) {
	    printf (STDERR "Error: Incorrect linecount, sendlines = %d, recvlines=%d\n",
		    $tlines, $nlines);
	}
	elsif (($computed_md5 = $context->hexdigest()) ne $md5) {
	    printf (STDERR "Error: server md5 = %s, computed md5 = %s\n",
		    $md5, $computed_md5);
	}
	else {
	    $error = 0;
	}
    }
    else {
	printf (STDERR "Error: unknown response: $response_str\n");
    }
    close ($socket);
    return ($error,@data);
}


sub wf_request
{
    my($sncl,$serverhost,$serverport,$fh) = @_;
    my($s,$n,$c,$l) = split(/\./,$sncl);
    my($format) = "mseed";
    my($trimflag) = ($opt_T) ? "exact" : ($opt_K) ? "record" : $DEFAULT_TRIMFLAG;
    my($request);
    $request = Waveform::swc::wf_request ($n,$s,$c,$l,$opt_f,$opt_t,
					  $id,$iddomain,$format,$trimflag);
    die "Invalid client request\n" if (! defined $request);
    print (STDERR "$serverhost $serverport $request") if ($opt_d);

    my($client, $socket);
    $socket = IO::Socket::INET->new(PeerAddr	=> $serverhost,
				    PeerPort	=> $serverport,
				    Proto	=> "tcp",
				    Type      => SOCK_STREAM)
	or die "Could not open connection to $serverhost:$serverport : $!\n";

    # Send request.
    $socket->print($request) || die ("Error: connection rejected by server.\n");
    
    my($response_str);
    my($error) = 1;
    # Read response string with server's bytecount and md5.
    # Compute md5 as data is read.
    # Compare received bytecount with server's bytecount.
    # Compare computed md5 with server's md5.
    $response_str = $socket->getline;
    die ("Error: connection rejected by server.\n") if ($response_str eq "");
    print (STDERR $response_str) if ($opt_d);
    my($resp_num,$tbytes,$md5);
    ($resp_num, $tbytes, $md5) = &parse_response ($response_str);
    my($close_file) = 0;
    if (! $fh->opened) {
	my($file) = &gen_filename ($n, $s, $c, $l, $opt_f, $id, $iddomain);
	print ("$file $tbytes bytes\n") if ($quiet);
	if (! $quiet) {
	    $fh->open("|qmerge -b 0 -o $file") ||  die "Error: Unable to pipe output to qmerge\n";
	}
	else {
	    $fh->open(">$file") || die "Error: Unable to open output file $file\n";
	}
	$close_file = 1;	
    }
    if ($tbytes >= 0 && $resp_num == $RESP_OK) {
	my($context) = Digest::MD5->new;
	my($nbytes, @data, $computed_md5);
	@data = $socket->getlines;
#:: 	printf ("# data elements = %d\n", scalar (@data)) if ($opt_d);
	for (my($i)=0; $i<@data; $i++) {
	    $context->add($data[$i]);
	    $nbytes += length($data[$i]);
	}
	if ($nbytes != $tbytes) {
	    printf (STDERR "Error: Incorrect bytecount, sendbytes = %d, recvbytes=%d\n",
		    $tbytes, $nbytes);
	}
	elsif (($computed_md5 = $context->hexdigest()) ne $md5) {
	    printf (STDERR "Error: server md5 = %s, computed md5 = %s\n",
		    $md5, $computed_md5);
	}
	else {
	    $fh->print(@data);
	    $error = 0;
	}
    }
    else {
	printf (STDERR "Error: unknown response: $response_str\n");
    }
    close ($socket);
    $fh->close if ($close_file);
    return ($error);
}

sub compute_endtime
{
    my($start,$span) = @_;
    my($qstart,$qend,$duration,$unit);
    my($qstart) = Qtime::set_time ($start);
    die ("Error: invalid starttime: $start\n") if (! defined ($qstart));
    my($duration,$unit) = $span =~ /^(\d+)(\D?)$/;
    die ("Error: invalid timespan: $span\n") if ($duration eq "");
    $qend = Qtime::new($qstart);
    {
	my ($dum);
	Qtime::set_second ($qend,Qtime::second($qend)+$duration), last 
	    if ($unit eq "S" || $unit eq "");
	Qtime::set_minute ($qend,Qtime::minute($qend)+$duration), last 
	    if ($unit eq "M");
	Qtime::set_hour ($qend,Qtime::hour($qend)+$duration), last 
	    if ($unit eq "H");
	Qtime::set_doy ($qend,Qtime::doy($qend)+$duration), last 
	    if ($unit eq "d");
	Qtime::set_month ($qend,Qtime::month($qend)+$duration), last 
	    if ($unit eq "m");
	Qtime::set_year ($qend,Qtime::year($qend)+$duration), last 
	    if ($unit eq "y" || $unit eq "Y");
	die ("Error: invalid timespan: $span\n");
    }
    my($end) = sprintf ("%04d.%03d,%02d:%02d:%02d.%06d",
	Qtime::year($qend), Qtime::doy($qend),
	Qtime::hour($qend), Qtime::minute($qend),
	Qtime::second($qend), Qtime::usec($qend));
    return ($end);
}

sub gen_filename {
    my($n,$s,$c,$l,$start) = @_;
    my($filename);
    if ($start ne "") {
	my($qt) = Qtime::set_time($start);
	$filename = sprintf ("%s.%s.%s.%s.D.%04d.%03d.%02d%02d%02d",
	    $s, $n, $c, $l, Qtime::year($qt), Qtime::doy($qt),
	    Qtime::hour($qt), Qtime::minute($qt), Qtime::second($qt));
    }
    else {
	my($domain) = ($iddomain ne "") ? $iddomain : "NONE";
	$filename = sprintf ("%s.%s.%s.%s.D.%s.%s",
	    $s, $n, $c, $l, $id, $domain);
	if ($use_event_dir) {
		my($outdir) = "$id.$domain";
		mkdir($outdir) || (die "Error $? from mkdir $outdir\n") if (! -d $outdir);
		$filename = File::Spec->catfile ($outdir, $filename);
	}
    }
    return ($filename);

}

sub parse_response {
    my($line) = @_;
    chomp($line);
    my($resp_num, $tbytes,$md5) = split(" ",$line);
    return ($resp_num,$tbytes,$md5);
}

########################################################################
#   parse_channel_spec:
#	Parse old or new style channel specifier.
#	Return new style channel specifier list with wildcards expanded.
########################################################################
sub parse_channel_spec {
    my(@argv) = @_;
    my($token,$dummy);
    my($oldstyle,$newstyle);
    foreach $token (@argv) {
	$dummy = ($token =~ m/\./) ? ++$newstyle : ++$oldstyle;
    }
    die "Error: bad SNCL syntax\n" if ($oldstyle);
    @argv = &gen_new_list(@argv);
    # Return only a sorted unique list.
    my(%argv,$argv);
    foreach $argv (@argv) {
	$argv{$argv} = 1;
    }
    @argv = sort(keys(%argv));
    return (@argv);
}

########################################################################
#   gen_new_list:
#	Generate new channel specifier list from wildcard specifier list.
#	Return new channel specifier list.
########################################################################
sub gen_new_list {
    my(@argv) = @_;
    my(@list,$argv);
    foreach $argv (@argv) {
	push(@list, &expand_newchan ($argv));
    }
    return (@list);
}

########################################################################
#   expand_newchan:
#	Expand comma-separated fields in SNCL components into full
#	cross product.  Do NOT do any wildcard expansion, since this
#	is done by sws.
#	Return matching channel specifier(s).
#	Allow station.net.channel to match station.net.channel.
########################################################################
sub expand_newchan {
    my($arg) = @_;
    $arg .= "." if ($arg =~ /^([^.]+\.[^.]+\.[^.]+)$/);
    return ($arg) if (! $arg =~ m/[\,\?\*]/);
    # Pick apart components, and create outer product from 
    # comma-delimted tokens.
    my($station,$network,$channel,$location);
    ($station,$network,$channel,$location) = $arg =~ /^([^.]+)\.([^.]+)\.([^.]+)\.([^.]*)$/; 
    $location = "-" if ($location eq "");
    my(@stations) = split(/,/,$station);
    my(@networks) = split(/,/,$network);
    my(@channels) = split(/,/,$channel);
    my(@locations) = split(/,/,$location);
    my(@argv,$chan,$pattern);
    foreach $station (@stations) {
	foreach $network (@networks) {
	    foreach $channel (@channels) {
		foreach $location (@locations) {
		    $location =~ s/-//g;
		    $chan = "$station.$network.$channel.$location";
		    print ("Expanded channel: $chan\n") if ($opt_d);
		    push (@argv, $chan);
		}
	    }
	}
    }
    return (@argv);
}
