""" This code is responsible for the computation of the cross-correlation
functions.

This script will group *jobs* marked "T"odo in the database by day and process
them using the following scheme. As soon as one day is selected, the
corresponding jobs are marked "I"n Progress in the database. This allows
running several instances of this script in parallel.


Waveform Preprocessing
~~~~~~~~~~~~~~~~~~~~~~
Pairs are first split and a station list is created. The database is then
queried to get file paths. For each station, all files potentially containing
data for the day are opened. The traces are then merged and splitted, to obtain
the most continuous chunks possible. The different chunks are then demeaned,
tapered and merged again to a 1-day long trace. If shorter than 1-day, the
trace is padded with zeros. If longer, it is cut to match the start/end of the
day.

Each 1-day long trace is then low-passed (at ``preprocess_lowpass`` Hz),
high-passed (at ``preprocess_highpass`` Hz), then decimated/downsampled.
Decimation/Downsampling are configurable (``resampling_method``) and users are
advised testing both. One advantage of Downsampling over Decimation is that
it is able to downsample the data by any factor, not only integer factors.

.. warning::
    For an unknown reason, the PAZ-correction has disappeard from the
    current sqlvolution on GitHub: CHECK!


Processing
~~~~~~~~~~

Once all traces are preprocessed, station pairs are processed sequentially.
If a component different from *ZZ* is to be computed, the traces are first
rotated. This supposes the user has provided the station coordinates in the
*station* table. The rotation is computed for Radial and Transverse components:

.. code-block:: python

    R = tramef_N * np.cos(cplAz * np.pi / 180.) + tramef_E * np.sin(cplAz * np.pi / 180.)
    T = tramef_N * np.sin(cplAz * np.pi / 180.) - tramef_E * np.cos(cplAz * np.pi / 180.)

Then, for each ``corr_duration`` window in the signal, and for each filter
configured in the database, the traces are clipped to ``windsorizing`` times
the RMS and then whitened (see :ref:`whiten`) between the frequency bounds.
When both traces are ready, the cross-correlation function is computed
(see :ref:`mycorr`). The function returned contains data for time lags 
corresponding to ``maxlag`` in the acausal (negative lags) and causal
(positive lags) parts.

If configured (setting ``keep_all`` to 'Y'), each ``corr_duration`` CCF is
saved to the hard disk. By default, the ``keep_days`` setting is set to True
and so "N = 1 day / corr_duration" CCF are stacked and saved to the hard disk
in the STACKS/001_DAYS folder.

Once done, each job is marked "D"one in the database.

To run this script:

.. code-block:: sh

    $ msnoise compute_cc
"""

import time
import calendar
import sys
### adds
import logging
### adde
### adds
import argparse
import scipy.fftpack
from obspy.signal import rotate_NE_RT
from obspy.signal import rotate_RT_NE
from obspy.signal.rotate import rotate2ZNE
from obspy.xseed import Parser
from obspy import UTCDateTime
### adde

from obspy.core import utcdatetime
from scikits.samplerate import resample

from api import *
from myCorr import myCorr
from whiten import whiten


#def preprocess(db, stations, comps, goal_day, params, tramef_Z, tramef_E = np.array([]), tramef_N = np.array([])):
def preprocess(db, stations, comps, goal_day, params, metadataOUT, tramef_Z, tramef_E = np.array([]), tramef_N = np.array([])):
    datafilesZ = {}
    datafilesE = {}
    datafilesN = {}
    ### adds
    datafiles3 = {}
    datafiles2 = {}
    ### addpk
    datafiles1 = {}
    ### adde

    for station in stations:
        datafilesZ[station] = []
        datafilesE[station] = []
        datafilesN[station] = []
        ### adds
        datafiles3[station] = []
        datafiles2[station] = []
        ## pkadd
        datafiles1[station] = []
        ### adde
        net, sta = station.split('.')
        gd = datetime.datetime.strptime(goal_day, '%Y-%m-%d')
        files = get_data_availability(
            db, net=net, sta=sta, starttime=gd, endtime=gd)
        for file in files:
            comp = file.comp
            fullpath = os.path.join(file.path, file.file)
            if comp[-1] == 'Z':
                datafilesZ[station].append(fullpath)
            elif comp[-1] == 'E':
                datafilesE[station].append(fullpath)
            elif comp[-1] == 'N':
                datafilesN[station].append(fullpath)
            elif comp[-1] == '3':
                datafiles3[station].append(fullpath)
            elif comp[-1] == '2':
                datafiles2[station].append(fullpath)
            ### adds_pk
            elif comp[-1] == '1':
                datafiles1[station].append(fullpath)
            ### adde_pk
                
    j = 0
    for istation, station in enumerate(stations):
        ### adds
        net, sta = station.split('.')
        datalessOUT = {}
        loc = '00' # def for C1
        if net == "BP":
            loc = '40'
        
        if net == "SF":
            loc = '01'
    
        loc_pre = '' # org loc 

        #loc_pre = ''
        net, sta = station.split('.')
        logging.debug("%s " %
                      (station))
        logging.debug("%s %s %s %s" %
                      (net,sta,loc,loc_pre))

        #loc = '00' # def for BP
        if net == "NC":
            loc = ''
            loc_pre = ''

        if net == "BK":
            loc = '00'
            loc_pre = ''

        if net == "PB":
            loc = ''
            loc_pre = ''

        if sta == "PH008":
            loc_pre = '01'
            
        if sta == "LME":
            loc = '02'
            loc_pre = ''

        if sta == "LGB":
            loc = '02'
            loc_pre = ''

        if sta == "LHE":
            loc_pre = '02'

        if sta == "LSH":
            loc_pre = ''
            loc = '03'

        
        dataless = None
        if net != "EN":
            #if net == "BP":
            #    dataless = Parser("/data/seis01/taira/msnoise-1.3.1/msnoise_work.pk/"+net+".dataless.seed")
            #    
            #elif net == "C1":
            #    dataless = Parser("/data/seis01/taira/msnoise-1.3.1/msnoise_work.c1/"+net+".dataless")
            dataless = Parser("/data/seis01/taira/msnoise-1.3.1/dataless/"+net+".dataless")
        
        #year, month, day, hourf, minf, secf, wday, yday, isdst = trace.stats.starttime.utctimetuple()
        # we could use trace.stats.starttime... instead of goal_day
        goal_day2 = goal_day+"T23:59:59"
        time_out = UTCDateTime(goal_day2) # if reseponse change on this day, then will be an issue, but we ignore this case this moment
        logging.debug("%s %s" %
                      (goal_day2,time_out))

        goal_day3 = goal_day+"T00:00:00"
        time_out3 = UTCDateTime(goal_day3) # if reseponse change on this day, then will be an issue, but we ignore this case this moment
        logging.debug("%s %s" %
                      (goal_day3,time_out3))

        decimation_factorOUT = params.decimation_factor
        resampling_methodOUT = params.resampling_method
        if net == "BG":
            decimation_factorOUT = 25

        if sta == "MOBB": # BH is 20 sps
            decimation_factorOUT = 1

        if sta == "PH008": # HH is 200 sps
            decimation_factorOUT = 10

        if sta == "PKD": # 2009 data HH is 80 sps
            decimation_factorOUT = 4

        #if station == "BK.VALB":
        #    decimation_factorOUT = 1
        #    
        #if station == "BK.PETB":
        #    decimation_factorOUT = 1
        #        
        #if station == "BK.CMAB":
        #    decimation_factorOUT = 1
        #
        ### adde
        for comp in comps:
            ### adde
            compOUT = comp
            # it should be ZNE. PB is Z12
            # it should be ZNE. SF is 123
            if comp == 'Z':
                compOUT = 'Z'
                #compOUT = '1'
                
            if comp == 'N':
                compOUT = '1'
                #compOUT = '2'
                
            if comp == 'E':
                compOUT = '2'
                #compOUT = '3'
    
            
            compD = comp # compD D is default
            logging.debug("comp = %s compOUT = %s compD = %s "%
                          (comp,compOUT,compD))
             #trace_id = net + "." + sta + "." + loc + "." comp
            if net == "BP":
                cout = "%s%s" % ('SP', comp)# def for BP
                cout_pre = "%s%s" % ('BP', comp)# old naming
                
            elif net == "NC":
                if sta == "LDH" or sta == "MCB" or sta == "MDPB" or sta == "MDY" or sta == "MINS" or sta == "MLI" or sta == "MMLB" or sta == "PAGB" or sta == "GDXB":
                    cout = "%s%s" % ('HH', comp)# def for BP
                    cout_pre = "%s%s" % ('HH', comp)# old naming

                else:
                    cout = "%s%s" % ('EH', comp)# def for BP
                    ### lp case black with EH 
                    ###cout_pre = "%s%s" % ('EH', comp)# old naming
                    ### 10.10.15 version for pre2012 data?
                    ### pre2012 LP
                    cout_pre = "%s%s" % ('SH', comp)# old naming

            elif net == "PB":
                # use compOUT becuase PB are processed with TS data
                cout = "%s%s" % ('EH', compOUT)# def for BP
                cout_pre = "%s%s" % ('EH', compOUT)# old naming
                compD = compOUT

            elif net == "SF":
                # use compOUT becuase PB are processed with TS data
                cout = "%s%s" % ('HH', compOUT)# def for BP
                cout_pre = "%s%s" % ('HH', compOUT)# old naming
                compD = compOUT

            elif net == "BG":
                cout = "%s%s" % ('DP', comp)# def for BP
                cout_pre = "%s%s" % ('DP', comp)# old naming
                
            elif net == "NN":
                cout = "%s%s" % ('HH', comp)# def for BP
                cout_pre = "%s%s" % ('HH', comp)# old naming
            
            elif net == "CL":
                cout = "%s%s" % ('HH', comp)# def for BP
                cout_pre = "%s%s" % ('HH', comp)# old naming

            elif net == "Z0":
                cout = "%s%s" % ('HH', comp)# def for BP
                cout_pre = "%s%s" % ('HH', comp)# old naming

            elif net == "UW":
                cout = "%s%s" % ('EH', comp)# def for BP
                cout_pre = "%s%s" % ('EH', comp)# old naming

            elif net == "BK":               
                #if sta == "WDC" or sta == "MOD" or sta == "PKD": # ss or shasta lake project
                if sta == "WDC" or sta == "MOD": # ss or shasta lake project and PKD -> BH
                    cout = "%s%s" % ('HH', comp)# def for BP
                    cout_pre = "%s%s" % ('HH', comp)# old naming

                else:
                    cout = "%s%s" % ('BH', comp)# 
                    cout_pre = "%s%s" % ('BH', comp)# 

            else:
                cout = "%s%s" % ('BH', comp)# def for BP
                cout_pre = "%s%s" % ('BH', comp)# old naming
                
            
            trace_id = "%s.%s.%s.%s" % (net, sta,
                                        loc, cout)
            
            trace_id_pre = "%s.%s.%s.%s" % (net, sta,
                                            loc_pre, cout_pre)
              
            logging.debug("%s " %(trace_id))            
            logging.debug("%s " % (trace_id_pre))

            logging.debug("comp = %s compOUT = %s compD = %s "%
                          (comp,compOUT,compD))

            datalessOUT['id'] = trace_id
            datalessOUT['id_pre'] = trace_id_pre
            saz = "%s%s" % ('az', comp)
            sdip = "%s%s" % ('dip', comp)
            datalessOUT[saz] = 0 # z                        
            datalessOUT[sdip] = -90 # z
           
            ### three zeor disp
            pazCONV = { 
                'poles': [  -14.1372 + 24.4863j,  -14.1372 + -24.4863j ],  
                'zeros': [  0 + 0j, 0 + 0j, 0 + 0j ],  
                'gain': 1,  
                'sensitivity': 4.008143e+08} 
            pazOUT = { 
                'poles': [  0 + 0j, -0 + 0j ],  
                'zeros': [  0 + 0j, 0 + 0j ],  
                'gain': 1,  
                'sensitivity': 1} 
        
            ### adds
            #files = eval("datafiles%s['%s']" % (comp, station))
            files = eval("datafiles%s['%s']" % (compD, station))
            ### adde
            ### orgs
            #logging.debug("check %s.%s Reading %i Files" %
            #              (station, comp, len(files)))
            ### orge
            ### adds
            logging.debug("check %s.%s Reading %i Files" %
                          (station, compD, len(files)))
            ### adde
            if len(files) != 0:
                ### adds
                # we now get pazOUT from dataless            
                #pazOUT = dataless.getPAZ(trace_id_pre, time_out3)
                #pazOUT = dataless.getPAZ(trace_id_pre, time_out)            
                #logging.debug("start with %s " %(pazOUT))                       
                if net != "EN":
                    time_outOUT = time_out
                    trace_idOUT = trace_id
                    try:
                        logging.debug("start with %s " %(trace_id))   
                        pazOUT = dataless.getPAZ(trace_id, time_out)
                        trace_idOUT = trace_id
                        time_outOUT = time_out
                        
                    except:
                        
                        try:
                            logging.debug("can not find with trace_id then try with %s " %(trace_id_pre))   
                            pazOUT = dataless.getPAZ(trace_id_pre, time_out)
                            trace_idOUT = trace_id_pre
                            time_outOUT = time_out                                            
                            
                        except:
                            
                            try:
                                logging.debug("time out3 start with %s " %(trace_id))   
                                pazOUT = dataless.getPAZ(trace_id, time_out3)
                                trace_idOUT = trace_id
                                time_outOUT = time_out3                                            
                                
                            except:
                                
                                logging.debug("time out3 can not find with trace_id then try with %s " %(trace_id_pre))
                                decimation_factorOUT = 5 # 50 Hz -> 10 Hz pre2012 LP
                                if sta == "PKD": # 2009 data HH is 80 sps
                                    decimation_factorOUT = 1 # 20 Hz -> 20 Hz pre200? BK PKD for parkfield project
                                
                                pazOUT = dataless.getPAZ(trace_id_pre, time_out3)
                                trace_idOUT = trace_id_pre
                                time_outOUT = time_out3                                            
                
                                
                    logging.debug("trace_idOUT = %s " %(trace_idOUT))                               
                    logging.debug("time_outOUT = %s " %(time_outOUT))

                    datalessOUT['idOUT'] = trace_idOUT
                    coord = dataless.getCoordinates(trace_idOUT, time_outOUT)
                    datalessOUT['coord'] = coord
                    
                    blockettes = dataless._select(trace_idOUT, time_outOUT)
                
                    #sys.exit()


                    for blkt in blockettes:
                        if blkt.id == 52:
                            #print blkt.latitude            
                            datalessOUT[saz] = blkt.azimuth
                            datalessOUT[sdip] = blkt.dip                        
                            break


            
 
                logging.debug("%s.%s pazOUT" %
                              (station, comp))
                logging.debug("%s " %
                              (pazOUT))
                logging.debug("%s " %
                              (datalessOUT))
                metadataOUT[istation] = datalessOUT
                ### adde

                logging.debug("%s.%s Reading %i Files" %
                              (station, comp, len(files)))
                stream = Stream()
                ### adds
                #logging.debug("%s " % 
                #              (stream[0])  )
                ### adde
                for file in sorted(files):
                    st = read(file, dytpe=np.float)                    
                    for tr in st:
                        tr.data = tr.data.astype(np.float)
                    stream += st
                    del st

                logging.debug("Checking sample alignment")
                for i, trace in enumerate(stream):
                    stream[i] = check_and_phase_shift(trace)

                stream.sort()
                logging.debug("Checking Gaps")
                if len(getGaps(stream)) > 0:
                    max_gap = 10
                    only_too_long=False
                    while getGaps(stream) and not only_too_long:
                        too_long = 0
                        gaps = getGaps(stream)
                        for gap in gaps:
                            if int(gap[-1]) <= max_gap:
                                stream[gap[0]] = stream[gap[0]].__add__(stream[gap[1]], method=0, fill_value="interpolate")
                                stream.remove(stream[gap[1]])
                                break
                            else:
                                too_long += 1
                        if too_long == len(gaps):
                            only_too_long = True

                taper_length = 20.0 #seconds
                for trace in stream:
                    if trace.stats.npts < 4 * taper_length*trace.stats.sampling_rate:
                        trace.data = np.zeros(trace.stats.npts)
                    else:
                        trace.detrend(type="demean")
                        trace.detrend(type="linear")
                        taper_1s = taper_length * float(trace.stats.sampling_rate) / trace.stats.npts
                        cp = cosTaper(trace.stats.npts, taper_1s)
                        trace.data *= cp
                stream.merge(method=0, fill_value=0.0)

                logging.debug("%s.%s Slicing Stream to %s:%s" % (station, comp, utcdatetime.UTCDateTime(
                    goal_day.replace('-', '')), utcdatetime.UTCDateTime(goal_day.replace('-', '')) + params.goal_duration - stream[0].stats.delta))
                stream[0].trim(utcdatetime.UTCDateTime(goal_day.replace('-', '')), utcdatetime.UTCDateTime(
                    goal_day.replace('-', '')) + params.goal_duration - stream[0].stats.delta, pad=True, fill_value=0.0,
                    nearest_sample=False)
                trace = stream[0]
                
                logging.debug(
                    "%s.%s Highpass at %.2f Hz" % (station, comp, params.preprocess_highpass))
                trace.filter("highpass", freq=params.preprocess_highpass, zerophase=True)
                
                ### orgs : orgs does not do lowpass if trace.stats.sampling_rate == params.goal_sampling_rate:
                ### add will alwayws do lowpass. lowass is before trace.stats.sampling_rate != params.goal_sampling_rate:
                #if trace.stats.sampling_rate != params.goal_sampling_rate:
                #    logging.debug(
                #        "%s.%s Lowpass at %.2f Hz" % (station, comp, params.preprocess_lowpass))
                #    trace.filter("lowpass", freq=params.preprocess_lowpass, zerophase=True)
                #
                ### orge
                ### adds
                logging.debug(
                    "%s.%s Lowpass at %.2f Hz" % (station, comp, params.preprocess_lowpass))
                trace.filter("lowpass", freq=params.preprocess_lowpass, zerophase=True)                    
                ### adds
                tr_sample_rate = trace.stats.sampling_rate;
                logging.debug("# tr_sample_rate = %s params.goal_sampling_rate = %s" %
                              (tr_sample_rate,params.goal_sampling_rate))
                decimation_factorOUT = int(tr_sample_rate / params.goal_sampling_rate);
                decimation_factor_frac = tr_sample_rate % params.goal_sampling_rate;
                if decimation_factor_frac == 0:
                    resampling_methodOUT = "Decimate"

                else:
                    resampling_methodOUT = "Resample"
                    
                logging.debug("# decimation_factorOUT  = %s decimation_factor_frac = %s resampling_methodOUT = %s" %
                              (decimation_factorOUT, decimation_factor_frac, resampling_methodOUT))

                if trace.stats.sampling_rate != params.goal_sampling_rate:
                #adde
                    

                
                    ### orgs
                    #if params.resampling_method == "Resample":
                    ### orge
                    ### adds
                    if resampling_methodOUT == "Resample":
                    ### adde

                        logging.debug("%s.%s Downsample to %.1f Hz" %
                                      (station, comp, params.goal_sampling_rate))
                        trace.data = resample(
                            trace.data, params.goal_sampling_rate / trace.stats.sampling_rate, 'sinc_fastest')

                    ### orgs
                    #elif params.resampling_method == "Decimate":
                    ### orge 
                    ### adds
                    elif resampling_methodOUT == "Decimate":
                    ### adde
                        ### orgs
                        #logging.debug("%s.%s Decimate by a factor of %i" %
                        #              (station, comp, params.decimation_factor))
                        #trace.data = trace.data[::params.decimation_factor]
                        ### orge
                        ### adde
                        logging.debug("%s.%s Decimate by a factor of %i" %
                                      (station, comp, decimation_factorOUT))
                        trace.data = trace.data[::decimation_factorOUT]
                        ### adds
                    trace.stats.sampling_rate = params.goal_sampling_rate
                    
                year, month, day, hourf, minf, secf, wday, yday, isdst = trace.stats.starttime.utctimetuple()
                ### adds
                ### remove resp
                mytrace = Trace(data=trace.data)
                mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                st = Stream(traces=[mytrace, ])
                st.detrend(type="demean")
                st.detrend(type="linear")
                # st.write('deci.sac', format='SAC')
                if args.convolv == 1:               
                    logging.debug("CONV")
                    st.simulate(paz_remove=pazOUT, paz_simulate=pazCONV)                     
                
                if args.convolv == 0:
                    logging.debug("NOCONV")       
                    st.simulate(paz_remove=pazOUT, paz_simulate=None)                    
                
                if args.convolv == 2:
                    logging.debug("NO REMOVE RESP")       
                    #st.simulate(paz_remove=pazOUT, paz_simulate=None)                    
                
                trace = st[0]                
                logging.debug(
                    "%s.%s Highpass after IR remove at %.2f Hz" % (station, comp, params.preprocess_highpass))
                trace.filter("highpass", freq=params.preprocess_highpass, zerophase=True)
                logging.debug(
                    "%s.%s Lowpass  after IR moreve at %.2f Hz" % (station, comp, params.preprocess_lowpass))
                trace.filter("lowpass", freq=params.preprocess_lowpass, zerophase=True)          
                ### adde
                                
                if j == 0:
                    t = time.strptime("%04i:%02i:%02i:%02i:%02i:%02i" %
                                      (year, month, day, hourf, minf, secf), "%Y:%m:%d:%H:%M:%S")
                    basetime = calendar.timegm(t)

                if len(trace.data) % 2 != 0:
                    trace.data = np.append(trace.data, 0.)

                ### adds
                logging.debug(
                    "# len_tramef_Z = %s le_traced.data = %s" % (len(tramef_Z[istation]),len(trace.data)))
                # xlen should be the same as xlen2
                xlen2 = int(params.goal_duration * params.goal_sampling_rate)
                #new_trace=np.zeros( xlen2 )
                #new_trace=trace.data
                miss_dp = xlen2 - len(trace.data)
                logging.debug(
                    "# miss_dp = %d" % (miss_dp))
                if miss_dp != 0:
                    logging.debug("# APPEND %s.%s Slicing Stream to %s:%s" % (station, comp, utcdatetime.UTCDateTime(goal_day.replace('-', '')), utcdatetime.UTCDateTime(goal_day.replace('-', '')) + params.goal_duration - stream[0].stats.delta))

                for idp in range(miss_dp):
                    trace.data = np.append(trace.data, 0.)
                
                logging.debug(
                    "# len_tramef_Z = %s le_traced.data = %s" % (len(tramef_Z[istation]),len(trace.data)))
                #sys.exit()
                ### adde
                if comp == "Z":
                    tramef_Z[istation] = trace.data
                ### orgs we may need it?   
                elif comp == "E":
                    tramef_E[istation] = trace.data
                elif comp == "N":
                    tramef_N[istation] = trace.data
                ### orge
                ### adds we will have an issue for Z12 system for PBO
                elif comp == "3":
                    tramef_E[istation] = trace.data
                elif comp == "2":
                    tramef_N[istation] = trace.data
                ### addpk
                elif comp == "1":
                    tramef_Z[istation] = trace.data
                ### addpk
                ### adde

                del trace, stream
    if len(tramef_E) != 0:
        ### orgs
        #return basetime, tramef_Z, tramef_E, tramef_N
        ### orge
        ### adds
        return basetime, tramef_Z, tramef_E, tramef_N, metadataOUT    
        ### adde
    else:
        ### orgs
        #return basetime, tramef_Z
        ### orge
        ### adds
        return basetime, tramef_Z, metadataOUT
        ### adde

class Params():
    pass

### orgs
#def main():
### orge
### adds
def main(convolv=0, azf=0.0, listch = 0, pairch = -1, stalist = 0, d1 = '1900-01-01', d2 = '2500-01-01'):
### adde
    ### adds
    #logging.basicConfig(level=logging.INFO,
    #                    format='%(asctime)s [%(levelname)s] %(message)s',
    #                    datefmt='%Y-%m-%d %H:%M:%S')
    ### adde
    ### adds
    logging.basicConfig(level=logging.DEBUG,
                        format='%(asctime)s [%(levelname)s] %(message)s',
                        datefmt='%Y-%m-%d %H:%M:%S')
    logging.info('Starting this program with: convolv=%i, azf=%f, listch=%i (0: zen 1: z23 2: 123), pairch=%i (0: all 1: zne 2 zrt -1: what we define in db), stalist=%i (0: all 1: select), d1=%s d2=%s'
                 % (args.convolv, args.azf, args.listch, args.pairch, args.stalist, args.d1, args.d2))       
    ### adde

    logging.info('*** Starting: Compute CC ***')

    # Connection to the DB
    db = connect()

    if len(get_filters(db, all=False)) == 0:
        logging.info("NO FILTERS DEFINED, exiting")
        sys.exit()

    # Get Configuration
    params = Params()
    params.goal_sampling_rate = float(get_config(db, "cc_sampling_rate"))
    params.goal_duration = float(get_config(db, "analysis_duration"))
    params.overlap = float(get_config(db, "overlap"))
    params.maxlag = float(get_config(db, "maxlag"))
    params.min30 = float(get_config(db, "corr_duration")) * params.goal_sampling_rate
    params.windsorizing = float(get_config(db, "windsorizing"))
    params.resampling_method = get_config(db, "resampling_method")
    params.decimation_factor = int(get_config(db, "decimation_factor"))
    params.preprocess_lowpass = float(get_config(db, "preprocess_lowpass"))
    params.preprocess_highpass = float(get_config(db, "preprocess_highpass"))
    params.keep_all = get_config(db, 'keep_all', isbool=True)
    params.keep_days = get_config(db, 'keep_days', isbool=True)
    ### orgs
    #params.components_to_compute = get_components_to_compute(db)
    ### orge

    ### adds
    
        
    ### add 3-5 opt 08.18.16
    if args.pairch == 0:
        params.components_to_compute = get_components_to_compute_all(db)
        components_given = "AL"
        
    elif args.pairch == 1:
        params.components_to_compute = get_components_to_compute_ZNE(db)
        components_given = "ZNE"
    
    elif args.pairch == 2:
        params.components_to_compute = get_components_to_compute_ZRT(db)
        components_given = "ZRT"
        
    elif args.pairch == 3: # 2 old
        params.components_to_compute = get_components_to_compute_SC(db)
        components_given = "SC"

    elif args.pairch == 4: # 3 old
        params.components_to_compute = get_components_to_compute_AC(db)
        components_given = "AC"

    elif args.pairch == 5: # 4 old
        params.components_to_compute = get_components_to_compute_ACSC(db)
        components_given = "ACSC"

    else:
        params.components_to_compute = get_components_to_compute(db)
        components_given = "DB"


    ### adde
    logging.info("Will compute %s" % " ".join(params.components_to_compute))
    #sys.exit()
    ### orgs
    # while is_next_job(db, jobtype='CC'):
    #    jobs = get_next_job(db, jobtype='CC')
    ### orge
    ### adds
    ### pairIN should have only one pair
    for station1, station2 in get_station_pairs(db, used=True):
        sta1 = "%s.%s" % (station1.net, station1.sta)
        sta2 = "%s.%s" % (station2.net, station2.sta)
        pairIn = "%s:%s" % (sta1, sta2)
        #pairIn = pair


    if args.stalist == 0:
        pairIn = None
    
    logging.info(
        "only conmpute this pair %s (None means all)" % pairIn)    
    #date1="1900-01-01"
    #date2="2025-01-01"
    date1 = args.d1
    date2 = args.d2

    #date1="2004-09-01"
    #date2="2004-09-02"
    #data1 = "2008-10-01"
    #data2 = "2008-10-02"

    logging.info(
        "date1 = %s date2 = %s" % (date1,date2))    
    # def
    # test for all sta all stalist = 0
    logging.info(
        "staslist %i" % args.stalist)    
    while is_next_job_pair(db, date1, date2, pair=pairIn, jobtype='CC'):
        jobs = get_next_job_pair(db, date1, date2, pair=pairIn, jobtype='CC')
    #while is_next_job(db, jobtype='CC'):
    #    jobs = get_next_job(db, jobtype='CC')       
    ### adde
        stations = []
        pairs = []
        refs = []

        for job in jobs:
            refs.append(job.ref)
            pairs.append(job.pair)
            netsta1, netsta2 = job.pair.split(':')
            stations.append(netsta1)
            stations.append(netsta2)
            goal_day = job.day

        stations = np.unique(stations)

        logging.info("New CC Job: %s (%i pairs with %i stations)" %
                     (goal_day, len(pairs), len(stations)))
        jt = time.time()

        xlen = int(params.goal_duration * params.goal_sampling_rate)

        ### orgs
        #if ''.join(params.components_to_compute).count('R') > 0 or ''.join(params.components_to_compute).count('T') > 0:
        ### orge
        ### adds
        ### enadd
        #if ''.join(params.components_to_compute).count('R') > 0 or ''.join(params.components_to_compute).count('T') > 0 or ''.join(params.components_to_compute).count('E') > 0 or ''.join(params.components_to_compute).count('N') > 0:
        ### pkadd
        if ''.join(params.components_to_compute).count('R') > 0 or ''.join(params.components_to_compute).count('T') > 0 or ''.join(params.components_to_compute).count('E') > 0 or ''.join(params.components_to_compute).count('N') > 0 or ''.join(params.components_to_compute).count('Z') > 0:
        ### adde

            ###orgs
            #comps = ['Z', 'E', 'N']
            ###orge
            ###adds
            logging.debug("listhch=%i"% (args.listch))
            if args.listch == 0:
                comps = ['Z', 'E', 'N']
            
            ### EN set
            elif args.listch == 1:
                comps = ['Z', '3', '2']

            ### PK set
            elif args.listch == 2:
                #logging.debug("pktest!")
                comps = ['1', '3', '2']
            
            else:
                comps = ['Z', 'E', 'N']

            
            
            ###adde
            tramef_Z = np.zeros((len(stations), xlen))
            tramef_E = np.zeros((len(stations), xlen))
            tramef_N = np.zeros((len(stations), xlen))
            metadataOUT = {}
            ### orgs
            #basetime, tramef_Z, tramef_E, tramef_N = preprocess(db, stations, comps, goal_day, params, tramef_Z, tramef_E, tramef_N)
            ### orge
            ### adds
            basetime, tramef_Z, tramef_E, tramef_N, metadataOUT = preprocess(db, stations, comps, goal_day, params, metadataOUT, tramef_Z, tramef_E, tramef_N)
            ### addse

        else:
            ### orgs
            comps = ['Z']
            ### orge
            ### adds
            # en set; same as the original
            #comps = ['Z']
            # pk set
            comps = ['1']
            metadataOUT = {}
            ### adde
            tramef_Z = np.zeros((len(stations), xlen))
            ### orgs
            #basetime, tramef_Z = preprocess(db, stations, comps, goal_day, params, tramef_Z)
            ### orge
            ### adds
            basetime, tramef_Z, metadataOUT = preprocess(db, stations, comps, goal_day, params, metadataOUT, tramef_Z)            
            ### addse 


        # print '##### STREAMS ARE ALL PREPARED AT goal Hz #####'
        dt = 1. / params.goal_sampling_rate
        # Calculate the number of slices

        slices = int(params.goal_duration * params.goal_sampling_rate / params.min30)
        begins = []
        ends = []
        i = 0
        while i <=  (params.goal_duration - params.min30/params.goal_sampling_rate):
            begins.append(int(i * params.goal_sampling_rate))
            ends.append(int(i * params.goal_sampling_rate + params.min30))
            i += int(params.min30/params.goal_sampling_rate * (1.0-params.overlap))
        slices = len(begins)

        #
        # Computing only ZZ components ? Then we can be much faster:
        #

        #if False:
        ### orgs
        #if len(params.components_to_compute) == 1 and params.components_to_compute[0] == "ZZ":
        ### orge
        ### adds
        ### len is 999 so that we will not go to this "if"
        if len(params.components_to_compute) == 999 and params.components_to_compute[0] == "XX":
        ### adde
            Nfft = params.min30
            if params.min30 / 2 % 2 != 0:
                Nfft = params.min30 + 2
            cp = cosTaper(int(params.min30), 0.04)

            logging.info("Pre-Whitening Traces")
            whitened_slices = np.zeros((len(stations), len(get_filters(db, all=False)), slices, int(Nfft)), dtype=np.complex)
            for istation, station in enumerate(stations):
                for islice, (begin, end) in enumerate(zip(begins,ends)):
                    tmp = tramef_Z[istation, begin:end]
                    rmsmat = np.std(np.abs(tmp))
                    if params.windsorizing == -1:
                        tmp = np.sign(tmp)
                    elif params.windsorizing != 0:
                        indexes = np.where(
                            np.abs(tmp) > (params.windsorizing * rmsmat))[0]
                        tmp[indexes] = (tmp[indexes] / np.abs(
                            tmp[indexes])) * params.windsorizing * rmsmat
                    tmp *= cp
                    for ifilter, filter in enumerate(get_filters(db, all=False)):
                        whitened_slices[istation, ifilter, islice,:] = whiten(tmp, Nfft, dt, float(filter.low), float(filter.high), plot=False)
                    del tmp
            del tramef_Z
            logging.info("Processing CC")
            for ifilter, filter in enumerate(get_filters(db, all=False)):
                for pair in pairs:
                    orig_pair = pair
                    if params.keep_all:
                        allcorr = {}
                    if params.keep_days:
                        daycorr = np.zeros(get_maxlag_samples(db,))
                        ndaycorr = 0
                    station1, station2 = pair.split(':')
                    pair = (np.where(stations == station1)
                            [0][0], np.where(stations == station2)[0][0])
                    for islice in range(slices):
                        tmp = np.vstack((whitened_slices[pair[0], ifilter, islice],
                                         whitened_slices[pair[1], ifilter, islice]))
                        corr = myCorr(tmp, np.ceil(params.maxlag / dt), plot=False)
                        tmptime = time.gmtime(basetime + begins[islice] /
                                                  params.goal_sampling_rate)
                        thisdate = time.strftime("%Y-%m-%d", tmptime)
                        thistime = time.strftime("%Y-%m-%d %H:%M:%S",
                                                 tmptime)
                        if not np.any(np.isnan(corr)) and not np.any(np.isinf(corr)):
                            if params.keep_all:
                                ccfid = "%s_%s_%s_%s_%s" % (station1, station2,
                                                            filter.ref, 'ZZ',
                                                            thisdate)
                                if ccfid not in allcorr:
                                    allcorr[ccfid] = {}
                                allcorr[ccfid][thistime] = corr

                            if params.keep_days:
                                daycorr += corr
                                ndaycorr += 1

                    if params.keep_all:
                        for ccfid in allcorr.keys():
                            export_allcorr(db, ccfid, allcorr[ccfid])

                    if params.keep_days:
                        thisdate = time.strftime(
                            "%Y-%m-%d", time.gmtime(basetime))
                        thistime = time.strftime(
                            "%H_%M", time.gmtime(basetime))
                        add_corr(
                            db, station1.replace(
                                '.', '_'), station2.replace('.', '_'), filter.ref,
                            thisdate, thistime, params.min30 / params.goal_sampling_rate, 'ZZ', daycorr, params.goal_sampling_rate, day=True, ncorr=ndaycorr)
                    update_job(db, goal_day, orig_pair, 'CC', 'D')
            logging.info("Job Finished. It took %.2f seconds" % (time.time() - jt))

        else:
            # ITERATING OVER PAIRS #####
            for pair in pairs:
                orig_pair = pair

                logging.info('Processing pair: %s' % pair.replace(':', ' vs '))
                tt = time.time()
                station1, station2 = pair.split(':')
                pair = (np.where(stations == station1)
                        [0][0], np.where(stations == station2)[0][0])

                ### adds
                logging.info('pair[0]: %d' % (pair[0]))
                logging.info('pair[1]: %d' % (pair[1]))
                m1 = metadataOUT[pair[0]]
                m2 = metadataOUT[pair[1]]
                logging.info('m1 metadatra pair[0]: %s' % (m1))
                logging.info('m2 metadatra pair[1]: %s' % (m2))
                ### adde
                s1 = get_station(db, station1.split('.')[0], station1.split('.')[1])
                s2 = get_station(db, station2.split('.')[0], station2.split('.')[1])

                if s1.X:
                    X0 = s1.X
                    Y0 = s1.Y
                    c0 = s1.coordinates

                    X1 = s2.X
                    Y1 = s2.Y
                    c1 = s2.coordinates

                    if c0 == c1:
                        coordinates = c0
                    else:
                        coordinates = 'MIX'

                    ### adds
                    logging.debug("X0 = %f Y0 = %f c0 = %s" % (X0,Y0,c0))
                    ### adde
                    cplAz = np.deg2rad(azimuth(coordinates, X0, Y0, X1, Y1))
                    logging.debug("Azimuth=%.1f"%np.rad2deg(cplAz))
                else:
                    # logging.debug('No Coordinates found! Skipping azimuth calculation!')
                    cplAz = 0.
                
                ### adds
                ##
                ## coordinate will be ginve here except for EN
                # x is lon y is lat c0 would be 'DEG'
                net1, sta1 = station1.split('.')
                net2, sta2 = station2.split('.')
                #net1 = "EN"
                #net2 = "EN"

                logging.debug("net1 = %s net2 = %s" % (net1, net2))
                #sys.exit()
                if net1 != "EN" and net2 != "EN":
                    X0 = m1['coord']['longitude']
                    Y0 = m1['coord']['latitude']
                    c0 = 'DEG'
                    X1 = m2['coord']['longitude']
                    Y1 = m2['coord']['latitude']
                    c1 = 'DEG'
                    logging.debug("X0 = %f Y0 = %f c0 = %s" % (X0,Y0,c0))
                    logging.debug("X1 = %f Y1 = %f c1 = %s" % (X1,Y1,c1))
                    coordinates = c0
                    cplAz = np.deg2rad(azimuth(coordinates, X0, Y0, X1, Y1))
                    logging.debug("Azimuth=%.1f"%np.rad2deg(cplAz))
                    
                ## coordinate will be ginve here except for EN
                # cplAz is from sta1 to sta2
                # i like to use sta1 is a source and sta2 is a receiver
                # then back azimuth for this case, baz = cplAz + 180 
                bAz = np.rad2deg(cplAz) + 180.0
                logging.debug("bAz (cplAz + 180) Azimuth=%.1f"%bAz) 
                if bAz < 0:
                    bAz = bAz + 360.0
                    
                if 360 < bAz :
                    bAz = bAz - 360.0

                logging.debug("bAz (cplAz + 180) Azimuth=%.1f"%bAz) 
                #args.azf = 47 # this is fault normal at PK-SAF. Doug's 2007 GRL 137 is strikem then we get 47 from 137 - 90 for fault normal
                logging.debug("azf: fault normal fixed Azimuth=%.1f"%args.azf)

                ch1Az = 0.0 # sta1 eh2
                ch1Az2 = 90.0 # sta1 eh3
                ch1AzZ = 0.0 # sta1 ehz
                
                ch1Dip = 0.0 # sta1 eh2
                ch1Dip2 = 0.0 # sta1 eh2
                ch1DipZ = -90.0 # sta1 eh2

                ch2Az = 0.0 # sta2 eh2 
                ch2Az2 = 90.0 # sta2 eh3
                ch2AzZ = 0.0 # sta1 ehz
                
                ch2Dip = 0.0 # sta1 eh2
                ch2Dip2 = 0.0 # sta1 eh2
                ch2DipZ = -90.0 # sta1 eh2

                if net1 != "EN" and net2 != "EN":
                    if args.listch == 0: #zne case
                        # ch2
                        ch1Az = m1['azN']
                        ch1Dip = m1['dipN']
                    # ch1
                        ch1AzZ = m1['azZ']
                        ch1DipZ = m1['dipZ']
                    # ch3
                        ch1Az2 = m1['azE']
                        ch1Dip2 = m1['dipE']
                
                    # ch2
                        ch2Az = m2['azN']
                        ch2Dip = m2['dipN']
                    # ch1
                        ch2AzZ = m2['azZ']
                        ch2DipZ = m2['dipZ']
                    # ch3
                        ch2Az2 = m2['azE']
                        ch2Dip2 = m2['dipE']
                        
                    if args.listch == 1: #z12 case en
                        # ch2
                        ch1Az = m1['az1']
                        ch1Dip = m1['dip1']
                    # ch1
                        ch1AzZ = m1['azZ']
                        ch1DipZ = m1['dipZ']
                    # ch3
                        ch1Az2 = m1['az2']
                        ch1Dip2 = m1['dip2']
                
                    # ch2
                        ch2Az = m2['az1']
                        ch2Dip = m2['dip1']
                    # ch1
                        ch2AzZ = m2['azZ']
                        ch2DipZ = m2['dipZ']
                    # ch3
                        ch2Az2 = m2['az2']
                        ch2Dip2 = m2['dip2']

                    if args.listch == 2: #pk case 123
                        # ch2
                        ch1Az = m1['az2']
                        ch1Dip = m1['dip2']
                    # ch1
                        ch1AzZ = m1['az1']
                        ch1DipZ = m1['dip1']
                    # ch3
                        ch1Az2 = m1['az3']
                        ch1Dip2 = m1['dip3']
                
                    # ch2
                        ch2Az = m2['az2']
                        ch2Dip = m2['dip2']
                    # ch1
                        ch2AzZ = m2['az1']
                        ch2DipZ = m2['dip1']
                    # ch3
                        ch2Az2 = m2['az3']
                        ch2Dip2 = m2['dip3']
                        
                
                if station1 == "EN.ELM":
                    ch1Az = 12.0 # eh2 az
                    ch1Az2 = 102.0
                
                if station1 == "EN.ENG":
                    ch1Az = 328.0 # eh2 az
                    ch1Az2 = 418.0 
 
                if station1 == "EN.HAT":
                    ch1Az = 151.0 # eh2 az
                    ch1Az2 = 241.0 
                
                if station1 == "EN.LIN":
                    ch1Az = 71.0 # eh2 az
                    ch1Az2 = 161.0
                
                if station1 == "EN.OBS":
                    ch1Az = 144.0 # eh2 az
                    ch1Az2 = 234.0
                    
                if station1 == "EN.RED":
                    ch1Az = 10.0 # eh2 az
                    ch1Az2 = 100.0

                if station1 == "EN.SIM":
                    ch1Az = 178.0 # # ch2
                    ch1Az2 = 88.0 # note this is eh3 az
                
                if station1 == "EN.YOU":
                    ch1Az = 132.0 # eh2 az
                    ch1Az2 = 222.0 
                
                if station2 == "EN.ELM":
                    ch2Az = 12.0 # eh2 az
                    ch2Az2 = 102.0
                
                if station2 == "EN.ENG":
                    ch2Az = 328.0 # eh2 az
                    ch2Az2 = 418.0 
 
                if station2 == "EN.HAT":
                    ch2Az = 151.0 # eh2 az
                    ch2Az2 = 241.0 
                
                if station2 == "EN.LIN":
                    ch2Az = 71.0 # eh2 az
                    ch2Az2 = 161.0
                
                if station2 == "EN.OBS":
                    ch2Az = 144.0 # eh2 az
                    ch2Az2 = 234.0
                    
                if station2 == "EN.RED":
                    ch2Az = 10.0 # eh2 az
                    ch2Az2 = 100.0

                if station2 == "EN.SIM":
                    ch2Az = 178.0 # # ch2
                    ch2Az2 = 88.0 # note this is eh3 az
                
                if station2 == "EN.YOU":
                    ch2Az = 132.0 # eh2 az
                    ch2Az2 = 222.0 
                
               
                

                logging.debug("ch1AzZ Azimuth=%.1f"%ch1AzZ)
                logging.debug("ch1DipZ Azimuth=%.1f"%ch1DipZ)

                logging.debug("ch1Az  Azimuth=%.1f"%ch1Az)
                logging.debug("ch1Dip  Azimuth=%.1f"%ch1Dip)
                
                logging.debug("ch1Az2 Azimuth=%.1f"%ch1Az2)
                logging.debug("ch1Dip2 Azimuth=%.1f"%ch1Dip2)


                logging.debug("ch2AzZ Azimuth=%.1f"%ch2AzZ)
                logging.debug("ch2DipZ Azimuth=%.1f"%ch2DipZ)

                logging.debug("ch2Az  Azimuth=%.1f"%ch2Az)
                logging.debug("ch2Dip  Azimuth=%.1f"%ch2Dip)

                logging.debug("ch2Az2 Azimuth=%.1f"%ch2Az2)
                logging.debug("ch2Dip2 Azimuth=%.1f"%ch2Dip2)

    


                # ch2 is tramef_n ch3 is tramef_e and ch1az is for ch2 azimuth
                # use obspy rotate_RT_NE to get n and e data
                # baz for rotate_RT_NE is bAz1 = ch2_az - 180. bAz1 has to be 0<=baz1<=360 
                bAz1 = ch1Az - 180
                if bAz1 < 0:
                    bAz1 = bAz1 + 360.0
                    
                if 360 < bAz1 :
                    bAz1 = bAz1 - 360.0
                    
                logging.debug("bAz1 (ch1Az - 180) Azimuth=%.1f"%bAz1) 

                logging.debug("ch2 Azimuth=%.1f"%ch2Az)        
                bAz2 = ch2Az - 180
                if bAz2 < 0:
                    bAz2 = bAz2 + 360.0
                    
                if 360 < bAz2 :
                    bAz2 = bAz2 - 360.0
                    
                logging.debug("bAz2 (ch2Az - 180) Azimuth=%.1f"%bAz2) 
                ### adde
                for components in params.components_to_compute:
                    
                    ### adds
                    #logging.debug("components %s " % (components)) 
                    ### adde
                    if components == "ZZ":
                        t1 = tramef_Z[pair[0]]
                        t2 = tramef_Z[pair[1]]
                    elif components[0] == "Z":
                        t1 = tramef_Z[pair[0]]
                        t2 = tramef_E[pair[1]]
                    elif components[1] == "Z":
                        t1 = tramef_E[pair[0]]
                        t2 = tramef_Z[pair[1]]
                    else:
                        t1 = tramef_E[pair[0]]
                        t2 = tramef_E[pair[1]]
                    if np.all(t1 == 0) or np.all(t2 == 0):
                        logging.debug("%s contains empty trace(s), skipping"%components)
                        continue
                    del t1, t2
                    
                    ### adds
                    ### tramef_N is ch2
                    ### tramef_E is ch3
                    if components [0] == "N" or components [0] == "E" or  components [0] == "R" or  components [0] == "T" or components [0] == "2" or  components [0] == "3" or components [0] == "Z" or components [0] == "1":
                        t1ztmp = tramef_Z[pair[0]]                                                
                        t1ntmp = tramef_N[pair[0]]
                        t1etmp = tramef_E[pair[0]]
                        #if station1 == "EN.SIM":
                            # ch3 is "north" and ch2 is "east"
                        #    t1ntmp = tramef_E[pair[0]]
                        #    t1etmp = tramef_N[pair[0]]
                        #
                        #outsac
                        #mytrace = Trace(data=t1ntmp)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t1ntmp.sac', format='SAC')
                        #mytrace = Trace(data=t1etmp)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t1etmp.sac', format='SAC')
                        #mytrace = Trace(data=t1ztmp)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t1ztmp.sac', format='SAC')
                        #outsac
                        #if ch1Az != 0:
                        #    t1n = t1ntmp * np.cos(ch1Az) -\
                        #          t1etmp * np.sin(ch1Az)
                        #    t1e = t1ntmp * np.sin(ch1Az) +\
                        #          t1etmp * np.cos(ch1Az)                            
                        #else:
                        #    t1n = t1ntmp
                        #    t1e = t1etmp
                        #
                        # rorate back to N and E component
                        #t1n , t1e = rotate_RT_NE(t1ntmp,t1etmp,bAz1)
                        #z, n, e = rotate2ZNE(tr,azimuth,dip,tr2,azimuth2,dip2,tr3,azimuth3,dip3)
                        t1z , t1n, t1e = rotate2ZNE(t1ztmp,ch1AzZ,ch1DipZ,t1ntmp,ch1Az,ch1Dip,t1etmp,ch1Az2,ch1Dip2)
                        ### outsac                        
                        #mytrace = Trace(data=t1n)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t1n.sac', format='SAC')
                        #mytrace = Trace(data=t1e)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t1e.sac', format='SAC')
                        #mytrace = Trace(data=t1z)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t1z.sac', format='SAC')
                        ### outsac
                        ### rotate to R T. We do here all time
                        ### sta1 is a souce and sta is a receiver and then R,T are defined above 
                        t1r , t1t = rotate_NE_RT(t1n,t1e,bAz)
                        t1fn , t1fp = rotate_NE_RT(t1n,t1e,args.azf)
                        ### outsac
                        #mytrace = Trace(data=t1r)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t1r.sac', format='SAC')
                        #mytrace = Trace(data=t1t)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t1t.sac', format='SAC')
                        ### outsac
                        del t1ntmp, t1etmp, t1ztmp
                    
                    if components [1] == "N" or components [1] == "E" or  components [1] == "R" or  components [1] == "T" or components [1] == "2" or  components [1] == "3" or components [1] == "Z" or components [1] == "1":
                        t2ztmp = tramef_Z[pair[1]]
                        t2ntmp = tramef_N[pair[1]]
                        t2etmp = tramef_E[pair[1]]
                        #if station2 == "EN.SIM":
                            # ch3 is "north" and ch2 is "east"
                        #    t2ntmp = tramef_E[pair[1]]
                        #    t2etmp = tramef_N[pair[1]]
                        #
                        ### outsac
                        #mytrace = Trace(data=t2ntmp)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t2ntmp.sac', format='SAC')
                        #mytrace = Trace(data=t2etmp)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t2etmp.sac', format='SAC')
                        #mytrace = Trace(data=t2ztmp)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t2ztmp.sac', format='SAC')
                        ### outsac
                        #if ch2Az != 0:
                        #    t2n = t2ntmp * np.cos(ch2Az) -\
                        #          t2etmp * np.sin(ch2Az)
                        #    t2e = t2ntmp * np.sin(ch2Az) +\
                        #          t2etmp * np.cos(ch2Az)                            
                        #else:
                        #    t2n = t2ntmp
                        #    t2e = t2etmp
                        #    
                        # rorate back to N and E component
                        #t2n , t2e = rotate_RT_NE(t2ntmp,t2etmp,bAz2)
                        #z, n, e = rotate2ZNE(tr,azimuth,dip,tr2,azimuth2,dip2,tr3,azimuth3,dip3)
                        t2z , t2n, t2e = rotate2ZNE(t2ztmp,ch2AzZ,ch2DipZ,t2ntmp,ch2Az,ch2Dip,t2etmp,ch2Az2,ch2Dip2)
                        ### outsac
                        #mytrace = Trace(data=t2n)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t2n.sac', format='SAC')
                        #mytrace = Trace(data=t2e)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t2e.sac', format='SAC')
                        #mytrace = Trace(data=t2z)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t2z.sac', format='SAC')
                        ### outsac
                        ### rotate to R T. We do here all time
                        ### sta1 is a souce and sta is a receiver and then R,T are defined above 
                        t2r , t2t = rotate_NE_RT(t2n,t2e,bAz)
                        t2fn , t2fp = rotate_NE_RT(t2n,t2e,args.azf)
                        ### outsac
                        #mytrace = Trace(data=t2r)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t2r.sac', format='SAC')
                        #mytrace = Trace(data=t2t)
                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                        #st = Stream(traces=[mytrace, ])
                        #st.write('t2t.sac', format='SAC')
                        ### outsac
                        del t2ntmp, t2etmp

                    ### adde
                    if components[0] == "Z":
                        #t1 = tramef_Z[pair[0]]
                        t1 = t1z
                    ### adds
                    elif components[0] == "N":
                        #t1 = tramef_N[pair[0]]
                        # now add rotation 
                        # north
                        #t1 = t1n
                        # pk-fault zone study
                        t1 = t1fn
                    elif components[0] == "E":
                        #t1 = tramef_E[pair[0]]                    
                        # now add rotation
                        #t1 = t1e
                        # pk-fault zone study
                        t1 = t1fp
                    ### adde
                    elif components[0] == "R":
                        ### orgs
                        #if cplAz != 0:
                        #    t1 = tramef_N[pair[0]] * np.cos(cplAz) +\
                        #         tramef_E[pair[0]] * np.sin(cplAz)
                        #else:
                        #    t1 = tramef_E[pair[0]]
                        #
                    ### orge
                        ### adds
                        t1 = t1r
                    ### adde
                    elif components[0] == "T":
                        ### orgs
                        #if cplAz != 0:
                        #    t1 = tramef_N[pair[0]] * np.sin(cplAz) -\
                        #         tramef_E[pair[0]] * np.cos(cplAz)
                        #else:
                        #    t1 = tramef_N[pair[0]]
                        #
                    # orge
                        ### adds
                        t1 = t1t
                    
                    # adde
                    if components[1] == "Z":
                        #t2 = tramef_Z[pair[1]]
                        t2 = t2z
                    ### adds
                    elif components[1] == "N":
                        #t2 = tramef_N[pair[1]]
                        #t2 = t2n
                        t2 = t2fn
                    elif components[1] == "E":
                        #t2 = tramef_E[pair[1]]
                        #t2 = t2e
                        t2 = t2fp
                    ### adde
                    elif components[1] == "R":
                        ### orgs
                        #if cplAz != 0:
                        #    t2 = tramef_N[pair[1]] * np.cos(cplAz) +\
                        #         tramef_E[pair[1]] * np.sin(cplAz)
                        #else:
                        #    t2 = tramef_E[pair[1]]
                        # orge
                        ### adds                        
                        #if cplAz != 0:
                        #    t2 = t2n * np.cos(cplAz) +\
                        #         t2e * np.sin(cplAz)
                        #else:
                        #    t2 = t2n
                        #
                        t2 = t2r
                        ### adde
                    elif components[1] == "T":
                        ### orgs
                      #  if cplAz != 0:
                      #      t2 = tramef_N[pair[1]] * np.sin(cplAz) -\
                      #           tramef_E[pair[1]] * np.cos(cplAz)
                      #  else:
                      #      t2 = tramef_N[pair[1]]
                      #      
                      ### orgs
                      ### adds
                      #  if cplAz != 0:
                      #      t2 = -1 * t2n * np.sin(cplAz) +\
                      #           t2e * np.cos(cplAz)
                      #  else:
                      #      t2 = t2e
                      #      
                        t2 = t2t
                    
                    # adde
                    trames = np.vstack((t1, t2))
                    del t1, t2
                    ### adds
                    if components [0] == "N" or components [0] == "E" or  components [0] == "R" or  components [0] == "T" or components [0] == "2" or  components [0] == "3" or components [0] == "Z" or components [0] == "1":
                        del t1n,t1e,t1r,t1t,t1z,t1fn,t1fp
                    
                    if components [1] == "N" or components [1] == "E" or  components [1] == "R" or  components [1] == "T" or components [1] == "2" or  components [1] == "3" or components [1] == "Z" or components [1] == "1":
                        del t2n,t2e,t2r,t2t,t2z,t2fn,t2fp
                    
                    ### adde
                    
                    daycorr = {}
                    ndaycorr = {}
                    allcorr = {}
                    for filterdb in get_filters(db, all=False):
                        filterid = filterdb.ref
                        daycorr[filterid] = np.zeros(get_maxlag_samples(db,))
                        ndaycorr[filterid] = 0

                    for islice, (begin, end) in enumerate(zip(begins, ends)):
                        # print "Progress: %#2d/%2d"% (islice+1,slices)
                        trame2h = trames[:, begin:end]

                        rmsmat = np.std(np.abs(trame2h), axis=1)
                        for filterdb in get_filters(db, all=False):
                            filterid = filterdb.ref
                            low = float(filterdb.low)
                            high = float(filterdb.high)
                            rms_threshold = filterdb.rms_threshold

                            Nfft = int(params.min30)
                            if params.min30 / 2 % 2 != 0:
                                Nfft = params.min30 + 2

                            trames2hWb = np.zeros((2, int(Nfft)), dtype=np.complex)
                            ### adds
                            trames2h_tmp = np.zeros((2, int(Nfft)), dtype=np.complex)
                            trames2h_tmp2 = np.zeros((2, int(Nfft)), dtype=np.complex)
                            ### adde
                            skip = False
                            for i, station in enumerate(pair):
                                if rmsmat[i] > rms_threshold:
                                    cp = cosTaper(len(trame2h[i]),0.04)
                                    trame2h[i] -= trame2h[i].mean()
                                    ### adds
				    #low = 2.5
				    #high = 5.5
                                    #logging.debug("components %s " % (components))
                                    #logging.info('station1: %s' % (station1))
                                    #logging.info('station2: %s' % (station2))
                                    #print (type(station1))                                    
                                    sta_opt = station1 == station2
                                    #print (sta_opt)
                                    #logging.info('sta_opt: %s' % str(sta_opt))                                                                        
                                    #sys.exit()
                                    if sta_opt and (components == "ZZ" or components == "NN" or  components == "EE" or components == "RR" or  components == "TT" or components == "11" or components == "22" or  components == "33"):
                                    #if components == "QQ":
                                        #for auto we cannot use whiten
                                        #logging.debug('No whiten. one more bandasss station1: %s station2: %s components: %s' % (station1,station2,components) )
                                        #low = 2.5
					#high = 5.5
					mytrace = Trace(data=trame2h[i])
                                        mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                                        #st = Stream(traces=[mytrace, ])
                                        #st.write('dmean.sac', format='SAC')
                                        #logging.debug(
                                        #    "%s Highpass for AUTO at %.2f Hz" % (station, low))
                                        #st.detrend(type="demean")
                                        #st.detrend(type="linear")
                                        mytrace.filter("highpass", freq=low, zerophase=True, corners=8)
                                        #st = Stream(traces=[mytrace, ])
                                        #st.write('hp.sac', format='SAC')
                                        #logging.debug(
                                        #    "%s Lowpass for AUTO at %.2f Hz" % (station, high))
                                        mytrace.filter("lowpass", freq=high, zerophase=True, corners=8)
                                        #st = Stream(traces=[mytrace, ])
                                        #st.write('lp.sac', format='SAC')
                                        #print type(mytrace.data)
                                        #print type(trames2h_tmp2[i])
                                        #logging.debug('before!')
                                        trames2h_tmp2[i]=scipy.fftpack.fft(mytrace.data, Nfft)
                                        trames2h_tmp2[i]=scipy.fftpack.ifft(trames2h_tmp2[i], Nfft)
                                        #logging.debug('after!')
                                        #sys.exit()
                                        #mytrace = Trace(data=trames2h_tmp2[i] )
                                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                                        #st = Stream(traces=[mytrace, ])
                                        #st.write('bp.sac', format='SAC')
                                        #sys.exit()
                                        #for auto we cannot use whiten
                                    else:
                                        #whiten first
                                        #logging.debug('Whiten')              
                                        #logging.debug('Whiten. %s station2: %s components: %s' % (station1,station2,components) )
                                        trames2h_tmp[i] = whiten(
                                            trame2h[i]*cp, Nfft, dt, low, high, plot=False)
                                        trames2h_tmp2[i]=scipy.fftpack.ifft(trames2h_tmp[i], Nfft)
                                        #whiten first
                                                                        
                                    ### adde                                    
                                    ### outsac
                                    #mytrace = Trace(data=trame2h[i])
                                    #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                                    #st = Stream(traces=[mytrace, ])
                                    #st.write('raw.sac', format='SAC')
                                    #mytrace = Trace(data=trames2h_tmp2[i])
                                    #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                                    #st = Stream(traces=[mytrace, ])
                                    #st.write('whiten.sac', format='SAC')                                    
                                    ### outsac
                                    ### adde
                                    
                                    if params.windsorizing == -1:
                                        ### orgs                                        
                                        #trame2h[i] = np.sign(trame2h[i])
                                        ### orge
                                        ### adds
                                        #logging.debug('before2!')
                                        trames2h_tmp2[i] = np.sign(trames2h_tmp2[i])                                    
                                        #logging.debug('after2!')
                                        ### outsac
                                        #mytrace = Trace(data=trames2h_tmp2[i])
                                        #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                                        #st = Stream(traces=[mytrace, ])
                                        #st.write('1bit.sac', format='SAC')
                                        #sys.exit()
                                        ### outsac
                                        ### adde
                                    elif params.windsorizing != 0:
                                        ### orgs    
                                        #indexes = np.where(
                                        #    np.abs(trame2h[i]) > (params.windsorizing * rmsmat[i]))[0]
                                        # clipping at windsorizing*rms
                                        #trame2h[i][indexes] = (trame2h[i][indexes] / np.abs(
                                        #    trame2h[i][indexes])) * params.windsorizing * rmsmat[i]
                                        #
                                        ### orgs    
                                        ### adds    
                                        indexes = np.where(
                                            np.abs(trames2h_tmp2[i]) > (params.windsorizing * rmsmat[i]))[0]
                                        # clipping at windsorizing*rms
                                        trames2h_tmp2[i][indexes] = (trames2h_tmp2[i][indexes] / np.abs(
                                            trames2h_tmp2[i][indexes])) * params.windsorizing * rmsmat[i]
                                    
                                        ### adde    
                                    ### orgs
                                    #trames2hWb[i] = whiten(
                                    #    trame2h[i]*cp, Nfft, dt, low, high, plot=False)
                                    ### orge
                                    ### adds
                                    ### whiten was done abobe
                                    #logging.debug('before3!')
                                    trames2hWb[i]=scipy.fftpack.fft(trames2h_tmp2[i]*cp, Nfft)
                                    #logging.debug('after3!')
                                    ### adde
                                else:
                                    trames2hWb[i] = np.zeros(int(Nfft))
                                    skip = True
                                    logging.debug('Slice is Zeros!')
                            if not skip:
                                ### adds
                                #logging.debug('before4!')
                                #print trames2h_tmp2[0]                               
                                #np.savetxt('w0.txt', trames2h_tmp2[0])
                                #mytrace = Trace(data=trames2h_tmp2[0] )
                                #logging.debug('before41!')
                                #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                                #logging.debug('before42!')
                                #st = Stream(traces=[mytrace, ])
                                #logging.debug('before43!')
                                #st.write('w0.sac', format='SAC')
                                #logging.debug('before44!')
                                #logging.debug('before5!')              
                                #np.savetxt('w1.txt', trames2h_tmp2[1])
                                #mytrace = Trace(data=trames2h_tmp2[1] )
                                #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                                #st = Stream(traces=[mytrace, ])
                                #st.write('w1.sac', format='SAC')
                                #logging.debug('before6!')                                                                
                                #mytrace = Trace(data=trames2hWb[0] )
                                #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                                #st = Stream(traces=[mytrace, ])
                                #st.write('wb0.sac', format='SAC')
                                #logging.debug('before7!')                                                                
                                #mytrace = Trace(data=trames2hWb[1] )
                                #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                                #st = Stream(traces=[mytrace, ])
                                #st.write('wb1.sac', format='SAC')
                                #np.savetxt('wb0.txt', trames2hWb[0])
                                #np.savetxt('wb1.txt', trames2hWb[1])
                                #sys.exit()
                                ### adde
                                ### orge
                                #logging.debug('before5!')
                                corr = myCorr(trames2hWb, np.ceil(params.maxlag / dt), plot=False)
                                ### orgs
                                ### adds
                                #mytrace = Trace(data=corr )
                                #mytrace.stats['sampling_rate'] = params.goal_sampling_rate
                                #st = Stream(traces=[mytrace, ])
                                #st.write('corr.sac', format='SAC')
                                #logging.debug('after4!')
                                #sys.exit()
                                #logging.debug('plot!')
                                #corr = myCorr(trames2hWb, np.ceil(params.maxlag / dt), plot=True)
                                #sys.exit()
                                ### adde
                                tmptime = time.gmtime(basetime + begin /
                                                      params.goal_sampling_rate)
                                thisdate = time.strftime("%Y-%m-%d", tmptime)
                                thistime = time.strftime("%Y-%m-%d %H:%M:%S",
                                                         tmptime)
                                if params.keep_all:
                                    ccfid = "%s_%s_%s_%s_%s" % (station1, station2,
                                                             filterid, components,
                                                             thisdate)
                                    if ccfid not in allcorr:
                                        allcorr[ccfid] = {}
                                    allcorr[ccfid][thistime] = corr

                                if params.keep_days:
                                    if not np.any(np.isnan(corr)) and \
                                            not np.any(np.isinf(corr)):
                                        daycorr[filterid] += corr
                                        ndaycorr[filterid] += 1

                            ### orgs
#                                del corr, thistime, trames2hWb
                            ### orge
                            ### adds
                                del corr, thistime, trames2hWb, trames2h_tmp, trames2h_tmp2
                            ### adde

                    if params.keep_all:
                        for ccfid in allcorr.keys():
                            export_allcorr(db, ccfid, allcorr[ccfid])

                    if params.keep_days:
                        try:
                            for filterdb in get_filters(db, all=False):
                                filterid = filterdb.ref
                                corr = daycorr[filterid]
                                ncorr = ndaycorr[filterid]
                                if ncorr > 0:
                                    logging.debug(
                                        "Saving daily CCF for filter %02i, comp %s (stack of %02i CCF)" % (filterid, components, ncorr))

                                    thisdate = time.strftime(
                                        "%Y-%m-%d", time.gmtime(basetime))
                                    thistime = time.strftime(
                                        "%H_%M", time.gmtime(basetime))
                                    add_corr(
                                        db, station1.replace('.', '_'),
                                        station2.replace('.', '_'), filterid,
                                        thisdate, thistime,  params.min30 /
                                        params.goal_sampling_rate,
                                        components, corr,
                                        params.goal_sampling_rate, day=True,
                                        ncorr=ncorr)
                                del corr, ncorr
                        except Exception as e:
                            logging.debug(str(e))
                    del trames, daycorr, ndaycorr
                logging.debug("Updating Job")
                update_job(db, goal_day, orig_pair, 'CC', 'D')

                logging.info("Finished processing this pair. It took %.2f seconds" %
                              (time.time() - tt))
            logging.info("Job Finished. It took %.2f seconds" % (time.time() - jt))
    logging.info('*** Finished: Compute CC ***')

if __name__ == "__main__":
    ### orgs
    #main()
    ### orge
    ### adds
    parser = argparse.ArgumentParser(description='Scan the data archive and insert the\
    metadata in the database')
    parser.add_argument('-c', '--convolv',
                        help='Compute DTT jobs have been modified in the last i days\
                        [default:0]', default=0, type=int)
    parser.add_argument('-f', '--azf',
                        help='azf. if azf is 45 then N is pointng to 45.\
                        [default:0.0]', default=0.0, type=float)
    parser.add_argument('-l', '--listch',
                        help='listch. 0: zen 1: z23 2: 123\
                        [default:0]', default=0, type=int)
    parser.add_argument('-p', '--pairch',
                        help='pairch. 0: all 1: zne ; -1 is what we define \
                        [default:-1]', default=-1, type=int)
    parser.add_argument('-s', '--stalist',
                        help='stalist. 0: all 1: select\
                        [default:0]', default=0, type=int)
    parser.add_argument('-d1', '--d1',
                        help='d1 start date def 1900-01-01.\
                        [default:1900-01-01]', default='1900-01-01', type=str)
    parser.add_argument('-d2', '--d2',
                        help='d2 end date def 2500-01-01.\
                        [default:2500-01-01]', default='2500-01-01', type=str)
    args = parser.parse_args()
       
    main(args.convolv, args.azf, args.listch, args.pairch, args.stalist, args.d1, args.d2)
    ### adde
    
