//  -*- Mode: C++;  -*-

//  File: grid.cpp

//  Author: Dino Bellugi (dino@geomorph.berkeley.edu)

//  Copyright  Dino Bellugi, BlueG SoftWear, U.C. Berkeley, 1999 (C)

// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

// * FUNCTION: implementation of all grid related functions

// *

// * RELATED PACKAGES: all

// *

// * HISTORY:

// * Created:  Mon Nov  6 1995 (dino, based on rob reiss' version)

// * Modified: Sat Jun 26 1999 (dino)

// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~



// the following include file (notice.h) contains important copyright 

// information, as well as other legal statements. it must be present 

// in all files and in any distribution. removal or modification is

// strictly forbidden and violates u.s. and international copyright law.

#include "notice.h"

//

#include "stabfun.h"



Grid::Grid(int x, int y, double xo, double yo, double spac, 

	double nodat, int flds) 

{

	// Creation of Grid

	xdim = x;

  	ydim = y;

  	xorg = xo;

  	yorg = yo;

  	xcur = ycur = xprev1 = yprev1 = xprev2 = yprev2 = 0;

  	spacing = xsize = ysize = spac;

  	nodata = nodat;

	fields = flds;



	for (int k = 0; k < fields; ++k)

	  	arrValues[k] = new double[x * y];



	timeseed();



}



Grid::Grid(Grid &z) 

{

	int i, j, k;



  	// Creation of Grid

  	xdim = z.xdim;

  	ydim = z.ydim;

  	xorg = z.xorg;

  	yorg = z.yorg;

  	xcur = ycur = xprev1 = yprev1 = xprev2 = yprev2 = 0;

  	spacing = z.spacing;

  	xsize = z.xsize;

  	ysize = z.ysize;

  	nodata = z.nodata;

	fields = z.fields;



 	for (k = 0; k < fields; ++k)

 		arrValues[k] = new double[z.xdim * z.ydim];



	for (k = 0; k < fields; ++k)

	{

	  	for (j = 0; j < ydim; j++) 

	  	{

	    	for(i = 0; i < xdim; i++) 

			{

	      		at(i, j, k) = z.at(i, j, k);

	    	}

	  	}

	}

	timeseed();

}



Grid::~Grid() 

{

	for (int k = 0; k < fields; ++k)

  		delete[] arrValues[k];

}



int Grid::addField(int srcfield)	// adds a field at the end of arrValues

{

	if (fields == MAXFIELDS ||

		(arrValues[fields] = new double[xdim * ydim]) == (double *)NULL)

	{

		cerr << "Not enough memory for " << fields + 1 << " fields ...\n";

		return(ERR_MEMALLOC);

	}

	++fields;

	if (srcfield >= 0)

		setValue(fields - 1, srcfield);

	return(OK);

}



int Grid::delField()	// deletes last field from arrValues

{

	if (fields < 1 || arrValues[fields - 1] == (double *)NULL)

		return(ERR_MEMFREE);



  	delete[] arrValues[--fields];

	return(OK);

}



int Grid::setSize(int rows, int cols, int init, int flds)

{

	int k;



	for (k = 0; k < fields; ++k)

  		delete[] arrValues[k];



	xdim = cols;

	ydim = rows;

	fields = flds;



 	for (k = 0; k < fields; ++k)

 		if ((arrValues[k] = new double[xdim * ydim]) == (double *)NULL)

			return(ERR_MEMALLOC);



	if (init)

	{

  		xorg = yorg = xcur = ycur = 

		xprev1 = yprev1 = xprev2 = yprev2 = 0;

		spacing = xsize = ysize = DEF_SIZE;

	}

	return(OK);

}



int Grid::setSize(Grid &z)

{

	int k;



	for (k = 0; k < fields; ++k)

  		delete[] arrValues[k];



	xdim = z.xdim;

	ydim = z.ydim;

  	xorg = z.xorg;

  	yorg = z.yorg;

  	xcur = ycur = xprev1 = yprev1 = xprev2 = yprev2 = 0;

  	spacing = z.spacing;

  	xsize = z.xsize;

  	ysize = z.ysize;

	nodata = z.nodata;

	fields = z.fields;



 	for (k = 0; k < fields; ++k)

 		if ((arrValues[k] = new double[xdim * ydim]) == (double *)NULL)

			return(ERR_MEMALLOC);



	return(OK);

}





void Grid::setNodata(double value)	{nodata = value;}



void Grid::chgNodata(double value, int set, int field)	

{

	chgValue(nodata, value, field);

	if (set)

		setNodata(value);

}



void Grid::setValue(double value, int field)

{

        int i, j;

 

        for (j = 0; j < ydim; j++)

                for(i = 0; i < xdim; i++)

                        at(i, j, field) = value;

}

 

void Grid::setValue(int field1, int field2)

{

	int i, j;



	for (j = 0; j < ydim; j++) 

		for(i = 0; i < xdim; i++) 

			at(i, j, field1) = at(i, j, field2);

}



void Grid::setValue(int field1, int field2, double value)

{

	int i, j;



	for (j = 0; j < ydim; j++) 

		for(i = 0; i < xdim; i++)

			if (isValue(i, j, field2))

				at(i, j, field1) = at(i, j, field2) + value;

			else

				at(i, j, field1) = nodata;

}



void Grid::setValue(int field1, double value, int field2)

{

	int i, j;



	for (j = 0; j < ydim; j++) 

		for(i = 0; i < xdim; i++) 

			if (isValue(i, j, field2))

				at(i, j, field1) = value;

			else

				at(i, j, field1) = nodata;

}



long Grid::countValue(double value, int field)

{

	int i, j;

	long cnt = 0L;



	for (j = 0; j < ydim; j++) 

		for(i = 0; i < xdim; i++) 

			if (at(i, j, field) == value)

				++cnt;

	return(cnt);

}



long Grid::countValue(int field)

{

	int i, j;

	long cnt = 0L;



	for (j = 0; j < ydim; j++) 

		for(i = 0; i < xdim; i++) 

			if (isValue(i, j, field))

				++cnt;

	return(cnt);

}



void Grid::chgValue(double old_value, double new_value, int field)

{

	int i, j;



	for (j = 0; j < ydim; j++) 

		for(i = 0; i < xdim; i++) 

			if (at(i, j, field) == old_value)

				at(i, j, field) = new_value;

}



void Grid::chgValue(int nvalues, double *old_value, double new_value, int field)

{

	int i, j, n;



	for (j = 0; j < ydim; ++j) 

	{

		for(i = 0; i < xdim; ++i) 

		{

			for (n = 0; n < nvalues; ++n)

			{

				if (at(i, j, field) == old_value[n])

				{

					at(i, j, field) = new_value;

					break;

				}

			}

		}

	}

}



void Grid::addValue(int field1, int field2)

{

	int i, j;



	for (j = 0; j < ydim; j++) 

		for(i = 0; i < xdim; i++)

			if (isValue(i, j, field1) && isValue(i, j, field2))

				at(i, j, field1) += at(i, j, field2);

}



void Grid::subValue(int field1, int field2)

{

	int i, j;



	for (j = 0; j < ydim; j++) 

		for(i = 0; i < xdim; i++) 

			if (isValue(i, j, field1) && isValue(i, j, field2))

				at(i, j, field1) -= at(i, j, field2);

}


void Grid::mkBinary(double threshold, int reverse, int infield, int outfield)
{
        int i, j;
 
        for (j = 0; j < ydim; j++)
	{
                for(i = 0; i < xdim; i++)
		{
			if (isValue(i, j, infield))
			{
                        	if (at(i, j, infield) > threshold)
					at(i, j, outfield) = (reverse ? 0 : 1);
				else
					at(i, j, outfield) = (reverse ? 1 : 0);
			}
			else
				at(i, j, outfield) = at(i, j, infield);
		}
	}
}



double Grid::getQoverT(double theta, double aoverb, int slope_mode, 

	double ps, double phi)

{

	if (theta == nodata || aoverb == nodata || ps <= 0.0 || phi <= 0.0)

		return(nodata);



	double ps_pw = ps / 1000.0, pw_ps = 1000.0 / ps, q_t;



	phi = deg2rad(phi);


	if (slope_mode != ATAN_RAD)
	{
		switch (slope_mode)
		{
			case TAN_RAD:
				theta = atan(theta);
				break;

			case SIN_RAD:
				theta = asin(theta);
				break;

			case COS_RAD:
				theta = acos(theta);
				break;

			case ATAN_DEG:
				theta = deg2rad(theta);
				break;

			case TAN_DEG:
				theta = deg2rad(atan(theta));
				break;

			case SIN_DEG:
				theta = deg2rad(asin(theta));
				break;

			case COS_DEG:
				theta = deg2rad(acos(theta));
				break;
		}
	}
 

	if (tan(theta) >= tan(phi))

	{

		q_t = DEF_STEEP;
	}

	else if (tan(theta) < ((1.0 - pw_ps) * tan(phi)))

	{

		q_t = DEF_FLAT;

	}

	else

	{

		q_t = log10((sin(theta) / aoverb) * ps_pw *

			(1.0 - (tan(theta) / tan(phi))));

	}



	return(q_t);

}



double Grid::getQoverT(int i, int j, int theta_field, int aoverb_field,

	int slope_mode, double ps, double phi)

{

	double theta, aoverb;



	if ((theta = at(i, j, theta_field)) == nodata ||

		(aoverb = at(i, j, aoverb_field)) == nodata ||

		ps <= 0.0 || phi <= 0.0)

		return(nodata);

 

	double ps_pw = ps / 1000.0, pw_ps = 1000.0 / ps, q_t;

 

	phi = deg2rad(phi);


	if (slope_mode != ATAN_RAD)
	{
		switch (slope_mode)
		{
			case TAN_RAD:
				theta = atan(theta);
				break;

			case SIN_RAD:
				theta = asin(theta);
				break;

			case COS_RAD:
				theta = acos(theta);
				break;

			case ATAN_DEG:
				theta = deg2rad(theta);
				break;

			case TAN_DEG:
				theta = deg2rad(atan(theta));
				break;

			case SIN_DEG:
				theta = deg2rad(asin(theta));
				break;

			case COS_DEG:
				theta = deg2rad(acos(theta));
				break;
		}
	}


	if (tan(theta) >= tan(phi))

	{
		q_t = DEF_STEEP;

	}

	else if (tan(theta) < ((1.0 - pw_ps) * tan(phi)))

	{

		q_t = DEF_FLAT;

	}

	else

	{

		q_t = log10((sin(theta) / aoverb) * ps_pw *

		(1.0 - (tan(theta) / tan(phi))));

	}

 

	return(q_t);

}

 

double Grid::getQoverT(int i, int j, int theta_field, double aoverb,

	int slope_mode, double ps, double phi)

{

	double theta;

 

	if ((theta = at(i, j, theta_field)) == nodata ||

		aoverb == nodata || ps <= 0.0 || phi <= 0.0)

		return(nodata);

 

	double ps_pw = ps / 1000.0, pw_ps = 1000.0 / ps, q_t;

 

	phi = deg2rad(phi);

 
	if (slope_mode != ATAN_RAD)
	{
		switch (slope_mode)
		{
			case TAN_RAD:
				theta = atan(theta);
				break;

			case SIN_RAD:
				theta = asin(theta);
				break;

			case COS_RAD:
				theta = acos(theta);
				break;

			case ATAN_DEG:
				theta = deg2rad(theta);
				break;

			case TAN_DEG:
				theta = deg2rad(atan(theta));
				break;

			case SIN_DEG:
				theta = deg2rad(asin(theta));
				break;

			case COS_DEG:
				theta = deg2rad(acos(theta));
				break;
		}
	}
 

	if (tan(theta) >= tan(phi))

	{

		q_t = DEF_STEEP;

	}

	else if (tan(theta) < ((1.0 - pw_ps) * tan(phi)))

	{

		q_t = DEF_FLAT;

	}

	else

	{

		q_t = log10((sin(theta) / aoverb) * ps_pw *

			(1.0 - (tan(theta) / tan(phi))));

	}

 

	return(q_t);

}

 

double Grid::getQoverT(int i, int j, double theta, int aoverb_field,

	int slope_mode, double ps, double phi)

{

	double aoverb;

 

	if ((aoverb = at(i, j, aoverb_field)) == nodata ||

		theta == nodata || ps <= 0.0 || phi <= 0.0)

		return(nodata);

 

	double ps_pw = ps / 1000.0, pw_ps = 1000.0 / ps, q_t;

 

	phi = deg2rad(phi);

 
	if (slope_mode != ATAN_RAD)
	{
		switch (slope_mode)
		{
			case TAN_RAD:
				theta = atan(theta);
				break;

			case SIN_RAD:
				theta = asin(theta);
				break;

			case COS_RAD:
				theta = acos(theta);
				break;

			case ATAN_DEG:
				theta = deg2rad(theta);
				break;

			case TAN_DEG:
				theta = deg2rad(atan(theta));
				break;

			case SIN_DEG:
				theta = deg2rad(asin(theta));
				break;

			case COS_DEG:
				theta = deg2rad(acos(theta));
				break;
		}
	}
 

	if (tan(theta) >= tan(phi))

	{

		q_t = DEF_STEEP;

	}

	else if (tan(theta) < ((1.0 - pw_ps) * tan(phi)))

	{

		q_t = DEF_FLAT;

	}

	else

	{

		q_t = log10((sin(theta) / aoverb) * ps_pw *

			(1.0 - (tan(theta) / tan(phi))));

	}

 

	return(q_t);

}

 

double Grid::getQoverTCF(int i, int j, int theta_field, 

	int aoverb_field, int slope_mode, double ps, double phi, 

	double depth, double coh)

{

	double theta, aoverb;



	if ((theta = at(i, j, theta_field)) == nodata ||

		(aoverb = at(i, j, aoverb_field)) == nodata ||

		ps <= 0.0 || phi <= 0.0)

		return(nodata);

 

	double ps_pw = ps / 1000.0, pw_ps = 1000.0 / ps, g = 9.81, q_t;

 

	phi = deg2rad(phi);

 
	if (slope_mode != ATAN_RAD)
	{
		switch (slope_mode)
		{
			case TAN_RAD:
				theta = atan(theta);
				break;

			case SIN_RAD:
				theta = asin(theta);
				break;

			case COS_RAD:
				theta = acos(theta);
				break;

			case ATAN_DEG:
				theta = deg2rad(theta);
				break;

			case TAN_DEG:
				theta = deg2rad(atan(theta));
				break;

			case SIN_DEG:
				theta = deg2rad(asin(theta));
				break;

			case COS_DEG:
				theta = deg2rad(acos(theta));
				break;
		}
	}
 

	if (tan(theta) >= 

		(tan(phi) + (coh / (ps * g * depth * sqr(cos(theta))))))

	{

		q_t = DEF_STEEP;

	}

	else if (tan(theta) < ((coh / (ps * g * depth * sqr(cos(theta))))

		+ ((1.0 - pw_ps) * tan(phi))))

	{

		q_t = DEF_FLAT;

	}

	else

	{

		q_t = log10((sin(theta) / aoverb) * ps_pw *

			(1.0 - (tan(theta) / tan(phi)) +

			(coh / (ps * g * depth * sqr(cos(theta)) * tan(phi)))));

	}

 

	return(q_t);

}

 

double Grid::getSlope(int i, int j, int field, int mode) 

// Our Method

{

	double slope, a, b, c, d, e, f, g, h, l;



 	if ((e = at(i,     j, field)) == nodata || 

		(d = at(i-1,   j, field)) == nodata || 

		(f = at(i+1,   j, field)) == nodata || 

		(h = at(i,   j-1, field)) == nodata || 

		(b = at(i,   j+1, field)) == nodata || 

		(c = at(i+1, j+1, field)) == nodata || 

		(g = at(i-1, j-1, field)) == nodata || 

		(a = at(i-1, j+1, field)) == nodata || 

		(l = at(i+1, j-1, field)) == nodata) 

	{

		return(nodata);

	}



	slope = (sqrt(sqr((f - d) / (2.0 * spacing)) +

		sqr((b - h) / (2.0 * spacing))) +

		sqrt(sqr((c - g) / (SQRT2_x2 * spacing)) +

		sqr((a - l) / (SQRT2_x2 * spacing)))) / 2.0;



	switch (mode)

	{

		case ATAN_RAD:  // theta, radians

			return(atan(slope));

		case TAN_RAD:   // tan(theta), radians

			return(slope);

		case SIN_RAD:   // sin(theta), radians

			return(sin(atan(slope)));

		case COS_RAD:   // cos(theta), radians

			return(cos(atan(slope)));

		case ATAN_DEG:  // theta, degrees

			return(rad2deg(atan(slope)));

		case TAN_DEG:   // tan(theta), degrees

			return(rad2deg(slope));

		case SIN_DEG:   // sin(theta), radians

			return(rad2deg(sin(atan(slope))));

		case COS_DEG:   // cos(theta), radians

			return(rad2deg(cos(atan(slope))));

		default:	// theta, radians

			return(atan(slope));

	}

}



double Grid::getSlope2(int i, int j, int field, int mode)

// Arc/Info Method

{

	double dz_dx, dz_dy, slope, a, b, c, d, e, f, g, h, l;

 

	if ((e = at(i,     j, field)) == nodata ||

		(d = at(i-1,   j, field)) == nodata ||

		(f = at(i+1,   j, field)) == nodata ||

		(h = at(i,   j-1, field)) == nodata ||

		(b = at(i,   j+1, field)) == nodata ||

		(c = at(i+1, j+1, field)) == nodata ||

		(g = at(i-1, j-1, field)) == nodata ||

		(a = at(i-1, j+1, field)) == nodata ||

		(l = at(i+1, j-1, field)) == nodata)

	{

		return(nodata);

	}

 

	dz_dx = ((a + 2 * d + g) - (c + 2 * f + l)) / (8 * xsize);

	dz_dy = ((a + 2 * b + c) - (g + 2 * h + l)) / (8 * ysize);

	slope = sqrt(sqr(dz_dx) + sqr(dz_dy));

 

	switch (mode)

	{

		case ATAN_RAD:  // theta, radians

			return(atan(slope));

		case TAN_RAD:   // tan(theta), radians

			return(slope);

		case SIN_RAD:   // sin(theta), radians

			return(sin(atan(slope)));

		case COS_RAD:   // cos(theta), radians

			return(cos(atan(slope)));

		case ATAN_DEG:  // theta, degrees

			return(rad2deg(atan(slope)));

		case TAN_DEG:   // tan(theta), degrees

			return(rad2deg(slope));

		case SIN_DEG:   // sin(theta), radians

			return(rad2deg(sin(atan(slope))));

		case COS_DEG:   // cos(theta), radians

			return(rad2deg(cos(atan(slope))));

		default:	// theta, radians

			return(atan(slope));

	}

}

 

double Grid::getSlope3(int i, int j, int field, int mode) 

// Downhill slope

{

	double a, b, c, d, e, f, g, h, l, val, dsize, slope = 0.0;

	int ndir = 0;



 	if ((e = at(i,     j, field)) == nodata || 

		(d = at(i-1,   j, field)) == nodata || 

		(f = at(i+1,   j, field)) == nodata || 

		(h = at(i,   j-1, field)) == nodata || 

		(b = at(i,   j+1, field)) == nodata || 

		(c = at(i+1, j+1, field)) == nodata || 

		(g = at(i-1, j-1, field)) == nodata || 

		(a = at(i-1, j+1, field)) == nodata || 

		(l = at(i+1, j-1, field)) == nodata) 

	{

		return(nodata);

	}



	dsize = sqrt(sqr(xsize) + sqr(ysize));

	

	if ((val = e - d) > 0.0)

	{

		slope += val / xsize;

		ndir++;

	}



	if ((val = e - f) > 0.0)

	{

		slope += val / xsize;

		ndir++;

	}



	if ((val = e - b) > 0.0)

	{

		slope += val / ysize;

		ndir++;

	}



	if ((val = e - h) > 0.0)

	{

		slope += val / ysize;

		ndir++;

	}



	if ((val = e - a) > 0.0)

	{

		slope += val / dsize;

		ndir++;

	}



	if ((val = e - c) > 0.0)

	{

		slope += val / dsize;

		ndir++;

	}



	if ((val = e - g) > 0.0)

	{

		slope += val / dsize;

		ndir++;

	}



	if ((val = e - l) > 0.0)

	{

		slope += val / dsize;

		ndir++;

	}

	

	slope /= ndir;



	switch (mode)

	{

		case ATAN_RAD:  // theta, radians

			return(atan(slope));

		case TAN_RAD:   // tan(theta), radians

			return(slope);

		case SIN_RAD:   // sin(theta), radians

			return(sin(atan(slope)));

		case COS_RAD:   // cos(theta), radians

			return(cos(atan(slope)));

		case ATAN_DEG:  // theta, degrees

			return(rad2deg(atan(slope)));

		case TAN_DEG:   // tan(theta), degrees

			return(rad2deg(slope));

		case SIN_DEG:   // sin(theta), radians

			return(rad2deg(sin(atan(slope))));

		case COS_DEG:   // cos(theta), radians

			return(rad2deg(cos(atan(slope))));

		default:	// theta, radians

			return(atan(slope));

	}

}



double Grid::getSlope4(int i, int j, int field, int mode) 

// positive average (1D use)

{

	double a, b, c, d, e, f, g, h, l, val, dsize, slope = 0.0;

	int ndir = 0;



 	if ((e = at(i, j, field)) == nodata) 

		return(nodata);



	dsize = sqrt(sqr(xsize) + sqr(ysize));

	

	if ((d = at(i-1, j, field)) != nodata)

	{

		val = e - d;

		slope += mfabs(val) / xsize;

		ndir++;

	}



	if ((f = at(i+1, j, field)) != nodata)

	{

		val = e - f;

		slope += mfabs(val) / xsize;

		ndir++;

	}



	if ((b = at(i, j+1, field)) != nodata)

	{

		val = e - b;

		slope += mfabs(val) / ysize;

		ndir++;

	}



	if ((h = at(i, j-1, field)) != nodata)

	{

		val = e - h;

		slope += mfabs(val) / ysize;

		ndir++;

	}



	if ((a = at(i-1, j+1, field)) != nodata)

	{

		val = e - a;

		slope += mfabs(val) / dsize;

		ndir++;

	}



	if ((c = at(i+1, j+1, field)) != nodata)

	{

		val = e - c;

		slope += mfabs(val) / dsize;

		ndir++;

	}



	if ((g = at(i-1, j-1, field)) != nodata)

	{

		val = e - g;

		slope += mfabs(val) / dsize;

		ndir++;

	}



	if ((l = at(i+1, j-1, field)) != nodata)

	{

		val = e - l;

		slope += mfabs(val) / dsize;

		ndir++;

	}



	if (!ndir)

		return(nodata);

	

	slope /= ndir;



	switch (mode)

	{

		case ATAN_RAD:  // theta, radians

			return(atan(slope));

		case TAN_RAD:   // tan(theta), radians

			return(slope);

		case SIN_RAD:   // sin(theta), radians

			return(sin(atan(slope)));

		case COS_RAD:   // cos(theta), radians

			return(cos(atan(slope)));

		case ATAN_DEG:  // theta, degrees

			return(rad2deg(atan(slope)));

		case TAN_DEG:   // tan(theta), degrees

			return(rad2deg(slope));

		case SIN_DEG:   // sin(theta), radians

			return(rad2deg(sin(atan(slope))));

		case COS_DEG:   // cos(theta), radians

			return(rad2deg(cos(atan(slope))));

		default:	// theta, radians

			return(atan(slope));

	}

}



double Grid::getCurvature(int i, int j, int field) 

{

	double a, b, c, d, e, f, g, h, l;



 	if ((e = at(i,     j, field)) == nodata || 

		(d = at(i-1,   j, field)) == nodata || 

		(f = at(i+1,   j, field)) == nodata || 

		(h = at(i,   j-1, field)) == nodata || 

		(b = at(i,   j+1, field)) == nodata || 

		(c = at(i+1, j+1, field)) == nodata || 

		(g = at(i-1, j-1, field)) == nodata || 

		(a = at(i-1, j+1, field)) == nodata || 

		(l = at(i+1, j-1, field)) == nodata) 

	{

		return(nodata);

	}



	return((2 * (f + d + b + h) + (c + a + g + l) - 12 * e) / 

		(4 * xsize * ysize));

}


double Grid::getMaxDHSlope(int i, int j, int field, Direction *dir) 
{
	int k;
	Direction d = NODIR;

	double slope = -1000000, val;

	if (isValue(i, j, field))
	{
		for (k = 0; k < 7; ++k)
		{
			if (isValue(i, j, (Direction)k), field)
			{
				val = at(i, j, field) - at(i, j, (Direction)k, field);
				val /= isDiag((Direction)k) ? spacing * SQRT2 : spacing;
				if (val > slope)
				{
					slope = val;
					d = (Direction)k;
				}
			}
		}
	}

	if (dir != (Direction *)NULL)
		*dir = d;
	return(slope);	// negative means no downhill
}


Direction Grid::getFlowDir(int i, int j, int field)
{
	Direction d;

	return((getMaxDHSlope(i, j, field, &d) < 0) ? NODIR : d);
}

double Grid::chDist2Out(int i, int j, int flow_field, int dist_field)
{
	Direction fdir = (Direction)((int)at(i, j, flow_field)); 
	double dist = at(i, j, dist_field);

	if (fdir < 1 || fdir > 8)	// can't follow
	{
		at(i, j, dist_field) = 0;
		return(0);
	}

	if (dist == nodata)		// follow (not yet computed)
		dist = chDist2Out(get_i(i, fdir), get_j(j, fdir), 
			flow_field, dist_field);
	dist += (isDiag(fdir) ? spacing * SQRT2 : spacing);
	at(i, j, dist_field) = dist;
	return(dist);
}


//

// Input/Output

//



void Grid::coutGrid(int field, int header, double zmin, double zmax) 

{

	cout.precision(15);

	switch (header)

	{

		case ARCINFO:

			cout << "ncols "	<< xdim		<< "\n";

			cout << "nrows "	<< ydim		<< "\n";

			cout << "xllcorner "	<< xorg		<< "\n";

			cout << "yllcorner "	<< yorg		<< "\n";

			cout << "cellsize "	<< spacing	<< "\n";

			cout << "NODATA_value " << nodata	<< "\n";

			break;



		case SURFER:

			if (zmin == 0.0 && zmax == 0.0)

			{

				zmin = findMin(field);

				zmax = findMax(field);

#if (GR_MESSAGES >= 1)

				cerr << "min: " << zmin << ", max: " << 

					zmax << "\n";

#endif

			}

			cout << "DSAA\n";

			cout << xdim << SEPARATOR << ydim				 << "\n";

			cout << xorg << SEPARATOR << xorg + xdim * xsize - 1 << "\n";

			cout << yorg << SEPARATOR << yorg + ydim * ysize - 1 << "\n";

			cout << zmin << SEPARATOR << zmax				 << "\n";

			break;


		case TURTLE:

			cout << "/** number of entries */\nindex {3,DEM}\n";

			cout << "/** pixel size */\n1: float array dem {" << spacing << "," << spacing << "}\n";

			cout << "/** nodata value */\n2: float array novalues {" << nodata << "," << nodata << "}\n";

			cout << "/** grid values */\n3: double matrix dem_values {" << xdim << "," << ydim << "}\n";

			break;


		default:

			break;

	}

	cout.precision(8); 



	for(int j = 0; j < ydim; ++j)  

	{

		for(int i = 0; i < xdim; ++i) 

		{

			cout << at(i, j, field) << "\t";

		}

		cout << "\n";	

	}

}



void Grid::cerrGrid(int field, int header, double zmin, double zmax)

{

        cerr.precision(15);

        switch (header)

        {

                case ARCINFO:

                        cerr << "ncols "        << xdim         << "\n";

                        cerr << "nrows "        << ydim         << "\n";

                        cerr << "xllcorner "    << xorg         << "\n";

                        cerr << "yllcorner "    << yorg         << "\n";

                        cerr << "cellsize "     << spacing      << "\n";

                        cerr << "NODATA_value " << nodata       << "\n";

                        break;

 

                case SURFER:

                        if (zmin == 0.0 && zmax == 0.0)

                        {

                                zmin = findMin(field);

                                zmax = findMax(field);

                        }

                        cerr << "DSAA\n";

                        cerr << xdim << SEPARATOR << ydim << "\n";

                        cerr << xorg << SEPARATOR << xorg + xdim * xsize - 1 << "\n";

                        cerr << yorg << SEPARATOR << yorg + ydim * ysize - 1 << "\n";

                        cerr << zmin << SEPARATOR << zmax << "\n";

                        break;

		case TURTLE:

			cerr << "/** number of entries */\nindex {3,DEM}\n";

			cerr << "/** pixel size */\n1: float array dem {" << spacing << "," << spacing << "}\n";

			cerr << "/** nodata value */\n2: float array novalues {" << nodata << "," << nodata << "}\n";

			cerr << "/** grid values */\n3: double matrix dem_values {" << xdim << "," << ydim << "}\n";

			break;

 

                default:

                        break;

        }

        cerr.precision(8);

 

        for(int j = 0; j < ydim; ++j)

        {

                for(int i = 0; i < xdim; ++i)

                {

                        cerr << at(i, j, field) << "\t";

                }

                cerr << "\n";

        }

}

 

void Grid::coutPoints(int field) 

{

	int i, j, n;



	cout.precision(15);



	// >= 0: write a single field

	if (field >= 0)

	{

		for(j = 0; j < ydim; ++j)  

		{

			for(i = 0; i < xdim; ++i) 

			{

				cout << xorg + i * xsize << SEPARATOR <<

					yorg + j * ysize << SEPARATOR <<

					at(i, j, field) << "\n";

			}

		}

	}

	// -1: write all fields

	else

	{

		for(j = 0; j < ydim; ++j)  

		{

			for(i = 0; i < xdim; ++i) 

			{

				cout << xorg + i * xsize << SEPARATOR <<

					yorg + j * ysize << SEPARATOR;

				for(n = 0; n < fields; ++n) 

					cout << at(i, j, n) << SEPARATOR;

				cout << "\n";

			}

		}

	}

	cout.precision(8); 

}




int Grid::readGrid(char* filename, int field, int header) 

{

	//  Reads in a Grid from the file "filename"

#ifdef AV_GIO_API

	double bnd_box[4], new_bnd_box[4];

	int tmpi, type;

	float MISSINGFLOAT;

#endif

	double txm, tym;

	float tmpf;

	ifstream infile;

	int i, j, in_grid = FALSE, ret_val;

	char tempstr[80];



	// open file(s)

	if (header == ARCINFO || header == SURFER)

	{

		ifstream testfile(filename);

		if (!testfile)

		{

#if (GR_MESSAGES >= 1)

			cerr << "readGrid: Can't open " << filename << " ...\n";

#endif

			return(ERR_FILEOPEN);

		}

		testfile.close();

		infile.open(filename);

		infile.precision(20);

		infile.width(20);

	}



#ifdef AV_GIO_API

	else if (header == ARCVIEW)

	{

		GridIOSetup();

		if ((in_grid = CellLayerOpen(filename, READONLY, ROWIO, 

			&type, &spacing)) < 0 || BndCellRead (filename, bnd_box) < 0)

		{

#if (GR_MESSAGES >= 1)

			cerr << "readGrid: Can't open " << filename << " ...\n";

#endif

			if (!in_grid)

				CellLyrClose(in_grid);

			GridIOExit();

			return(ERR_AV_CLOPEN);

		}

		if(AccessWindowSet(bnd_box, spacing, new_bnd_box) < 0)

		{

#if (GR_MESSAGES >= 1)

			cerr << "readGrid: Could not set window ...\n";

#endif

			CellLyrClose(in_grid);

			GridIOExit();

			return(ERR_AV_AWSET);

		}

	}

#endif



	// read info

	if (!field)

	{

		switch (header)

		{		

			case ARCINFO:

				infile >> tempstr >> xdim;

				infile >> tempstr >> ydim;

				infile >> tempstr >> xorg;

				infile >> tempstr >> yorg;

				infile >> tempstr >> spacing;

				infile >> tempstr >> nodata;

				xsize = ysize = spacing;

				break;



			case SURFER:

				infile >> tempstr;

				infile >> xdim >> ydim;

				infile >> xorg >> txm;

				infile >> yorg >> tym;

				infile >> tmpf >> tmpf;

				xsize = ((txm - xorg + 1) / xdim);

				ysize = ((tym - yorg + 1) / ydim);

				spacing = (xsize + ysize) / 2.0;

				nodata = SURF_VAL;

				break;



#ifdef AV_GIO_API

			case ARCVIEW:

				xorg = bnd_box[0];

				yorg = bnd_box[1];

				xdim = WindowCols();

				ydim = WindowRows();

				nodata = DEF_VAL;

				xsize = ysize = spacing;

				GetMissingFloat(&MISSINGFLOAT);

 				break;

#endif



			default:

				return(ERR_BADTYPE);

		}



		// initialize data structure

		if ((ret_val = setSize(ydim, xdim)) != OK)

		{

			if (header == ARCINFO || header == SURFER)

				infile.close();

#ifdef AV_GIO_API

			else if (header == ARCVIEW)

			{

				CellLyrClose(in_grid);

				GridIOExit();

			}

#endif

			return(ERR_MEMALLOC);

		}

	}

	else

	{

		int txdim, tydim;

		double txorg, tyorg, tspacing, tnodata, txsize, tysize;



		switch (header)

		{

			case ARCINFO:

				infile >> tempstr >> txdim;

				infile >> tempstr >> tydim;

				infile >> tempstr >> txorg;

				infile >> tempstr >> tyorg;

				infile >> tempstr >> tspacing;

				infile >> tempstr >> tnodata;

				txsize = tysize = tspacing;

				break;



			case SURFER:

				infile >> tempstr;

				infile >> txdim >> tydim;

				infile >> txorg >> txm;

				infile >> tyorg >> tym;

				infile >> tmpf   >> tmpf;

				txsize = ((txm - txorg + 1) / txdim);

				tysize = ((tym - tyorg + 1) / tydim);

				tspacing = (txsize + tysize) / 2.0;

				tnodata = nodata;

				break;



#ifdef AV_GIO_API

			case ARCVIEW:

				txorg = bnd_box[0];

				tyorg = bnd_box[1];

				txdim = WindowCols();

				tydim = WindowRows();

				tnodata = DEF_VAL;

				txsize = tysize = tspacing = spacing; // temporary

				GetMissingFloat(&MISSINGFLOAT);

 				break;

#endif



			default:

				return(ERR_BADTYPE);

		}

		if (txdim != xdim || tydim != ydim ||

			txorg != xorg || tyorg != yorg ||

			txsize != xsize || tysize != ysize ||

			tspacing != spacing || tnodata != nodata)

		{

#if (GR_MESSAGES >= 1)

			cerr << "readGrid: grid " << filename << 

				" has a different header ...\n";

#endif

			if (header == ARCINFO || header == SURFER)

				infile.close();

#ifdef AV_GIO_API

			else if (header == ARCVIEW)

			{

				CellLyrClose(in_grid);

				GridIOExit();

			}

#endif

			return(ERR_BADHEADER);

		}

	}



	// read data

	if (header == ARCINFO || header == SURFER)

	{

		for(j = 0; j < ydim; j++) 

			for(i = 0; i < xdim; i++) 

				infile >> at(i, j, field);

		infile.close();

	}

#ifdef AV_GIO_API

	else if (header == ARCVIEW)

	{

		CELLTYPE *row;

		int *irow;

		float *frow;



		if (type != CELLINT && type != CELLFLOAT)

        {

#if (GR_MESSAGES >= 1)

			cerr << "readGrid: unknown cell type\n";

#endif

			CellLyrClose(in_grid);

			GridIOExit();

			return(ERR_BADTYPE);

        }


		// allocate memory for row

		if ((row = (CELLTYPE *)new char[xdim * sizeof(CELLTYPE)]) == (CELLTYPE *)NULL)

        {

#if (GR_MESSAGES >= 1)

			cerr << "readGrid: Could not allocate memory\n";

#endif

			CellLyrClose(in_grid);

			GridIOExit();

			return(ERR_MEMALLOC);

        }		



		irow = (int *)row;

		frow = (float *)row;



		for(j = 0; j < ydim; j++) 

		{

			GetWindowRow(in_grid, j, row);



			for(i = 0; i < xdim; i++) 

			{

				if (type == CELLINT)

				{

					tmpi = irow[i];					

					at(i, j, field) = (tmpi == MISSINGINT ?

						nodata : (double)tmpi);

				}

				else if (type == CELLFLOAT)

				{

					tmpf = frow[i];					

					at(i, j, field) = (tmpf == MISSINGFLOAT ?

						nodata : tmpf);

				}

			}

		}

		// free memory for row

		delete[] row;

		CellLyrClose(in_grid);

		GridIOExit();

	}

#endif



	if (fields <= 0)

		fields = 1;

	

	return(OK);

}




int Grid::writeGrid(char* filename, int field, int header, 

	double zmin, double zmax) 

{

	// saves grid to filename

#ifdef AV_GIO_API

	float MISSINGFLOAT;

	double bnd_box[4], new_bnd_box[4], tmp;

#endif
	char tmpstr[80];

	ofstream outfile;

	int i, j, out_grid = FALSE;



	// open files

	if (header == ARCINFO || header == SURFER || header == TURTLE)

	{

		// check if file exists and if so rename it
		FILE *fp;
		if ((fp = fopen(filename, "r")) != (FILE *)NULL)
		{
			fclose(fp);
			sprintf(tmpstr,"%s.old",filename);
			if ((fp = fopen(tmpstr, "r")) != (FILE *)NULL)
			{
				fclose(fp);
				remove(tmpstr);
			}
			rename(filename, tmpstr);
		}

		ofstream testfile(filename);

		if (!testfile)

		{

#if (GR_MESSAGES >= 1)

			cerr << "writeGrid: Can't create " << filename << " ...\n";

#endif

			return(ERR_FILECREATE);

		}

		testfile.close();

		outfile.open(filename);

		outfile.precision(20);

	}

#ifdef AV_GIO_API

	else if (header == ARCVIEW)

	{

		GridIOSetup();

		// check if file exists and if so rename it
		if (CellLyrExists(filename))

		{

			sprintf(tmpstr,"%s_old",filename);

			if (CellLyrExists(tmpstr))

				GridDelete(tmpstr);

			GridRename(filename, tmpstr);

#if (GR_MESSAGES >= 1)

			cerr << "writeGrid: Existing " << filename << 

				" has been renamed " << tmpstr << " ...\n";

#endif

			GridDelete(filename);

		}

		// set bounding box to xll, yll, xur, yur

		bnd_box[0] = xorg;

		bnd_box[1] = yorg;

		bnd_box[2] = xorg + xdim * spacing;

		bnd_box[3] = yorg + ydim * spacing;

		if((out_grid = CellLayerCreate(filename, WRITEONLY, 

			ROWIO, CELLFLOAT, spacing, bnd_box)) < 0)

		{

#if (GR_MESSAGES >= 1)

			cerr << "writeGrid: Can't create " << 

				filename << " ...\n";

#endif

			CellLyrClose(out_grid);

			GridIOExit();

			return(ERR_AV_CLCREATE);

		}

		if(AccessWindowSet(bnd_box, spacing, new_bnd_box) < 0)

		{

#if (GR_MESSAGES >= 1)

			cerr << "writeGrid: Could not set window ...\n";

#endif

			CellLyrClose(out_grid);

			GridIOExit();

			return(ERR_AV_AWSET);

		}

	}

#endif



	switch (header)

	{

		case ARCINFO:

			outfile << "ncols "			<< xdim		<< "\n";

			outfile << "nrows "			<< ydim		<< "\n";

			outfile << "xllcorner "		<< xorg		<< "\n";

			outfile << "yllcorner "		<< yorg		<< "\n";

			outfile << "cellsize "		<< spacing	<< "\n";

			outfile << "NODATA_value "	<< nodata	<< "\n";

			outfile.precision(8);  

			break;



		case SURFER:

			if (zmin == 0.0 && zmax == 0.0)

			{

				zmin = findMin(field);

				zmax = findMax(field);

#if (GR_MESSAGES >= 1)

				cerr << "min: " << zmin << ", max: " << 

					zmax << "\n";

#endif

			}

			outfile << "DSAA\n";

			outfile << xdim << SEPARATOR << ydim				<< "\n";

			outfile << xorg << SEPARATOR << xorg + xdim * xsize - 1 << "\n";

			outfile << yorg << SEPARATOR << yorg + ydim * ysize - 1 << "\n";

			outfile << zmin << SEPARATOR << zmax				<< "\n";

			outfile.precision(8);  

			break;

		case TURTLE:

			outfile << "/** number of entries */\nindex {3,DEM}\n";

			outfile << "/** pixel size */\n1: float array dem {" << spacing << "," << spacing << "}\n";

			outfile << "/** nodata value */\n2: float array novalues {" << nodata << "," << nodata << "}\n";

			outfile << "/** grid values */\n3: double matrix dem_values {" << xdim << "," << ydim << "}\n";

			break;



#ifdef AV_GIO_API

		case ARCVIEW:

			GetMissingFloat(&MISSINGFLOAT);

			break;

#endif



		default:

			return(ERR_BADTYPE);

	}



	// write data

	if (header == ARCINFO || header == SURFER)

	{

		for(j = 0; j < ydim; j++)  

		{

			for(i = 0; i < xdim; i++) 

				outfile << at(i, j, field) << "\t";

			outfile << "\n";

		}

		outfile.close();

	}

#ifdef AV_GIO_API

	else if (header == ARCVIEW)

	{

		CELLTYPE *row;

		float *frow;



		// allocate memory for row

 		if ((row = (CELLTYPE *)new char[xdim * sizeof(CELLTYPE)]) == (CELLTYPE *)NULL)

        {

#if (GR_MESSAGES >= 1)

			cerr << "writeGrid: Could not allocate memory\n";

#endif

			CellLyrClose(out_grid);

			GridIOExit();

			return(ERR_MEMALLOC);

        }		

		

		for(j = 0, frow = (float *)row; j < ydim; j++)  

		{

			for(i = 0; i < xdim; i++)

				frow[i] = (float)((tmp = at(i, j, field)) == 

					nodata ? (double)MISSINGFLOAT : tmp);

			PutWindowRow(out_grid, j, row);

		}

		// release memory for row

		delete[] row;

		CellLyrClose(out_grid);

		GridIOExit();

	}

#endif



	return(OK);

}




//

// Initialization

//



int Grid::initGrid(int rows, int cols, int init, double csize, 

	double value, int field) 

{

	int ret_val;



	if ((ret_val = setSize(rows, cols, init)) != OK)

		return(ret_val);

	setValue(value, field);	

	spacing = xsize = ysize = csize;

	timeseed();

	return(OK);

}



int Grid::initGrid(Grid &z, double value, int field) 

{

	int ret_val;



	if ((ret_val = setSize(z)) != OK)

		return(ret_val);



	setValue(value, field);

	timeseed();

	return(OK);

}



int Grid::initGrid(Grid &z) 

{

	int i, j, k, ret_val;



	if ((ret_val = setSize(z)) != OK)

		return(ret_val);



	for (k = 0; k < fields; ++k)

  		for (i = 0; i < xdim; ++i) 

    			for (j = 0; j < ydim; ++j)

      				at(i,j,k) = z.at(i,j,k);

	timeseed();

	return(OK);

}



//

// Access

//



double& Grid::at(int i, int j, int field) 

{

	if (i >= xdim || i < 0 || j >= ydim || j < 0 || field < 0 || field >= fields)

		return(nodata);

	return(arrValues[field][i + (j * xdim)]);

}



double& Grid::at(int field) 

{

	return(arrValues[field][xcur + (ycur * xdim)]);

}



double& Grid::at(Direction dir, int field) 

{

	return(at(xcur, ycur, dir, field));

}



double& Grid::at(int i, int j, Direction dir, int field) 

{

	switch (dir) 

  	{

  		case R:

    			return(at(i+1, j, field));

    			break;

  		case L:

    			return(at(i-1, j, field));

    			break;

  		case U: 

    			return(at(i, j+1, field));

    			break;

  		case D:

    			return(at(i, j-1, field));

    			break;

  		case UR: 

    			return(at(i+1, j+1, field));

    			break;

  		case DL:

    			return(at(i-1, j-1, field));

    			break;

  		case DR: 

    			return(at(i+1, j-1, field));

    			break;

  		case UL:

    			return(at(i-1, j+1, field));

    			break;

		default:

#if (GR_MESSAGES >= 1)

			cerr << "at: invalid direction ...\n";

#endif

			return(at(i, j, field));

			break;

  	}

}



double& Grid::get(int i, int j, Direction dir, int field) 	// For compatibility

{

	return(at(i, j, dir, field));

}



double& Grid::goTo(int i, int j, int field) 

{

	setCurPos(i, j); 

	return(at(field));

}



double& Grid::goTo(Direction dir, int field) 

{

	return(goTo(xcur, ycur, dir, field));

}



double& Grid::goTo(int i, int j, Direction dir, int field) 

{

	switch (dir) 

  	{

  		case R:

    		return(goTo(i+1, j));

		case L:

			return(goTo(i-1, j));

		case U: 

			return(goTo(i, j+1));

		case D:

    		return(goTo(i, j-1));

		case UR: 

    		return(goTo(i+1, j+1));

		case DL:

			return(goTo(i-1, j-1));

		case DR: 

			return(goTo(i+1, j-1));

		case UL:

			return(goTo(i-1, j+1));

		default:

#if (GR_MESSAGES >= 1)

			cerr << "goTo: invalid direction ...\n";

#endif

			return(nodata);

			break;

	}

}



double& Grid::goToPrev(int field) 

{

	// ATTENTION: do we need a setcurpos since we have prev2?

	setCurPos(xcur, ycur);

	return(goTo(xprev1, yprev1, field));

}



int Grid::get_i(int i, Direction direc) 

{

  	switch (direc) 

  	{

  		case R:

    		return(i+1);

    		break;

  		case L:

    		return(i-1);

    		break;

  		case U: 

    		return(i);

    		break;

  		case D:

    		return(i);

    		break;

  		case UR: 

    		return(i+1);

    		break;

  		case DL:

    		return(i-1);

    		break;

  		case DR: 

    		return(i+1);

    		break;

  		case UL:

    		return(i-1);

    		break;

		default:

#if (GR_MESSAGES >= 1)

			cerr << "get_i: invalid direction ...\n";

#endif

			return(ERR_BADDIREC);

			break;

  	}

}



int Grid::get_i(Direction direc) 

{

  	switch (direc) 

  	{

  		case R:

    		return(xcur+1);

    		break;

  		case L:

    		return(xcur-1);

    		break;

  		case U: 

    		return(xcur);

    		break;

  		case D:

    		return(xcur);

    		break;

  		case UR: 

    		return(xcur+1);

    		break;

  		case DL:

    		return(xcur-1);

    		break;

  		case DR: 

    		return(xcur+1);

    		break;

  		case UL:

    		return(xcur-1);

    		break;		default:

#if (GR_MESSAGES >= 1)

			cerr << "get_i: invalid direction ...\n";

#endif

			return(ERR_BADDIREC);

			break;

  	}

}



int Grid::get_j(int j, Direction direc) 

{

  	switch (direc) 

  	{

  		case R:

    		return(j);

    		break;

  		case L:

    		return(j);

    		break;

  		case U: 

    		return(j+1);

    		break;

  		case D:

    		return(j-1);

    		break;

  		case UR: 

    		return(j+1);

    		break;

  		case DL:

    		return(j-1);

    		break;

  		case DR: 

    		return(j-1);

    		break;

  		case UL:

    		return(j+1);

   			break;		

		

		default:

#if (GR_MESSAGES >= 1)

			cerr << "get_j: invalid direction ...\n";

#endif						

			return(ERR_BADDIREC);

			break;



  	}

}



int Grid::get_j(Direction direc) 

{

  	switch (direc) 

  	{

  		case R:

    		return(ycur);

    		break;

  		case L:

    		return(ycur);

    		break;

  		case U: 

    		return(ycur+1);

    		break;

  		case D:

    		return(ycur-1);

    		break;

  		case UR: 

    		return(ycur+1);

    		break;

  		case DL:

    		return(ycur-1);

    		break;

  		case DR: 

    		return(ycur-1);

    		break;

  		case UL:

    		return(ycur+1);			

			break;		

		

		default:

#if (GR_MESSAGES >= 1)

			cerr << "get_j: invalid direction ...\n";

#endif						

			return(ERR_BADDIREC);

			break;

  	}

}



int Grid::saveCurPos(POS *pos) 

{

	if (pos == (POS *)NULL)

	{

#if (GR_MESSAGES >= 1)

		cerr << "storeCurPos: position not stored!\n";

#endif

		return(ERR_GENERIC);	// must deal with error messages

	}

	pos->xcur = xcur; 

	pos->ycur = ycur;

	pos->xprev1 = xprev1; 

	pos->yprev1 = yprev1; 

	pos->xprev2 = xprev2;

	pos->yprev2 = yprev2;

	return(OK);

}



int Grid::restoreCurPos(POS *pos) 

{

	if (pos == (POS *)NULL)

	{

#if (GR_MESSAGES >= 1)

		cerr << "restoreCurPos: position not restored!\n";

#endif

		return(ERR_GENERIC);	// must deal with error messages

	}

	xcur = pos->xcur; 

	ycur = pos->ycur;

	xprev1 = pos->xprev1; 

	yprev1 = pos->yprev1; 

	xprev2 = pos->xprev2;

	yprev2 = pos->yprev2;

	return(OK);

}



int Grid::setCurPos(int i, int j) 

{

	if (i < 0 || i >= xdim || j < 0 || j >= ydim)

		return(ERR_BADVALUE);	// must deal with error messages

	xprev2 = xprev1;

	yprev2 = yprev1;

	xprev1 = xcur; 

	yprev1 = ycur; 

	xcur = i; 

	ycur = j;

	return(OK);

}



int Grid::setCurPos(Grid *grid) 

{

	if (grid->xcur < 0 || grid->xcur >= xdim || 

		grid->ycur < 0 || grid->ycur >= ydim)

		return(ERR_BADVALUE);	// must deal with error messages

	xprev2 = xprev1;

	yprev2 = yprev1;

	xprev1 = xcur; 

	yprev1 = ycur; 

	xcur = grid->xcur; 

	ycur = grid->ycur;

	return(OK);

}



double Grid::getLocMin(int i, int j, int field)	// returns min val in 8 directions

{

	double val, val0 = at(i, j, field);

	int h;



	for (h = 0; h < 8; ++h)

	{

		if (isValue(i, j, (Direction)h, field) &&

			(val = at(i, j, (Direction)h, field)) < val0)

			val0 = val;

	}

	return(val);

}



double Grid::getLocMin(int field)

{

	return(getLocMin(xcur, ycur, field));

}



double Grid::getLocMax(int i, int j, int field)	// returns max val in 8 directions

{

	double val, val0 = at(i, j, field);

	int h;



	for (h = 0; h < 8; ++h)

	{

		if (isValue(i, j, (Direction)h, field) &&

			(val = at(i, j, (Direction)h, field)) > val0)

			val0 = val;

	}

	return(val);

}



double Grid::getLocMax(int field)

{

	return(getLocMax(xcur, ycur, field));

}



double Grid::findMax(int field, int xinit, int yinit, int backw)

{

	int i, j;

	double value = nodata, maxval = -10.0e+100;



	if (setCurPos(xinit, yinit) != OK)

		return(nodata);

	for (i = xcur; backw ? i >= 0 : i < xdim; backw ? --i : ++i)

	{

		for (j = ycur; backw ? j >= 0 : j < ydim; backw ? --j : ++j)

		{

			if ((value = at(i, j, field)) != nodata && 

				value > maxval)

			{

				maxval = value;

				setCurPos(i, j);

			}

		}

	}

	return(maxval);

}





double Grid::findMin(int field, int xinit, int yinit, int backw)

{

	int i, j;

	double value = nodata, minval = 10.0e+100;



	if (setCurPos(xinit, yinit) != OK)

		return(nodata);

	for (i = xcur; backw ? i >= 0 : i < xdim; backw ? --i : ++i)

	{

		for (j = ycur; backw ? j >= 0 : j < ydim; backw ? --j : ++j)

		{

			if ((value = at(i, j, field)) != nodata && 

				value < minval)

			{

				minval = value;

				setCurPos(i, j);

			}

		}

	}

	

	return(minval);

}





int Grid::findCell(double value, int backw, int xinit, int yinit, int field)

{

	int i, j;



	if (xinit < 0 || xinit >= xdim || yinit < 0 || yinit >= ydim ||

		field < 0 || field >= fields)

		return(ERR_BADVALUE);	// must deal with error messages



	for (i = xinit; backw ? i >= 0 : i < xdim; backw ? --i : ++i)

	{

		for (j = yinit; backw ? j >= 0 : j < ydim; backw ? --j : ++j)

		{

			if (at(i, j, field) == value)

			{

				return(setCurPos(i, j));

			}

		}

	}

	

	return(ERR_NOTFOUND);

}



int Grid::findNext(double value, int backw, int field)

{

	return(findCell(value, backw, xcur, ycur, field));

}


int Grid::isDiag(Direction dir, int field)
{
	return((dir == U || dir == D || dir == R || dir == L) ? 
		FALSE : TRUE);
}


int Grid::isValue(int field)

{

	return(at(field) == nodata ? FALSE : TRUE);

}



int Grid::isValue(Direction dir, int field)

{

	return(at(dir, field) == nodata ? FALSE : TRUE);

}



int Grid::isValue(int i, int j, int field)

{

	return(at(i, j, field) == nodata ? FALSE : TRUE);

}



int Grid::isValue(int i, int j, Direction dir, int field)

{

	return(at(i, j, dir, field) == nodata ? FALSE : TRUE);

}



//

// Operators

//



Grid& Grid::operator = (Grid& v) 

{

	int i, j, k;



  	xdim = v.xdim;

  	ydim = v.ydim;

  	xorg = v.xorg;

  	yorg = v.yorg;

  	spacing = v.spacing;

  	xsize = v.xsize;

  	ysize = v.ysize;

  	nodata = v.nodata;

	setSize(v);

	for (k = 0; k < fields; ++k)

  		for(i = 0; i < xdim; i++) 

    			for(j = 0; j < ydim; j++) 

      				at(i,j,k) = v.at(i,j,k);

  	return(*this);

}



Grid& Grid::operator = (double num) 

{

  	for(int i = 0; i < xdim; i++) 

    		for(int j = 0; j < ydim; j++) 

      			at(i,j) = num;

	return(*this);

}



Grid& Grid::operator -= (double num) 

{

  	for(int i = 0; i < xdim; i++) 

    		for(int j = 0; j < ydim; j++) 

      			at(i,j) = at(i,j) - num;

  	return(*this);

}



Grid& Grid::operator += (double num) 

{

  	for(int i = 0; i < xdim; i++) 

    		for(int j = 0; j < ydim; j++) 

      			at(i,j) = at(i,j) + num;

  	return(*this);

}



Grid& Grid::operator *= (double num) 

{

  	for(int i = 0; i < xdim; i++) 

    		for(int j = 0; j < ydim; j++) 

      			at(i,j) = at(i,j) * num;

  	return(*this);

}



Grid& Grid::operator /= (double num) 

{

  	for(int i = 0; i < xdim; i++) 

    		for(int j = 0; j < ydim; j++) 

      			at(i,j) = at(i,j) / num;

  	return(*this);

}



Grid& Grid::operator - (Grid& v) 

{

  	if(xdim != v.xdim || ydim != v.ydim) 

  	{

    		cerr << "Error- Attempt to perform subtraction on grids of unequal dimension.\n";

    		exit(ERR_BADHEADER);

  	}

  	for (int k = 0; k < fields; ++k) 

	{

  		for (int i = 0; i < xdim; i++) 

  		{

    		for (int j = 0; j < ydim; j++) 

    		{

      			if (at(i,j) == nodata || v.at(i,j) == v.nodata) 

      			{

					at(i,j) = nodata;

      			}

      			else 

      			{

					at(i,j) = at(i,j) - v.at(i,j);

      			}

    		}

  		}

	}

  	return(*this);

}



Grid& Grid::operator + (Grid& v) 

{

  	if(xdim != v.xdim || ydim != v.ydim) 

  	{

    		cerr << "Error- Attempt to perform subtraction on grids of unequal dimension.\n";

    		exit(ERR_BADHEADER);

  	}

	for (int k = 0; k < fields; ++k)

	{

  		for(int i = 0; i < xdim; i++) 

  		{

    		for(int j = 0; j < ydim; j++) 

    		{

      			if(at(i,j,k) == nodata || v.at(i,j,k) == v.nodata) 

      			{

					at(i,j,k) = nodata;

      			}

      			else 

      			{

					at(i,j,k) = at(i,j,k) + v.at(i,j,k);

      			}

    		}

  		}

	}

  	return(*this);

}



Grid& Grid::setBorders(double val, int field) 

{

  	for(int i = 0; i < xdim; i++) 

  	{

    	at(i, 0, field) = val;

    	at(i, ydim - 1, field) = val;

  	}

  	for(int j = 0; j < ydim; j++) 

  	{

    	at(0, j, field) = val;

    	at(xdim - 1, j, field) = val;

  	}

  	return(*this);

}



// Random number generation

#define RANDOM_IA		16807

#define RANDOM_IM		2147483647

#define RANDOM_AM		(1.0 / RANDOM_IM)

#define RANDOM_IQ		127773

#define RANDOM_IR		2836

#define RANDOM_RMASK	123459876



double Grid::dRandino()

{

	long l;

	double num;



	seedino ^= RANDOM_RMASK;

	l = seedino / RANDOM_IQ;

	seedino = RANDOM_IA * (seedino - l * RANDOM_IQ) - RANDOM_IR * l;

	if (seedino < 0L)

		seedino += RANDOM_IM;

	num = RANDOM_AM * seedino;

	seedino ^= RANDOM_RMASK;

	return(num);

}



int  Grid::getRandInt(int n)   

{

	return(int((dRandino()  * (double)n) + 0.5));

}



long Grid::getRandLong(long n) 

{

	return(long((dRandino() * (double)n) + 0.5));

}



void  Grid::doRndGrd(int hits, int range, int unique, int field)

{

	int n, rx, ry, rn;



	if (!hits || (unique && (hits > xdim * ydim)))

		hits = xdim * ydim;



	for (n = 0; n < hits; ++n)

	{

		// get a random number between 0 and xdim - 1

		rx = getRandInt(xdim - 1);

		// get a random number between 0 and ydim - 1

		ry = getRandInt(ydim - 1);

		// already assigned?

		if (unique)

		{

			if (isValue(rx, ry, field))

			{

				--n;

				continue;

			}

		}

		// get a random number between 1 and range

		rn = getRandInt(range - 1) + 1;

		// assign value

		at(rx, ry, field) = rn;

	}

}



void Grid::doRndGrd(int hits, int range, int unique, int infield, int outfield)

{

	int n, rx, ry, rn;



	for (n = 0; n < hits; ++n)

	{

		// get a random number between 0 and xdim - 1

		rx = getRandInt(xdim - 1);

		// get a random number between 0 and ydim - 1

		ry = getRandInt(ydim - 1);

		// already assigned?

		if (!(isValue(rx, ry, infield)) || 

			(unique && isValue(rx, ry, outfield)))

		{

			--n;

			continue;

		}

		// get a random number between 1 and range

		rn = getRandInt(range - 1) + 1;

		// assign value

		at(rx, ry, outfield) = rn;

	}

}



// Sink removal functions



void Grid::removeSinks(double inc, int maxpasses, int neighbors,

	int field, int disp)

{

	int cnt = 0, snk = 0;



	if (disp >= 1)

		cerr << "Removing sinks (max. passes: " <<

			maxpasses << ") ...\n";



	while (maxpasses--)

	{

		if (disp >= 1)

			cerr << "Removing sinks, pass number: " <<

				++cnt << "\n";



		snk = remSinksPass(inc, neighbors, field);

#if (RS_CHECKSINKS == 1)

 		if (!snk)

			break;

#else 

#if (RS_CHECKSINKS == 2)

		if (disp >= 1)

			cerr << "Checking for sinks ...\n";

		if (checkForSinks(field, FALSE))

			break;

#endif

#endif

	}

	if (disp >= 1)

		cerr << "Done removing sinks ...\n";

}



int Grid::remSinksPass(double addinc, int neighbors, int field, int disp) 

{

	// This is a simple Sink removal procedure.

	// Remove sinks by adding elevation to all sinks

	// untill there are no sinks left in the grid.

  	int i, j, stillSinks = 0, perc_comp = 0, last_comp = 0, 

		cnt = 0, tot = xdim * ydim;



	if (disp >= 1)

		cerr << "Completed with: " << perc_comp << " %\r";



    	for (j = 0; j < ydim; ++j) 

	{ 

      		for (i = 0; i < xdim; ++i) 

		{

			if (disp >= 1)

			{

				// Let the user know how far we are

				perc_comp = int(((double)(++cnt) /

					(double)tot) * 100.0);

				if (last_comp != perc_comp)

				{

					last_comp = perc_comp;

					cerr << "Completed with: " <<

						perc_comp << " %\r";

				}

			}

			if (isSink(i, j, field)) 

			{

	  			++stillSinks;

	 			at(i, j, field) = sinkAddValue(i, j, field) + addinc;

				if (neighbors)

	  				checkSinkNeighbors(i, j, addinc, field);

			}

      		}

    	}

	if (disp >= 1)

    		cerr << "\nNumber of sinks: " << stillSinks << "\n";



	return(stillSinks);

}



void Grid::checkSinkNeighbors(int i, int j, double addinc, int field) 

{

	if (isSink(i - 1, j, field)) 

	{

		at(i - 1, j, field) = sinkAddValue(i - 1, j, field) + addinc;

		checkSinkNeighbors(i - 1, j, addinc, field);

	}

	if (isSink(i, j - 1, field)) 

	{

		at(i, j - 1, field) = sinkAddValue(i, j - 1, field) + addinc;

		checkSinkNeighbors(i, j - 1, addinc, field);

	}

	if (isSink(i - 1, j - 1, field)) 

	{

		at(i - 1, j - 1, field) = sinkAddValue(i - 1, j - 1, field) + addinc;

		checkSinkNeighbors(i - 1, j - 1, addinc, field);

	}

	if (isSink(i + 1, j + 1, field)) 

	{

		at(i + 1, j + 1, field) = sinkAddValue(i + 1, j + 1, field) + addinc;

		checkSinkNeighbors(i + 1, j + 1, addinc, field);

	}

	if (isSink(i + 1, j, field)) 

	{

		at(i + 1, j, field) = sinkAddValue(i + 1, j, field) + addinc;

		checkSinkNeighbors(i + 1, j, addinc, field);

	}

	if (isSink(i, j + 1, field)) 

	{

		at(i, j + 1, field) = sinkAddValue(i, j + 1, field) + addinc;

		checkSinkNeighbors(i, j + 1,addinc, field);

	}

	if (isSink(i + 1, j - 1, field)) 

	{

		at(i + 1, j - 1, field) = sinkAddValue(i + 1, j - 1, field) + addinc;

		checkSinkNeighbors(i + 1, j - 1, addinc, field);

	}

	if (isSink(i - 1, j + 1, field)) 

	{

		at(i - 1, j + 1, field) = sinkAddValue(i - 1, j + 1, field) + addinc;

		checkSinkNeighbors(i - 1, j + 1, addinc, field);

	}

}





double Grid::sinkAddValue(int i, int j, int field) 

{

	double minneigh = get(i, j, L, field);

	for (int k = 0; k < 8; ++k) 

		if (get(i, j, (Direction)k, field) < minneigh) 

			minneigh = get(i, j, (Direction)k, field);

  	return(minneigh);

}



int Grid::isSink(int i, int j, int field)

{

	if (at(i,     j, field) == nodata ||

		at(i-1,   j, field) == nodata ||

		at(i+1,   j, field) == nodata ||

		at(i,   j-1, field) == nodata ||

		at(i,   j+1, field) == nodata ||

		at(i+1, j+1, field)     == nodata ||

		at(i-1, j-1, field) == nodata ||

		at(i-1, j+1, field) == nodata ||

		at(i+1, j-1, field) == nodata)

	{

                return(FALSE);

	}

 

	if (at(i, j, field) <= at(i-1,   j, field) &&

		at(i, j, field) <= at(i+1,   j, field) &&

		at(i, j, field) <= at(i,   j-1, field) &&

		at(i, j, field) <= at(i,   j+1, field) &&

		at(i, j, field) <= at(i+1, j+1, field) &&

		at(i, j, field) <= at(i-1, j+1, field) &&

		at(i, j, field) <= at(i+1, j-1, field) &&

		at(i, j, field) <= at(i-1, j-1, field))

	{

		return(TRUE);

	}

	else

	{

		return(FALSE);

	}

}

 

int Grid::checkForSinks(int field, int disp)
// if disp == 0 the function returns true  or false
// if disp > 0 the function returns true  or false and displays messages
// if disp == -1 the function returns the number of sinks
// if disp < -1 the function returns the number of sinks and displays messages
{

	int totsinks = 0, perc_comp = 0, last_comp = 0,

		cnt = 0, tot = xdim * ydim;



	for (int j = 0; j < ydim; j++)

	{

		for(int i = 0; i < xdim; i++)

		{

			if (disp >= 2 || disp < -2)

			{

				// Let the user know how far we are

				perc_comp = int(((double)(++cnt) /

					(double)tot) * 100.0);



				if (last_comp != perc_comp)

				{

					last_comp = perc_comp;

					cerr << "Completed with: " << perc_comp << " %\r";

				}

			}



			if (isSink(i, j, field))

			{

				if (!disp)

					return(FALSE);

				++totsinks;

				if (disp >= 3 || disp < -3)

					cerr << "Sink found in elevation grid at(" << i << ", " << j << ") ...\n";

			}

		}

	}

 

	if (disp >= 2 || disp < -2)

		cerr << "\n";



	if (totsinks)

	{

		if  (disp >= 1 || disp < -1)
			
			cerr << "checkForSinks: " << totsinks << " sinks found in elevation grid!\n";

		return(disp >= 0 ? FALSE : totsinks);

	}
 
	return(disp >= 0 ? TRUE : FALSE);

}

// Sink removal functions for channels (modified to consider nodata as a "wall")

void Grid::removeSinks2(double inc, int maxpasses, int neighbors,
	int field, int disp)
{
	int cnt = 0, snk = 0;

	if (disp >= 1)
		cerr << "Removing sinks (max. passes: " <<
			maxpasses << ") ...\n";

	while (maxpasses--)
	{
		if (disp >= 1)
			cerr << "Removing sinks, pass number: " <<
				++cnt << "\n";

		snk = remSinksPass2(inc, neighbors, field);
#if (RS_CHECKSINKS == 1)
 		if (!snk)
			break;
#else 
#if (RS_CHECKSINKS == 2)
		if (disp >= 1)
			cerr << "Checking for sinks ...\n";
		if (checkForSinks2(field, FALSE))
			break;
#endif
#endif
	}
	if (disp >= 1)
		cerr << "Done removing sinks ...\n";
}

int Grid::remSinksPass2(double addinc, int neighbors, int field, int disp) 
{
	// This is a simple Sink removal procedure.
	// Remove sinks by adding elevation to all sinks
	// untill there are no sinks left in the grid.
  	int i, j, stillSinks = 0, perc_comp = 0, last_comp = 0, 
		cnt = 0, tot = xdim * ydim;

	if (disp >= 1)
		cerr << "Completed with: " << perc_comp << " %\r";

    	for (j = 0; j < ydim; ++j) 
	{ 
      		for (i = 0; i < xdim; ++i) 
		{
			if (disp >= 1)
			{
				// Let the user know how far we are
				perc_comp = int(((double)(++cnt) /
					(double)tot) * 100.0);
				if (last_comp != perc_comp)
				{
					last_comp = perc_comp;
					cerr << "Completed with: " <<
						perc_comp << " %\r";
				}
			}
			if (isSink2(i, j, field)) 
			{
	  			++stillSinks;
	 			at(i, j, field) = sinkAddValue(i, j, field) + addinc;
				if (neighbors)
	  				checkSinkNeighbors2(i, j, addinc, field);
			}
      		}
    	}
	if (disp >= 1)
    		cerr << "\nNumber of sinks: " << stillSinks << "\n";

	return(stillSinks);
}

void Grid::checkSinkNeighbors2(int i, int j, double addinc, int field) 
{
	if (isSink2(i - 1, j, field)) 
	{
		at(i - 1, j, field) = sinkAddValue(i - 1, j, field) + addinc;
		checkSinkNeighbors2(i - 1, j, addinc, field);
	}
	if (isSink2(i, j - 1, field)) 
	{
		at(i, j - 1, field) = sinkAddValue(i, j - 1, field) + addinc;
		checkSinkNeighbors2(i, j - 1, addinc, field);
	}
	if (isSink2(i - 1, j - 1, field)) 
	{
		at(i - 1, j - 1, field) = sinkAddValue(i - 1, j - 1, field) + addinc;
		checkSinkNeighbors2(i - 1, j - 1, addinc, field);
	}
	if (isSink2(i + 1, j + 1, field)) 
	{
		at(i + 1, j + 1, field) = sinkAddValue(i + 1, j + 1, field) + addinc;
		checkSinkNeighbors2(i + 1, j + 1, addinc, field);
	}
	if (isSink2(i + 1, j, field)) 
	{
		at(i + 1, j, field) = sinkAddValue(i + 1, j, field) + addinc;
		checkSinkNeighbors2(i + 1, j, addinc, field);
	}
	if (isSink2(i, j + 1, field)) 
	{
		at(i, j + 1, field) = sinkAddValue(i, j + 1, field) + addinc;
		checkSinkNeighbors2(i, j + 1,addinc, field);
	}
	if (isSink2(i + 1, j - 1, field)) 
	{
		at(i + 1, j - 1, field) = sinkAddValue(i + 1, j - 1, field) + addinc;
		checkSinkNeighbors2(i + 1, j - 1, addinc, field);
	}
	if (isSink2(i - 1, j + 1, field)) 
	{
		at(i - 1, j + 1, field) = sinkAddValue(i - 1, j + 1, field) + addinc;
		checkSinkNeighbors2(i - 1, j + 1, addinc, field);
	}
}

int Grid::checkForSinks2(int field, int disp)
{
        int totsinks = 0, perc_comp = 0, last_comp = 0,
                cnt = 0, tot = xdim * ydim;
 
        for (int j = 0; j < ydim; j++)
        {
                for(int i = 0; i < xdim; i++)
                {
			if (disp >= 2)
			{
                        	// Let the user know how far we are
                        	perc_comp = int(((double)(++cnt) /
					(double)tot) * 100.0);

                        	if (last_comp != perc_comp)
                        	{
                               		last_comp = perc_comp;
                               	 	cerr << "Completed with: "
						<< perc_comp << " %\r";
                        	}
			}

                        if (isSink2(i, j, field))
                        {
                                if (!disp)
                                        return(0);
                                ++totsinks;
                                if (disp == 3)
                                        cerr << "Sink found in elevation grid at (" << i
                                                << ", " << j << ") ...\n";
                        }
                }
        }
 
	if (disp >= 2)
		cerr << "\n";

        if (disp && totsinks)
        {
                cerr << "checkForSinks2: " << totsinks <<
                        " sinks found in elevation grid!\n";
                return(0);
        }
 
        return(1);
}

int Grid::isSink2(int i, int j, int field) // modified for edges
{
	int k, vals = 0;

	if (!isValue(i, j, field))
		return(0);

	for (k = 0; k < 8; ++k)
	{
		if (isValue(i, j, (Direction)k, field))
		{
			if (at(i, j, (Direction)k, field) < at(i, j, field))
				return(0);
			++vals;
		}
	}
	return(vals > 1 ? 1 : 0); // try to exclude outlet
}
