//  -*- Mode: C++;  -*-
//  File: stabfun.cc
//  Author: Dino Bellugi (dino@geomorph.berkeley.edu)
//  Copyright  Dino Bellugi, BlueG SoftWear, U.C. Berkeley, 1999 (C)
// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// * FUNCTION: main interface for shalstab library. functions include
// *           sink removal, computation of A/b, slope, q/T, curvature
// *
// * RELATED PACKAGES: area_class.cc grid.cc gridedges.cc
// *
// * HISTORY:
// * Created:  Thu Apr  1 18:15:00 1999 (dino)
// * Modified: Mon May 10 17:40:00 1999 (dino)
// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

// the following include file (notice.h) contains important copyright 
// information, as well as other legal statements. it must be present 
// in all files and in any distribution. removal or modification is
// strictly forbidden and violates u.s. and international copyright law.
#include "notice.h"
//
#include <stdlib.h>
#include <stdio.h>
#include <fstream.h>
#include "grid.h"
#include "area_class.h"
#include "stabfun.h"
#include "macros.h"
#include "display.h"

// Removes Sinks
void calcRemoveSinks(Grid *grd, int rem_field, double soil_incr,
	int rem_max_pass, int disp_mode)
{
        if (!grd->checkForSinks(rem_field, disp_mode))
        	grd->removeSinks(soil_incr, rem_max_pass, RS_NEIGHBORS,
			rem_field, disp_mode);
}

int do_RemoveSinks(char *elv_grid, char *rem_grid, int in_grid_mode,
	int out_grid_mode, double soil_incr, int rem_max_pass,
	int disp_mode)
{
	Grid grd;

	doDisp("Reading elevation grid ...\n", disp_mode);

        if (!grd.readGrid(elv_grid, 0, in_grid_mode))
	{
		doDisp("Error reading elevation grid!", disp_mode);
		return(FALSE);
	}

        if (!grd.addField(0))
        {
                doDisp("Insufficient memory!\n", disp_mode);
                return(FALSE);
        }
        
	calcRemoveSinks(&grd, 1, soil_incr, rem_max_pass, disp_mode);
 
        doDisp("Writing elevation grid ...\n", disp_mode);

	if (rem_grid == (char *)NULL)
		grd.coutGrid(1, out_grid_mode);

        else if (!grd.writeGrid(rem_grid, 1, out_grid_mode))
        {
                doDisp("Error writing sink removed grid!", disp_mode);
                return(FALSE);
        }

        return(1); 
}
 

// Calculates Slope 
void calcSlope(Grid *grd, int rem_field, int slp_field, 
	int slp_mode, int disp_mode)
{
        int i, j;
 
        doDisp("Computing slope ...\n", disp_mode);

        for (j = 0; j < grd->ydim; j++)
                for (i = 0; i < grd->xdim; i++)
                        grd->at(i, j, slp_field) =
				grd->getSlope(i, j, rem_field, slp_mode);
}
 
int do_Slope(char *rem_grid, char *slp_grid, int in_grid_mode,
        int out_grid_mode, int slp_mode, int disp_mode)
{
        Grid grd;
 
        doDisp("Reading elevation grid ...\n", disp_mode);

        if (!grd.readGrid(rem_grid, 0, in_grid_mode))
        {
                doDisp("Error reading elevation grid!", disp_mode);
                return(FALSE);
        }
 
        if (!grd.addField())
        {
                doDisp("Insufficient memory!\n", disp_mode);
                return(FALSE);
        }
 
        calcSlope(&grd, 0, 1, slp_mode, disp_mode);
 
        doDisp("Writing slope grid ...\n", disp_mode);
 
        if (slp_grid == (char *)NULL)
                grd.coutGrid(1, out_grid_mode);
 
        else if (!grd.writeGrid(slp_grid, 1, out_grid_mode))
        {
                doDisp("Error writing slope grid!", disp_mode);
                return(FALSE);
        }
 
        return(1);
}

// Calculates Curvature
void calcCurvature(Grid *grd, int rem_field, int crv_field, int disp_mode)
{
        int i, j;
 
        doDisp("Computing curvature ...\n", disp_mode);
 
        for (j = 0; j < grd->ydim; j++)
                for (i = 0; i < grd->xdim; i++)
                        grd->at(i, j, crv_field) =
                                grd->getCurvature(i, j, rem_field);
}
 
int do_Curvature(char *rem_grid, char *crv_grid, int in_grid_mode,
        int out_grid_mode, int disp_mode)
{
        Grid grd;
 
        doDisp("Reading elevation grid ...\n", disp_mode);
 
        if (!grd.readGrid(rem_grid, 0, in_grid_mode))
        {
                doDisp("Error reading elevation grid!", disp_mode);
                return(FALSE);
        }
 
        if (!grd.addField())
        {
                doDisp("Insufficient memory!\n", disp_mode);
                return(FALSE);
        }
 
        calcCurvature(&grd, 0, 1, disp_mode);
 
        doDisp("Writing curvature grid ...\n", disp_mode);
 
        if (crv_grid == (char *)NULL)
                grd.coutGrid(1, out_grid_mode);
 
        else if (!grd.writeGrid(crv_grid, 1, out_grid_mode))
        {
                doDisp("Error writing curvature grid!", disp_mode);
                return(FALSE);
        }
 
        return(1);
}
 
// Calculates Area
void calcArea(Grid *grd, int rem_field, int area_field,
	int area_mode, int check_sinks, int disp_mode)
{
	Contributing_Area totalArea(grd, rem_field, area_field,
		area_mode, check_sinks, disp_mode);
}

int do_Area(char *rem_grid, char *area_grid, int in_grid_mode,
        int out_grid_mode, int area_mode, int check_sinks, int disp_mode)
{
        Grid grd;
 
        doDisp("Reading sink removed grid ...\n", disp_mode);

        if (!grd.readGrid(rem_grid, 0, in_grid_mode))
        {
                doDisp("Error reading sink removed grid!", disp_mode);
                return(FALSE);
        }
 
        if (!grd.addField())
        {
                doDisp("Insufficient memory!\n", disp_mode);
                return(FALSE);
        }
 
        calcArea(&grd, 0, 1, area_mode, check_sinks, disp_mode);
 
        doDisp("Writing area grid ...\n", disp_mode);
 
        if (area_grid == (char *)NULL)
                grd.coutGrid(1, out_grid_mode);
 
        else if (!grd.writeGrid(area_grid, 1, out_grid_mode))
        {
                doDisp("Error writing area grid!", disp_mode);
                return(FALSE);
        }
 
        return(1);
}

 
// Calculates q/T
void calcQoverT(Grid *grd, int slp_field, int area_field,
	int qt_field, double qt_ps, double qt_phi, int disp_mode)
{
        int i, j;
 
        doDisp("Computing q/T ...\n", disp_mode);
 
        for (j = 0; j < grd->ydim; j++)
                for (i = 0; i < grd->xdim; i++)
                        grd->at(i, j, qt_field) =
                                grd->getQoverT(i, j, slp_field,
					area_field, qt_ps, qt_phi);
}
 
int do_QoverT(char *slp_grid, char *area_grid, char *qt_grid,
        int in_grid_mode, int out_grid_mode, double qt_ps,
        double qt_phi, int disp_mode)
{
        Grid grd;
 
        doDisp("Reading slope grid ...\n", disp_mode);
 
        if (!grd.readGrid(slp_grid, 0, in_grid_mode))
        {
                doDisp("Error reading slope grid!", disp_mode);
                return(FALSE);
        }

        if (!grd.addField() || !grd.addField())
        {
                doDisp("Insufficient memory!\n", disp_mode);
                return(FALSE);
        }

        doDisp("Reading area grid ...\n", disp_mode);
 
        if (!grd.readGrid(area_grid, 1, in_grid_mode))
        {
                doDisp("Error reading area grid!", disp_mode);
                return(FALSE);
        }
 
        calcQoverT(&grd, 0, 1, 2, qt_ps, qt_phi, disp_mode);
 
        doDisp("Writing q/T grid ...\n", disp_mode);
 
        if (qt_grid == (char *)NULL)
                grd.coutGrid(2, out_grid_mode);
 
        else if (!grd.writeGrid(qt_grid, 2, out_grid_mode))
        {
                doDisp("Error writing q/T grid!", disp_mode);
                return(FALSE);
        }
 
        return(1);
}


// Calculates all
int do_Shalstab(char *elv_grid, char *rem_grid, char *slp_grid,
	char *area_grid, char *qt_grid, int in_grid_mode,
	int out_grid_mode, double soil_incr, int rem_max_pass,
	double qt_ps, double qt_phi, int disp_mode)
{
        Grid grd;
	int fields = FIELDS;
 
        doDisp("Reading elevation grid ...\n", disp_mode);
 
        if (!grd.readGrid(elv_grid, ELEV, in_grid_mode))
        {
                doDisp("Error reading elevation grid!", disp_mode);
                return(FALSE);
        }

	while (fields--)
	{
        	if (!grd.addField())
        	{
               	 	doDisp("Insufficient memory!\n", disp_mode);
               	 	return(FALSE);
        	}
	}

	grd.setValue(REMOVE, ELEV);

	calcRemoveSinks(&grd, REMOVE, soil_incr, rem_max_pass, disp_mode);
 
        if (rem_grid != (char *)NULL)
	{
        	doDisp("Writing sink removed grid ...\n", disp_mode);
 
        	if (!grd.writeGrid(rem_grid, REMOVE, out_grid_mode))
        	{
                	doDisp("Error writing sink removed grid!", disp_mode);
                	return(FALSE);
        	}
	}

        calcSlope(&grd, REMOVE, SLOPE, ATAN_RAD, disp_mode);
 
        if (slp_grid != (char *)NULL)
        {
        	doDisp("Writing slope grid ...\n", disp_mode);
 
        	if (!grd.writeGrid(slp_grid, SLOPE, out_grid_mode))
        	{
                	doDisp("Error writing slope grid!", disp_mode);
                	return(FALSE);
        	}
	} 

        calcArea(&grd, REMOVE, AREA, CA_AOVERB, FALSE, disp_mode);
 
        if (area_grid != (char *)NULL)
        {
        	doDisp("Writing area grid ...\n", disp_mode);
 
        	if (!grd.writeGrid(area_grid, AREA, out_grid_mode))
        	{
                	doDisp("Error writing area grid!", disp_mode);
                	return(FALSE);
        	}
	}

        calcQoverT(&grd, SLOPE, AREA, QOVERT, qt_ps, qt_phi, disp_mode);
 
        doDisp("Writing q/T grid ...\n", disp_mode);
 
        if (qt_grid == (char *)NULL)
                grd.coutGrid(QOVERT, out_grid_mode);

        else if (!grd.writeGrid(qt_grid, QOVERT, out_grid_mode))
        {
                doDisp("Error writing q/T grid!", disp_mode);
                return(FALSE);
        }

	return(1);
} 
