//  -*- Mode: C++;  -*-
//  File: soil_class.h
//  Author: Dino Bellugi (dino@alba.geo.berkeley.edu), Rob Reiss
//  Copyright (C) Dino Bellugi 1997
// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// * FUNCTION:  Prediction of Soil Depth Due to Diffusion
// *		New Version, Includes Stream Lines
// * CLASSES: DIFFUSION
// * HISTORY:
// * Created: Wed Apr 30 12:50:38 1997 (dino)
// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#ifndef	hydrosim_DIFFUSION
#define	hydrosim_DIFFUSION

#include "grid.h"

// Switches
#define MESSAGES	1	// 0 = none, <n> = level<n>
#define FLUX_1D 	0	// 0 = none, 1 = boundary, 2 = half, 3 = all
#define PRODUCTION	1	// 0 = no prod., 1 = fixed, 2 = variable
#define LOWERING	2	// 0 = no lowering, 1 = at boundary, 2 = use status
#define TRIMMING	0	// 0 = no trimming, 1 = trimming
#define TRANSPORT	1	// 0 = no transport, 1 = transport
#define STABILITY	0	// 0 = don't use slope stability grid, 1 = use it
#define COUPLING	0	// 0 = independent, 1 = coupled
#define CLIPPING	1	// 0 = don't clip values, 1 = clip
#define ORDER		0	// 0 = l->r, 1 = from divide, 2 = from boundary
#if (ORDER)
#define locX(n)		xOrd[n]
#define locY(n)		yOrd[n]
#endif

// Fields
#define SC_FIELDS		12	// Number of fields in use
#define SC_ELEV		0	// Elevation (initial)
#define SC_SOIL		1	// Soil depth (initial)
#define SC_ZPAST	2	// Elevation 
#define SC_HPAST	3	// Soil depth (previous iteration)
#define SC_HFUTURE	4	// Soil depth
#define SC_EPAST	5	// Bedrock elev. (previous iteration)
#define SC_EFUTURE	6	// Bedrock elevation
#define SC_FLUX		7	// Flux out
#define SC_P_INT	8	// Production function intercept
#define SC_P_EXP	9	// Production function exponent
#define SC_P_MUL	10	// Production function multiplier
#define SC_STATUS	11	// Status for production and lowering
//#define SC_SLOPE	12	// Slope

// Status Flags (Production)
#define P_STND		0	// Standard production parameters
#define P_BLK1		1	// Block production hard then soft
#define P_BLK2		2	// Block production soft then hard (1)
#define P_BLK3		3	// Block production soft then hard (2)
#define P_FLP1_1	4	// Layered production soft then hard (1)
#define P_FLP1_2	5	// Layered production soft then hard (2)
#define P_FLP2_1	6	// Layered production soft then hard (1)
#define P_FLP2_2	7	// Layered production soft then hard (2)
#define P_FLP3_1	8	// 1 layer, soft then hard at elev_var (1)
#define P_FLP3_2	9	// 1 layer, soft then hard at elev_var (2)

// Status Flags (Boundary Conditions)
#define B_LWR_RT1	-1	// Use lowering rate 1
#define B_LWR_RT2	-2	// Use lowering rate 2
#define B_SPL_INF	-3	// Infinite supply
#define B_SPL_LIM	-4	// Limited supply
#define B_REM_INF	-5	// Infinite removal

// Constants
#define ADJ_8DIR	2.0			// Adjustment for 8 directions
#define MAX_PSAV	10	 		// Max number of partial saves 
#define MAX_FSAV	10	 		// Max number of flux saves 
#define MAX_ITER	(MAXINT>>1) 		// Max number of iterations
#define MAX_DT		100			// Max diffusion step
#define MAX_DTP		2000			// Max production step
#define MAX_PT		(MAX_ITER/MAX_PSAV)	// Max partial save iterations
#define MAX_PO		(MAX_PT-1)		// Max partial save offset
#define MAX_FT		(MAX_ITER/MAX_FSAV)	// Max flux save iterations

// Macros
#define srandino(l)	seedino = l
#define timeseed()	{time_t tt; time(&tt); srandino(tt);}

class Diffusion;
typedef double (Diffusion::*Prod_Func)(int, int, int, int); 
typedef double (Diffusion::*Tran_Func)(int, int, int, double); 

class Diffusion 
{
private:
	long	seedino;		// Random number seed
	double	D;			// Diffusion coefficient
	double	Kf;			// Transport coefficient or Critical Slope
	double	Kx;			// Threshold coefficient
	double	Kc;			// Non linearity exponent
	double	rho_ratio;		// Bulk density ratio
	double	low_rate1;		// Boundary lowering rate 1
	double	low_rate2;		// Boundary lowering rate 2
	double	up_rate;		// Uplift rate
	double	depth_ini;		// Initial soil depth (< 0 to use grid)
	int	depth_pert;		// Number of soil depth perturbations
	double	depth_var;		// Threshold soil depth
	double	elev_var;		// Elevation of 1st layer
	double	numIter;		// The number of time iterations 
	int	dT;			// The time step
	int	dTp;			// The production step (mult. of diff.)
	int	pT;			// Partial save interval
	int	pO;			// Partial save offset
	int	fT;			// 1D-Flux save interval
	Grid	*dGrid;			// Grid data structure
	Prod_Func	fPr; 		// Production function
	Tran_Func	fTr; 		// Transport function
	double	*sim_var_args;		// Production fuction coefficients
#if (ORDER)
	int	*xOrd, *yOrd;		// Arrays for x and y ordering
#endif
#if (FLUX_1D == 1)
	double	*flux;			// flux array (for last cell)
#else
#if (FLUX_1D == 2)
	double	**flux;			// flux array (for every cell)
#endif
#endif
	void setTranFunc(char*); 
	void setProdFunc(char*); 
	void AllPropTran();  
	double prodFuncExp(int i, int j, int t, int maxp = 0);	// Exponential
	double prodFuncVar(int i, int j, int t, int maxp = 0);	// Variable
	double prodFuncPoly5(int i, int j, int t, int maxp = 0);// Polynomial
	double prodFuncSto(int i, int j, int t, int maxp = 0);	// Stochastic
	double prodFuncArj(int i, int j, int t, int maxp = 0);	// Exponential
	double prodFuncHum(int i, int j, int t, int maxp = 0);	// Humped
	double prodFuncExpND(int i, int j, int t, int maxp = 0);// Normal Depth
	double tranFuncDif(int i, int j, int k, double dx);	// Diffusive
	double tranFuncDdd(int i, int j, int k, double dx);	// Depth Dep. Diff.
	double tranFuncHow(int i, int j, int k, double dx);	// Howardian
	double tranFuncRoe(int i, int j, int k, double dx);	// Roeringian
	double sDiff(int i, int j, int k, double dx);
	double sDdd(int i, int j, int k, double dx);
	double sSlide(int i, int j, int k, double dx);
	double sJosh(int i, int j, int k, double dx);
//	double getKf(int i, int j);
#if (ORDER)
	void setXord();
	void setYord();
#endif
	int isLowering(int i, int j);
	int isLowering(int i, int j, Direction k);
	int isBoundary(int i, int j);
	int isBoundary(int i, int j, Direction k);
	int isRemoval(int i, int j);
	int isRemoval(int i, int j, Direction k);
	int isSupply(int i, int j);
	int isSupply(int i, int j, Direction k);
	double dRandino(void);
	int doPerturb(int iter = 0, int field = 0, int clip = 0);
public:
	void writeElev(char* name);	// Write out the elevation grid
	void writeDepth(char* name);	// Write out the depth grid
	void writeFlux(char* name);	// Write out the flux grid
	void writeSlope(char* name);	// Write out the flux grid
	void writeBedrock(char* name);	// Write out the bedrock grid
	void write1DFlux(char* name);	// Write out the flux array
	void coutDepth();		// Write out the Soil Depth grid to cout
	int addFlux(double *flux);	// Adds up soil depth on streams
					// Does Diffusion
	Diffusion(Grid &grd, char* tranfunc, double D_in, double Kf_in, double Kx_in, 
		double Kc_in, double init_soil, int pert_soil, double var_soil, 
		double var_elev, double rho_ratio_in, double low_rate1_in,
		double low_rate2_in, double up_rate_in, int num_iter_in, 
		int dt_in, int dtp_in, int pt_in, int po_in, int ft_in, 
		char* prodfunc, double sim_var_args_in[]);
	~Diffusion();
};

#endif

