//  -*- Mode: C++;  -*-
//  File: soil_class.cc
//  Author: Dino Bellugi (dino@alba.geo.berkeley.edu), Rob Reiss
//  Copyright (C) Dino Bellugi 1997
// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// * FUNCTION:  Prediction of Soil Depth Due to Diffusion
// * 		New Version, Includes Stream Lines, Variable 
// * 		Production, and Landsliding
// * CLASSES:  Diffusion
// * HISTORY:
// * Created: Wed Apr 30 12:50:38 1997 (dino)
// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <fstream.h>
#include <values.h>
#include <macros.h>
#include "soil_class.h"

// Calculates the soil depth after dT * numIter years of
// diffusion.
Diffusion::Diffusion(Grid &grd, char* tranfunc, double D_in, double Kf_in, 
	double Kx_in, double Kc_in, double init_soil, int pert_soil, 
	double var_soil, double var_elev, double rho_ratio_in, 
	double low_rate1_in, double low_rate2_in, double up_rate_in, 
	int num_iter_in, int dt_in, int dtp_in, int pt_in, int po_in, 
	int ft_in, char* prodfunc, double sim_var_args_in[]) 
{
	int i, j;
#if (MESSAGES)
	cerr << "Starting soil depth simulation ...\n";
#endif
#if (MESSAGES >= 1)
	cerr << "Initializing ...\n";
#endif
	timeseed();
	dGrid = &grd;
	D  = D_in;
	Kf = Kf_in;
	Kx = Kx_in;
	Kc = Kc_in;
	depth_ini = init_soil;
	depth_pert = pert_soil;
	depth_var = var_soil;
	elev_var = var_elev;
	rho_ratio = rho_ratio_in;
	low_rate1 = low_rate1_in;
	low_rate2 = low_rate2_in;
	up_rate = up_rate_in;
#if (CLIPPING)
	numIter = clip(num_iter_in,  0, MAX_ITER);
	dT  = clip(dt_in,  1, MAX_DT);
	dTp = clip(dtp_in, 1, MAX_DTP);
	pT  = clip(pt_in,  0, MAX_PT);
	pO  = clip(po_in,  0, MAX_PO);
	fT  = clip(ft_in,  0, MAX_FT);
#else
	numIter = num_iter_in;
	dT = dt_in;
	dTp = dtp_in;
	pT = pt_in;
	pO = po_in;
	fT = ft_in;
#endif
	sim_var_args = sim_var_args_in;
	setTranFunc(tranfunc);
	setProdFunc(prodfunc);
	
	// Initializations
	dGrid->setValue(SC_ZPAST, SC_ELEV);
	if (depth_ini >= 0)
		dGrid->setValue(SC_SOIL, depth_ini, SC_ELEV);
	dGrid->setValue(SC_HPAST, SC_SOIL);

	if (depth_pert)
	{
#if (MESSAGES)
		cerr << "Perturbing soil depth ...\n";
#endif
		doPerturb(depth_pert, SC_HPAST, TRUE);
	}
	dGrid->setValue(SC_HFUTURE, SC_HPAST);
	//dGrid->setValue(SC_EPAST, SC_ELEV, -depth_ini);
	dGrid->setValue(SC_EPAST, SC_ELEV);
	dGrid->subValue(SC_EPAST, SC_HPAST);
	dGrid->setValue(SC_EFUTURE, SC_EPAST);
	dGrid->setValue(SC_FLUX, 0.0, SC_STATUS);
#ifdef SC_SLOPE
	dGrid->setValue(SC_SLOPE, 0.0, SC_ELEV);
#endif

#if (ORDER)
	// allocate array for ordering x and y
	if ((xOrd = (int *)malloc(dGrid->xdim * sizeof(int))) != (int *)NULL)
		setXord();
#if (MESSAGES)
	else
		cerr << "Memory allocation failure ...\n";
#endif

	if ((yOrd = (int *)malloc(dGrid->ydim * sizeof(int))) != (int *)NULL)
		setYord();
#if (MESSAGES)
	else
		cerr << "Memory allocation failure ...\n";
#endif
#endif

#if (FLUX_1D == 1)
	if (fT)
	{
		int n = rnd((numIter * dT) / fT);
		if ((flux = (double *)malloc(n * sizeof(double))) == (double *)NULL)
			fT = 0;
		else
			for (int i = 0; i < n; ++i)
				flux[i] = 0.0;
	}
#endif
#if (FLUX_1D == 2)
	if (fT)
	{
		int n = rnd((numIter * dT) / fT), m = dGrid->xdim << 1;
		// first allocate the pointer array
		if ((flux = (double **)malloc(m * sizeof(double **))) == 
			(double **)NULL)
			fT = 0;
		for (int j = 0; fT && j < m; ++j)
		{
			if ((flux[j] = (double *)malloc(n * sizeof(double))) == 
				(double *)NULL)
				fT = 0;
			else
				for (int i = 0; i < n; ++i)
					flux[j][i] = 0.0;
		}
	}
#endif
#if (FLUX_1D == 3)
	if (fT)
	{
		int n = rnd((numIter * dT) / fT), m = dGrid->xdim;
		// first allocate the pointer array
		if ((flux = (double **)malloc(m * sizeof(double **))) == 
			(double **)NULL)
			fT = 0;
		for (int j = 0; fT && j < m; ++j)
		{
			if ((flux[j] = (double *)malloc(n * sizeof(double))) == 
				(double *)NULL)
				fT = 0;
			else
				for (int i = 0; i < n; ++i)
					flux[j][i] = 0.0;
		}
	}
#endif
	AllPropTran();
}

Diffusion::~Diffusion() 
{
#if (ORDER)
	free((char *)xOrd);
	free((char *)yOrd);
#endif
#if (FLUX_1D == 1)
	if (fT)
		free((char *)flux);
#endif
#if (FLUX_1D == 2)
	// free arrays and pointer array
	if (fT)
	{
		int n = dGrid->xdim << 1;
		for (int i = 0; i < n; ++i)
			free((char *)flux[i]);
		free((char *)flux);
	}
#endif
#if (FLUX_1D == 3)
	// free arrays and pointer array
	if (fT)
	{
		int n = dGrid->xdim;
		for (int i = 0; i < n; ++i)
			free((char *)flux[i]);
		free((char *)flux);
	}
#endif
}

// ***************************************************************************
//  Soil Transport by Diffusion Simulation
// **************************************************************************
// Perfoms a diffusive soil transport simulation 
// using the diffusion equation: dz/dt = D * dz^2/dt^2
// a production law: dz/dt = prodfunc(soildepth)
// and conservation of mass. (i.e. soil limited)
// D  = Diffusion Coeff.
// numiterations = The number of iterations to run through.
// dT = time step
// Rho_Ratio = bulk density ratio.
// zpast = Grid holding the surface elevation values.
// hpast = Grid holding the soil depth values.
// epast = Grid holding the bedrock elevation values.
// prodfunc = Name of the Production Function to use.
// WARNING: this procedure does not properly handle
// the production limited case (when diffusion > amount of soil)
// The extent of bedrock will depend on dT. eek! 

void Diffusion::AllPropTran() 
{
	int i, j, k, m, n, t, status = P_STND;
	double val = 0.0, val2 = 0.0;
	double low1 = low_rate1 * dT, low2 = low_rate2 * dT;
	double dl = dGrid->spacing;
	double dz[8], dx[8], soilavail, totalsoil;
	int perc_comp = 0;
	int last_comp = 0;
#if (MESSAGES >= 2)
	cerr << "Lowering rate 1: " << low1 << ", Lowering rate 2: " << low2 << "\n";
#endif
#if (MESSAGES)
	cerr << "Completed with: " << perc_comp << " %\r";
#endif

	// For every iteration
	for (t = 0; t < numIter; ++t) 
	{
#if (MESSAGES)
		// Let the user know how far we are
		perc_comp = int(((double) t / (double) numIter) * 100.0);
		if (last_comp != perc_comp)
		{
			last_comp = perc_comp;
			cerr << "Completed with: " << perc_comp << " %\r";
		}
#endif

		// For every cell
#if (ORDER)
		for (m = 0; m < dGrid->ydim; ++m) 
		{
			j = locY(m);
#else
//		for (j = 1; j < dGrid->ydim - 1; ++j) 
		for (j = 0; j < dGrid->ydim; ++j) 
		{
#endif
#if (ORDER)
			for (n = 0; n < dGrid->xdim; ++n) 
			{
				i = locX(n);
#else
//			for (i = 1; i < dGrid->xdim - 1; ++i) 
			for (i = 0; i < dGrid->xdim; ++i) 
			{
#endif
#if (MESSAGES >= 3)
				cerr << "At (" << i << "," << j << ") ...\n";
				cerr << "Present z: " << dGrid->at(i, j, SC_ZPAST) << "\n";
				cerr << "Present h: " << dGrid->at(i, j, SC_HPAST) << "\n";
				cerr << "Present e: " << dGrid->at(i, j, SC_EPAST) << "\n";
#endif
// Lowering here to speed up things ...
#if (LOWERING == 2)
				status = trn(dGrid->at(i, j, SC_STATUS));
#if (MESSAGES >= 3)
				cerr << "Status at (" << i << "," << j
							<< "): " << status << "\n";
#endif
				if (status == B_LWR_RT1)
				{
#if (MESSAGES >= 3)
					cerr << "Old efuture: " << 
						dGrid->at(i, j, SC_EFUTURE) << "\n";
#endif
					dGrid->at(i, j, SC_EFUTURE) -= low1;
#if (MESSAGES >= 3)
					cerr << "New efuture: " << 
						dGrid->at(i, j, SC_EFUTURE) << "\n";
#endif
				}
				else if (status == B_LWR_RT2)
				{
#if (MESSAGES >= 3)
					cerr << "Old efuture: " << 
						dGrid->at(i, j, SC_EFUTURE) << "\n";
#endif
					dGrid->at(i, j, SC_EFUTURE) -= low2;
#if (MESSAGES >= 3)
					cerr << "New efuture: " << 
						dGrid->at(i, j, SC_EFUTURE) << "\n";
#endif
				}
#endif
// Done lowering
				if (!dGrid->isValue(i, j, SC_ZPAST))
					continue;

#if (LOWERING == 2)
				// Check if it's a removal cell (no diffusion)
				if (isRemoval(i, j))
				{
#if (MESSAGES >= 2)
					cerr << "Removal cell at (" << i << "," << j
							<< "), skipping ...\n";
#endif
					continue;
				}
#endif
				totalsoil = soilavail = 0.0;

				for (k = 0; k < 8; ++k) 
				{
					// For Every Direction
					dz[k] = 0.0;
					dx[k] = dl * angleFactor((Direction)k);

					// Need a special case for giving soil
					// away at the boundary. For 2-D:
					if (dGrid->ydim == 1 && !i && k == 4)
					{
#if (TRANSPORT)
						dz[k] = (this->*fTr)(i + 1, j, k, dl);
#if (MESSAGES >= 2)
						cerr << "Soil removed from boundary (left): " 
							<< dz[k] << "\n";
#endif
#endif
					}

					else if (dGrid->ydim == 1 && 
						i == dGrid->xdim - 1 && !k)
					{
#if (TRANSPORT)
						dz[k] = (this->*fTr)(i - 1, j, k, dl);
#if (MESSAGES >= 2)
						cerr << "Soil removed from boundary (right): " 
							<< dz[k] << "\n";
#endif
#endif
					}


					else if (!dGrid->isValue(i, j, (Direction)k, SC_ZPAST)
						|| !dGrid->isValue(i, j, SC_HPAST))
						continue;

					else
					{
/*
						// No diffusion when on a stream
						if (streams->isValue(i, j))
							continue;
*/
#if (TRANSPORT)
						dz[k] = (this->*fTr)(i, j, k, dx[k]);
#endif
#if (MESSAGES >= 3)
						cerr << "Transport at (" << i << "," << j
							<< "," << k << "): " << dz[k] << "\n";
#endif
						if (dz[k] <= 0.0)
							continue;
					}

					totalsoil += dz[k];
				}


				// is there anything to do?
				if (totalsoil > 0.0 || 
					status == B_SPL_INF ||
					status == B_SPL_LIM)
				{
				// Limit the amount of soil transported 
				// to the amount availaible
				val = dGrid->at(i, j, SC_HPAST);
#if (MESSAGES >= 3)
				if (val < 0.0)
					cerr << "Soil available at (" << i << ", " << j 
							<< "): " << val << "\n";
				if (totalsoil < 0.0)
					cerr << "Total soil at (" << i << ", " << j 
							<< "): " << totalsoil << "\n";
#endif
#if (CLIPPING)
				val = lclip(val, 0.0);
#endif
				if (totalsoil <= val || status == B_SPL_INF)
					soilavail = totalsoil;
				else if (status == B_SPL_LIM)
				{
					// limit to max. production
					val = ((this->*fPr)(i, j, t, TRUE)) 
							* dT * dTp * rho_ratio;
					soilavail = uclip(totalsoil, val);
				}
				else 
					soilavail = val;

				for (k = 0; k < 8; ++k) 
				{
					// NO!!! Need a special case for giving soil
					// away at the boundary. For 2-D:
					if (dGrid->ydim == 1 && !i && k == 4)	// left
					{
#if (FLUX_1D == 1)
						// Keep track of flux
						if (fT)
						{
							flux[int(((double)t * dT) / fT)] += 
								(dz[k] / totalsoil) 
								* soilavail;
						}
#endif
					}

					else if (dGrid->ydim == 1 && 
						i == dGrid->xdim - 1 && !k)	// right
					{
					}

					else if (!dGrid->isValue(i, j, (Direction)k, SC_ZPAST))
						continue;

					else
					{
/*
						// No diffusion when on a stream
						if (dz[k] <= 0.0 || streams->isValue(i, j))
							continue;
*/
						if (dz[k] <= 0.0)
							continue;
					}

					val = (dz[k] / totalsoil) * soilavail;

					if (val <= 0.0)
						continue;

#if (FLUX_1D == 2)
					// Keep track of flux out (left hand only)
					if (fT && k == 4 && dGrid->ydim == 1 && 
						i < dGrid->xdim >> 1 && val > 0.0)
					{
						flux[i][int(((double)t * dT) / fT)] += 
							val * dx[k];
					}
#endif
#if (FLUX_1D == 3)
					// Keep track of flux out (whole profile)
					if (fT && dGrid->ydim == 1 && val > 0.0)
					{
						flux[i][int(((double)t * dT) / fT)] += 
							val * dx[k];
					}
#endif
#if (MESSAGES >= 3)
					cerr << "Soil moved at (" << i << ", " << j 
						<< ", " << k << "): " << val << "\n";
#else
#if (MESSAGES >= 2)
					if (val < 0.0)
						cerr << "Soil moved at (" << i << ", " << j 
							<< ", " << k << "): " << val << "\n";
#endif
#endif
					// remove (if not an infinite supply cell)
					if (status != B_SPL_INF &&
						status != B_SPL_LIM)
					{
						dGrid->at(i, j, SC_HFUTURE) -= val;
						if (up_rate < 0)
							dGrid->at(i, j, SC_EFUTURE) += val;
					}

#if (!LOWERING)					
					dGrid->at(i, j, (Direction)k, SC_HFUTURE) += val;
					if (up_rate < 0)
						dGrid->at(i, j, (Direction)k, SC_EFUTURE) -= val;

#else					// make sure it's not a lowering cell
#if (LOWERING == 1)
					// 2-D case is already taken care of above
					if (dGrid->ydim > 1 && i != 0 && j != 0 &&
						i != dGrid->xdim - 1 && j != dGrid->ydim - 1)
					{
						dGrid->at(i, j, (Direction)k, SC_HFUTURE) 
							+= val;
						if (up_rate < 0)
							dGrid->at(i, j, (Direction)k, SC_EFUTURE) 
								-= val;
					}
#else
#if (LOWERING == 2)
					if (!isBoundary(i, j, (Direction)k))
					{
						dGrid->at(i, j, (Direction)k, SC_HFUTURE) 
							+= val;
						if (up_rate < 0)
							dGrid->at(i, j, (Direction)k, SC_EFUTURE) 
								-= val;
					}

					// add flux here
					else if (isRemoval(i, j, (Direction)k))
						dGrid->at(i, j, (Direction)k, SC_FLUX) 
							+= val;

#endif						
#endif						
#endif						
				}
				}
#if (PRODUCTION)

				// Soil Production for internal cells.
				// No production when on a stream
//				if (!streams->isValue(i, j))
#if (!COUPLING)
				// Produce according to the prod. time step (dTp)
				// the dTp check needs to be in the production
				// function to allow for a stochastic production
				//if (t && !(t % dTp))
#endif
				{
					// don't produce on boundary!!
#if (LOWERING == 2)
					if (!isBoundary(i, j) &&
						(dGrid->ydim != 1 || 
						(i && i != dGrid->xdim - 1)))
#else
					if (dGrid->ydim != 1 || 
						(i && i != dGrid->xdim - 1))
#endif
					{
						val = ((this->*fPr)(i, j, t, FALSE)) 
							* dT * dTp;
#if (MESSAGES >= 2)
						if (val < 0.0)
							cerr << "Production at (" << 
							i << ", " << j << "): " << 
							val << "\n";
#endif

#if (CLIPPING)
						val = lclip(val, 0.0);
#endif
						if (val != 0.0)
						{
#if (MESSAGES >= 2)
							cerr << "Production at (" 
								<< i << "," << j
								<< "): " << val << "\n";
#endif
#if (MESSAGES >= 3)
							cerr << "Old hfuture: " << 
								dGrid->at(i, j, SC_HFUTURE) << "\n";
							cerr << "Old efuture: " << 
								dGrid->at(i, j, SC_EFUTURE) << "\n";
#endif
							dGrid->at(i, j, SC_HFUTURE) += 
								rho_ratio * val;
							dGrid->at(i, j, SC_EFUTURE) -= val;
#if (MESSAGES >= 3)
							cerr << "New hfuture: " << 
								dGrid->at(i, j, SC_HFUTURE) << "\n";
							cerr << "New efuture: " << 
								dGrid->at(i, j, SC_EFUTURE) << "\n";
#endif
						}
					}
				}
#endif
			}
		}

#if (LOWERING)    
		// DEAL WITH LOWERING RATE HERE!!!
#if (LOWERING == 1)    
		// Lowering rate applied at the boundary

		// the 2-D case is implemented
		if (dGrid->ydim == 1)
		{
#if (MESSAGES >= 2)
			cerr << "Lowering at the boundary (left): " << low1 << "\n";
#endif
			dGrid->at(0, 0, SC_EFUTURE) -= low1;
#if (MESSAGES >= 2)
			cerr << "Lowering at the boundary (right): " << low2 << "\n";
#endif
			dGrid->at(dGrid->xdim - 1, 0, SC_EFUTURE) -= low2;
		}

		// the 3-D case is implemented
		else
		{
#if (MESSAGES >= 2)
			cerr << "Lowering at the boundary (left): " << low1 << "\n";
			cerr << "Lowering at the boundary (right): " << low2 << "\n";
#endif
			for (j = 0; j < dGrid->ydim; ++j) 
			{
				// Left Boundary
				dGrid->at(0, j, SC_EFUTURE) -= low1;
				// Right Boundary
				dGrid->at(dGrid->xdim - 1, j, SC_EFUTURE) -= low2;
			}
#if (MESSAGES >= 2)
			cerr << "Lowering at the boundary (bottom): " << low1 << "\n";
			cerr << "Lowering at the boundary (top): " << low2 << "\n";
#endif
			for (i = 0; i < dGrid->xdim; ++i) 
			{
				// Bottom Boundary
				dGrid->at(i, 0, SC_EFUTURE) -= low1;
				// Top Boundary
				dGrid->at(i, dGrid->ydim - 1, SC_EFUTURE) -= low2;
			}
		}

#else
// LOWERING with status grid has been moved at the beginning for speed
#if 0
#if (LOWERING == 2)
		// Lowering rate applied according to the status grid
#if (MESSAGES >= 2)
		cerr << "Lowering according to status grid\n";
		cerr << "rate 1: " << low1 << ", rate 2: " << low2 << "\n";
#endif
		for (j = 0; j < dGrid->ydim; ++j) 
		{
			for (i = 0; i < dGrid->xdim; ++i)
			{
				if ((status = trn(dGrid->at(i, j, SC_STATUS))) == B_LWR_RT1)
				{
					dGrid->at(i, j, SC_EFUTURE) -= low1;
#if (MESSAGES >= 2)
					cerr << ".";
#endif
				}
				else if (status == B_LWR_RT2)
				{
					dGrid->at(i, j, SC_EFUTURE) -= low2;
#if (MESSAGES >= 2)
					cerr << ".";
#endif
				}
			}
		}
#if (MESSAGES >= 2)
		cerr << "\n";
#endif
#endif
#endif
#endif
#endif

		// FUTURE BECOMES PRESENT HERE!!
#if (ORDER)
		for (m = 0; m < dGrid->ydim; ++m) 
		{
			j = locY(m);
#else
		for (j = 0; j < dGrid->ydim; ++j) 
		{
#endif
#if (ORDER)
			for (n = 0; n < dGrid->xdim; ++n) 
			{
				i = locX(n);
#else
			for (i = 0; i < dGrid->xdim; ++i) 
			{
#endif
				if (!dGrid->isValue(i, j, SC_ZPAST)) 
					continue;
#if (MESSAGES >= 3)
				cerr << "Preparing (" << i << "," << j << 
					") for next iteration ...\n";
				cerr << "Old z: " << dGrid->at(i, j, SC_ZPAST) << "\n";
				cerr << "Old h: " << dGrid->at(i, j, SC_HPAST) << "\n";
				cerr << "Old e: " << dGrid->at(i, j, SC_EPAST) << "\n";
#endif
				dGrid->at(i, j, SC_ZPAST) = dGrid->at(i, j, SC_EFUTURE) +
					dGrid->at(i, j, SC_HFUTURE);
				dGrid->at(i, j, SC_HPAST) = dGrid->at(i, j, SC_HFUTURE);
				dGrid->at(i, j, SC_EPAST) = dGrid->at(i, j, SC_EFUTURE);
#if (MESSAGES >= 3)
				cerr << "New z: " << dGrid->at(i, j, SC_ZPAST) << "\n";
				cerr << "New h: " << dGrid->at(i, j, SC_HPAST) << "\n";
				cerr << "New e: " << dGrid->at(i, j, SC_EPAST) << "\n";
#endif
			}
		}

		// Save partial values (depth and elev)
		if (pT && t && !(((long)(t * dT) % pT) - pO))
		{
			char filename[10];
			int n = int(((double)t * dT) / pT);
#if (MESSAGES)
			cerr << "Saving partial results (elevation " << n << ") ...\n";
#endif
			sprintf(filename, "z%03d.tmp", n);
			writeElev(filename);
#if (MESSAGES)
                        cerr << "Saving partial results (depth " << n << ") ...\n";
#endif
			filename[0] = 'h';
			writeDepth(filename);
		}

                // Save partial values (flux)
                if (fT && t && !(((long)(t * dT) % fT) - pO))
                {
                        char filename[10];
                        int n = int(((double)t * dT) / pT);
#if (MESSAGES)
                        cerr << "Saving partial results (flux " << n << ") ...\n";
#endif
                        sprintf(filename, "f%03d.tmp", n);
                        writeFlux(filename);
                }
	}

#if (TRIMMING)
	if (dGrid->xdim >= 3)
	{

		for (j = 0; j < dGrid->ydim; ++j) 
		{
			// Left Boundary
			dGrid->at(0, j, SC_HFUTURE) = dGrid->nodata;
			dGrid->at(0, j, SC_HPAST) = dGrid->nodata;
			dGrid->at(0, j, SC_EPAST) = dGrid->nodata;
			dGrid->at(0, j, SC_EFUTURE) = dGrid->nodata;
			dGrid->at(0, j, SC_ZPAST) = dGrid->nodata;

			// Right Boundary
			dGrid->at(dGrid->xdim - 1, j, SC_HFUTURE) = dGrid->nodata;
			dGrid->at(dGrid->xdim - 1, j, SC_HPAST) = dGrid->nodata;
			dGrid->at(dGrid->xdim - 1, j, SC_EFUTURE) = dGrid->nodata;
			dGrid->at(dGrid->xdim - 1, j, SC_EPAST) = dGrid->nodata;
			dGrid->at(dGrid->xdim - 1, j, SC_ZPAST) = dGrid->nodata;
		}
	}

	if (dGrid->ydim >= 3)
	{
		for (i = 0; i < dGrid->xdim; ++i) 
		{
			// Bottom Boundary
			dGrid->at(i, 0, SC_HFUTURE) = dGrid->nodata;
			dGrid->at(i, 0, SC_HPAST)   = dGrid->nodata;
			dGrid->at(i, 0, SC_EFUTURE) = dGrid->nodata;
			dGrid->at(i, 0, SC_EPAST)   = dGrid->nodata;
			dGrid->at(i, 0, SC_ZPAST)   = dGrid->nodata;

			// Top Boundary
			dGrid->at(i, dGrid->ydim - 1, SC_HFUTURE) = dGrid->nodata;
			dGrid->at(i, dGrid->ydim - 1, SC_HPAST)   = dGrid->nodata;
			dGrid->at(i, dGrid->ydim - 1, SC_EFUTURE) = dGrid->nodata;
			dGrid->at(i, dGrid->ydim - 1, SC_EPAST)   = dGrid->nodata;
			dGrid->at(i, dGrid->ydim - 1, SC_ZPAST)   = dGrid->nodata;
		}
	}
#endif

//	for (j = 1; j < dGrid->ydim - 1; ++j) 
	for (j = 0; j < dGrid->ydim; ++j) 
	{    
//		for (i = 1; i < dGrid->xdim - 1; ++i) 
		for (i = 0; i < dGrid->xdim; ++i) 
		{
			if (!dGrid->isValue(i, j, SC_ZPAST)) 
			{
				dGrid->at(i, j, SC_HPAST) = dGrid->nodata;
				dGrid->at(i, j, SC_EPAST) = dGrid->nodata;
			}
		}
	}

#if (MESSAGES)
	cerr << "\nDone!\n";
#endif
}

double Diffusion::tranFuncDif(int i, int j, int k, double dx)
{
	return(sDiff(i, j, k, dx));
}

double Diffusion::tranFuncDdd(int i, int j, int k, double dx)
{
	return(sDdd(i, j, k, dx));
}

double Diffusion::tranFuncHow(int i, int j, int k, double dx)
{
	return(sDiff(i, j, k, dx) + sSlide(i, j, k, dx));
}

double Diffusion::tranFuncRoe(int i, int j, int k, double dx)
{
	return(sJosh(i, j, k, dx));
}

double Diffusion::sDiff(int i, int j, int k, double dx)
{
	double qs, slope, zdiff;

	zdiff = dGrid->at(i, j, SC_ZPAST) - dGrid->at(i, j, (Direction)k, SC_ZPAST);
	if (zdiff == 0.0)
		return(0.0);
	slope = zdiff / dx;
	qs = D * slope * dT;

	// 3-D case: adjust for 8 directions
	 if (dGrid->ydim != 1)
		qs /= ADJ_8DIR;

	qs /= dx;

#if (MESSAGES >= 3)
	cerr << "Diff. term at (" << i << ", " << j << ", " << k << "): " << qs << 
		"\n\tzdiff: " << zdiff << "\n\tslope: " << slope << "\n\tD: " << D << 
		"\n\tdT: " << dT << "\n\tdx: " << dx << "\n";
#endif
	return(qs);
}

double Diffusion::sDdd(int i, int j, int k, double dx)
{
	double qs, slope, zdiff, diff;

	zdiff = dGrid->at(i, j, SC_ZPAST) - dGrid->at(i, j, (Direction)k, SC_ZPAST);
	if (zdiff == 0.0)
		return(0.0);
	slope = zdiff / dx;

	// depth dependency	
	diff = D * dGrid->at(i, j, SC_HPAST);
	qs = diff * slope * dT;

	// 3-D case: adjust for 8 directions
	 if (dGrid->ydim != 1)
		qs /= ADJ_8DIR;

	qs /= dx;

#if (MESSAGES >= 3)
	cerr << "Diff. term at (" << i << ", " << j << ", " << k << "): " << qs << 
		"\n\tzdiff: " << zdiff << "\n\tslope: " << slope << "\n\tD: " << D << 
		"\n\tdT: " << dT << "\n\tdx: " << dx << "\n";
#endif
	return(qs);
}

/*
#if (STABILITY)
	if (val1 <= 0.0 || (val2 = getKf(i, j)) <= 0.0)
		val2 = 1.0;
	else
		val2 /= 1.0 - val1;

//	cerr << "Slide term at (" << i << ", " << j << ", " << k << "): " << val2 << "\n";
#else

	val2 = 1.0 / val1 - 1.0;
	qs = Kf * val2;

	cerr << "Slide term at (" << i << ", " << j << ", " << k << "): " << qs << "\n";
#endif
	return(qs);
}
*/

double Diffusion::sSlide(int i, int j, int k, double dx)
{
	// Computes Qs = Kf * ((1 / (1 - Kx * S^Kc)) - 1)

	double val1, val2, slope, qs;

	slope = (dGrid->at(i, j, SC_ZPAST) - dGrid->at(i, j, (Direction)k, SC_ZPAST)) / dx;

	// val1 = (1 - Kx * S^Kc)
	if ((val1 = 1.0 - Kx * pow(slope, Kc)) == 1.0)
		val1 = 0.99999999;

	val2 = 1.0 / val1 - 1.0;
	qs = Kf * val2 * dT;
#if (MESSAGES >= 3)
	if (qs < 0.0)
	{
		cerr << "Slide term at (" << i << ", " << j << ", " << k << "): " 
			<< qs << "\n";
		cerr << "Slope = " << slope << ", val1 = " << val1 << ", val2 = "
			<< val2 << "\n";
	}
#endif
	return(qs / dx);
}

double Diffusion::sJosh(int i, int j, int k, double dx)
{
	double qs, slope, zdiff, scrit = Kf;

	zdiff = dGrid->at(i, j, SC_ZPAST) - dGrid->at(i, j, (Direction)k, SC_ZPAST);

	if (zdiff == 0.0)
		return(0.0);

	slope = zdiff / dx;

	if (slope >= scrit)
		slope = scrit - (0.000000001 * scrit);

	qs = (D * slope * dT) / (1 - sqr(slope / scrit));

	// 3-D case: adjust for 8 directions
	 if (dGrid->ydim != 1)
		qs /= ADJ_8DIR;

#if (MESSAGES >= 3)
	if (qs < 0.0)
		cerr << "Diff. term at (" << i << ", " << j << ", " << k << "): " 
			<< qs << "\n";
#endif
	return(qs / dx);
}

/*
double Diffusion::getKf(int i, int j)
{
	double val;

	if (!slpstab->isValue(i, j) || (val = slpstab->at(i, j)) >= 0.0)
		return(-1.0);
	val = 1.0 / exp10(val);
	return(val >= 100.0 ? val * 0.01 : 1.0);
}
*/

// Exponential Production function
double Diffusion::prodFuncExp(int i, int j, int t, int maxp) 
{
	double pval, pint, pexp, depth;

	// time to produce?
	if (!t || (t % dTp))
		return(0.0);

#if (LOWERING == 1)
	// 2-D case
	if (dGrid->ydim == 1)
	{
		if (!i || i == dGrid->xdim - 1)
			return(0.0);
	}
	// 3-D case
	else if (dGrid->ydim > 1)
	{
		if (!i || !j || i == dGrid->xdim - 1 || j == dGrid->ydim - 1)
			return(0.0);
	}
 
#else
#if (LOWERING == 2)
	// Check if it's a lowering cell (no production)
	if (isBoundary(i, j))
		return(0.0);
#endif
#endif

	// Set for max prod. or convert from m to cm for function
	if (maxp)
		depth = 0.0;
	else
		depth = dGrid->at(i, j, SC_HPAST) * 100;

	if (depth >= 0.0 && depth < sim_var_args[0]) 
	{
#if (PRODUCTION == 2)
		if (dGrid->isValue(i, j, SC_P_INT))
			pint = dGrid->at(i, j, SC_P_INT);
		else
#endif
			pint = sim_var_args[1];
#if (PRODUCTION == 2)
		if (dGrid->isValue(i, j, SC_P_EXP))
			pexp = dGrid->at(i, j, SC_P_EXP);
		else
#endif
			pexp = sim_var_args[2];
		pval = pint * exp(-pexp * depth);

		if (pval < 0.0) 
			return(0.0);
		else
			return(pval / 100.0);
	}
	else 
		return(0.0);
}

// Normal Depth Exponential Production Function
double Diffusion::prodFuncExpND(int i, int j, int t, int maxp) 
{
	double pval, pint, pexp, depth, costh;

	// time to produce?
	if (!t || (t % dTp))
		return(0.0);

#if (LOWERING == 1)
	// 2-D case
	if (dGrid->ydim == 1)
	{
		if (!i || i == dGrid->xdim - 1)
			return(0.0);
	}
	// 3-D case
	else if (dGrid->ydim > 1)
	{
		if (!i || !j || i == dGrid->xdim - 1 || j == dGrid->ydim - 1)
			return(0.0);
	}
 
#else
#if (LOWERING == 2)
	// Check if it's a lowering cell (no production)
	if (isBoundary(i, j))
		return(0.0);
#endif
#endif

	// Set for max prod. or convert from m to cm for function
	if (maxp)
		depth = 0.0;
	else
		depth = dGrid->at(i, j, SC_HPAST) * 100;

	if (depth >= 0.0 && depth < sim_var_args[0] && 
		(costh = dGrid->getSlope4(i, j, SC_ZPAST, COS_RAD))
		!= dGrid->nodata )
	{
#if (PRODUCTION == 2)
		if (dGrid->isValue(i, j, SC_P_INT))
			pint = dGrid->at(i, j, SC_P_INT);
		else
#endif
			pint = sim_var_args[1];
#if (PRODUCTION == 2)
		if (dGrid->isValue(i, j, SC_P_EXP))
			pexp = dGrid->at(i, j, SC_P_EXP);
		else
#endif
			pexp = sim_var_args[2];
		pval = pint * exp(-pexp * depth * costh) / costh;

		if (pval < 0.0) 
			return(0.0);
		else
			return(pval / 100.0);
	}
	else 
		return(0.0);
}

// Variable (Exponential) Production function
double Diffusion::prodFuncVar(int i, int j, int t, int maxp) 
{
	double pval, pint, pexp, depth, down, down2, pmul = 1.0;
	int status = P_STND;

	// time to produce?
	if (!t || (t % dTp))
		return(0.0);

	if (!dGrid->isValue(i, j, SC_STATUS))
		return(0.0);

#if (LOWERING == 1)
	// 2-D case
	if (dGrid->ydim == 1)
	{
		if (!i || i == dGrid->xdim - 1)
			return(0.0);
	}
	// 3-D case
	else if (dGrid->ydim > 1)
	{
		if (!i || !j || i == dGrid->xdim - 1 || j == dGrid->ydim - 1)
			return(0.0);
	}
 
#else
#if (LOWERING == 2)
	// return if a lowering cell
	if (isBoundary(i, j))
		return(0.0);
#endif
#endif

	status = trn(dGrid->at(i, j, SC_STATUS));

	// Set for max prod. or convert from m to cm for function
	if (maxp)
		depth = 0.0;
	else
		depth = dGrid->at(i, j, SC_HPAST) * 100;

	if (depth < 0.0 || depth > sim_var_args[0])
		return(0.0);
 
	// get downcutting
	if (status)
		down = dGrid->at(i, j, SC_ELEV) - dGrid->at(i, j, SC_EPAST) - dGrid->at(i, j, SC_SOIL);

	if (status >= P_FLP3_1)
		down2 = dGrid->at(i, j, SC_ELEV) - elev_var - dGrid->at(i, j, SC_SOIL);

	// resolve production type
	switch (status)
	{
		case P_STND:	// standard, no variations
			break;

		case P_BLK1:	// single block, hard then soft
			if (odd(trn(down / depth_var)))
			{
				dGrid->at(i, j, SC_STATUS) = P_STND;
				pmul = 1.0;
#if (MESSAGES >= 2)
				cerr << "Cell (" << i << ", " << j << 
					"), switching to standard ...\n";
#endif
			}
			else
			{
				if (dGrid->isValue(i, j, SC_STATUS))
					pmul = dGrid->at(i, j, SC_P_MUL);
			}
			break;

		case P_BLK2:	// single block, soft then hard (state 1)
			if (odd(trn(down / depth_var)))
			{
				dGrid->at(i, j, SC_STATUS) = P_BLK3;
				if (dGrid->isValue(i, j, SC_STATUS))
					pmul = dGrid->at(i, j, SC_P_MUL);
#if (MESSAGES >= 2)
				cerr << "Cell (" << i << ", " << j << 
					"), switching to hard block ...\n";
#endif
			}
			break;

		case P_BLK3:	// single block, soft then hard (state 2)
			if (even(trn(down / depth_var)))
			{
				dGrid->at(i, j, SC_STATUS) = P_STND;
				pmul = 1.0;
#if (MESSAGES >= 2)
				cerr << "Cell (" << i << ", " << j << 
					"), switching to standard ...\n";
#endif
			}
			else
			{
				if (dGrid->isValue(i, j, SC_STATUS))
					pmul = dGrid->at(i, j, SC_P_MUL);
			}
			break;

		case P_FLP1_1:	// layered, soft then hard (state 1)
			if (odd(trn(down / depth_var)))
			{
				dGrid->at(i, j, SC_STATUS) = P_FLP1_2;
				if (dGrid->isValue(i, j, SC_STATUS))
					pmul = dGrid->at(i, j, SC_P_MUL);
			}
			break;

		case P_FLP1_2:	// layered, soft then hard (state 2)
			if (even(trn(down / depth_var)))
			{
				dGrid->at(i, j, SC_STATUS) = P_FLP1_1;
				pmul = 1.0;
			}
			else
			{
				if (dGrid->isValue(i, j, SC_STATUS))
					pmul = dGrid->at(i, j, SC_P_MUL);
			}
			break;

		case P_FLP2_1:	// layered, hard then soft (state 1)
			if (odd(trn(down / depth_var)))
			{
				dGrid->at(i, j, SC_STATUS) = P_FLP2_2;
				pmul = 1.0;
			}
			else
			{
				if (dGrid->isValue(i, j, SC_STATUS))
					pmul = dGrid->at(i, j, SC_P_MUL);
			}
			break;

		case P_FLP2_2:	// layered, hard then soft (state 2)
			if (odd(trn(down / depth_var)))
			{
				dGrid->at(i, j, SC_STATUS) = P_FLP2_1;
				if (dGrid->isValue(i, j, SC_STATUS))
					pmul = dGrid->at(i, j, SC_P_MUL);
			}
			break;

		case P_FLP3_1:	// 1 layer, soft then hard at elev_var (state 1)
			if (odd(trn(down / down2)))
			{
				dGrid->at(i, j, SC_STATUS) = P_FLP3_2;
				if (dGrid->isValue(i, j, SC_STATUS))
					pmul = dGrid->at(i, j, SC_P_MUL);
			}
			break;

		case P_FLP3_2:	// 1 layer, soft then hard at elev_var (state 2)
			if (even(trn(down / depth_var)))
			{
				dGrid->at(i, j, SC_STATUS) = P_STND;
				pmul = 1.0;
			}
			else
			{
				if (dGrid->isValue(i, j, SC_STATUS))
					pmul = dGrid->at(i, j, SC_P_MUL);
			}
			break;
	}

	// get standard parameters
	if (dGrid->isValue(i, j, SC_P_INT))
		pint = dGrid->at(i, j, SC_P_INT);
	else
		pint = sim_var_args[1];
	if (dGrid->isValue(i, j, SC_P_EXP))
		pexp = dGrid->at(i, j, SC_P_EXP);
	else
		pexp = sim_var_args[2];
		
	pval = pmul * pint * exp(-pexp * depth);

	if (pval < 0.0) 
		return(0.0);
	else
		return(pval / 100.0);
}

// Fifth Order Polynomial Production Function
// Ahnert
double Diffusion::prodFuncPoly5(int i, int j, int t, int maxp) 
{
	double pval, depth;

	// time to produce?
	if (!t || (t % dTp))
		return(0.0);

#if (LOWERING == 1)
	// 2-D case
	if (dGrid->ydim == 1)
	{
		if (!i || i == dGrid->xdim - 1)
			return(0.0);
	}
	// 3-D case
	else if (dGrid->ydim > 1)
	{
		if (!i || !j || i == dGrid->xdim - 1 || j == dGrid->ydim - 1)
			return(0.0);
	}
 
#else
#if (LOWERING == 2)
	// Check if it's a lowering cell (no production)
	if (isBoundary(i, j))
		return(0.0);
#endif
#endif
	// Set for max prod. or convert from m to cm for function
	if (maxp)
		depth = 0.0;
	else
		depth = dGrid->at(i, j, SC_HPAST) * 100;

	if (depth >= 0.0 && depth < sim_var_args[0]) 
	{
		pval =  sim_var_args[1] + (depth * (sim_var_args[2] + 
			(depth * (sim_var_args[3] + (depth * (sim_var_args[4] + 
			(depth * (sim_var_args[5] + (depth * sim_var_args[6])))))))));

		if (pval < 0.0) 
			return(0.0);
		else
			return(pval / 100.0);
	}
 	else
		return(0.0);
}

// Stochastic (exponential) Production function
double Diffusion::prodFuncSto(int i, int j, int t, int maxp) 
{
	double rn, pval, pint, pexp, depth;
	unsigned long n, rl;

	// time to produce? random test
	rn = dRandino();	// pick a # between 0 and 1
	rl = ulrnd(rn * dTp);	// convert it to a # between 0 and dTp
	if (rl != (dTp / 2))	// pick a # in that range and compare
		return(0.0);
	
#if (LOWERING == 1)
	// 2-D case
	if (dGrid->ydim == 1)
	{
		if (!i || i == dGrid->xdim - 1)
			return(0.0);
	}
	// 3-D case
	else if (dGrid->ydim > 1)
	{
		if (!i || !j || i == dGrid->xdim - 1 || j == dGrid->ydim - 1)
			return(0.0);
	}
 
#else
#if (LOWERING == 2)
	// Check if it's a lowering cell (no production)
	if (isBoundary(i, j))
		return(0.0);
#endif
#endif
	// Set for max prod. or convert from m to cm for function
	if (maxp)
		depth = 0.0;
	else
		depth = dGrid->at(i, j, SC_HPAST) * 100;

#if (PRODUCTION == 2)
	if (dGrid->isValue(i, j, SC_P_INT))
			pint = dGrid->at(i, j, SC_P_INT);
	else
#endif
		pint = sim_var_args[1];
#if (PRODUCTION == 2)
	if (dGrid->isValue(i, j, SC_P_EXP))
		pexp = dGrid->at(i, j, SC_P_EXP);
	else
#endif
		pexp = sim_var_args[2];
	// compute probability
	pval = exp(pexp * depth);
#if (COUPLING)
	pval *= (double)dTp;
#endif
	n = ulrnd(pval * 100);

	// pick a random number
	rn = dRandino();
	rl = ulrnd(rn * n * pval);

	// is it in range?
	if (rl > n)
		return(0.0);		// don't produce
	else
		return(pint / 100.0);	// max. production
}


/*
// Stochastic (exponential) Production function (Sklar's method)
double Diffusion::prodFuncSto(int i, int j, int t, int maxp) 
{
	double rn;
	unsigned long n, rl;

	// time to produce?
	if (!t || (t % dTp))
		return(0.0);

#if (LOWERING == 1)
	// 2-D case
	if (dGrid->ydim == 1)
	{
		if (!i || i == dGrid->xdim - 1)
			return(0.0);
	}
	// 3-D case
	else if (dGrid->ydim > 1)
	{
		if (!i || !j || i == dGrid->xdim - 1 || j == dGrid->ydim - 1)
			return(0.0);
	}
 
#else
#if (LOWERING == 2)
	// Check if it's a lowering cell (no production)
	if (isBoundary(i, j))
		return(0.0);
#endif
#endif
	n = ulrnd(sim_var_args[3]);
	rn = dRandino();
	rl = ulrnd(rn * MAXLONG);
	if (rl % n)
		return(0.0);
	else
		return((double)n * prodFuncExp(i, j, t, TRUE));	// Max. production?
}
*/

// Arjun's Exponential Production function
// (production curve flattens when depth < threshold)
double Diffusion::prodFuncArj(int i, int j, int t, int maxp) 
{
	double pval, pint, pexp, depth;

	// time to produce?
	if (!t || (t % dTp))
		return(0.0);

#if (LOWERING == 1)
	// 2-D case
	if (dGrid->ydim == 1)
	{
		if (!i || i == dGrid->xdim - 1)
			return(0.0);
	}
	// 3-D case
	else if (dGrid->ydim > 1)
	{
		if (!i || !j || i == dGrid->xdim - 1 || j == dGrid->ydim - 1)
			return(0.0);
	}
 
#else
#if (LOWERING == 2)
	// Check if it's a boundary cell (no production) or 
	if (isBoundary(i, j))
		return(0.0);
#endif
#endif

	// Set for max prod. or convert from m to cm for function
	if (maxp)
		depth = 0.0;
	else
		depth = dGrid->at(i, j, SC_HPAST) * 100;

	if (depth >= 0.0 && depth < sim_var_args[0]) 
	{
		if (depth >= sim_var_args[3])
		{
#if (PRODUCTION == 2)
			if (dGrid->isValue(i, j, SC_P_INT))
				pint = dGrid->at(i, j, SC_P_INT);
			else
#endif
				pint = sim_var_args[1];
		}
		else
				pint = sim_var_args[4];
#if (PRODUCTION == 2)
		if (dGrid->isValue(i, j, SC_P_EXP))
			pexp = dGrid->at(i, j, SC_P_EXP);
		else
#endif
			pexp = sim_var_args[2];
		pval = pint * exp(-pexp * depth);

		if (pval < 0.0) 
			return(0.0);
		else
			return(pval / 100.0);
	}
	else 
		return(0.0);
}

// Bill's Humped Production function
// rate	= a * 10^b * h^2 + c * h + d
//	= -3 * 10^-5 * x^2 + 0.0017x + 0.1599
// where rate is in mm/yr (don't forget to convert!!)

double Diffusion::prodFuncHum(int i, int j, int t, int maxp) 
{
	double pval, depth;

	// time to produce?
	if (!t || (t % dTp))
		return(0.0);

#if (LOWERING == 1)
	// 2-D case
	if (dGrid->ydim == 1)
	{
		if (!i || i == dGrid->xdim - 1)
			return(0.0);
	}
	// 3-D case
	else if (dGrid->ydim > 1)
	{
		if (!i || !j || i == dGrid->xdim - 1 || j == dGrid->ydim - 1)
			return(0.0);
	}
 
#else
#if (LOWERING == 2)
	// Check if it's a lowering cell (no production)
	if (isBoundary(i, j))
		return(0.0);
#endif
#endif

	// Set for max prod. or convert from m to cm for function
	if (maxp)
		depth = 0.0;
	else
		depth = dGrid->at(i, j, SC_HPAST) * 100;

	if (depth >= 0.0 && depth < sim_var_args[0])
	{ 
		pval = sim_var_args[1] * exp10(sim_var_args[2]) * sqr(depth) +
			sim_var_args[3] * depth + sim_var_args[4];

		if (pval < 0.0) 
			return(0.0);
		else
			return(pval / 1000.0);	// convert back to meters
	}
	else
		return(0.0);
}

// Returns a pointer to the Soil Production Function
void Diffusion::setProdFunc(char* prodfunc) 
{
	fPr = &Diffusion::prodFuncExp;

	if (!strcmp(prodfunc,"Polynomial")) 
	{
		fPr = &prodFuncPoly5;
#if (MESSAGES)
		cerr << "Using production punction: " << prodfunc << "\n";
#endif
	} 
	else if (!strcmp(prodfunc,"Exponential")) 
	{
		fPr = &prodFuncExp;
#if (MESSAGES)
		cerr << "Using production function: " << prodfunc << "\n";
		cerr << "Parameters:\n\tcutoff: \t" << sim_var_args[0] << 
			"\n\tintercept: \t" << sim_var_args[1] << 
			"\n\texponent: \t" << sim_var_args[2] << "\n";
#endif
	} 
	else if (!strcmp(prodfunc,"Variable")) 
	{
		fPr = &prodFuncVar;
#if (MESSAGES)
		cerr << "Using production function: " << prodfunc << "\n";
#endif
	} 
	else if (!strcmp(prodfunc,"Stochastic")) 
	{
		fPr = &prodFuncSto;
#if (MESSAGES)
		cerr << "Using production function: " << prodfunc << " (" << dTp << ")\n";
#endif
	} 
	else if (!strcmp(prodfunc,"Arjuns")) 
	{
		fPr = &prodFuncArj;
#if (MESSAGES)
		cerr << "Using production function: " << prodfunc << "\n";
#endif
	} 
	else if (!strcmp(prodfunc,"Humped")) 
	{
		fPr = &prodFuncHum;
#if (MESSAGES)
		cerr << "Using production function: " << prodfunc << "\n";
#endif
	} 
	else if (!strcmp(prodfunc,"ExpNormDep")) 
	{
		fPr = &prodFuncExpND;
#if (MESSAGES)
		cerr << "Using production function: " << prodfunc << "\n";
#endif
	} 
	else 
	{
#if (MESSAGES)
		cerr << "Unknown production function:" << prodfunc << "\n";
#endif
		exit(1);
	}
}  

// Returns a pointer to the Soil Transport Function
void Diffusion::setTranFunc(char* tranfunc) 
{
	fTr = &Diffusion::tranFuncDif;

	if (!strcmp(tranfunc,"Diffusive")) 
	{
		fTr = &tranFuncDif;
#if (MESSAGES)
		cerr << "Using transport function: " << tranfunc << "\n";
		cerr << "Parameters:\n\tdiffusivity: \t" << D <<
			"\n\ttime step: \t" << dT << 
			"\n\tgrid size: \t" << dGrid->spacing << "\n";
#endif
	} 
	else if (!strcmp(tranfunc,"Howardian")) 
	{
		fTr = &tranFuncHow;
#if (MESSAGES)
		cerr << "Using transport function: " << tranfunc << "\n";
		cerr << "Parameters:\n\tdiffusivity: \t" << D << 
			"\n\ttrans. coeff.: \t" << Kf << 
			"\n\tthres. coeff.: \t" << Kx <<
			"\n\tnonlin. exp.: \t" << Kc << "\n";
#endif
	} 
	else if (!strcmp(tranfunc,"Roeringian")) 
	{
		fTr = &tranFuncRoe;
#if (MESSAGES)
		cerr << "Using transport function: " << tranfunc << "\n";
		cerr << "Parameters:\n\tdiffusivity: \t" << D << 
			"\n\tcrit. slope: \t" << Kf << "\n";
#endif
	} 
	else if (!strcmp(tranfunc,"DiffDepth")) 
	{
		fTr = &tranFuncDdd;
#if (MESSAGES)
		cerr << "Using transport function: " << tranfunc << "\n";
		cerr << "Parameters:\n\tdiffusivity: \t" << D <<
			"\n\ttime step: \t" << dT << 
			"\n\tgrid size: \t" << dGrid->spacing << "\n";
#endif
	} 
	else 
	{
#if (MESSAGES)
		cerr << "Unknown transport function:" << tranfunc << "\n";
#endif
		exit(1);
	}
}  

#if (ORDER)
void Diffusion::setXord() 
{
	int i;

	// 2-D case
	if (dGrid->ydim == 1)
	{
		int div =  dGrid->xdim >> 1;
#if (ORDER == 1)
		for (i = 0; i <= div; ++i)
			xOrd[i] = div - i;
		for (i = div + 1; i < dGrid->xdim; ++i)
			xOrd[i] = i;
#else
#if (ORDER == 2)
		for (i = 0; i < dGrid->xdim; ++i)
			xOrd[i] = i % 2 ? i / 2 : dGrid->xdim - 1 - (i >> 1);
#endif
#endif
	}
	else
		for (i = 0; i < dGrid->xdim; ++i)
			xOrd[i] = i;
#if (MESSAGES > 1)
	cerr << "x order:";
	for (i = 0; i < dGrid->xdim; ++i)
		cerr << " " << locX(i);
	cerr << "\n";
#endif
}


void Diffusion::setYord() 
{
	int i;

	for (i = 0; i < dGrid->ydim; ++i)
		yOrd[i] = i;
#if (MESSAGES > 1)
	cerr << "y order:";
	for (i = 0; i < dGrid->ydim; ++i)
		cerr << " " << locY(i);
	cerr << "\n";
#endif
}
#endif

int Diffusion::isLowering(int i, int j)
{
	int status;

	if (dGrid->isValue(i, j, SC_STATUS))
	{
		status = trn(dGrid->at(i, j, SC_STATUS));
		if (status == B_LWR_RT1 || status == B_LWR_RT2)
			return(1);
	}
	return(0);
}

int Diffusion::isLowering(int i, int j, Direction k)
{
	int status;

	if (dGrid->isValue(i, j, k, SC_STATUS))
	{
		status = trn(dGrid->at(i, j, k, SC_STATUS));
		if (status == B_LWR_RT1 || status == B_LWR_RT2)
			return(1);
	}
	return(0);
}

int Diffusion::isBoundary(int i, int j)
{
	int status;

	if (dGrid->isValue(i, j, SC_STATUS))
	{
		status = trn(dGrid->at(i, j, SC_STATUS));
		if (status == B_LWR_RT1 || 
			status == B_LWR_RT2 ||
			status == B_REM_INF ||
			status == B_SPL_INF ||
			status == B_SPL_LIM)
			return(1);
	}
	return(0);
}

int Diffusion::isBoundary(int i, int j, Direction k)
{
	int status;

	if (dGrid->isValue(i, j, k, SC_STATUS))
	{
		status = trn(dGrid->at(i, j, k, SC_STATUS));
		if (status == B_LWR_RT1 || 
			status == B_LWR_RT2 ||
			status == B_REM_INF ||
			status == B_SPL_INF ||
			status == B_SPL_LIM)
			return(1);
	}
	return(0);
}

int Diffusion::isSupply(int i, int j)
{
	int status;

	if (dGrid->isValue(i, j, SC_STATUS))
	{
		status = trn(dGrid->at(i, j, SC_STATUS));
		if (status == B_SPL_INF || 
			status == B_SPL_LIM)
			return(1);
	}
	return(0);
}

int Diffusion::isSupply(int i, int j, Direction k)
{
	int status;

	if (dGrid->isValue(i, j, k, SC_STATUS))
	{
		status = trn(dGrid->at(i, j, k, SC_STATUS));
		if (status == B_SPL_INF ||
			status == B_SPL_LIM)
			return(1);
	}
	return(0);
}

int Diffusion::isRemoval(int i, int j)
{
	int status;

	if (dGrid->isValue(i, j, SC_STATUS))
	{
		status = trn(dGrid->at(i, j, SC_STATUS));
		if (status == B_LWR_RT1 || 
			status == B_LWR_RT2 ||
			status == B_REM_INF)
			return(1);
	}
	return(0);
}

int Diffusion::isRemoval(int i, int j, Direction k)
{
	int status;

	if (dGrid->isValue(i, j, k, SC_STATUS))
	{
		status = trn(dGrid->at(i, j, k, SC_STATUS));
		if (status == B_LWR_RT1 || 
			status == B_LWR_RT2 ||
			status == B_REM_INF)
			return(1);
	}
	return(0);
}

void Diffusion::coutDepth() 
{
	dGrid->coutGrid(SC_HPAST);
}

void Diffusion::writeElev(char* gridname) 
{
	dGrid->writeGrid(gridname, SC_ZPAST);
}

void Diffusion::writeDepth(char* gridname) 
{
	dGrid->writeGrid(gridname, SC_HPAST);
}

void Diffusion::writeFlux(char* gridname) 
{
	dGrid->writeGrid(gridname, SC_FLUX);
}

void Diffusion::writeSlope(char* gridname) 
{
#ifdef SC_SLOPE
	dGrid->writeGrid(gridname, SC_SLOPE);
#endif
}

void Diffusion::writeBedrock(char* gridname) 
{
	dGrid->writeGrid(gridname, SC_EPAST);
}

void Diffusion::write1DFlux(char* filename) 
{
#if (!FLUX_1D)
	return;
#endif
	FILE *fp;
	int i, j, m, n = rnd((numIter * dT) / fT);

	if ((fp = fopen(filename, "w")) != NULL)
	{

#if (FLUX_1D == 1)
		for (i = 0; i < n; ++i)
			fprintf(fp, "%ld\t%lf\n", (long)(i + 1) * fT, flux[i]);
#endif
#if (FLUX_1D == 2)
		for (i = 0, m = dGrid->xdim / 2; i < n; ++i)
		{
			fprintf(fp, "%ld", (long)(i + 1) * fT);
			for (j = 0; j < m; ++j)
				fprintf(fp, "\t%.8lf", flux[j][i]);
			fprintf(fp, "\n");
		}
#endif
#if (FLUX_1D == 3)
		for (i = 0, m = dGrid->xdim; i < n; ++i)
		{
			fprintf(fp, "%ld", (long)(i + 1) * fT);
			for (j = 0; j < m; ++j)
				fprintf(fp, "\t%.8lf", flux[j][i]);
			fprintf(fp, "\n");
		}
#endif
		fclose(fp);
	}
#if (MESSAGES)
	else
		cerr << "Can't open " << filename << " ...\n";
#endif
}

int Diffusion::addFlux(double *flux)
{
	double sum = 0.0;
	long cells = 0L;
	int i, j;

	for (j = 0; j < dGrid->ydim; ++j) 
	{
		for (i = 0; i < dGrid->xdim; ++i)
		{
			if (isLowering(i, j))
			{
				sum += dGrid->at(i, j, SC_FLUX);
				++cells;
			}
		}
	}

	if (cells)
		*flux = sum;
	else
		*flux = 0.0;

	return(cells);
}

// Random number generation
#define IA	16807
#define IM	2147483647
#define AM	(1.0 / IM)
#define IQ	127773
#define IR	2836
#define RMASK	123459876

double Diffusion::dRandino()
{
	long l;
	double num;

	seedino ^= RMASK;
	l = seedino / IQ;
	seedino = IA * (seedino - l * IQ) - IR * l;
	if (seedino < 0L)
		seedino += IM;
	num = AM * seedino;
	seedino ^= RMASK;
	return(num);
}

int Diffusion::doPerturb(int iter, int field, int clip)
{
	double val, rn;
	int i, j, n;

	for (n = 0; n < iter; ++n)
	{
		for (j = 0; j < dGrid->ydim; ++j) 
		{
			for (i = 0; i < dGrid->xdim; ++i)
			{
				// get a random number between -0.5 and 0.5
				rn = dRandino() - 0.5;
				// get current value
				val = dGrid->at(i, j, field);
				// perturb value
				val += val * rn;
				// clip value
				if (clip)
					val = lclip(val, 0.0);
				// reassign value
				dGrid->at(i, j, field) = val;
			}
		}
	}
}
