//  -*- Mode: C++;  -*-
//  File: profile.cc
//  Author: Dino Bellugi (dino@geomorph.berkeley.edu)
//  Copyright  Dino Bellugi, BlueG SoftWear, U.C. Berkeley, 1999 (C)
// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// * FUNCTION: windows-like profile functions for ".ini" files
// *
// * RELATED PACKAGES: 
// *
// * HISTORY:
// * Created:  Mon Apr 28 1997 (dino)
// * Modified: Sat Jun 26 1999 (dino)
// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

// the following include file (notice.h) contains important copyright 
// information, as well as other legal statements. it must be present 
// in all files and in any distribution. removal or modification is
// strictly forbidden and violates u.s. and international copyright law.
#include "notice.h"
//
#include <stdio.h>
//#include <io.h>
#include <stdlib.h>
#include <string.h>
#include "profile.h"

int readProfileString(char *pAppName, char *pKeyName, char *pString,
	int nSize, char *pDefault)
{
	return(readPrivateProfileString(INI_FILE, pAppName, pKeyName,
		pString, nSize, pDefault));
}

int readProfileInt(char *pAppName, char *pKeyName, int *pNum, 
	int nDefault)
{
	return(readPrivateProfileInt(INI_FILE, pAppName, pKeyName, 
		pNum, nDefault));
}

int readProfileDouble(char *pAppName, char *pKeyName, double *pDbl, 
	double dDefault)
{
	return(readPrivateProfileDouble(INI_FILE, pAppName, pKeyName, 
		pDbl, dDefault));
}

int writeProfileString(char *pAppName, char *pKeyName, char *pString)
{
	char buf[NAME_SIZE];
	
	readProfileString(pAppName, pKeyName, buf, sizeof(buf), "");
	return(strcmp(buf, pString) ?
		writePrivateProfileString(INI_FILE, pAppName, pKeyName, pString) :
		TRUE);
}

int writeProfileInt(char *pAppName, char *pKeyName, int nNum)
{
	int old;
	
	readProfileInt(pAppName, pKeyName, &old, ~nNum);
	return(nNum == old ? TRUE :
		writePrivateProfileInt(INI_FILE, pAppName, pKeyName, nNum));
}

int writeProfileDouble(char *pAppName, char *pKeyName, double dNum)
{
	double old;
	
	readProfileDouble(pAppName, pKeyName, &old, -dNum);
	return(dNum == old ? TRUE :
		writePrivateProfileDouble(INI_FILE, pAppName, pKeyName, dNum));
}

int readPrivateProfileString(char *pFileName, char *pAppName,
	char *pKeyName, char *pString, int nSize, char *pDefault)
{
	FILE *fp = NULL;
	char file[NAME_SIZE], *c;
	int n;

	if (!strrchr(strcpy(file, pFileName), '.'))
		strcat(file, PRF_EXT);

	if (((fp = fopen(file, "rt")) == NULL) ||
		!findAppName(fp, pAppName) ||
		!findKeyName(fp, pKeyName))
	{
		if (pDefault)
			*(strncpy(pString, pDefault, nSize - 1) + nSize - 1) = 0;
		if (fp)
			fclose(fp);
		return(FALSE);
	}
	while ((n = fgetc(fp)) != EOF && iswhite(n));
	if (n == '\n')
	{
		if (pDefault)
			*(strncpy(pString, pDefault, nSize - 1) + nSize - 1) = 0;
		if (fp)
			fclose(fp);
		return(FALSE);
	}
	fseek(fp, -1L, SEEK_CUR);
	fgets(pString, nSize, fp);
	if (c = strrchr(pString, '\n'))	
		*c = 0;
	fclose(fp);
	return(TRUE);
}

int readPrivateProfileInt(char *pFileName, char *pAppName, char *pKeyName,
	int *pNum, int  nDefault)
{
	FILE *fp = NULL;
	char file[NAME_SIZE];
	int n;

	if (!strrchr(strcpy(file, pFileName), '.'))
		strcat(file, PRF_EXT);

	if (((fp = fopen(file, "rt")) == NULL) ||
		!findAppName(fp, pAppName) ||
		!findKeyName(fp, pKeyName))
	{
		*pNum = nDefault;
		if (fp)
			fclose(fp);
		return(FALSE);
	}
	while ((n = fgetc(fp)) != EOF && iswhite(n));
	if (n == '\n')
	{
		*pNum = nDefault;
		if (fp)
			fclose(fp);
		return(FALSE);
	}
	fseek(fp, -1L, SEEK_CUR);
	fscanf(fp, "%d", pNum);
	fclose(fp);
	return(TRUE);
}

int readPrivateProfileDouble(char *pFileName, char *pAppName,
	char *pKeyName, double *pDbl, double dDefault)
{
	FILE *fp = NULL;
	char file[NAME_SIZE];
	int n;

	if (!strrchr(strcpy(file, pFileName), '.'))
		strcat(file, PRF_EXT);

	if (((fp = fopen(file, "rt")) == NULL) ||
		!findAppName(fp, pAppName) ||
		!findKeyName(fp, pKeyName))
	{
		*pDbl = dDefault;
		if (fp)
			fclose(fp);
		return(FALSE);
	}
	while ((n = fgetc(fp)) != EOF && iswhite(n));
	if (n == '\n')
	{
		*pDbl = dDefault;
		if (fp)
			fclose(fp);
		return(FALSE);
	}
	fseek(fp, -1L, SEEK_CUR);
	fscanf(fp, "%lf", pDbl);
	fclose(fp);
	return(TRUE);
}

int writePrivateProfileString(char *pFileName, char *pAppName,
	char *pKeyName, char *pString)
{
	FILE *ifp = NULL, *ofp = NULL;
	char line[LINE_SIZE], filetemp[TEMP_SIZE], *filename, *c,
		file[NAME_SIZE], bak_file[NAME_SIZE];
	int oldfile, n;

	if (!strcpy(filetemp, "TMPXXXXXX") ||
		//((filename = mktemp(filetemp)) == NULL) ||
		((ofp = fopen(filename, "wt")) == NULL))
		return(FALSE);

	if (!strrchr(strcpy(file, pFileName), '.'))
		strcat(file, PRF_EXT);

	if (c = strrchr(strcpy(bak_file, pFileName), '.'))
		strcpy(c, BAK_EXT);
	else
		strcat(bak_file, BAK_EXT);

	oldfile = (ifp = fopen(file, "rt")) ? TRUE : FALSE;

	if (!findAppName(ifp, pAppName))
	{
		backupFile(ofp, ifp, SEEK_BEG);
		sprintf(line, "\n[%s]\n", pAppName);
		fputs(line, ofp);
		sprintf(line, "%s=%s\n", pKeyName, pString);
		fputs(line, ofp);
	}
	else if (!findKeyName(ifp, pKeyName))
	{
		backupFile(ofp, ifp, SEEK_CUR);
		sprintf(line, "%s=%s\n", pKeyName, pString);
		fputs(line, ofp);
		backupFile(ofp, ifp, SEEK_END);
	}
	else
	{
		if ((n = fgetc(ifp)) == '\n')
				fseek(ifp, -2L, SEEK_CUR);
        else
				fseek(ifp, -1L, SEEK_CUR);
		backupFile(ofp, ifp, SEEK_CUR);
		sprintf(line, "%s\n", pString);
		fputs(line, ofp);
		nextLine(ifp);
		backupFile(ofp, ifp, SEEK_END);
	}
	if (ifp)
		fclose(ifp);
	fclose(ofp);
	remove(bak_file);
	return((((oldfile && rename(file, bak_file)) ||
		rename(filename, file)) ? FALSE : TRUE));
}

int writePrivateProfileInt(char *pFileName, char *pAppName, 
	char *pKeyName, int nNum)
{
	char pString[LINE_SIZE];
	
	sprintf(pString, "%d", nNum);
	return(writePrivateProfileString(pFileName, pAppName, 
		pKeyName, pString));
}

int writePrivateProfileDouble(char *pFileName, char *pAppName, 
	char *pKeyName, double dNum)
{
	char pString[LINE_SIZE];
	
	sprintf(pString, "%lf", dNum);
	return(writePrivateProfileString(pFileName, pAppName, 
		pKeyName, pString));
}

int findAppName(FILE *fp, char *name)
{
	char line[LINE_SIZE];
	int n;

	while (TRUE)
	{
		while ((n = fgetc(fp)) != '[' && n != EOF)
			if (n != '\n')
				nextLine(fp);
		if ((n == EOF) || (n = fscanf(fp, "%[^]\n]", line)) == EOF)
			return(FALSE);
		if (!n)
			continue;
		if (!strcmp(line, name))
		{
			while ((n = fgetc(fp)) == ']' || n == '\n' || iswhite(n));
			fseek(fp, -1L, SEEK_CUR);
			return(TRUE);
		}
	}
}

int findKeyName(FILE *fp, char *name)
{
	char line[LINE_SIZE], *c;
	int n;

	if (fp == NULL)
		return(FALSE);

	while (TRUE)
	{
		if ((n = fscanf(fp, "%[^=\n]", line)) == EOF)
			return(FALSE);
		if (*line == '[')
		{
			fseek(fp, -(long)strlen(line), SEEK_CUR);
			return(FALSE);
		}
		if (n > 0)
		{
			c = line + strlen(line) - 1;
			while (iswhite(*c))
				--c;
			*(++c) = 0;
			if (!strcmp(line, name))
			{
				while ((n = fgetc(fp)) == '=' || iswhite(n));
				fseek(fp, -1L, SEEK_CUR);
				return(TRUE);
			}
		}
		nextLine(fp);
	}
}

int backupFile(FILE *ofp, FILE *ifp, int mode)
{
	int n;
	long offset;

	if (ifp == NULL || ofp == NULL)
		return(FALSE);

	if (mode == SEEK_BEG)		// backup the entire file
	{
		rewind(ifp);
		rewind(ofp);
		while((n = fgetc(ifp)) != EOF)
			if (fputc(n, ofp) == EOF)
				return(FALSE);
		fflush(ofp);
		return(TRUE);
	}
	else if (mode == SEEK_CUR)	// from start to current position
	{
		offset = ftell(ifp);
		rewind(ifp);
		while (ftell(ifp) < offset)
			if (((n = fgetc(ifp)) == EOF) || (fputc(n, ofp) == EOF))
				return(FALSE);
		fflush(ofp);
		return(TRUE);
	}
	else if (mode == SEEK_END)	// from current position to end
	{
		while((n = fgetc(ifp)) != EOF)
			if (fputc(n, ofp) == EOF)
				return(FALSE);
		fflush(ofp);
		return(TRUE);
	}
	else
		return(FALSE);
}
