//  -*- Mode: C++;  -*-
//  File: gridedges.cc
//  Author: Dino Bellugi (dino@geomorph.berkeley.edu)
//  Copyright  Dino Bellugi, BlueG SoftWear, U.C. Berkeley, 1999 (C)
// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// * FUNCTION: grid cells edge weights, used in 8-direction distribution
// *
// * RELATED PACKAGES: grid.cc area_class.cc
// *
// * HISTORY:
// * Created:  Wed Jul 14 1993 (dino, based on rob reiss' version)
// * Modified: Tue Mar 30 1999 (dino)
// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

// the following include file (notice.h) contains important copyright 
// information, as well as other legal statements. it must be present 
// in all files and in any distribution. removal or modification is
// strictly forbidden and violates u.s. and international copyright law.
#include "notice.h"
//
#include <fstream.h> 
#include "gridedges.h"
#if (GE_DEBUG)
#include <stdio.h>
#endif

GridEdges::GridEdges(int x, int y) 
{
	long m, n;

	xdim = x;
	ydim = y;
	m = x * y * 4L;
#if (GE_DEBUG)
	fprintf(stderr, "Gridedges: allocating %ld bytes ...\n", 
		m * sizeof(double));
#endif
	arr = new double[m];

	for (n = 0; n < m; ++n)
		arr[n] = 0.0;
}

GridEdges::~GridEdges() 
{
	delete[] arr;
}

double GridEdges::get(int x,int y,Direction direc) 
{
// Returns the weight of the edge at location x,y
// in the grid and direction = direc.
	switch (direc) 
	{
		case R:
			if (x >= (xdim - 1)) 
			{
				return(0.0);
			}
			return(-1.0 * arr[(((y * xdim) + x) * 4)]);
		case L:
			if (x < 1) 
			{
				return(0.0);
			}
			return(arr[(((y * xdim) + x - 1) * 4)]);
		case U: 
			if (y >= (ydim - 1)) 
			{
				return(0.0);
			}
			return(-1.0 * arr[(((y * xdim) + x) * 4 + 1)]);
		case D:
			if (y < 1) 
			{
				return(0.0);
			}
			return(arr[((((y - 1) * xdim) + x) * 4 + 1)]);
		case UR: 
			if (y >= (ydim - 1) || x >= (xdim - 1)) 
			{
				return(0.0);
			}
			return(-1.0 * arr[(((y * xdim) + x) * 4 + 2)]);
		case DL:
			if (y < 1 || x < 1) 
			{
				return(0.0);
			}
			return(arr[((((y - 1) * xdim) + (x - 1)) * 4 + 2)]);
		case DR: 
			if (y < 1 || x >= (xdim - 1)) 
			{
				return(0.0);
			}
			return(-1.0 * arr[(((y * xdim) + x) * 4 + 3)]);
		case UL:
			if (y >= (ydim - 1) || x < 1) 
			{
				return(0.0);
			}
			return(arr[((((y + 1) * xdim) + (x - 1)) * 4 + 3)]);
	}
	return(0.0);
}

void GridEdges::set(int x,int y,Direction direc,double val) 
{
// Sets the weight of the edge at location x,y
// in the grid and direction = direc to val.
	switch (direc) 
	{
		case R:
			arr[(((y * xdim) + x) * 4)] = -1.0 * val;
			break;
		case L:
			arr[(((y * xdim) + x - 1) * 4)] = val;
			break;
		case U: 
			arr[(((y * xdim) + x) * 4 + 1)] = -1.0 * val;
			break;
		case D:
			arr[((((y - 1) * xdim) + x) * 4 + 1)] = val;
			break;
		case UR: 
			arr[(((y * xdim) + x) * 4 + 2)] = -1.0 * val;
			break;
		case DL:
			arr[((((y - 1) * xdim) + (x - 1)) * 4 + 2)] = val;
			break;
		case DR: 
			arr[(((y * xdim) + x) * 4 + 3)] = -1.0 * val;
			break;
		case UL:
			arr[((((y + 1) * xdim) + (x - 1)) * 4 + 3)] = val; 
			break;
	}
}

