//  -*- Mode: C++;  -*-

//  File: stabfun.cpp

//  Author: Dino Bellugi (dino@geomorph.berkeley.edu)

//  Copyright  Dino Bellugi, BlueG SoftWear, U.C. Berkeley, 1999 (C)

// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

// * FUNCTION: main interface for shalstab library. functions include

// *           sink removal, computation of A/b, slope, q/T, curvature

// *

// * RELATED PACKAGES: area_class.cpp grid.cpp gridedges.cpp

// *

// * HISTORY:

// * Created:  Thu Apr  1 18:15:00 1999 (dino)

// * Modified: Mon May 10 17:40:00 1999 (dino)

// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~



// the following include file (notice.h) contains important copyright 

// information, as well as other legal statements. it must be present 

// in all files and in any distribution. removal or modification is

// strictly forbidden and violates u.s. and international copyright law.

#include "notice.h"

//



#include "stabfun.h"



// Removes Sinks

void calcRemoveSinks(Grid *grd, int rem_field, double soil_incr,

	int rem_max_pass, int disp_mode)

{

	if (!grd->checkForSinks(rem_field, disp_mode))

		grd->removeSinks(soil_incr, rem_max_pass, RS_NEIGHBORS,

		rem_field, disp_mode);

}



short int do_RemoveSinks(char *elv_grid, char *rem_grid, short int grid_mode,

	float soil_incr, short int rem_max_pass, short int disp_mode)

{

	Grid grd;

	int ret_val;



	doDisp("Reading elevation grid ...\n", disp_mode);



	if ((ret_val = grd.readGrid(elv_grid, 0, grid_mode)) != OK)

	{

		doDisp("Error reading elevation grid!", disp_mode);

		return((short int)ret_val);

	}



	if ((ret_val = grd.addField(0)) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}

        

	calcRemoveSinks(&grd, 1, (double)soil_incr, rem_max_pass, disp_mode);

 

	doDisp("Writing elevation grid ...\n", disp_mode);



	if (rem_grid == (char *)NULL)

		grd.coutGrid(1, grid_mode);



	else if ((ret_val = grd.writeGrid(rem_grid, 1, grid_mode)) != OK)

	{

		doDisp("Error writing sink removed grid!", disp_mode);

		return((short int)ret_val);

	}



	return(OK); 

}


short int do_CheckSinks(char *elv_grid, short int grid_mode, short int disp_mode)

// if disp_mode == 0 the function returns true  or false
// if disp_mode > 0 the function returns true  or false and displays messages
// if disp_mode == -1 the function returns the number of sinks
// if disp_mode < -1 the function returns the number of sinks and displays messages

{

	Grid grd;

	int ret_val;


	doDisp("Reading elevation grid ...\n", disp_mode);



	if ((ret_val = grd.readGrid(elv_grid, 0, grid_mode)) != OK)

	{

		doDisp("Error reading elevation grid!", disp_mode);

		return((short int)ret_val);

	}

	return(grd.checkForSinks(0, disp_mode));


}




// Calculates Slope 

void calcSlope(Grid *grd, int rem_field, int slp_field, 

	int slp_mode, int disp_mode)

{

	int i, j;

 

	doDisp("Computing slope ...\n", disp_mode);



	for (j = 0; j < grd->ydim; j++)

		for (i = 0; i < grd->xdim; i++)

			grd->at(i, j, slp_field) =

				grd->getSlope(i, j, rem_field, slp_mode);

}

 

short int do_Slope(char *rem_grid, char *slp_grid, short int grid_mode,

	short int slp_mode, short int disp_mode)

{

	Grid grd;

	int ret_val;

 

	doDisp("Reading elevation grid ...\n", disp_mode);



	if ((ret_val = grd.readGrid(rem_grid, 0, grid_mode)) != OK)

	{

		doDisp("Error reading elevation grid!", disp_mode);

		return((short int)ret_val);

	}

 

	if ((ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}

 

	calcSlope(&grd, 0, 1, slp_mode, disp_mode);

 

	doDisp("Writing slope grid ...\n", disp_mode);

 

	if (slp_grid == (char *)NULL)

		grd.coutGrid(1, grid_mode);

 

	else if ((ret_val = grd.writeGrid(slp_grid, 1, grid_mode)) != OK)

	{

		doDisp("Error writing slope grid!", disp_mode);

		return((short int)ret_val);

	}

 

	return(OK);

}



// Calculates Curvature

void calcCurvature(Grid *grd, int rem_field, int crv_field, int disp_mode)

{

	int i, j;

 

	doDisp("Computing curvature ...\n", disp_mode);

 

	for (j = 0; j < grd->ydim; j++)

		for (i = 0; i < grd->xdim; i++)

			grd->at(i, j, crv_field) =

				grd->getCurvature(i, j, rem_field);

}

 

short int do_Curvature(char *rem_grid, char *crv_grid, short int grid_mode,

	short int disp_mode)

{

	Grid grd;

	int ret_val;

 

	doDisp("Reading elevation grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(rem_grid, 0, grid_mode)) != OK)

	{

		doDisp("Error reading elevation grid!", disp_mode);

		return((short int)ret_val);

	}

 

	if ((ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}

 

	calcCurvature(&grd, 0, 1, disp_mode);

 

	doDisp("Writing curvature grid ...\n", disp_mode);

 

	if (crv_grid == (char *)NULL)

		grd.coutGrid(1, grid_mode);

 

	else if ((ret_val = grd.writeGrid(crv_grid, 1, grid_mode)) != OK)

	{

		doDisp("Error writing curvature grid!", disp_mode);

		return((short int)ret_val);

	}

 

	return(OK);

}

 

// Calculates Area

void calcArea(Grid *grd, int rem_field, int area_field,

	int area_mode, int check_sinks, int *sinks, int disp_mode)

{

	Contributing_Area totalArea(grd, rem_field, area_field,

		area_mode, check_sinks, sinks, disp_mode);

}



short int do_Area(char *rem_grid, char *area_grid, short int grid_mode,

	short int area_mode, short int check_sinks, short int disp_mode)

{

	Grid grd;

	int ret_val, sinks;

 

	doDisp("Reading sink removed grid ...\n", disp_mode);



	if ((ret_val = grd.readGrid(rem_grid, 0, grid_mode)) != OK)

	{

		doDisp("Error reading sink removed grid!", disp_mode);

		return((short int)ret_val);

	}

 

	if ((ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}

 

	calcArea(&grd, 0, 1, area_mode, check_sinks, &sinks, disp_mode);

 

	doDisp("Writing area grid ...\n", disp_mode);

 

	if (area_grid == (char *)NULL)

		grd.coutGrid(1, grid_mode);

 

	else if ((ret_val = grd.writeGrid(area_grid, 1, grid_mode)) != OK)

	{

		doDisp("Error writing area grid!", disp_mode);

		return((short int)ret_val);

	}

 
	return(check_sinks ? (disp_mode >= 0 ? (sinks ? OK : MSG_SINKS) : (sinks ? MSG_SINKS : OK)) : OK);

}

 

// Calculates q/T

void calcQoverT(Grid *grd, int slp_field, int area_field, int qt_field, 

	int slope_mode, double qt_ps, double qt_phi, int disp_mode)

{

	int i, j;

 

	doDisp("Computing q/T ...\n", disp_mode);

 

	for (j = 0; j < grd->ydim; j++)

		for (i = 0; i < grd->xdim; i++)

			grd->at(i, j, qt_field) =

				grd->getQoverT(i, j, slp_field, area_field, 

					slope_mode, qt_ps, qt_phi);

}

 

short int do_QoverT(char *slp_grid, char *area_grid, char *qt_grid,

	short int grid_mode, short int slope_mode, float qt_ps, float qt_phi, 

	short int disp_mode)

{

	Grid grd;

	int ret_val;

 

	doDisp("Reading slope grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(slp_grid, 0, grid_mode)) != OK)

	{

		doDisp("Error reading slope grid!", disp_mode);

		return((short int)ret_val);

	}



	if ((ret_val = grd.addField()) != OK || 

			(ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}



	doDisp("Reading area grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(area_grid, 1, grid_mode)) != OK)

	{

		doDisp("Error reading area grid!", disp_mode);

		return((short int)ret_val);

	}



	calcQoverT(&grd, 0, 1, 2, slope_mode, (double)qt_ps, (double)qt_phi, disp_mode);

 

	doDisp("Writing q/T grid ...\n", disp_mode);

 

	if (qt_grid == (char *)NULL)

		grd.coutGrid(2, grid_mode);

 

	else if ((ret_val = grd.writeGrid(qt_grid, 2, grid_mode)) != OK)

	{

		doDisp("Error writing q/T grid!", disp_mode);

		return((short int)ret_val);

	}

 

	return(OK);

}



// Calculates all

short int do_Shalstab(char *elv_grid, char *rem_grid, char *slp_grid,

	char *area_grid, char *qt_grid, short int grid_mode, float soil_incr, 

	float qt_ps, float qt_phi, short int disp_mode)

{

	Grid grd;

	int fields = FIELDS, ret_val, sinks;

 

	doDisp("Reading elevation grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(elv_grid, ELEV, grid_mode)) != OK)

	{

		doDisp("Error reading elevation grid!", disp_mode);

		return((short int)ret_val);

	}



	while (fields--)

	{

		if ((ret_val = grd.addField()) != OK)

		{

			doDisp("Insufficient memory!\n", disp_mode);

			return((short int)ret_val);

		}

	}



	grd.setValue(REMOVE, ELEV);



	if (*rem_grid != NOREMSINK)
		
		calcRemoveSinks(&grd, REMOVE, (double)soil_incr, RS_MAXPASSES, disp_mode);
	


	if (rem_grid != (char *)NULL && *rem_grid != NOREMSINK)

	{

		doDisp("Writing sink removed grid ...\n", disp_mode);

 

		if ((ret_val = grd.writeGrid(rem_grid, REMOVE, grid_mode)) != OK)

		{

			doDisp("Error writing sink removed grid!", disp_mode);

			return((short int)ret_val);

		}

	}



	calcSlope(&grd, REMOVE, SLOPE, ATAN_RAD, disp_mode);

 

	if (slp_grid != (char *)NULL)

	{

		doDisp("Writing slope grid ...\n", disp_mode);

 

		if ((ret_val = grd.writeGrid(slp_grid, SLOPE, grid_mode)) != OK)

		{

			doDisp("Error writing slope grid!", disp_mode);

			return((short int)ret_val);

		}

	} 



	calcArea(&grd, REMOVE, AREA, CA_AOVERB, TRUE, &sinks, disp_mode);

 

	if (area_grid != (char *)NULL)

	{

		doDisp("Writing area grid ...\n", disp_mode);

 

		if ((ret_val = grd.writeGrid(area_grid, AREA, grid_mode)) != OK)

		{

			doDisp("Error writing area grid!", disp_mode);

			return((short int)ret_val);

		}

	}



	calcQoverT(&grd, SLOPE, AREA, QOVERT, ATAN_RAD,

		(double)qt_ps, (double)qt_phi, disp_mode);

 

	doDisp("Writing q/T grid ...\n", disp_mode);

 

	if (qt_grid == (char *)NULL)

		grd.coutGrid(QOVERT, grid_mode);



	else if ((ret_val = grd.writeGrid(qt_grid, QOVERT, grid_mode)) != OK)

	{

		doDisp("Error writing q/T grid!", disp_mode);

		return((short int)ret_val);

	}



	return(disp_mode >= 0 ? (sinks ? OK : MSG_SINKS) : (sinks ? MSG_SINKS : OK));

} 

 

// Calculates q/T with cohesion and constant soil depth

void calcQoverTCF(Grid *grd, int slp_field, int area_field,

	int qt_field, int slope_mode, double qt_ps, double qt_phi, 

	double qt_depth, double qt_coh, int disp_mode)

{

	int i, j;

 

	doDisp("Computing q/T ...\n", disp_mode);

 

	for (j = 0; j < grd->ydim; j++)

		for (i = 0; i < grd->xdim; i++)

			grd->at(i, j, qt_field) =

				grd->getQoverTCF(i, j, slp_field, area_field, 

					slope_mode, qt_ps, qt_phi, qt_depth, qt_coh);

}

 

short int do_QoverTCF(char *slp_grid, char *area_grid, char *qt_grid,

	short int grid_mode, short int slope_mode, float qt_ps, float qt_phi, 

	float qt_depth, float qt_coh, short int disp_mode)

{

	Grid grd;

	int ret_val;

 

	doDisp("Reading slope grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(slp_grid, 0, grid_mode)) != OK)

	{

		doDisp("Error reading slope grid!", disp_mode);

		return((short int)ret_val);

	}



	if ((ret_val = grd.addField()) != OK || 

			(ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}



	doDisp("Reading area grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(area_grid, 1, grid_mode)) != OK)

	{

		doDisp("Error reading area grid!", disp_mode);

		return((short int)ret_val);

	}

 

	calcQoverTCF(&grd, 0, 1, 2, slope_mode, (double)qt_ps, 

		(double)qt_phi, (double)qt_depth, (double)qt_coh, disp_mode);

 

	doDisp("Writing q/T grid ...\n", disp_mode);

 

	if (qt_grid == (char *)NULL)

		grd.coutGrid(2, grid_mode);

 

	else if ((ret_val = grd.writeGrid(qt_grid, 2, grid_mode)) != OK)

	{

		doDisp("Error writing q/T grid!", disp_mode);

		return((short int)ret_val);

	}

 

	return(OK);

}





// Calculates all with cohesion and constant soil depth

short int do_ShalstabCF(char *elv_grid, char *rem_grid, char *slp_grid,

	char *area_grid, char *qt_grid, short int grid_mode, float soil_incr, 

	float qt_ps, float qt_phi, float qt_depth, float qt_coh,

	short int disp_mode)

{

	Grid grd;

	int fields = FIELDS, ret_val, sinks;

 

	doDisp("Reading elevation grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(elv_grid, ELEV, grid_mode)) != OK)

	{

		doDisp("Error reading elevation grid!", disp_mode);

		return((short int)ret_val);

	}



	while (fields--)

	{

		if ((ret_val = grd.addField()) != OK)

		{

			doDisp("Insufficient memory!\n", disp_mode);

			return((short int)ret_val);

		}

	}



	grd.setValue(REMOVE, ELEV);



	if (*rem_grid != NOREMSINK)
		
		calcRemoveSinks(&grd, REMOVE, (double)soil_incr, 

			RS_MAXPASSES, disp_mode);

 

	if (rem_grid != (char *)NULL && *rem_grid != NOREMSINK)

	{

		doDisp("Writing sink removed grid ...\n", disp_mode);

 

		if ((ret_val = grd.writeGrid(rem_grid, REMOVE, grid_mode)) != OK)

		{

			doDisp("Error writing sink removed grid!", disp_mode);

			return((short int)ret_val);

		}

	}



	calcSlope(&grd, REMOVE, SLOPE, ATAN_RAD, disp_mode);

 

	if (slp_grid != (char *)NULL)

	{

		doDisp("Writing slope grid ...\n", disp_mode);

 

		if ((ret_val = grd.writeGrid(slp_grid, SLOPE, grid_mode)) != OK)

		{

			doDisp("Error writing slope grid!", disp_mode);

			return((short int)ret_val);

		}

	} 



	calcArea(&grd, REMOVE, AREA, CA_AOVERB, TRUE, &sinks, disp_mode);

 

	if (area_grid != (char *)NULL)

	{

		doDisp("Writing area grid ...\n", disp_mode);

 

		if ((ret_val = grd.writeGrid(area_grid, AREA, grid_mode)) != OK)

		{

			doDisp("Error writing area grid!", disp_mode);

			return((short int)ret_val);

		}

	}



	calcQoverTCF(&grd, SLOPE, AREA, QOVERT, ATAN_RAD, (double)qt_ps, 

		(double)qt_phi, (double)qt_depth, (double)qt_coh, disp_mode);

 

	doDisp("Writing q/T grid ...\n", disp_mode);

 

	if (qt_grid == (char *)NULL)

		grd.coutGrid(QOVERT, grid_mode);



	else if ((ret_val = grd.writeGrid(qt_grid, QOVERT, grid_mode)) != OK)

	{

		doDisp("Error writing q/T grid!", disp_mode);

		return((short int)ret_val);

	}



	return(disp_mode >= 0 ? (sinks ? OK : MSG_SINKS) : (sinks ? MSG_SINKS : OK));

} 

/*
// Calculates all with constant soil depth, variable cohesion, phi and rho

void calcQoverTCRPF(Grid *grd, int slp_field, int area_field,

	int qt_field, int slope_mode, double qt_ps, double qt_phi, 

	double qt_depth, double qt_coh, int disp_mode)

{

	int i, j;

 

	doDisp("Computing q/T ...\n", disp_mode);

 

	for (j = 0; j < grd->ydim; j++)

		for (i = 0; i < grd->xdim; i++)

			grd->at(i, j, qt_field) =

				grd->getQoverTCF(i, j, slp_field, area_field, 

					slope_mode, qt_ps, qt_phi, qt_depth, qt_coh);

}


short int do_ShalstabCRPF(char *elv_grid, char *rho_grid, char *phi_grid, char *coh_grid, 

	char *slp_grid, char *area_grid, char *qt_grid, short int grid_mode, float soil_incr, 

	float qt_ps, float qt_phi, float qt_depth, float qt_coh,

	short int disp_mode)

{

	Grid grd;

	int fields = FIELDS, ret_val, sinks;

 

	doDisp("Reading elevation grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(elv_grid, ELEV, grid_mode)) != OK)

	{

		doDisp("Error reading elevation grid!", disp_mode);

		return((short int)ret_val);

	}



	while (fields--)

	{

		if ((ret_val = grd.addField()) != OK)

		{

			doDisp("Insufficient memory!\n", disp_mode);

			return((short int)ret_val);

		}

	}



	grd.setValue(REMOVE, ELEV);



	if (*rem_grid != NOREMSINK)
		
		calcRemoveSinks(&grd, REMOVE, (double)soil_incr, 

		RS_MAXPASSES, disp_mode);

 

	if (rem_grid != (char *)NULL && *rem_grid != NOREMSINK)

	{

		doDisp("Writing sink removed grid ...\n", disp_mode);

 

		if ((ret_val = grd.writeGrid(rem_grid, REMOVE, grid_mode)) != OK)

		{

			doDisp("Error writing sink removed grid!", disp_mode);

			return((short int)ret_val);

		}

	}



	calcSlope(&grd, REMOVE, SLOPE, ATAN_RAD, disp_mode);

 

	if (slp_grid != (char *)NULL)

	{

		doDisp("Writing slope grid ...\n", disp_mode);

 

		if ((ret_val = grd.writeGrid(slp_grid, SLOPE, grid_mode)) != OK)

		{

			doDisp("Error writing slope grid!", disp_mode);

			return((short int)ret_val);

		}

	} 



	calcArea(&grd, REMOVE, AREA, CA_AOVERB, TRUE, &sinks, disp_mode);

 

	if (area_grid != (char *)NULL)

	{

		doDisp("Writing area grid ...\n", disp_mode);

 

		if ((ret_val = grd.writeGrid(area_grid, AREA, grid_mode)) != OK)

		{

			doDisp("Error writing area grid!", disp_mode);

			return((short int)ret_val);

		}

	}



	calcQoverTCRPF(&grd, SLOPE, AREA, QOVERT, ATAN_RAD, (double)qt_ps, 

		(double)qt_phi, (double)qt_depth, (double)qt_coh, disp_mode);

 

	doDisp("Writing q/T grid ...\n", disp_mode);

 

	if (qt_grid == (char *)NULL)

		grd.coutGrid(QOVERT, grid_mode);



	else if ((ret_val = grd.writeGrid(qt_grid, QOVERT, grid_mode)) != OK)

	{

		doDisp("Error writing q/T grid!", disp_mode);

		return((short int)ret_val);

	}



	return(disp_mode >= 0 ? (sinks ? OK : MSG_SINKS) : (sinks ? MSG_SINKS : OK));

} 
*/


short int do_Rangrid(char *in_grid, short int hits, short int range,

	char *out_grid, short int grid_mode, short int disp_mode)

{

	Grid grd;

	int ret_val;

 

	doDisp("Reading input grid ...\n", disp_mode);



	// read input grid

	if ((ret_val = grd.readGrid(in_grid, 0, grid_mode)) != OK)

	{

		doDisp("Error reading input grid!", disp_mode);

		return((short int)ret_val);

	}

 

	// add the output layer

	if ((ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}

 

	// initialize output to nodata

	doDisp("Initializing output grid ...\n", disp_mode);

	grd.setValue(grd.nodata, 1);



	// generate random 'hits'

	doDisp("Randomizing output grid ...\n", disp_mode);

	grd.doRndGrd(hits, range, TRUE, 0, 1); 



	// write output grid

	doDisp("Writing output grid ...\n", disp_mode);

 

	// stdout

	if (out_grid == (char *)NULL)

		grd.coutGrid(1, grid_mode);

 

	// file

	else if ((ret_val = grd.writeGrid(out_grid, 1, grid_mode)) != OK)

	{

		doDisp("Error writing output grid!", disp_mode);

		return((short int)ret_val);

	}

 

	return(OK);

}



// Adds two grids

void calcAddGrid(Grid *grd, int in_field1, int in_field2, 

	int out_field, int disp_mode)

{

	int i, j;

 

	doDisp("Adding grids ...\n", disp_mode);

 

	for (j = 0; j < grd->ydim; j++)

		for (i = 0; i < grd->xdim; i++)

			if (grd->isValue(i, j, in_field1) && 

					grd->isValue(i, j, in_field2))

				grd->at(i, j, out_field) =

					grd->at(i, j, in_field1) + 

					grd->at(i, j, in_field2);

			else

				grd->at(i, j, out_field) = grd->nodata;

}

 

short int do_GridAdd(char *in_grid1, char *in_grid2, char *out_grid,

	short int grid_mode, short int disp_mode)

{

	Grid grd;

	int ret_val;

 

	doDisp("Reading first input grid...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(in_grid1, 0, grid_mode)) != OK)

	{

		doDisp("Error reading first input grid!", disp_mode);

		return((short int)ret_val);

	}



	if ((ret_val = grd.addField()) != OK || 

			(ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}



	doDisp("Reading second input grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(in_grid2, 1, grid_mode)) != OK)

	{

		doDisp("Error reading second input grid!", disp_mode);

		return((short int)ret_val);

	}

 

	calcAddGrid(&grd, 0, 1, 2, disp_mode);

 

	doDisp("Writing output grid ...\n", disp_mode);

 

	if (out_grid == (char *)NULL)

		grd.coutGrid(2, grid_mode);

 

	else if ((ret_val = grd.writeGrid(out_grid, 2, grid_mode)) != OK)

	{

		doDisp("Error writing output grid!", disp_mode);

		return((short int)ret_val);

	}

 

	return(OK);

}



// Subtracts two grids

void calcSubGrid(Grid *grd, int in_field1, int in_field2, 

	int out_field, int disp_mode)

{

	int i, j;

 

	doDisp("Subtracting grids ...\n", disp_mode);

 

	for (j = 0; j < grd->ydim; j++)

		for (i = 0; i < grd->xdim; i++)

			if (grd->isValue(i, j, in_field1) && 

				grd->isValue(i, j, in_field2))

				grd->at(i, j, out_field) =

					grd->at(i, j, in_field1) - 

					grd->at(i, j, in_field2);

			else

				grd->at(i, j, out_field) = grd->nodata;

}

 

short int do_GridSub(char *in_grid1, char *in_grid2, char *out_grid,

	short int grid_mode, short int disp_mode)

{

	Grid grd;

	int ret_val;

 

	doDisp("Reading first input grid...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(in_grid1, 0, grid_mode)) != OK)

	{

		doDisp("Error reading first input grid!", disp_mode);

		return((short int)ret_val);

	}



	if ((ret_val = grd.addField()) != OK || 

			(ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}



	doDisp("Reading second input grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(in_grid2, 1, grid_mode)) != OK)

	{

		doDisp("Error reading second input grid!", disp_mode);

		return((short int)ret_val);

	}

 

	calcSubGrid(&grd, 0, 1, 2, disp_mode);

 

	doDisp("Writing output grid ...\n", disp_mode);

 

	if (out_grid == (char *)NULL)

		grd.coutGrid(2, grid_mode);

 

	else if ((ret_val = grd.writeGrid(out_grid, 2, grid_mode)) != OK)

	{

		doDisp("Error writing output grid!", disp_mode);

		return((short int)ret_val);

	}

 

	return(OK);

}



// Multiplies two grids

void calcMulGrid(Grid *grd, int in_field1, int in_field2, 

	int out_field, int disp_mode)

{

	int i, j;

 

	doDisp("Multiplying grids ...\n", disp_mode);

 

	for (j = 0; j < grd->ydim; j++)

		for (i = 0; i < grd->xdim; i++)

			if (grd->isValue(i, j, in_field1) && 

				grd->isValue(i, j, in_field2))

				grd->at(i, j, out_field) =

					grd->at(i, j, in_field1) * 

					grd->at(i, j, in_field2);

			else

				grd->at(i, j, out_field) = grd->nodata;

}

 

short int do_GridMul(char *in_grid1, char *in_grid2, char *out_grid,

	short int grid_mode, short int disp_mode)

{

	Grid grd;

	int ret_val;

 

	doDisp("Reading first input grid...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(in_grid1, 0, grid_mode)) != OK)

	{

		doDisp("Error reading first input grid!", disp_mode);

		return((short int)ret_val);

	}



	if ((ret_val = grd.addField()) != OK || 

			(ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}



	doDisp("Reading second input grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(in_grid2, 1, grid_mode)) != OK)

	{

		doDisp("Error reading second input grid!", disp_mode);

		return((short int)ret_val);

	}

 

	calcMulGrid(&grd, 0, 1, 2, disp_mode);

 

	doDisp("Writing output grid ...\n", disp_mode);

 

	if (out_grid == (char *)NULL)

		grd.coutGrid(2, grid_mode);

 

	else if ((ret_val = grd.writeGrid(out_grid, 2, grid_mode)) != OK)

	{

		doDisp("Error writing output grid!", disp_mode);

		return((short int)ret_val);

	}

 

	return(OK);

}



// Divides two grids

void calcDivGrid(Grid *grd, int in_field1, int in_field2, 

	int out_field, int disp_mode)

{

	int i, j;

 

	doDisp("Dividing grids ...\n", disp_mode);

 

	for (j = 0; j < grd->ydim; j++)

		for (i = 0; i < grd->xdim; i++)

			if (grd->isValue(i, j, in_field1) && 

				grd->isValue(i, j, in_field2))

				grd->at(i, j, out_field) =

					grd->at(i, j, in_field1) / 

					grd->at(i, j, in_field2);

			else

				grd->at(i, j, out_field) = grd->nodata;

}

 

short int do_GridDiv(char *in_grid1, char *in_grid2, char *out_grid,

	short int grid_mode, short int disp_mode)

{

	Grid grd;

	int ret_val;

 

	doDisp("Reading first input grid...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(in_grid1, 0, grid_mode)) != OK)

	{

		doDisp("Error reading first input grid!", disp_mode);

		return((short int)ret_val);

	}



	if ((ret_val = grd.addField()) != OK || 

			(ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}



	doDisp("Reading second input grid ...\n", disp_mode);

 

	if ((ret_val = grd.readGrid(in_grid2, 1, grid_mode)) != OK)

	{

		doDisp("Error reading second input grid!", disp_mode);

		return((short int)ret_val);

	}

 

	calcDivGrid(&grd, 0, 1, 2, disp_mode);

 

	doDisp("Writing output grid ...\n", disp_mode);

 

	if (out_grid == (char *)NULL)

		grd.coutGrid(2, grid_mode);

 

	else if ((ret_val = grd.writeGrid(out_grid, 2, grid_mode)) != OK)

	{

		doDisp("Error writing output grid!", disp_mode);

		return((short int)ret_val);

	}

 

	return(OK);

}


// test reading and writing, can be used to convert formats

short int do_ReadWrite(char *in_grid, char *out_grid, 
	short int in_mode, short int out_mode, short int disp_mode)

{

	Grid grd;

	int ret_val;

 
	// check parameters
	if (in_mode < 0 || in_mode >= FORMATS || out_mode < 0 || out_mode >= FORMATS)
		return(ERR_BADTYPE);

	// read grid

	doDisp("Reading input grid ...\n", disp_mode);

	if ((ret_val = grd.readGrid(in_grid, 0, in_mode)) != OK)

	{

		doDisp("Error reading input grid!", disp_mode);

		return((short int)ret_val);

	}

/*
	// deal with nodata and flipping
	switch(in_mode)
	{
		case ARCINFO:
			break;
		case SURFER:
			break;
		case ARCVIEW:
			break;
	}
*/
	// write grid
	
	if (out_grid == (char *)NULL)

		grd.coutGrid(0, out_mode);

	else if ((ret_val = grd.writeGrid(out_grid, 0, out_mode)) != OK)

	{

		doDisp("Error writing output grid!", disp_mode);

		return((short int)ret_val);

	}

 

	return(OK);

}

short int do_Arc2Surf(char *in_grid, char *out_grid, short int disp_mode)
{

	Grid grd;

	int i, j, ret_val;

	double val, minz = 10.0e+100, maxz = -10.0e+100;


	doDisp("Reading input arcinfo grid ...\n", disp_mode);


	if ((ret_val = grd.readGrid(in_grid, 0, ARCINFO)) != OK)

	{

		doDisp("Error reading input grid!", disp_mode);

		return((short int)ret_val);

	}


	doDisp("Changing nodata value ...\n", disp_mode);


	grd.chgNodata(SURF_VAL, TRUE, 0);



	doDisp("Initializing output grid ...\n", disp_mode);


    if ((ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}


	doDisp("Converting grid ...\n", disp_mode);


	for (i = 0; i < grd.xdim; ++i)

	{

		for (j = 0; j < grd.ydim; ++j)

		{

			val = grd.at(i, j, 1) = 

				grd.at(i, grd.ydim - j - 1, 0);

			if (val != grd.nodata && val < minz)

				minz = val;

			if (val != grd.nodata && val > maxz)

				maxz = val;

		}

	}  


	doDisp("Writing output surfer grid ...\n", disp_mode);


	if ((ret_val = grd.writeGrid(out_grid, 1, SURFER, minz, maxz)) != OK)

	{

		doDisp("Error writing output grid!", disp_mode);

		return((short int)ret_val);

	}

    return(OK);
}


short int do_Surf2Arc(char *in_grid, char *out_grid, short int disp_mode)
{

	Grid grd;

	int i, j, ret_val;


	doDisp("Reading input surfer grid ...\n", disp_mode);


	if ((ret_val = grd.readGrid(in_grid, 0, SURFER)) != OK)

	{

		doDisp("Error reading input grid!", disp_mode);

		return((short int)ret_val);

	}


	doDisp("Changing nodata value ...\n", disp_mode);


	grd.chgNodata(DEF_VAL, TRUE, 0);



	doDisp("Initializing output grid ...\n", disp_mode);


    if ((ret_val = grd.addField()) != OK)

	{

		doDisp("Insufficient memory!\n", disp_mode);

		return((short int)ret_val);

	}


	doDisp("Converting grid ...\n", disp_mode);


	for (i = 0; i < grd.xdim; ++i)

	{

		for (j = 0; j < grd.ydim; ++j)

		{

			grd.at(i, j, 1) = 

				grd.at(i, grd.ydim - j - 1, 0);

		}

	}  


	doDisp("Writing output arcinfo grid ...\n", disp_mode);


	if ((ret_val = grd.writeGrid(out_grid, 1, ARCINFO)) != OK)

	{

		doDisp("Error writing output grid!", disp_mode);

		return((short int)ret_val);

	}

    return(OK);
}

// Calculates flow direction
void calcFlowDir(Grid *grd, int elev_field, int flow_field, int disp_mode)
{
        int i, j;
 
        doDisp("Computing flow direction ...\n", disp_mode);
 
        for (j = 0; j < grd->ydim; j++)
	{
                for (i = 0; i < grd->xdim; i++)
		{
                        if (grd->isValue(i, j, elev_field))
			{
				grd->at(i, j, flow_field) = 
					grd->getFlowDir(i, j, elev_field);
			}
			else
				grd->at(i, j, flow_field) = grd->nodata;
		}
	}
}



// Calculates distance to outlet on channels
void calcChDist2Out(Grid *grd, int flow_field, int dist_field, int disp_mode)
{
	// Notes:
	// (1) 	flow_field must reference an arcinfo type of flow
	//	direction grid calculated on channels only (or nodata)
	// (2)	dist_field must be initialized to nodata and will
	//	contain distances to the end of continuous flows

        int i, j;
 
        doDisp("Computing distance to outlet ...\n", disp_mode);
 
        for (j = 0; j < grd->ydim; j++)
	{
                for (i = 0; i < grd->xdim; i++)
		{
                        if (!grd->isValue(i, j, flow_field) ||
				grd->isValue(i, j, dist_field) ||
				!grd->isValue(grd->get_i(i, 
				(Direction)(int)grd->at(i, j, 
				flow_field)), grd->get_j(j, 
				(Direction)(int)grd->at(i, j, 
				flow_field)), flow_field))
				continue;
                        grd->at(i, j, dist_field) = 
				grd->chDist2Out(i, j, flow_field, dist_field);
		}
	}
}

int do_ChDist2Out(char *elv_grid, char *rem_grid, char *flow_grid,
	char *dist_grid, int in_grid_mode, int out_grid_mode, 
	double soil_incr, int rem_max_pass, int disp_mode)
{
        Grid grd;
	int fields = FIELDS2 + 1, ret_val;
 
        doDisp("Reading elevation grid ...\n", disp_mode);
 
        if ((ret_val = grd.readGrid(elv_grid, ELEV, in_grid_mode)) != OK)
        {
                doDisp("Error reading elevation grid!", disp_mode);
                return(ret_val);
        }

	while (fields--)
	{
        	if ((ret_val = grd.addField()) != OK)
        	{
               	 	doDisp("Insufficient memory!\n", disp_mode);
               	 	return(ret_val);
        	}
	}

	grd.setValue(REMOVE, ELEV);

	calcRemoveSinks2(&grd, REMOVE, soil_incr, rem_max_pass, disp_mode);
 
        if (rem_grid != (char *)NULL)
	{
        	doDisp("Writing sink removed grid ...\n", disp_mode);
 
        	if ((ret_val = grd.writeGrid(rem_grid, REMOVE, out_grid_mode)) != OK)
        	{
                	doDisp("Error writing sink removed grid!", disp_mode);
                	return(ret_val);
        	}
	}

        calcFlowDir(&grd, REMOVE, FLOW, disp_mode);
 
        if (flow_grid != (char *)NULL)
        {
        	doDisp("Writing flow direction grid ...\n", disp_mode);
 
        	if ((ret_val = grd.writeGrid(flow_grid, FLOW, out_grid_mode)) != OK)
        	{
                	doDisp("Error writing flow direction grid!", disp_mode);
                	return(ret_val);
        	}
	} 

	grd.setValue(grd.nodata, DIST);

        calcChDist2Out(&grd, FLOW, DIST, disp_mode);

        doDisp("Writing distance to outlet grid grid ...\n", disp_mode);
 
        if (dist_grid == (char *)NULL)
                grd.coutGrid(DIST, out_grid_mode);

        else if ((ret_val = grd.writeGrid(dist_grid, DIST, out_grid_mode)) != OK)
        {
                	doDisp("Error writing distance to outlet grid!", 
				disp_mode);
                return(ret_val);
        }
	return(OK);
}

// Removes Sinks for channels only
void calcRemoveSinks2(Grid *grd, int rem_field, double soil_incr,
	int rem_max_pass, int disp_mode)
{
        if (!grd->checkForSinks2(rem_field, disp_mode))
        	grd->removeSinks2(soil_incr, rem_max_pass, RS_NEIGHBORS,
			rem_field, disp_mode);
}

short int do_RemoveSinks2(char *elv_grid, char *rem_grid, int in_grid_mode,
	int out_grid_mode, double soil_incr, int rem_max_pass,
	int disp_mode)
{
	Grid grd;
	int ret_val;

	doDisp("Reading elevation grid ...\n", disp_mode);

        if ((ret_val = grd.readGrid(elv_grid, 0, in_grid_mode)) != OK)
	{
		doDisp("Error reading elevation grid!", disp_mode);
		return(ret_val);
	}

        if ((ret_val = grd.addField(0)) != OK)
        {
                doDisp("Insufficient memory!\n", disp_mode);
                return(ret_val);
        }
        
	calcRemoveSinks2(&grd, 1, soil_incr, rem_max_pass, disp_mode);
 
        doDisp("Writing elevation grid ...\n", disp_mode);

	if (rem_grid == (char *)NULL)
		grd.coutGrid(1, out_grid_mode);

        else if ((ret_val = grd.writeGrid(rem_grid, 1, out_grid_mode)) != OK)
        {
                doDisp("Error writing sink removed grid!", disp_mode);
                return(ret_val);
        }

        return(OK); 
}
 
