//  -*- Mode: C++;  -*-

//  File: area_class.cpp

//  Author: Dino Bellugi (dino@geomorph.berkeley.edu)

//  Copyright  Dino Bellugi, BlueG SoftWear, U.C. Berkeley, 1999 (C)

// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

// * FUNCTION: Computes the contributing area value for each

// *           cell. Contributing area is calculated using an

// *           8 neighbour slope (tan) weighted function.

// * 

// * RELATED PACKAGES: grid.cpp

// *

// * HISTORY:

// * Created:  Wed Jul 21 1993 (dino, based on rob reiss' version)

// * Modified: Mon May 10 1999 (dino)

// *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~



// the following include file (notice.h) contains important copyright 

// information, as well as other legal statements. it must be present 

// in all files and in any distribution. removal or modification is

// strictly forbidden and violates u.s. and international copyright law.

#include "notice.h"

//

#include <fstream.h>

#include "area_class.h"

#include "display.h"



#ifdef CA_SORT

// Globals for sorting:

int compElev(const void *n1, const void *n2);

Grid *pSGrd;

int iSFld;

#endif



/* 

 * find the contributing area given the elevation grid

 * initialize to 0.0 then make the gridedges then find

 * contributing area

 */

Contributing_Area::Contributing_Area(Grid *inz, int infield, int outfield,

	int mode, int check, int *sinks, int disp)

{

    doDisp("Initializing area grid ...\n", disp);



	pGrd = inz;

	pGrd->setValue(outfield, 0.0, infield);



#if (CA_SORT)

	int i, j, *p;

	pSInd = new int[xdim * ydim];

	for (j = 0, p = pSInd; j < ydim; ++j) 

		for (i = 0; i < xdim; ++i, ++p) 

			*p = j * xdim + i;

#endif



	if (check)

	{
		int n;

        doDisp("Checking for sinks ...\n", disp);

		n = pGrd->checkForSinks(infield, disp);

		if (sinks)
			*sinks = n;

	}



	pWeights = new GridEdges(pGrd->xdim, pGrd->ydim);



	makeWeights(infield, disp);



#if (CA_SORT)

	sortIndex(disp);

#endif



	initContributingArea(infield, outfield, mode, disp);

}



Contributing_Area::~Contributing_Area() 

{

	delete pWeights;

}



/*

 *  function initializes "weights". It sets the

 *  weight of each edge propotional to the slope between

 *  two neighboring cells. The edge weights are normalized

 *  so that the sum of all the weights leaving a cell = 1.0.

*/

void Contributing_Area::makeWeights(int field, int disp) 

{

	double ztot = 0.0, zdiff = 0.0, perc;

	int i, j, h, k;



        doDisp("Making weights ...\n", disp);



	for (j = 1; j < pGrd->ydim - 1; j++)

	{

		for (i = 1; i < pGrd->xdim - 1; i++) 

		{

			if (pGrd->at(i,j, field) == pGrd->nodata) 

			{

				for (k = 0; k < 8; k++) 

					pWeights->set(i, j, (Direction) k, 0.0);

			}

			else 

			{

				for (ztot = 0.0, k = 0; k < 8; k++) 

				{

					if (pGrd->get(i, j, (Direction) k, field) !=

						pGrd->nodata) 

					{

						zdiff = (pGrd->at(i,j, field) -

							pGrd->get(i ,j, (Direction) k, field))

							/ angleFactor((Direction) k);

						if (zdiff > 0.0) 

							ztot += zdiff;

					}

				}

				for (h = 0; h < 8; h++) 

				{

					if(pGrd->get(i,j,(Direction) h, field) !=

						pGrd->nodata) 

					{

						zdiff = (pGrd->at(i,j, field) -

							pGrd->get(i,j,(Direction) h, field))

							/ angleFactor((Direction) h);

						if (zdiff > 0.0) 

						{

							perc = zdiff / ztot;

							pWeights->set(i,j,

								(Direction) h,-perc);

						}

					}

				}

      			}

    		}

	}

}



/*

 * Recursively add up all the contributing cells area to the

 * cell i,j.

 */

double Contributing_Area::getContributingArea(int i, int j, int field) 

{

	double area = pGrd->at(i, j, field);

	if (area == 0.0) 

	{

		area = (pGrd->spacing * pGrd->spacing);

		if (pWeights->get(i,j,L) > 0.0) 

			area += (pWeights->get(i,j,L) * getContributingArea(i-1,j,field));

		if (pWeights->get(i,j,R) > 0.0) 

      			area += (pWeights->get(i,j,R) * getContributingArea(i+1,j,field));

		if (pWeights->get(i,j,U) > 0.0) 

			area += (pWeights->get(i,j,U) * getContributingArea(i,j+1,field));

		if (pWeights->get(i,j,D) > 0.0) 

			area += (pWeights->get(i,j,D) * getContributingArea(i,j-1,field));

		if (pWeights->get(i,j,UL) > 0.0) 

			area += (pWeights->get(i,j,UL) * getContributingArea(i-1,j+1,field));

		if (pWeights->get(i,j,UR) > 0.0) 

			area += (pWeights->get(i,j,UR) * getContributingArea(i+1,j+1,field));

		if (pWeights->get(i,j,DL) > 0.0) 

			area += (pWeights->get(i,j,DL) * getContributingArea(i-1,j-1,field));

		if (pWeights->get(i,j,DR) > 0.0) 

			area += (pWeights->get(i,j,DR) * getContributingArea(i+1,j-1,field));

		pGrd->at(i,j,field) = area;

	}

	return(area);

}



void Contributing_Area::initContributingArea(int infield, int outfield,

	int mode, int disp) 

{

	int i, j;



	doDisp("Computing contributing area ...\n", disp);



#if (CA_SORT)

	int n = xdim * ydim, *p = pSInd;



	while (n--)

	{

		i = getIndI(*p);

		j = getIndJ(*p);

		getContributingArea(i, j, outfield);

		++p;

	}

#else

	for(j = 1; j < pGrd->ydim - 1; j++) 

		for (i = 1; i < pGrd->xdim - 1; i++) 

			getContributingArea(i, j, outfield);

#endif



	for (j = 0; j < pGrd->ydim; j++) 

	{ 

		for (i = 0; i < pGrd->xdim; i++) 

		{

			if (j == 0 || j == pGrd->ydim - 1 ||

				i == 0 || i == pGrd->xdim - 1 ||

				!pGrd->isValue(i, j, infield) ||

				pGrd->at(i, j, outfield) == 0)

			{

				pGrd->at(i, j, outfield) = pGrd->nodata;

			}

			else if (mode == CA_AOVERB)

			{

				pGrd->at(i,j,outfield) /= pGrd->spacing;

			}

		}

	}

}



#if (CA_SORT)

/*

 *  function initializes "index". It sorts the grid in

 *  descending elevation order, storing the array's index

*/

void Contributing_Area::sortIndex(int field, int disp) 

{

	// sort descending mirror indexes

	doDisp("Sorting elevations ...\n", disp);

	pSGrd = pGrd;

	iSFld = field;

	qsort(index, xdim * ydim, sizeof(int), compElev);

}



int compElev(const void *ind1, const void *ind2)

{

	// array is sorted descending

	double z1, z2, diff;

	z1 = pSGrd->getArrVal(*((int *)ind1), iSFld);

	z2 = pSGrd->getArrVal(*((int *)ind2), iSFld);



	if (z1 == z2)

		return(0);

	if (z2 == pSGrd->nodata)

			return(-1);

	if (z1 == pSGrd->nodata)

			return(1);

	if (z2 > z1)

		return(1);

	return(-1);

}

#endif



