#include "DMMessageSender.h"

DMMessageSender::DMMessageSender(const string& bURI,
				 const string& dName,
				 bool uTopic,
				 bool sTransacted,
				 enum eewSystemName sysName) : connection(NULL),
							       session(NULL),
							       destination(NULL),
							       producer(NULL),
							       brokerURI(bURI),
							       destinationName(dName),
							       useTopic(uTopic),
							       sessionTransacted(sTransacted),
							       systemName(sysName) {}

void DMMessageSender::run()
{
    try {
	// Create a ConnectionFactory
	auto_ptr<ConnectionFactory> connectionFactory(ConnectionFactory::createCMSConnectionFactory( brokerURI ) );
	
	// Create a Connection
	connection = connectionFactory->createConnection();
	connection->start();
	
	// Create a Session
	if( sessionTransacted ) {
	    session = connection->createSession( Session::SESSION_TRANSACTED );
	} else {
	    session = connection->createSession( Session::AUTO_ACKNOWLEDGE );
	}
	
	// Create the destination (Topic or Queue)
	if( useTopic ) {
	    destination = session->createTopic( destinationName );
	} else {
	    destination = session->createQueue( destinationName );
	}
	
	// Create a MessageProducer from the Session to the Topic or Queue
	producer = session->createProducer( destination );
	
	producer->setDeliveryMode( DeliveryMode::NON_PERSISTENT );
    } catch ( CMSException& e ) {
	cout << "CMSException run" << endl;
	
	e.printStackTrace();
    }
}


DMMessageSender::~DMMessageSender()
{
    this->cleanup();
}

void DMMessageSender::close()
{
    this->cleanup();
}


void DMMessageSender::cleanup() 
{
    // Destroy resources.
    try{
	if( destination != NULL ) delete destination;
    }catch ( CMSException& e ) { e.printStackTrace(); }
    destination = NULL;
    
    try{
	if( producer != NULL ) delete producer;
    }catch ( CMSException& e ) { e.printStackTrace(); }
    producer = NULL;
    
    // Close open resources.
    try{
	if( session != NULL ) session->close();
	if( connection != NULL ) connection->close();
    }catch ( CMSException& e ) { e.printStackTrace(); }
    
    try{
	if( session != NULL ) delete session;
    }catch ( CMSException& e ) { e.printStackTrace(); }
    session = NULL;
    
    try{
	if( connection != NULL ) delete connection;
    }catch ( CMSException& e ) { e.printStackTrace(); }
    connection = NULL;
}

int DMMessageSender::sendString(string s) {
    // Create a messages
    TextMessage* message = session->createTextMessage( s );
    //message->setIntProperty( "Integer", ix );
    
    // Tell the producer to send the message
    //printf( "Sent message #%d from thread %s\n", ix+1, threadIdStr.c_str() );
    producer->send( message );
    
    delete message;
    return 0;
    
}

int DMMessageSender::sendMessage(ElarmsMessage &EM) 
{
    //makes sure sender is sending appropriate message type
    if (this->systemName != ELARMS) {
	return 1;
    }
	
    string t_orig_sys = "elarms";
    string t_message_type;
    
    if (EM.getVersion() > 0) {
	t_message_type = "update";
    } else {
	t_message_type = "new";
    }

    stringstream text;
    
    text << " <event_message orig_sys=\" " << t_orig_sys << "  \" message_type=\" " << t_message_type << "  \">\n";
    text << " \t<core_info id=\" " << EM.getID() << "  \">\n";
    text << "  \t\t<mag units=\" " << EM.getMagnitudeUnits() << " \"> " << EM.getMagnitude() << " </mag>\n";
    text << " \t\t<mag_err units=\" " << EM.getMagnitudeUncertaintyUnits() << " \"> " << EM.getMagnitudeUncertainty() << " </mag_err>\n";
    text << " \t\t<lat units=\" " << EM.getLatitudeUnits() << " \"> " << EM.getLatitude() << " </lat>\n";
    text << " \t\t<lat_err units=\" " << EM.getLatitudeUncertaintyUnits() << " \"> " << EM.getLatitudeUncertainty() << " </lat_err>\n";
    text << " \t\t<lon units=\" " << EM.getLongitudeUnits() << " \"> " << EM.getLongitude() << " </lon>\n";
    text << " \t\t<lon_err units=\" " << EM.getLongitudeUncertaintyUnits() << " \"> " << EM.getLongitudeUncertainty() << " </lon_err>\n";
    text << " \t\t<depth units=\" " << EM.getDepthUnits() << " \"> " << EM.getDepth() << " </depth>\n";
    text << " \t\t<depth_err units=\" " << EM.getDepthUncertaintyUnits() << " \"> " << EM.getDepthUncertainty() << " </depth_err>\n";
    text << " \t\t<orig_time units=\" " << EM.getOriginTimeUnits() << " \"> " << EM.getOriginTime() << " </orig_time>\n";
    text << " \t\t<orig_time_err units=\" " << EM.getOriginTimeUncertaintyUnits() << " \"> " << EM.getOriginTimeUncertainty() << " </orig_time_err>\n";
    text << " \t\t<likelyhood> " << EM.getLikelyhood() << " </likelyhood>\n";
    text << " \t</core_info>\n";
    
    
    
    text << " \t<elarms_info>\n";
    
    int num_vel_obs = EM.getNumberVelocityObservations();
    int num_acc_obs = EM.getNumberAccelerationObservations();
    
    if(num_vel_obs > 0) {
	PGVObsIter bv = EM.getPGVObservationsIteratorBegin();
	PGVObsIter ev = EM.getPGVObservationsIteratorEnd();
	text << " \t\t<pgv_obs number=\" " << num_vel_obs << " \">\n";
	for (bv; bv != ev; bv++) {
	    text << " \t\t\t<obs>\n";
	    text << " \t\t\t\t<SNCL> " << bv->getSNCL() << " </SNCL>\n";
	    text << " \t\t\t\t<value units=\" " << bv->getObservationUnits() << " \"> " << bv->getObservation() << " </obs>\n";
	    text << " \t\t\t\t<lat units=\" " << bv->getLatitudeUnits() << " \"> " << bv->getLatitude() << " </lat>\n";
	    text << " \t\t\t\t<lon units=\" " << bv->getLongitudeUnits() << " \"> " << bv->getLongitude() << " </lon>\n";
	    text << " \t\t\t\t<time units=\" " << bv->getObservationTimeUnits() << " \"> " << bv->getObservationTime() << " </time>\n";
	    text << " \t\t\t</obs>\n";
	}
	text << " \t\t</pgv_obs>\n";
    }
    
    if(num_acc_obs > 0) {
	PGAObsIter ba = EM.getPGAObservationsIteratorBegin();
	PGAObsIter ea = EM.getPGAObservationsIteratorEnd();
	text << " \t\t<pga_obs number=\" " << num_acc_obs << " \">\n";
	for (ba; ba != ea; ba++) {
	    text << " \t\t\t<obs>\n";
	    text << " \t\t\t\t<SNCL> " << ba->getSNCL() << " </SNCL>\n";
	    text << " \t\t\t\t<value units=\" " << ba->getObservationUnits() << " \"> " << ba->getObservation() << " </obs>\n";
	    text << " \t\t\t\t<lat units=\" " << ba->getLatitudeUnits() << " \"> " << ba->getLatitude() << " </lat>\n";
	    text << " \t\t\t\t<lon units=\" " << ba->getLongitudeUnits() << " \"> " << ba->getLongitude() << " </lon>\n";
	    text << " \t\t\t\t<time units=\" " << ba->getObservationTimeUnits() << " \"> " << ba->getObservationTime() << " </time>\n";
	    text << " \t\t\t</obs>\n";
	}
	text << " \t\t</pga_obs>\n";
    }
    
    
    text << " \t</elarms_info>\n";
    text << " </event_message>\n";
    
    this->sendString(text.str());
    EM.updateSent();
    return 0;
}

int DMMessageSender :: sendMessage(VSMessage &VSM) 
{
    if (this->systemName != VS) {
	return 1;
    }

    string t_orig_sys = "vs";
    string t_message_type;
    //string t_orig_time = VSM.getOriginTime();
    
    if (VSM.getVersion() != 0) {
	t_message_type = "update";
    } else {
	t_message_type = "new";
    }

    stringstream text;
    
    text << " <event_message orig_sys=\" " << t_orig_sys << " \" message_type=\" " << t_message_type << " \">\n";
    text << " \t<core_info id=\" " << VSM.getID() << " \">\n";
    text << " \t\t<mag units=\" " << VSM.getMagnitudeUnits() << " \"> " << VSM.getMagnitude() << " </mag>\n";
    text << " \t\t<mag_err units=\" " << VSM.getMagnitudeUncertaintyUnits() << " \"> " << VSM.getMagnitudeUncertainty() << " </mag_err>\n";
    text << " \t\t<lat units=\" " << VSM.getLatitudeUnits() << " \"> " << VSM.getLatitude() << " </lat>\n";
    text << " \t\t<lat_err units=\" " << VSM.getLatitudeUncertaintyUnits() << " \"> " << VSM.getLatitudeUncertainty() << " </lat_err>\n";
    text << " \t\t<lon units=\" " << VSM.getLongitudeUnits() << " \"> " << VSM.getLongitude() << " </lon>\n";
    text << " \t\t<lon_err units=\" " << VSM.getLongitudeUncertaintyUnits() << " \"> " << VSM.getLongitudeUncertainty() << " </lon_err>\n";
    text << " \t\t<depth units=\" " << VSM.getDepthUnits() << " \"> " << VSM.getDepth() << " </depth>\n";
    text << " \t\t<depth_err units=\" " << VSM.getDepthUncertaintyUnits() << " \"> " << VSM.getDepthUncertainty() << " </depth_err>\n";
    text << " \t\t<orig_time units=\" " << VSM.getOriginTimeUnits() << " \"> " << VSM.getOriginTime() << " </orig_time>\n";
    text << " \t\t<orig_time_err units=\" " << VSM.getOriginTimeUncertaintyUnits() << " \"> " << VSM.getOriginTimeUncertainty() << " </orig_time_err>\n";
    text << " \t\t<likelyhood> " << VSM.getLikelyhood() << " </likelyhood>\n";
    text << " \t</core_info>\n";
    
    
    text << " \t<vs_info>\n";
    
    int num_vel_obs = VSM.getNumberVelocityObservations();
    int num_acc_obs = VSM.getNumberAccelerationObservations();
    
    if(num_vel_obs > 0) {
	PGVObsIter bv = VSM.getPGVObservationsIteratorBegin();
	PGVObsIter ev = VSM.getPGVObservationsIteratorEnd();
	text << " \t\t<pgv_obs number=\" " << num_vel_obs << " \">\n";
	for (bv; bv != ev; bv++) {
	    text << " \t\t\t<obs>\n";
	    text << " \t\t\t\t<SNCL> " << bv->getSNCL() << " </SNCL>\n";
	    text << " \t\t\t\t\t<value units=\" " << bv->getObservationUnits() << " \"> " << bv->getObservation() << " </value>\n";
	    text << " \t\t\t\t<lat units=\" " << bv->getLatitudeUnits() << " \"> " << bv->getLatitude() << " </lat>\n";
	    text << " \t\t\t\t<lon units=\" " << bv->getLongitudeUnits() << " \"> " << bv->getLongitude() << " </lon>\n";
	    text << " \t\t\t\t<time units=\" " << bv->getObservationTimeUnits() << " \"> " << bv->getObservationTime() << " </time>\n";
	    text << " \t\t\t</obs>\n";
	}
	text << " \t\t</pgv_obs>\n";
    }
    
    if(num_acc_obs > 0) {
	PGAObsIter ba = VSM.getPGAObservationsIteratorBegin();
	PGAObsIter ea = VSM.getPGAObservationsIteratorEnd();
	text << " \t\t<pga_obs number=\" " << num_acc_obs << " \">\n";
	for (ba; ba != ea; ba++) {
	    text << " \t\t\t<obs>\n";
	    text << " \t\t\t\t<SNCL> " << ba->getSNCL() << " </SNCL>\n";
	    text << " \t\t\t\t<value units=\" " << ba->getObservationUnits() << " \"> " << ba->getObservation() << " </value>\n";
	    text << " \t\t\t\t<lat units=\" " << ba->getLatitudeUnits() << " \"> " << ba->getLatitude() << " </lat>\n";
	    text << " \t\t\t\t<lon units=\" " << ba->getLongitudeUnits() << " \"> " << ba->getLongitude() << " </lon>\n";
	    text << " \t\t\t\t<time units=\" " << ba->getObservationTimeUnits() << " \"> " << ba->getObservationTime() << " </time>\n";
	    text << " \t\t\t</obs>\n";
	}
	text << " \t\t</pga_obs>\n";
    }
    
    
    text << " \t</vs_info>\n";
    text << " </event_message>\n";

    this->sendString(text.str());
    VSM.updateSent();
    return 0;
    
}

int DMMessageSender :: sendMessage(OnSiteMessage &OSM) 
{
    if (this->systemName != ONSITE) {
	return 1;
    }
    string t_orig_sys = "onsite";
    string t_message_type;


    if (OSM.getVersion() != 0) {
	t_message_type = "update";
    } else {
	t_message_type = "new";
    }

    stringstream text;
    
    text << " <event_message orig_sys=\" " << t_orig_sys << " \" message_type=\" " << t_message_type << " \">\n";
    text << " \t<core_info id=\" " << OSM.getID() << " \">\n";
    text << " \t\t<mag units=\" " << OSM.getMagnitudeUnits() << " \"> " << OSM.getMagnitude() << " </mag>\n";
    text << " \t\t<mag_err units=\" " << OSM.getMagnitudeUncertaintyUnits() << " \"> " << OSM.getMagnitudeUncertainty() << " </mag_err>\n";
    text << " \t\t<lat units=\" " << OSM.getLatitudeUnits() << " \"> " << OSM.getLatitude() << " </lat>\n";
    text << " \t\t<lat_err units=\" " << OSM.getLatitudeUncertaintyUnits() << " \"> " << OSM.getLatitudeUncertainty() << " </lat_err>\n";
    text << " \t\t<lon units=\" " << OSM.getLongitudeUnits() << " \"> " << OSM.getLongitude() << " </lon>\n";
    text << " \t\t<lon_err units=\" " << OSM.getLongitudeUncertaintyUnits() << " \"> " << OSM.getLongitudeUncertainty() << " </lon_err>\n";
    text << " \t\t<depth units=\" " << OSM.getDepthUnits() << " \"> " << OSM.getDepth() << " </depth>\n";
    text << " \t\t<depth_err units=\" " << OSM.getDepthUncertaintyUnits() << " \"> " << OSM.getDepthUncertainty() << " </depth_err>\n";
    text << " \t\t<orig_time units=\" " << OSM.getOriginTimeUnits() << " \"> " << OSM.getOriginTime() << " </orig_time>\n";
    text << " \t\t<orig_time_err units=\" " << OSM.getOriginTimeUncertaintyUnits() << " \"> " << OSM.getOriginTimeUncertainty() << " </orig_time_err>\n";
    text << " \t\t<likelyhood> " << OSM.getLikelyhood() << " </likelyhood>\n";
    text << " \t</core_info>\n";
    
    
    text << " \t<onsite_info>\n";
    
    int num_vel_pred = OSM.getNumberVelocityPredictions();
    
    if(num_vel_pred > 0) {
	PGVPredIter bv = OSM.getPGVPredictionsIteratorBegin();
	PGVPredIter ev = OSM.getPGVPredictionsIteratorEnd();
	text << " \t\t<pgv_pred number=\" " << num_vel_pred << " \">\n";
	for (bv; bv != ev; bv++) {
	    text << " \t\t\t<pred>\n";
	    text << " \t\t\t<SNCL> " << bv->getSNCL() << " </SNCL>\n";
	    text << " \t\t\t<value units=\" " << bv->getPredictionUnits() << " \"> " << bv->getPrediction() << " </value>\n";
	    text << " \t\t\t<value_uncertainty units=\" " << bv->getPredictionUncertaintyUnits() << " \"> " << bv->getPredictionUncertainty() <<"</value_uncertainty>\n";
	    text << " \t\t\t<lat units=\" " << bv->getLatitudeUnits() << " \"> " << bv->getLatitude() << " </lat>\n";
	    text << " \t\t\t<lon units=\" " << bv->getLongitudeUnits() << " \"> " << bv->getLongitude() << " </lon>\n";
	    text << " \t\t\t</pred>\n";
	}
	text << " \t\t</pgv_pred>\n";
    }
    
    text << " \t</onsite_info>\n";
    text << " </event_message>\n";
    
    this->sendString(text.str());
    OSM.updateSent();
    return 0;
    
}

int DMMessageSender :: sendDeleteMessage(int EventID) 
{
    string t_orig_sys;
    string t_message_type = "delete";
    if (this->systemName == ELARMS) {
	t_orig_sys = "elarms";
    }
    else if (this->systemName == ONSITE) {
	t_orig_sys = "onsite";
    }
    else if (this->systemName == VS) {
	t_orig_sys = "vs";
    } else {
	return 1;
    }
    
    stringstream text;
    
    text << " <event_message orig_sys=\" " << t_orig_sys << " \" message_type=\" " << t_message_type << " \">\n";
    text << " \t<core_info id=\" " << EventID << " \">\n";
    text << " \t</core_info>\n";
    text << " </event_message>\n";
    
    this->sendString(text.str());
    return 0;
    
}
